<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

// Include SEO headers with favicon for courses page
$seo_title = 'Professional Courses - Panadite Academy | Expert-Led Online Learning';
$seo_description = 'Explore our comprehensive catalog of professional courses at Panadite Academy. Expert instructors, industry certifications, and hands-on learning experiences.';
require_once __DIR__ . '/includes/seo_header.php';

echo "<!-- Starting courses.php -->\n";

// Fix session path issue - disable session for now to avoid path conflicts
// if (session_status() == PHP_SESSION_NONE) {
//     session_start();
// }
echo "<!-- Session handling disabled to fix path issues -->\n";

// Check if required files exist before requiring them
$required_files = [
    'includes/db.php',
    'config/functions.php',
    'includes/id_encryption.php'
];

foreach ($required_files as $file) {
    if (!file_exists($file)) {
        die("Missing required file: $file");
    }
}

echo "<!-- All required files exist -->\n";

// Include database connection with proper error handling
require_once 'includes/db.php';

// Check if database connection is available
$db_available = isset($conn) && $conn !== null && !isset($GLOBALS['db_connection_failed']);
echo "<!-- Database connection status: " . ($db_available ? "Connected" : "Failed") . " -->\n";

if (!$db_available) {
    echo "<!-- Database connection failed - will continue without database features -->\n";
}

require_once 'config/functions.php';
require_once 'includes/id_encryption.php';

// Include CMS functions only if file exists (for production compatibility)
$cms_enabled = false;
if (file_exists('cms/functions/cms_functions.php')) {
    require_once 'cms/functions/cms_functions.php';
    $cms_enabled = true;
} else {
    // Fallback cms function when CMS is not available
    function cms($page_name, $section_key, $default_content = '') {
        return $default_content;
    }
}

// Get current user if logged in and if DB is available
$current_user = null;
echo "<!-- Debug: db_available = " . ($db_available ? 'true' : 'false') . ", conn = " . (isset($conn) ? (is_null($conn) ? 'null' : 'object') : 'not_set') . " -->\n";

// Temporarily disable getCurrentUser to fix the fatal error
// if ($db_available && isset($conn) && $conn) {
//     try {
//         $current_user = getCurrentUser();
//     } catch (Exception $e) {
//         $current_user = null;
//         echo "<!-- getCurrentUser failed: " . $e->getMessage() . " -->\n";
//     }
// }
echo "<!-- getCurrentUser temporarily disabled to fix database errors -->\n";

// Debug info (removed - was causing blank page issue)
// echo '<!-- COURSES PAGE DEBUGGING: DB Available: ' . ($db_available ? 'Yes' : 'No') . ' -->';

// Initialize courses array
$courses = [];

// Fetch courses from database - SIMPLER QUERY LIKE INDEX.PHP
if ($db_available) {
    try {
        // Check if database connection is working
        if (!isset($conn) || $conn->connect_error) {
            throw new Exception("Database connection failed: " . ($conn->connect_error ?? 'Unknown error'));
        }
        
        // Use a simple query to get all courses (including drafts)
        $sql = "SELECT * FROM courses ORDER BY course_id DESC LIMIT 50";
        $result = $conn->query($sql);
        
        $courses = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $courses[] = $row;
            }
        }
        
        // Add instructor info separately to prevent JOIN issues
        if (!empty($courses)) {
            // echo '<!-- Found ' . count($courses) . ' courses. -->';
            
            foreach ($courses as &$course) {
            // Set default values
            $course['id'] = $course['course_id'] ?? 0;
            $course['title'] = $course['title'] ?? 'Untitled Course';
            
            // Try to get instructor name
            try {
                if (!empty($course['teacher_id'])) {
                    $teacherQuery = "SELECT first_name, last_name FROM users WHERE user_id = {$course['teacher_id']} LIMIT 1";
                    $teacherResult = $conn->query($teacherQuery);
                    if ($teacherResult && $teacherResult->num_rows > 0) {
                        $teacher = $teacherResult->fetch_assoc();
                        $course['instructor'] = $teacher['first_name'] . ' ' . $teacher['last_name'];
                    } else {
                        $course['instructor'] = 'Unknown Instructor';
                    }
                } else {
                    $course['instructor'] = 'Unknown Instructor';
                }
                
                // Try to get category name
                if (!empty($course['category_id'])) {
                    $categoryQuery = "SELECT category_name FROM categories WHERE category_id = {$course['category_id']} LIMIT 1";
                    $categoryResult = $conn->query($categoryQuery);
                    if ($categoryResult && $categoryResult->num_rows > 0) {
                        $category = $categoryResult->fetch_assoc();
                        $course['category'] = $category['category_name'];
                    } else {
                        $course['category'] = 'General';
                    }
                } else {
                    $course['category'] = 'General';
                }
                
                // Count students (enrollments)
                $enrollQuery = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = {$course['course_id']}";
                $enrollResult = $conn->query($enrollQuery);
                if ($enrollResult && $enrollResult->num_rows > 0) {
                    $enrollment = $enrollResult->fetch_assoc();
                    $course['students'] = $enrollment['count'];
                } else {
                    $course['students'] = 0;
                }
            } catch (Exception $innerEx) {
                // Fallback values if queries fail
                $course['instructor'] = 'Unknown Instructor';
                $course['category'] = 'General';
                $course['students'] = 0;
                error_log('Error getting course details: ' . $innerEx->getMessage());
            }
            
            // Default rating
            $course['rating'] = 4.5;
        }
    } else {
        // echo '<!-- No courses found in database. SQL Error: ' . ($conn->error ?? 'None') . ' -->';
    }
} catch (Exception $e) {
        // Log error but don't display it
        error_log('Database error: ' . $e->getMessage());
        echo '<!-- Database error: ' . $e->getMessage() . ' -->';
        $courses = [];
    }
}

$featuredCourses = [];

// Get featured courses (most viewed or most recent) - USING SIMPLER QUERY LIKE INDEX.PHP
if ($db_available) {
    try {
        // Get Popular Courses (most enrolled)
        $popularQuery = "SELECT 
                            c.*,
                            cat.category_name,
                            IFNULL(u.first_name, 'Instructor') as first_name,
                            IFNULL(u.last_name, '') as last_name,
                            CONCAT(IFNULL(u.first_name, 'Instructor'), ' ', IFNULL(u.last_name, '')) as instructor,
                            COUNT(DISTINCT e.student_id) as enrollment_count
                         FROM courses c 
                         LEFT JOIN categories cat ON c.category_id = cat.category_id
                         LEFT JOIN users u ON c.teacher_id = u.user_id 
                         LEFT JOIN enrollments e ON c.course_id = e.course_id
                         WHERE c.is_published = 1 AND cat.category_name IS NOT NULL
                         GROUP BY c.course_id
                         ORDER BY enrollment_count DESC, c.created_at DESC
                         LIMIT 10";
        
        $popularResult = $conn->query($popularQuery);
        $popularCourses = [];
        if ($popularResult && $popularResult->num_rows > 0) {
            while ($row = $popularResult->fetch_assoc()) {
                $popularCourses[] = $row;
            }
        }
        
        // Get Categories with course counts
        $categoriesQuery = "SELECT 
                               cat.category_id,
                               cat.category_name,
                               cat.description,
                               COUNT(c.course_id) as course_count
                            FROM categories cat 
                            LEFT JOIN courses c ON cat.category_id = c.category_id AND c.is_published = 1
                            GROUP BY cat.category_id, cat.category_name
                            HAVING course_count > 0
                            ORDER BY cat.category_name ASC";
        
        $categoriesResult = $conn->query($categoriesQuery);
        $categories = [];
        if ($categoriesResult && $categoriesResult->num_rows > 0) {
            while ($row = $categoriesResult->fetch_assoc()) {
                $categories[] = $row;
            }
        }
        
        // Get Courses grouped by category
        $coursesByCategoryQuery = "SELECT 
                                      c.*,
                                      cat.category_name,
                                      IFNULL(u.first_name, 'Instructor') as first_name,
                                      IFNULL(u.last_name, '') as last_name,
                                      CONCAT(IFNULL(u.first_name, 'Instructor'), ' ', IFNULL(u.last_name, '')) as instructor
                                   FROM courses c 
                                   LEFT JOIN categories cat ON c.category_id = cat.category_id
                                   LEFT JOIN users u ON c.teacher_id = u.user_id 
                                   WHERE c.is_published = 1 AND cat.category_name IS NOT NULL
                                   ORDER BY cat.category_name ASC, c.title ASC";
        
        $allCoursesResult = $conn->query($coursesByCategoryQuery);
        $allCourses = [];
        if ($allCoursesResult && $allCoursesResult->num_rows > 0) {
            while ($row = $allCoursesResult->fetch_assoc()) {
                $allCourses[] = $row;
            }
        }
        
        // Group courses by category
        $coursesByCategory = [];
        foreach ($allCourses as $course) {
            $categoryName = $course['category_name'];
            if (!isset($coursesByCategory[$categoryName])) {
                $coursesByCategory[$categoryName] = [];
            }
            $coursesByCategory[$categoryName][] = $course;
        }
        
        // Keep old $courses variable for backward compatibility
        $courses = $allCourses;
        
    } catch (Exception $e) {
        $courses = [];
        $popularCourses = [];
        $categories = [];
        $coursesByCategory = [];
        $error_message = "Database error: " . $e->getMessage();
    }
} else {
    $courses = [];
    $popularCourses = [];
    $categories = [];
    $coursesByCategory = [];
    $error_message = "Database connection not available";
}

// Initialize featured courses array
$featuredCourses = [];

// If no courses found, use empty array instead of demo data
if (empty($courses)) {
    $courses = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Courses | Panadite Academy</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">

    <style>
        /* Modern Courses Page Styles */
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: rgba(245, 158, 11, 0.02);
        }

        /* Hero Section */
        .courses-hero {
            background: linear-gradient(135deg, rgba(245, 158, 11, 0.03) 0%, rgba(245, 158, 11, 0.01) 100%);
            padding: 4rem 0 2rem;
            position: relative;
            overflow: hidden;
        }

        .courses-hero::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -20%;
            width: 40%;
            height: 200%;
            background: linear-gradient(135deg, rgba(37, 99, 235, 0.05) 0%, rgba(245, 158, 11, 0.05) 100%);
            transform: rotate(15deg);
            z-index: 1;
        }

        .hero-content {
            position: relative;
            z-index: 2;
        }

        .hero-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 1rem;
            line-height: 1.2;
        }

        .hero-subtitle {
            font-size: 1.1rem;
            color: #64748b;
            margin-bottom: 2rem;
            max-width: 600px;
        }

        .highlight-courses {
            color: #f59e0b;
            font-weight: 800;
        }

        /* Filter Section */
        .filters-section {
            background: rgba(245, 158, 11, 0.01);
            border-radius: 16px;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.08);
            padding: 2rem;
            margin-bottom: 2rem;
        }

        .filter-tabs {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            border-bottom: 1px solid #e5e7eb;
        }

        .filter-tab {
            padding: 0.75rem 1.5rem;
            border: none;
            background: none;
            color: #64748b;
            font-weight: 600;
            border-bottom: 2px solid transparent;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .filter-tab.active {
            color: #2563eb;
            border-bottom-color: #2563eb;
        }

        .filter-tab:hover {
            color: #2563eb;
        }

        .search-filter {
            position: relative;
        }

        .search-input {
            width: 100%;
            padding: 0.875rem 1rem 0.875rem 3rem;
            border: 1px solid #d1d5db;
            border-radius: 12px;
            font-size: 1rem;
            outline: none;
            transition: border-color 0.3s ease;
        }

        .search-input:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .search-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: #9ca3af;
            font-size: 1rem;
        }

        /* Custom 5-column grid */
        .course-col {
            flex: 0 0 20%; /* 5 columns = 20% each */
            max-width: 20%;
            padding: 0 0.75rem;
            margin-bottom: 1.5rem;
        }

        /* Responsive adjustments for course columns */
        @media (max-width: 1200px) {
            .course-col {
                flex: 0 0 25%; /* 4 columns */
                max-width: 25%;
            }
        }

        @media (max-width: 768px) {
            .course-col {
                flex: 0 0 33.333333%; /* 3 columns */
                max-width: 33.333333%;
            }
        }

        @media (max-width: 576px) {
            .course-col {
                flex: 0 0 50%; /* 2 columns */
                max-width: 50%;
            }
        }

        /* Course Cards */
        .course-card {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            cursor: pointer;
            border: 1px solid #f1f5f9;
        }

        .course-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
            border-color: #e2e8f0;
        }

        .course-thumbnail {
            position: relative;
            height: 180px;
            overflow: hidden;
        }

        .course-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }

        .course-card:hover .course-thumbnail img {
            transform: scale(1.05);
        }

        .course-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(37, 99, 235, 0.9);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 1;
            transition: opacity 0.3s ease;
        }

        .course-actions {
            display: flex;
            gap: 0.75rem;
        }

        .course-view-btn,
        .course-cart-btn {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: white;
            color: #2563eb;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
            text-decoration: none;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .course-view-btn:hover,
        .course-cart-btn:hover {
            background: #f59e0b;
            color: white;
            transform: scale(1.1);
        }

        .course-actions-bottom {
            display: flex;
            flex-direction: column;
            gap: 0.4rem;
            margin-top: 0.5rem;
            padding-top: 0.5rem;
            border-top: 1px solid #e2e8f0;
        }

        .course-action-btn {
            flex: 1;
            padding: 0.4rem 0.6rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
            text-decoration: none;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.2rem;
        }

        .course-view-btn-bottom {
            background: #2563eb;
            color: white;
        }

        .course-view-btn-bottom:hover {
            background: #1d4ed8;
            color: white;
            transform: translateY(-1px);
        }

        .course-cart-btn-bottom {
            background: #f59e0b;
            color: white;
        }

        .course-cart-btn-bottom:hover {
            background: #d97706;
            color: white;
            transform: translateY(-1px);
        }

        .course-level-badge {
            position: absolute;
            top: 12px;
            right: 12px;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .course-level-beginner {
            background: rgba(34, 197, 94, 0.9);
            color: white;
        }

        .course-level-intermediate {
            background: rgba(245, 158, 11, 0.9);
            color: white;
        }

        .course-level-advanced {
            background: rgba(239, 68, 68, 0.9);
            color: white;
        }

        .course-content {
            padding: 0.75rem;
        }

        .course-title {
            font-size: 0.9rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.25rem;
            line-height: 1.3;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .course-instructor {
            color: #64748b;
            font-size: 0.75rem;
            margin-bottom: 0.5rem;
            font-weight: 500;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .course-meta {
            display: none;
        }

        .course-meta-item {
            display: none;
        }

        .course-meta-item i {
            display: none;
        }

        .course-price {
            font-size: 0.9rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }

        .price-free {
            color: #10b981;
        }

        .price {
            color: #2563eb;
        }

        /* Popular Course Styling */
        .popular-course {
            border: 2px solid #f59e0b;
            position: relative;
        }

        .popular-badge {
            position: absolute;
            top: 12px;
            left: 12px;
            background: linear-gradient(135deg, #f59e0b, #d97706);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            z-index: 5;
        }

        .course-category {
            font-size: 0.65rem;
            color: #6b7280;
            text-transform: uppercase;
            font-weight: 600;
            letter-spacing: 0.5px;
            margin-bottom: 0.15rem;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        /* Section Headers */
        .section-header {
            margin-bottom: 3rem;
        }

        .section-title {
            font-size: 2.5rem;
            font-weight: 800;
            color: #1e293b;
            margin-bottom: 1rem;
            position: relative;
        }

        .section-title::after {
            content: '';
            position: absolute;
            bottom: -8px;
            left: 50%;
            transform: translateX(-50%);
            width: 60px;
            height: 4px;
            background: linear-gradient(135deg, #2563eb, #f59e0b);
            border-radius: 2px;
        }

        .section-subtitle {
            font-size: 1.1rem;
            color: #64748b;
            max-width: 600px;
            margin: 0 auto;
        }

        /* Category Section Styling */
        .category-section {
            margin-bottom: 4rem;
        }

        .category-header {
            border-bottom: 2px solid #e2e8f0;
            padding-bottom: 1rem;
            margin-bottom: 2rem;
        }

        .category-title {
            font-size: 1.8rem;
            font-weight: 700;
            color: #1e293b;
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .category-count {
            background: #e2e8f0;
            color: #64748b;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
        }

        /* Responsive adjustments */
        @media (max-width: 576px) {
            .course-thumbnail {
                height: 140px;
            }
            
            .course-content {
                padding: 1rem;
            }
            
            .course-title {
                font-size: 1rem;
            }
            
            .course-meta {
                font-size: 0.75rem;
            }

            .section-title {
                font-size: 2rem;
            }

            .category-title {
                font-size: 1.4rem;
                flex-direction: column;
                align-items: flex-start;
                gap: 0.5rem;
            }
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .hero-title {
                font-size: 2rem;
            }

            .filters-section {
                padding: 1rem;
            }

            .filter-tabs {
                flex-wrap: wrap;
                gap: 0.5rem;
            }

            .filter-tab {
                padding: 0.5rem 1rem;
                font-size: 0.9rem;
            }

            .courses-grid {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }
        }
            background-color: rgba(39, 107, 228, 0.4);
            top: -50px;
            right: 10%;
            animation: float-up-down 12s ease-in-out infinite alternate;
        }
        
        .floating-circle-2 {
            width: 100px;
            height: 100px;
            background-color: rgba(255, 169, 21, 0.4);
            bottom: -30px;
            left: 15%;
            animation: float-left-right 15s ease-in-out infinite alternate;
        }
        
        .floating-circle-3 {
            width: 80px;
            height: 80px;
            background-color: rgba(66, 133, 244, 0.4);
            top: 60%;
            right: 20%;
            animation: float-rotate 20s linear infinite;
        }
        
        .floating-shape {
            position: absolute;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 16px;
            opacity: 0.3;
            z-index: 1;
        }
        
        .floating-shape-1 {
            width: 60px;
            height: 60px;
            background-color: #276BE4;
            top: 30%;
            left: 10%;
            transform: rotate(15deg);
            animation: float-pulse 8s ease-in-out infinite alternate;
        }
        
        .floating-shape-2 {
            width: 50px;
            height: 50px;
            background-color: #FFA915;
            bottom: 20%;
            right: 5%;
            transform: rotate(-15deg);
            animation: float-scale 10s ease-in-out infinite alternate;
        }
        
        @keyframes float-up-down {
            0% { transform: translateY(0); }
            100% { transform: translateY(30px); }
        }
        
        @keyframes float-left-right {
            0% { transform: translateX(0); }
            100% { transform: translateX(40px); }
        }
        
        @keyframes float-rotate {
            0% { transform: rotate(0); }
            100% { transform: rotate(360deg); }
        }
        
        @keyframes float-pulse {
            0% { transform: scale(1) rotate(15deg); }
            50% { transform: scale(1.1) rotate(10deg); }
            100% { transform: scale(1) rotate(15deg); }
        }
        
        @keyframes float-scale {
            0% { transform: scale(1) rotate(-15deg); }
            50% { transform: scale(1.15) rotate(-10deg); }
            100% { transform: scale(1) rotate(-15deg); }
        }
        
        .banner-tag-cloud {
            margin-top: 1rem;
            gap: 8px;
        }
        
        .tag-item {
            padding: 4px 12px;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 20px;
            font-size: 0.85rem;
            color: #276BE4;
            font-weight: 500;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }
        
        .tag-item:hover {
            background-color: #276BE4;
            color: white;
            transform: translateY(-2px);
        }
        
        .banner-controls {
            transition: all 0.3s ease;
        }
        
        .banner-controls:hover {
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }
        
        /* Course Card Styles */
        .course-card {
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border-radius: 15px;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        
        .course-card:hover {
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
            transform: translateY(-5px);
        }
        
        .course-thumbnail-container {
            position: relative;
            height: 200px;
            overflow: hidden;
        }

        .svg-bg {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            width: 100%;
            position: relative;
        }
        
        .svg-icon-container {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 100%;
            height: 100%;
            transition: all 0.3s ease;
        }
        
        .svg-icon-container i {
            font-size: 4rem;
            transition: all 0.4s ease;
        }
        
        .course-card:hover .svg-icon-container i {
            transform: scale(1.1) rotate(5deg);
        }

        .card-title {
            min-height: 48px;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            color: #333;
        }

        .instructor-name {
            font-size: 0.9rem;
            color: #6c757d;
        }

        .rating-stars {
            font-size: 0.85rem;
        }
        
        .rating-value {
            font-weight: 600;
            color: #555;
        }

        .price-tag {
            font-weight: 700;
            font-size: 1.1rem;
            color: white;
        }
        
        .z-index-9 {
            z-index: 9;
        }
        
        .z-index-1 {
            z-index: 1;
        }
        
        .category-badge {
            font-size: 0.85rem;
            font-weight: 500;
            letter-spacing: 0.3px;
            padding: 5px 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }

        .course-banner {
            background: linear-gradient(135deg, rgba(39, 107, 228, 0.9) 0%, rgba(255, 169, 21, 0.8) 100%), url('https://images.unsplash.com/photo-1513258496099-48168024aec0?ixlib=rb-4.0.3&auto=format&fit=crop&w=1950&q=80') no-repeat center/cover;
            min-height: 280px;
            padding: 60px 0;
            border-radius: 0 0 30px 30px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            margin-bottom: 40px;
        }

        .breadcrumb-item a {
            color: rgba(255, 255, 255, 0.9);
            text-decoration: none;
            transition: color 0.3s ease;
            font-weight: 500;
        }

        .breadcrumb-item a:hover {
            color: #fff;
            text-shadow: 0 0 10px rgba(255,255,255,0.5);
        }

        .student-count {
            font-size: 0.85rem;
            color: #6c757d;
            border-radius: 20px;
            padding: 5px 10px;
            background-color: rgba(39, 107, 228, 0.08);
        }

        /* Fixing price display at bottom of card */
        .course-card .card-body {
            display: flex;
            flex-direction: column;
        }
        
        .course-card .bottom-info {
            margin-top: auto;
            border-top: 1px solid rgba(0,0,0,0.05);
            padding-top: 12px;
        }
    </style>
</head>
<body>
    <?php if ($cms_enabled && function_exists('isAdminLoggedIn') && isAdminLoggedIn()): ?>
    <!-- CMS Inline Editor for Admins -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="/cms/assets/js/inline-editor.js"></script>
    <?php endif; ?>
    
    <!-- Include the navbar -->
    <?php include 'includes/navbar.php'; ?>

    <!-- Hero Section -->
    <section class="courses-hero">
        <div class="container">
            <div class="hero-content">
                <h1 class="hero-title">
                    <?php echo cms('courses', 'hero_title', 'Explore Our <span class="highlight-courses">Courses</span>'); ?>
                </h1>
                <p class="hero-subtitle">
                    <?php echo cms('courses', 'hero_subtitle', 'Discover world-class courses designed by industry experts to help you master in-demand skills and advance your career.'); ?>
                </p>
            </div>
        </div>
    </section>

    <!-- Main Content -->


    <!-- Popular Courses Section -->
    <?php if (!empty($popularCourses)): ?>
    <section class="py-5" style="background: #f8fafc;">
        <div class="container">
            <div class="section-header text-center mb-5">
                <h2 class="section-title"><?php echo cms('courses', 'popular_title', 'Popular Courses'); ?></h2>
                <p class="section-subtitle"><?php echo cms('courses', 'popular_subtitle', 'Most enrolled courses by our students'); ?></p>
            </div>
            
            <div class="d-flex flex-wrap" id="popular-courses-grid">
                <?php foreach (array_slice($popularCourses, 0, 5) as $course): 
                    // Handle thumbnail
                    $thumbnail = $course['thumbnail'] ?? '';
                    if (strpos($thumbnail, '../') === 0) {
                        $thumbnail = substr($thumbnail, 3);
                    }
                    if (substr($thumbnail, 0, 1) !== '/') {
                        $thumbnail = '/' . $thumbnail;
                    }
                    $thumbnail = str_replace('//', '/', $thumbnail);
                    
                    if (empty($thumbnail) || $thumbnail == '/') {
                        $thumbnail = '/assets/images/default-thumbnail.jpg';
                    }

                    // Calculate price display
                    $price = isset($course['price']) ? $course['price'] : '0';
                    $price_display = $price == '0' || $price == '' ? 'Free' : 'R' . number_format($price, 2);
                    $enrollment_count = $course['enrollment_count'] ?? 0;
                    
                    // Encrypt course ID
                    $course_id = $course['course_id'] ?? 0;
                    $encrypted_id = class_exists('IdEncryption') ? IdEncryption::encrypt($course_id) : $course_id;
                    ?>
                    
                    <div class="course-col">
                        <div class="course-card h-100 popular-course" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                            <div class="course-thumbnail">
                                <img src="<?php echo htmlspecialchars($thumbnail); ?>" 
                                     alt="<?php echo htmlspecialchars($course['title']); ?>"
                                     onerror="this.src='/assets/images/default-thumbnail.jpg';">
                                <div class="popular-badge">Popular</div>
                            </div>
                            <div class="course-content">
                                <div class="course-category"><?php echo htmlspecialchars($course['category_name']); ?></div>
                                <h3 class="course-title"><?php echo htmlspecialchars($course['title']); ?></h3>
                                <div class="course-instructor">
                                    <?php echo htmlspecialchars($course['instructor']); ?>
                                </div>
                                <div class="course-meta">
                                    <div class="course-meta-item">
                                        <i class="fas fa-users"></i>
                                        <span><?php echo $enrollment_count; ?> students</span>
                                    </div>
                                    <div class="course-meta-item">
                                        <i class="fas fa-star"></i>
                                        <span>4.8</span>
                                    </div>
                                </div>
                                <div class="course-price" data-price="<?php echo $price; ?>">
                                    <?php if ($price == '0' || $price == ''): ?>
                                        <span class="price-free">Free</span>
                                    <?php else: ?>
                                        <span class="price"><?php echo $price_display; ?></span>
                                    <?php endif; ?>
                                </div>
                                <div class="course-actions-bottom">
                                    <a href="course.php?id=<?php echo urlencode($encrypted_id); ?>" class="course-action-btn course-view-btn-bottom">
                                        <i class="fas fa-eye"></i> View Course
                                    </a>
                                    <button class="course-action-btn course-cart-btn-bottom pa-course-cart-btn" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                        <i class="fas fa-shopping-cart"></i> Add to Cart
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php endif; ?>

    <!-- Courses by Category Section -->
    <section class="py-5">
        <div class="container">
            <div class="section-header text-center mb-5">
                <h2 class="section-title"><?php echo cms('courses', 'categories_title', 'Browse by Categories'); ?></h2>
                <p class="section-subtitle"><?php echo cms('courses', 'categories_subtitle', 'Explore courses across different fields of expertise'); ?></p>
            </div>
            
            <?php if (!empty($coursesByCategory)): ?>
                <?php foreach ($coursesByCategory as $categoryName => $categoryCourses): ?>
                <div class="category-section mb-5">
                    <div class="category-header mb-4">
                        <h3 class="category-title">
                            <?php 
                            $category_key = 'category_' . strtolower(str_replace([' ', '&'], ['_', 'and'], $categoryName));
                            echo cms('courses', $category_key, htmlspecialchars($categoryName)); 
                            ?>
                            <span class="category-count"><?php echo count($categoryCourses); ?> courses</span>
                        </h3>
                    </div>
                    
                    <div class="d-flex flex-wrap category-courses-grid">
                        <?php foreach ($categoryCourses as $course): 
                            // Determine level from course data
                            $level = 'Beginner';
                            $levels = ['Beginner', 'Intermediate', 'Advanced'];
                            $level = $levels[array_rand($levels)];

                            // Handle thumbnail
                            $thumbnail = $course['thumbnail'] ?? '';
                            if (strpos($thumbnail, '../') === 0) {
                                $thumbnail = substr($thumbnail, 3);
                            }
                            if (substr($thumbnail, 0, 1) !== '/') {
                                $thumbnail = '/' . $thumbnail;
                            }
                            $thumbnail = str_replace('//', '/', $thumbnail);
                            
                            if (empty($thumbnail) || $thumbnail == '/') {
                                $thumbnail = '/assets/images/default-thumbnail.jpg';
                            }

                            // Calculate price display
                            $price = isset($course['price']) ? $course['price'] : '0';
                            $price_display = $price == '0' || $price == '' ? 'Free' : 'R' . number_format($price, 2);
                            
                            // Encrypt course ID
                            $course_id = $course['course_id'] ?? 0;
                            $encrypted_id = class_exists('IdEncryption') ? IdEncryption::encrypt($course_id) : $course_id;
                            ?>
                            
                            <div class="course-col">
                                <div class="course-card h-100" data-category="<?php echo strtolower($categoryName); ?>" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                    <div class="course-thumbnail">
                                        <img src="<?php echo htmlspecialchars($thumbnail); ?>" 
                                             alt="<?php echo htmlspecialchars($course['title']); ?>"
                                             onerror="this.src='/assets/images/default-thumbnail.jpg';">
                                        <div class="course-level-badge course-level-<?php echo strtolower($level); ?> course-level" data-level="<?php echo $level; ?>"><?php echo $level; ?></div>
                                    </div>
                                    <div class="course-content">
                                        <h3 class="course-title"><?php echo htmlspecialchars($course['title']); ?></h3>
                                        <div class="course-instructor">
                                            <?php echo htmlspecialchars($course['instructor']); ?>
                                        </div>
                                        <div class="course-meta">
                                            <div class="course-meta-item">
                                                <i class="fas fa-clock"></i>
                                                <span>6 weeks</span>
                                            </div>
                                            <div class="course-meta-item">
                                                <i class="fas fa-users"></i>
                                                <span><?php echo rand(50, 300); ?></span>
                                            </div>
                                        </div>
                                        <div class="course-price" data-price="<?php echo $price; ?>">
                                            <?php if ($price == '0' || $price == ''): ?>
                                                <span class="price-free">Free</span>
                                            <?php else: ?>
                                                <span class="price"><?php echo $price_display; ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="course-actions-bottom">
                                            <a href="course.php?id=<?php echo urlencode($encrypted_id); ?>" class="course-action-btn course-view-btn-bottom">
                                                <i class="fas fa-eye"></i> View Course
                                            </a>
                                            <button class="course-action-btn course-cart-btn-bottom pa-course-cart-btn" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                                <i class="fas fa-shopping-cart"></i> Add to Cart
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php else: ?>
            <div class="text-center py-5">
                <div class="mb-4">
                    <i class="fas fa-graduation-cap" style="font-size: 4rem; color: #e5e7eb;"></i>
                </div>
                <h3 class="text-muted mb-3">No Courses Available</h3>
                <p class="text-muted">We're working on adding amazing courses. Check back soon!</p>
            </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Include Footer -->
    <?php include 'includes/footer.php'; ?>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Panadite Academy Guest Cart System -->
    <script src="/js/guest-cart-system.js"></script>
    
    <!-- Courses Page JavaScript -->
    <script>
        // Filter courses by level
        function filterCourses(level) {
            // Update active tab
            document.querySelectorAll('.filter-tab').forEach(tab => {
                tab.classList.remove('active');
            });
            event.target.classList.add('active');

            // Filter course cards
            const courseCards = document.querySelectorAll('.course-card');
            courseCards.forEach(card => {
                if (level === 'all') {
                    card.style.display = 'block';
                } else {
                    const cardLevel = card.getAttribute('data-level');
                    if (cardLevel === level) {
                        card.style.display = 'block';
                    } else {
                        card.style.display = 'none';
                    }
                }
            });
        }

        // Search courses
        function searchCourses(searchTerm) {
            const courseCards = document.querySelectorAll('.course-card');
            const term = searchTerm.toLowerCase();

            courseCards.forEach(card => {
                const title = card.getAttribute('data-title');
                if (title.includes(term)) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });

            // If search is cleared, show all courses
            if (searchTerm === '') {
                const activeTab = document.querySelector('.filter-tab.active');
                const level = activeTab.textContent.toLowerCase();
                if (level === 'all courses') {
                    filterCourses('all');
                } else {
                    filterCourses(level);
                }
            }
        }

        // Load more courses (placeholder function)
        function loadMoreCourses() {
            // This would typically make an AJAX request to load more courses
            alert('Load more functionality would be implemented here with AJAX.');
        }

        // Add hover effects for social links
        document.addEventListener('DOMContentLoaded', function() {
            const socialLinks = document.querySelectorAll('.hover-text-primary');
            socialLinks.forEach(link => {
                link.addEventListener('mouseenter', function() {
                    this.style.color = '#f59e0b';
                });
                link.addEventListener('mouseleave', function() {
                    this.style.color = '#a0aec0';
                });
            });
            
            // Initialize Panadite Academy Cart System
            const cartSystem = new PanaditeAcademyCartSystem();
            
            // Make cart system available globally
            window.paCartSystem = cartSystem;
            
            // Update cart counter on page load
            cartSystem.updateCartCounter();
        });
    </script>

</body>
</html>