<?php
/**
 * Automatic Backup Cron Job
 * 
 * This file should be called by a system cron job at regular intervals
 * (e.g., every hour) to check if a backup needs to be performed.
 * 
 * Example cron setup:
 * 0 * * * * php /path/to/cron/run_backup.php > /dev/null 2>&1
 */

// Set script to run without time limit
set_time_limit(0);
// Disable output buffering
if (ob_get_level()) ob_end_clean();

// Include required files
$root = dirname(__DIR__);
require_once $root . '/auth/functions.php';
require_once $root . '/includes/classes/DatabaseBackup.php';
require_once $root . '/includes/classes/BackupScheduler.php';

// Log function
function log_message($message, $type = 'INFO') {
    $date = date('Y-m-d H:i:s');
    $formattedMessage = "[$date] [$type] $message";
    echo $formattedMessage . PHP_EOL;
    
    // Write to log file
    $logFile = dirname(__DIR__) . '/logs/backup_log.txt';
    $dir = dirname($logFile);
    
    // Create logs directory if it doesn't exist
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true)) {
            // Can't log to file if directory creation failed
            echo "[$date] [ERROR] Failed to create log directory at $dir" . PHP_EOL;
            return;
        }
    }
    
    // Check if log directory is writable
    if (!is_writable($dir)) {
        echo "[$date] [ERROR] Log directory is not writable: $dir" . PHP_EOL;
        return;
    }
    
    // Append to log file
    if (file_put_contents($logFile, $formattedMessage . PHP_EOL, FILE_APPEND) === false) {
        echo "[$date] [ERROR] Failed to write to log file: $logFile" . PHP_EOL;
    }
    
    // Rotate log if it's too large (over 10MB)
    if (file_exists($logFile) && filesize($logFile) > 10 * 1024 * 1024) {
        $archiveFile = $logFile . '.' . date('Y-m-d') . '.bak';
        rename($logFile, $archiveFile);
    }
}

// Initialize backup scheduler
try {
    log_message("Starting backup check...");
    
    // Verify database connection
    global $conn;
    if (!$conn || $conn->connect_error) {
        throw new Exception("Database connection failed: " . ($conn ? $conn->connect_error : 'Connection not established'));
    }
    
    // Check if backup directory exists and is writable
    $backupDir = dirname(__DIR__) . '/backups';
    if (!file_exists($backupDir)) {
        if (!mkdir($backupDir, 0755, true)) {
            throw new Exception("Failed to create backup directory at {$backupDir}");
        }
        log_message("Created backup directory: {$backupDir}");
    }
    
    if (!is_writable($backupDir)) {
        throw new Exception("Backup directory is not writable: {$backupDir}");
    }
    
    // Initialize scheduler
    $scheduler = new BackupScheduler();
    log_message("Backup scheduler initialized successfully.");
    
    // Get current settings
    $settings = $scheduler->getSettings();
    log_message("Loaded backup settings: Frequency={$settings['frequency']}, Format={$settings['format']}, Enabled=" . ($settings['enabled'] ? 'Yes' : 'No'));
    
    // Check if automatic backups are enabled
    if (!$settings['enabled']) {
        log_message("Automatic backups are disabled. Exiting.", 'NOTICE');
        exit(0);
    }
    
    // Check if it's time for a scheduled backup
    if ($scheduler->shouldRunBackup()) {
        log_message("It's time for a scheduled backup. Running backup process...", 'NOTICE');
        
        // Run the backup
        $startTime = microtime(true);
        $backupResult = $scheduler->runAutomaticBackup();
        $endTime = microtime(true);
        $executionTime = round($endTime - $startTime, 2);
        
        if ($backupResult) {
            // Get the most recent backup file info
            $history = $scheduler->getBackupHistory(1, 0);
            if (!empty($history)) {
                $backupFile = $history[0];
                log_message("Automatic backup completed successfully in {$executionTime} seconds. File: {$backupFile['filename']} ({$backupFile['size']})", 'SUCCESS');
            } else {
                log_message("Automatic backup completed successfully in {$executionTime} seconds, but backup record not found.", 'WARNING');
            }
        } else {
            log_message("Automatic backup failed after {$executionTime} seconds.", 'ERROR');
        }
    } else {
        log_message("No backup needed at this time.");
        
        // Show when next backup is scheduled
        if (!empty($settings['next_backup'])) {
            $nextBackup = new DateTime($settings['next_backup']);
            $now = new DateTime();
            $interval = $now->diff($nextBackup);
            $timeUntilNextBackup = '';
            
            if ($interval->days > 0) {
                $timeUntilNextBackup = "{$interval->days} days, ";
            }
            $timeUntilNextBackup .= "{$interval->h} hours, {$interval->i} minutes";
            
            log_message("Next backup scheduled for: {$settings['next_backup']} (in {$timeUntilNextBackup})");
        } else {
            log_message("Next backup time not set. Recalculating...", 'WARNING');
            $scheduler->calculateNextBackupTime();
            $scheduler->saveSettings();
            $updatedSettings = $scheduler->getSettings();
            log_message("Updated next backup time: {$updatedSettings['next_backup']}");
        }
    }
    
    // Clean up old backups
    log_message("Checking for old backups to clean up...");
    try {
        $scheduler->cleanupOldBackups();
        log_message("Cleanup process completed.");
    } catch (Exception $cleanupError) {
        log_message("Cleanup process error: " . $cleanupError->getMessage(), 'WARNING');
        // Continue execution - cleanup failure shouldn't stop the script
    }
    
    log_message("Backup check completed successfully.", 'SUCCESS');
} catch (Exception $e) {
    log_message("Critical error during backup process: " . $e->getMessage(), 'ERROR');
    
    // Log detailed error information
    log_message("Error details: " . $e->getTraceAsString(), 'ERROR');
    
    // Exit with error code
    exit(1);
}
