<?php
/**
 * Automatic Reminder Email System - EXACT SAME APPROACH AS WORKING ADMIN COURSE EMAILS
 * This script sends emails for scheduled study reminders when they're due
 * Run this as a cron job every hour or every 30 minutes
 */

require_once '../includes/session_start.php';
require_once '../config/db_connect.php';

// Include email service for automatic reminders - EXACT SAME AS ADMIN COURSES
require_once '../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but this uses MySQLi - EXACT SAME AS ADMIN COURSES
if (!isset($pdo)) {
    try {
        $pdo = new PDO("mysql:host=$host;dbname=$db_name", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } catch (Exception $e) {
        error_log("PDO connection failed: " . $e->getMessage());
        die();
    }
}

/**
 * Send automatic study reminder notification to student - EXACT SAME APPROACH AS ADMIN COURSE EMAILS
 */
function sendAutomaticReminderEmail($user_id, $reminder_data) {
    global $conn;
    
    try {
        // Get student details - EXACT SAME QUERY PATTERN AS WORKING ADMIN COURSE EMAILS
        $stmt = $conn->prepare("
            SELECT u.first_name, u.last_name, u.email
            FROM users u 
            WHERE u.user_id = ?
        ");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $student = $result->fetch_assoc();
        
        if (!$student) {
            error_log("Student not found for automatic reminder email: user_id=$user_id");
            return false;
        }
        
        $student_name = $student['first_name'] . ' ' . $student['last_name'];
        $student_email = $student['email'];
        
        // Parse reminder data
        $reminder_date = date('l, M j, Y', strtotime($reminder_data['reminder_date']));
        $reminder_time = date('g:i A', strtotime($reminder_data['reminder_time']));
        $reminder_type = ucwords(str_replace('_', ' ', $reminder_data['reminder_type']));
        $reminder_description = $reminder_data['description'];
        $course_name = $reminder_data['course_name'] ?? 'Your Course';
        
        // Create email content - EXACT SAME STRUCTURE AS WORKING ADMIN COURSE EMAILS
        $subject = "⏰ Study Reminder: Time for $reminder_type - $course_name";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#f39c12; color:white; padding:20px; text-align:center;'>
                <h1>⏰ Study Reminder!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hello " . htmlspecialchars($student_name) . "! 📚</h2>
                <p>This is your scheduled reminder for <strong>\"" . htmlspecialchars($course_name) . "\"</strong> - it's time to study!</p>
                <div style='background:#fff3cd; padding:15px; border-radius:5px; margin:20px 0; border-left: 4px solid #f39c12;'>
                    <h3>📋 Your Scheduled Study Session:</h3>
                    <p><strong>Type:</strong> " . htmlspecialchars($reminder_type) . "</p>
                    <p><strong>Date:</strong> " . htmlspecialchars($reminder_date) . "</p>
                    <p><strong>Time:</strong> " . htmlspecialchars($reminder_time) . "</p>
                    <p><strong>Description:</strong> " . htmlspecialchars($reminder_description) . "</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_name) . "</p>
                </div>
                <div style='background:#d1ecf1; padding:15px; border-radius:5px; margin:20px 0; border-left: 4px solid #bee5eb;'>
                    <h4>🚀 Ready to Continue Your Learning Journey?</h4>
                    <p>Don't let this study session slip by! Consistent learning is the key to mastering any skill.</p>
                    <p>Click the button below to access your course and make the most of your scheduled study time.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id=" . urlencode(base64_encode($reminder_data['course_id'] ?? '')) . "' 
                       style='background:#f39c12; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Start Studying Now
                    </a>
                </p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <p style='margin:0; color:#6c757d; font-size:14px; text-align:center;'>
                        <strong>💡 Study Tip:</strong> Set a specific goal for this session to stay focused and productive!
                    </p>
                </div>
            </div>
        </div>";
        
        // Try direct email sending - EXACT COPY FROM ADMIN COURSES.PHP (lines 82-87)
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT COPY FROM ADMIN COURSES.PHP (lines 89-94)
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        $mail_result = mail($student_email, $subject, $message, $headers);
        
        return $mail_result;
        
    } catch (Exception $e) {
        error_log("Error sending automatic study reminder email: " . $e->getMessage());
        return false;
    }
}

// Main automatic reminder processing
function processScheduledReminders() {
    global $conn;
    
    try {
        // Create reminder_emails table if it doesn't exist
        $create_table = "
            CREATE TABLE IF NOT EXISTS reminder_emails (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                course_id INT,
                reminder_type VARCHAR(100),
                reminder_date DATE,
                reminder_time TIME,
                description TEXT,
                course_name VARCHAR(255),
                email_sent TINYINT DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                sent_at TIMESTAMP NULL,
                INDEX idx_reminder_schedule (reminder_date, reminder_time, email_sent)
            )
        ";
        $conn->query($create_table);
        
        // Get current time and calculate reminder window (send reminders 15 minutes before scheduled time)
        $current_time = date('Y-m-d H:i:s');
        $reminder_window_start = date('Y-m-d H:i:s', strtotime('-15 minutes'));
        $reminder_window_end = date('Y-m-d H:i:s', strtotime('+15 minutes'));
        
        // Current time calculations for reminder window
        // Send reminders within 15 minutes of scheduled time
        
        // Find reminders that need to be sent (within 30-minute window around scheduled time)
        $stmt = $conn->prepare("
            SELECT * FROM reminder_emails 
            WHERE CONCAT(reminder_date, ' ', reminder_time) BETWEEN ? AND ?
            AND email_sent = 0
            ORDER BY reminder_date, reminder_time
        ");
        $stmt->bind_param("ss", $reminder_window_start, $reminder_window_end);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $sent_count = 0;
        
        while ($reminder = $result->fetch_assoc()) {
            // Send automatic reminder email
            $email_result = sendAutomaticReminderEmail($reminder['user_id'], $reminder);
            
            if ($email_result) {
                // Mark as sent
                $update_stmt = $conn->prepare("UPDATE reminder_emails SET email_sent = 1, sent_at = NOW() WHERE id = ?");
                $update_stmt->bind_param("i", $reminder['id']);
                $update_stmt->execute();
                
                $sent_count++;
                error_log("Automatic reminder email sent successfully to user {$reminder['user_id']} for reminder {$reminder['id']}");
            } else {
                error_log("Failed to send automatic reminder email to user {$reminder['user_id']} for reminder {$reminder['id']}");
            }
        }
        
        if ($sent_count > 0) {
            error_log("Automatic reminder system processed $sent_count reminders successfully");
        }
        
    } catch (Exception $e) {
        error_log("Error in automatic reminder processing: " . $e->getMessage());
    }
}

// Function to add reminder to database (called from course-detail.php)
function addReminderToDatabase($user_id, $reminder_data) {
    global $conn;
    
    try {
        // Create reminder_emails table if it doesn't exist
        $create_table = "
            CREATE TABLE IF NOT EXISTS reminder_emails (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                course_id INT,
                reminder_type VARCHAR(100),
                reminder_date DATE,
                reminder_time TIME,
                description TEXT,
                course_name VARCHAR(255),
                email_sent TINYINT DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                sent_at TIMESTAMP NULL,
                INDEX idx_reminder_schedule (reminder_date, reminder_time, email_sent)
            )
        ";
        $conn->query($create_table);
        
        // Insert reminder into database for automatic sending
        $stmt = $conn->prepare("
            INSERT INTO reminder_emails (user_id, course_id, reminder_type, reminder_date, reminder_time, description, course_name) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->bind_param("iisssss", 
            $user_id,
            $reminder_data['courseId'],
            $reminder_data['type'],
            $reminder_data['date'],
            $reminder_data['time'],
            $reminder_data['description'],
            $reminder_data['courseName']
        );
        
        return $stmt->execute();
        
    } catch (Exception $e) {
        error_log("Error adding reminder to database: " . $e->getMessage());
        return false;
    }
}

// QUICK TEST FEATURE - Create test reminder if requested
if (isset($_GET['create_test'])) {
    echo "\n🧪 CREATING TEST REMINDER\n";
    echo "========================\n";
    
    $test_time = date('H:i:s', strtotime('+1 minute'));
    $test_date = date('Y-m-d');
    
    echo "Creating test reminder for: $test_date $test_time (in 1 minute - should be in current window)\n";
    
    try {
        // Create table first
        $create_table = "
            CREATE TABLE IF NOT EXISTS reminder_emails (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                course_id INT,
                reminder_type VARCHAR(100),
                reminder_date DATE,
                reminder_time TIME,
                description TEXT,
                course_name VARCHAR(255),
                email_sent TINYINT DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                sent_at TIMESTAMP NULL,
                INDEX idx_reminder_schedule (reminder_date, reminder_time, email_sent)
            )
        ";
        $conn->query($create_table);
        
        // Insert test reminder
        $stmt = $conn->prepare("
            INSERT INTO reminder_emails (user_id, course_id, reminder_type, reminder_date, reminder_time, description, course_name) 
            VALUES (?, 1, 'test', ?, ?, 'Quick test reminder', 'Test Course')
        ");
        
        // Use user_id 1 for test, or get from session if available
        $test_user_id = $_SESSION['user_id'] ?? 1;
        $stmt->bind_param("iss", $test_user_id, $test_date, $test_time);
        
        $stmt->execute();
    } catch (Exception $e) {
        error_log("Error creating test reminder: " . $e->getMessage());
    }
}

// Check if we should process reminders (CLI or manual trigger)
$should_process = (php_sapi_name() === 'cli' || isset($_GET['cron']) || isset($_POST['cron']) || true);

if ($should_process) {
    try {
        processScheduledReminders();
    } catch (Exception $e) {
        error_log("Error during reminder processing: " . $e->getMessage());
    }
}
?>
