<?php
/**
 * Create Reviews System Tables
 * This script creates all necessary tables for the course review system
 */

// Include database configuration
require_once '../includes/config.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Function to execute SQL query with error handling
function executeQuery($conn, $sql, $description) {
    echo "<div style='margin: 10px 0; padding: 10px; border-left: 4px solid #2563eb; background: #f8fafc;'>";
    echo "<strong>$description</strong><br>";
    
    if ($conn->query($sql) === TRUE) {
        echo "<span style='color: #059669;'>✓ Success</span>";
    } else {
        echo "<span style='color: #dc2626;'>✗ Error: " . $conn->error . "</span>";
    }
    echo "</div>";
}

// Start HTML output
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Reviews System Tables</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; margin: 40px; background: #f9fafb; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        h1 { color: #1f2937; border-bottom: 2px solid #2563eb; padding-bottom: 10px; }
        .status { padding: 15px; margin: 20px 0; border-radius: 8px; }
        .success { background: #f0fdf4; border: 1px solid #bbf7d0; color: #15803d; }
        .error { background: #fef2f2; border: 1px solid #fecaca; color: #dc2626; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🌟 Create Reviews System Tables</h1>
        <p>This script will create all necessary database tables for the course review system.</p>

<?php

// Check database connection
if (!$conn) {
    echo "<div class='status error'><strong>❌ Database Connection Failed!</strong><br>";
    echo "Error: " . mysqli_connect_error() . "</div>";
    exit();
}

echo "<div class='status success'><strong>✅ Database Connected Successfully!</strong></div>";

// 1. Create reviews table
$reviews_table = "
CREATE TABLE IF NOT EXISTS course_reviews (
    review_id INT AUTO_INCREMENT PRIMARY KEY,
    course_id INT NOT NULL,
    student_id INT NOT NULL,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    review_title VARCHAR(255) NOT NULL,
    review_text TEXT NOT NULL,
    is_approved BOOLEAN DEFAULT TRUE,
    helpful_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_course_review (course_id, student_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

executeQuery($conn, $reviews_table, "Creating 'course_reviews' table");

// 2. Create review likes table (for helpful votes)
$review_likes_table = "
CREATE TABLE IF NOT EXISTS review_likes (
    like_id INT AUTO_INCREMENT PRIMARY KEY,
    review_id INT NOT NULL,
    user_id INT NOT NULL,
    is_helpful BOOLEAN NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (review_id) REFERENCES course_reviews(review_id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_review_like (review_id, user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

executeQuery($conn, $review_likes_table, "Creating 'review_likes' table");

// 3. Create review replies table (for instructor responses)
$review_replies_table = "
CREATE TABLE IF NOT EXISTS review_replies (
    reply_id INT AUTO_INCREMENT PRIMARY KEY,
    review_id INT NOT NULL,
    teacher_id INT NOT NULL,
    reply_text TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (review_id) REFERENCES course_reviews(review_id) ON DELETE CASCADE,
    FOREIGN KEY (teacher_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

executeQuery($conn, $review_replies_table, "Creating 'review_replies' table");

// 4. Create review statistics view for easy querying
$review_stats_view = "
CREATE OR REPLACE VIEW course_review_stats AS
SELECT 
    c.course_id,
    COUNT(cr.review_id) as total_reviews,
    ROUND(AVG(cr.rating), 1) as average_rating,
    COUNT(CASE WHEN cr.rating = 5 THEN 1 END) as five_star_count,
    COUNT(CASE WHEN cr.rating = 4 THEN 1 END) as four_star_count,
    COUNT(CASE WHEN cr.rating = 3 THEN 1 END) as three_star_count,
    COUNT(CASE WHEN cr.rating = 2 THEN 1 END) as two_star_count,
    COUNT(CASE WHEN cr.rating = 1 THEN 1 END) as one_star_count
FROM courses c
LEFT JOIN course_reviews cr ON c.course_id = cr.course_id AND cr.is_approved = TRUE
GROUP BY c.course_id";

executeQuery($conn, $review_stats_view, "Creating 'course_review_stats' view");

// 5. Insert sample review data for testing
$sample_reviews = "
INSERT IGNORE INTO course_reviews (course_id, student_id, rating, review_title, review_text) VALUES
(1, 1, 5, 'Excellent Course!', 'This course exceeded my expectations. The content is well-structured and the instructor explains concepts clearly. Highly recommended!'),
(1, 4, 4, 'Very Good Content', 'Great course with practical examples. The pace is perfect and the materials are comprehensive. Would love to see more advanced topics.'),
(1, 5, 5, 'Outstanding Learning Experience', 'Absolutely fantastic! The instructor is knowledgeable and the course content is up-to-date. Perfect for beginners and intermediates.'),
(2, 1, 4, 'Solid Foundation', 'Good course that covers the basics thoroughly. The assignments are helpful and the video quality is excellent.'),
(2, 4, 5, 'Highly Recommended!', 'One of the best courses I have taken. Clear explanations, good examples, and excellent support from the instructor.')";

executeQuery($conn, $sample_reviews, "Inserting sample review data");

// 6. Insert sample review replies
$sample_replies = "
INSERT IGNORE INTO review_replies (review_id, teacher_id, reply_text) VALUES
(1, 2, 'Thank you so much for your wonderful feedback! I am thrilled that you found the course helpful. Keep up the great work!'),
(2, 2, 'I appreciate your review and suggestions. I am planning to add more advanced topics in the next update. Stay tuned!'),
(4, 3, 'Thank you for taking the time to review the course. I am glad you found the foundation solid. Feel free to reach out if you have any questions!')";

executeQuery($conn, $sample_replies, "Inserting sample review replies");

// 7. Create indexes for better performance
$indexes = [
    "CREATE INDEX idx_course_reviews_course_id ON course_reviews(course_id)",
    "CREATE INDEX idx_course_reviews_student_id ON course_reviews(student_id)",
    "CREATE INDEX idx_course_reviews_rating ON course_reviews(rating)",
    "CREATE INDEX idx_course_reviews_created_at ON course_reviews(created_at)",
    "CREATE INDEX idx_review_likes_review_id ON review_likes(review_id)",
    "CREATE INDEX idx_review_replies_review_id ON review_replies(review_id)"
];

foreach ($indexes as $index) {
    executeQuery($conn, $index, "Creating database index");
}

// Close connection
$conn->close();

echo "<div class='status success'>";
echo "<h3>🎉 Reviews System Setup Complete!</h3>";
echo "<p><strong>Tables Created:</strong></p>";
echo "<ul>";
echo "<li>✅ course_reviews - Main reviews table</li>";
echo "<li>✅ review_likes - Review helpful votes</li>";
echo "<li>✅ review_replies - Instructor replies</li>";
echo "<li>✅ course_review_stats - Statistics view</li>";
echo "</ul>";
echo "<p><strong>Sample Data:</strong> Added sample reviews and replies for testing</p>";
echo "<p><strong>Next Step:</strong> The review system is now ready to be integrated into the course detail page!</p>";
echo "</div>";

?>

        <div style="margin-top: 30px; padding: 20px; background: #f0f9ff; border: 1px solid #7dd3fc; border-radius: 8px;">
            <h4 style="color: #0369a1; margin-top: 0;">📋 Database Schema Overview</h4>
            <p><strong>course_reviews table:</strong> Stores student reviews with ratings, titles, and text</p>
            <p><strong>review_likes table:</strong> Tracks helpful votes on reviews</p>
            <p><strong>review_replies table:</strong> Stores instructor responses to reviews</p>
            <p><strong>course_review_stats view:</strong> Provides aggregated review statistics for courses</p>
        </div>

        <div style="margin-top: 20px; padding: 15px; background: #fffbeb; border: 1px solid #fed7aa; border-radius: 8px;">
            <p><strong>⚠️ Important:</strong> Make sure to update your course-detail.php file to use the new review system!</p>
        </div>
    </div>
</body>
</html>
