<?php
/**
 * Debug Verification Token - Check what's stored in database
 */

require_once __DIR__ . '/../includes/db_connection.php';

echo "<h1>🔍 Verification Token Debug Tool</h1>";

if (!$conn) {
    die("❌ Database connection failed!");
}

// Get token from URL or show form
$token = $_GET['token'] ?? '';

if (empty($token)) {
    echo "<form method='GET'>";
    echo "<p>Enter verification token to debug:</p>";
    echo "<input type='text' name='token' style='width: 500px; padding: 8px;' placeholder='Paste verification token here...'>";
    echo "<br><br><button type='submit' style='padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px;'>Debug Token</button>";
    echo "</form>";
    
    // Show recent verification records
    echo "<h2>📋 Recent Verification Records</h2>";
    $result = $conn->query("SELECT verification_id, email, username, user_role, created_at, expires_at, additional_data FROM email_verifications ORDER BY created_at DESC LIMIT 10");
    
    if ($result && $result->num_rows > 0) {
        echo "<div style='overflow-x: auto; max-height: 400px; border: 1px solid #ddd; padding: 10px;'>";
        echo "<table style='width: 100%; border-collapse: collapse; font-size: 12px;'>";
        echo "<tr style='background: #f1f1f1;'>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>ID</th>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>Email</th>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>Username</th>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>Role</th>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>Created</th>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>Expires</th>";
        echo "<th style='border: 1px solid #ddd; padding: 5px;'>Has Data</th>";
        echo "</tr>";
        
        while ($row = $result->fetch_assoc()) {
            $hasData = !empty($row['additional_data']) ? '✅ Yes' : '❌ No';
            $isExpired = strtotime($row['expires_at']) < time() ? 'style="background: #ffcccc;"' : '';
            
            echo "<tr $isExpired>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['verification_id']) . "</td>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['email']) . "</td>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['username']) . "</td>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['user_role']) . "</td>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['created_at']) . "</td>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['expires_at']) . "</td>";
            echo "<td style='border: 1px solid #ddd; padding: 5px;'>$hasData</td>";
            echo "</tr>";
        }
        echo "</table>";
        echo "</div>";
        echo "<p><em>Red rows are expired. Copy a verification token from the email to debug it.</em></p>";
    } else {
        echo "<p>❌ No verification records found</p>";
    }
    
} else {
    echo "<p><strong>🔍 Debugging Token:</strong> " . htmlspecialchars($token) . "</p>";
    
    // Look up the verification record
    $stmt = $conn->prepare("SELECT * FROM email_verifications WHERE verification_token = ?");
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px; color: #721c24;'>";
        echo "<h3>❌ Token Not Found</h3>";
        echo "<p>This verification token does not exist in the database or has been deleted.</p>";
        echo "</div>";
        
        // Check if token exists but is expired
        $expiredStmt = $conn->prepare("SELECT * FROM email_verifications WHERE verification_token = ?");
        $expiredStmt->bind_param("s", $token);
        $expiredStmt->execute();
        $expiredResult = $expiredStmt->get_result();
        
        if ($expiredResult->num_rows > 0) {
            echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; color: #856404; margin-top: 10px;'>";
            echo "<h4>⚠️ Token Found But Expired</h4>";
            $expiredRecord = $expiredResult->fetch_assoc();
            echo "<p><strong>Email:</strong> " . htmlspecialchars($expiredRecord['email']) . "</p>";
            echo "<p><strong>Expired at:</strong> " . htmlspecialchars($expiredRecord['expires_at']) . "</p>";
            echo "</div>";
        }
        
    } else {
        $verification = $result->fetch_assoc();
        
        echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; color: #155724;'>";
        echo "<h3>✅ Verification Record Found</h3>";
        echo "</div>";
        
        echo "<h3>📊 Basic Information:</h3>";
        echo "<table style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><td style='border: 1px solid #ddd; padding: 8px; font-weight: bold; background: #f8f9fa;'>Field</td><td style='border: 1px solid #ddd; padding: 8px; font-weight: bold; background: #f8f9fa;'>Value</td></tr>";
        
        $basicFields = ['verification_id', 'email', 'username', 'first_name', 'last_name', 'user_role', 'created_at', 'expires_at'];
        foreach ($basicFields as $field) {
            $value = htmlspecialchars($verification[$field] ?? 'NULL');
            echo "<tr><td style='border: 1px solid #ddd; padding: 8px;'>$field</td><td style='border: 1px solid #ddd; padding: 8px;'>$value</td></tr>";
        }
        echo "</table>";
        
        echo "<h3>🔍 Additional Data Analysis:</h3>";
        $additionalData = $verification['additional_data'];
        
        if (empty($additionalData)) {
            echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px; color: #721c24;'>";
            echo "<h4>❌ PROBLEM FOUND: additional_data is NULL/Empty</h4>";
            echo "<p>This explains the 'No data supplied for parameters' error!</p>";
            echo "<p><strong>Cause:</strong> Student registration data was not stored properly during registration.</p>";
            echo "<p><strong>Solution:</strong> The student needs to register again with all the additional fields.</p>";
            echo "</div>";
        } else {
            echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; color: #155724;'>";
            echo "<h4>✅ Additional Data Present</h4>";
            echo "</div>";
            
            echo "<h4>📋 Raw JSON Data:</h4>";
            echo "<pre style='background: #f8f9fa; padding: 10px; border-radius: 5px; overflow-x: auto; font-size: 12px;'>";
            echo htmlspecialchars($additionalData);
            echo "</pre>";
            
            // Try to parse the JSON
            $parsedData = json_decode($additionalData, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px; color: #721c24;'>";
                echo "<h4>❌ JSON Parse Error</h4>";
                echo "<p><strong>Error:</strong> " . json_last_error_msg() . "</p>";
                echo "</div>";
            } else {
                echo "<h4>📊 Parsed Student Data:</h4>";
                echo "<table style='border-collapse: collapse; width: 100%;'>";
                echo "<tr><td style='border: 1px solid #ddd; padding: 8px; font-weight: bold; background: #f8f9fa;'>Field</td><td style='border: 1px solid #ddd; padding: 8px; font-weight: bold; background: #f8f9fa;'>Value</td></tr>";
                
                $studentFields = ['title', 'cell_number', 'id_number', 'suburb', 'province', 'communication_method', 'owns_laptop', 'training_preference', 'job_field_interest', 'job_field_other', 'education_status', 'employment_status', 'heard_about_us', 'referral_person_name'];
                
                $missingFields = [];
                foreach ($studentFields as $field) {
                    $value = $parsedData[$field] ?? '';
                    $displayValue = empty($value) ? '<em>empty</em>' : htmlspecialchars($value);
                    $rowStyle = empty($value) ? 'background: #ffcccc;' : '';
                    
                    if (empty($value)) {
                        $missingFields[] = $field;
                    }
                    
                    echo "<tr style='$rowStyle'><td style='border: 1px solid #ddd; padding: 8px;'>$field</td><td style='border: 1px solid #ddd; padding: 8px;'>$displayValue</td></tr>";
                }
                echo "</table>";
                
                if (empty($missingFields)) {
                    echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; color: #155724; margin-top: 10px;'>";
                    echo "<h4>✅ All Student Fields Present</h4>";
                    echo "<p>The verification should work. If it's still failing, the issue is with the SQL query or column names.</p>";
                    echo "</div>";
                } else {
                    echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; color: #856404; margin-top: 10px;'>";
                    echo "<h4>⚠️ Some Fields Are Empty</h4>";
                    echo "<p><strong>Missing/Empty:</strong> " . implode(', ', $missingFields) . "</p>";
                    echo "<p>This might cause issues if these fields are required.</p>";
                    echo "</div>";
                }
            }
        }
        
        echo "<h3>🔧 Test Verification:</h3>";
        echo "<p><a href='/verify-email.php?token=" . urlencode($token) . "' target='_blank' style='display: inline-block; padding: 10px 20px; background: #28a745; color: white; text-decoration: none; border-radius: 5px;'>🚀 Try Verification Now</a></p>";
    }
}

$conn->close();
?>

<style>
body { font-family: Arial, sans-serif; max-width: 1200px; margin: 0 auto; padding: 20px; line-height: 1.6; }
h1 { color: #007bff; }
h2 { color: #28a745; margin-top: 30px; }
h3 { color: #6f42c1; margin-top: 25px; }
h4 { color: #dc3545; margin-top: 20px; }
</style>
