<?php
/**
 * ROBUST Student Registration Fields Migration
 * This version uses basic ALTER TABLE statements that work on all MySQL versions
 */

// Include database connection
require_once __DIR__ . '/../includes/db_connection.php';

// Set time limit
set_time_limit(300);

echo "<h1>🔧 Robust Migration: Student Fields</h1>";
echo "<p>Using MySQL-compatible ALTER TABLE syntax...</p>";
flush();

if (!$conn) {
    die("❌ Database connection failed!");
}

// Function to check if column exists (works on older MySQL)
function columnExistsRobust($conn, $table, $column) {
    $result = $conn->query("SHOW COLUMNS FROM `$table` LIKE '$column'");
    return $result && $result->num_rows > 0;
}

echo "<h2>📊 Current Database Info:</h2>";
echo "<strong>MySQL Version:</strong> " . $conn->server_info . "<br>";
echo "<strong>Database Name:</strong> " . $conn->query("SELECT DATABASE()")->fetch_row()[0] . "<br><br>";

// List of columns to add with individual ALTER TABLE statements
$columns = [
    ['name' => 'title', 'sql' => "ALTER TABLE `users` ADD `title` VARCHAR(10) NULL AFTER `last_name`"],
    ['name' => 'cell_number', 'sql' => "ALTER TABLE `users` ADD `cell_number` VARCHAR(15) NULL AFTER `email`"],
    ['name' => 'id_number', 'sql' => "ALTER TABLE `users` ADD `id_number` VARCHAR(20) NULL AFTER `cell_number`"],
    ['name' => 'suburb', 'sql' => "ALTER TABLE `users` ADD `suburb` VARCHAR(100) NULL AFTER `id_number`"],
    ['name' => 'province', 'sql' => "ALTER TABLE `users` ADD `province` VARCHAR(50) NULL AFTER `suburb`"],
    ['name' => 'communication_method', 'sql' => "ALTER TABLE `users` ADD `communication_method` ENUM('WhatsApp', 'Email') NULL AFTER `province`"],
    ['name' => 'owns_laptop', 'sql' => "ALTER TABLE `users` ADD `owns_laptop` ENUM('Yes', 'No') NULL AFTER `communication_method`"],
    ['name' => 'training_preference', 'sql' => "ALTER TABLE `users` ADD `training_preference` ENUM('Onsite', 'Online') NULL AFTER `owns_laptop`"],
    ['name' => 'job_field_interest', 'sql' => "ALTER TABLE `users` ADD `job_field_interest` VARCHAR(100) NULL AFTER `training_preference`"],
    ['name' => 'job_field_other', 'sql' => "ALTER TABLE `users` ADD `job_field_other` VARCHAR(100) NULL AFTER `job_field_interest`"],
    ['name' => 'education_status', 'sql' => "ALTER TABLE `users` ADD `education_status` VARCHAR(100) NULL AFTER `job_field_other`"],
    ['name' => 'employment_status', 'sql' => "ALTER TABLE `users` ADD `employment_status` TEXT NULL AFTER `education_status`"],
    ['name' => 'heard_about_us', 'sql' => "ALTER TABLE `users` ADD `heard_about_us` VARCHAR(50) NULL AFTER `employment_status`"],
    ['name' => 'referral_person_name', 'sql' => "ALTER TABLE `users` ADD `referral_person_name` VARCHAR(100) NULL AFTER `heard_about_us`"]
];

$successCount = 0;
$skippedCount = 0;
$errorCount = 0;

echo "<div style='background: #f8f9fa; padding: 15px; border-left: 4px solid #007bff; margin: 10px 0;'>";
echo "<h3>Adding Columns:</h3>";

foreach ($columns as $column) {
    $columnName = $column['name'];
    $sql = $column['sql'];
    
    // Check if column already exists
    if (columnExistsRobust($conn, 'users', $columnName)) {
        echo "⚠️ <strong>$columnName:</strong> Already exists - skipped<br>";
        $skippedCount++;
        continue;
    }
    
    // Execute ALTER TABLE
    echo "⏳ <strong>$columnName:</strong> Adding... ";
    flush();
    
    if ($conn->query($sql) === TRUE) {
        echo "✅ Success<br>";
        $successCount++;
    } else {
        echo "❌ Error - " . $conn->error . "<br>";
        $errorCount++;
        // Log the exact error for debugging
        error_log("MIGRATION ERROR for $columnName: " . $conn->error . " | SQL: " . $sql);
    }
    
    flush();
}

echo "</div>";

// Summary
echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<h2>🎉 Migration Summary</h2>";
echo "<p><strong>✅ Added Successfully:</strong> $successCount</p>";
echo "<p><strong>⚠️ Already Existed:</strong> $skippedCount</p>";
echo "<p><strong>❌ Failed:</strong> $errorCount</p>";
echo "<p><strong>Total Processed:</strong> " . count($columns) . "</p>";

if ($errorCount > 0) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; border-radius: 3px; margin: 10px 0;'>";
    echo "<h4>⚠️ Errors Detected:</h4>";
    echo "<p>Check your server error logs or database permissions.</p>";
    echo "<p><strong>Possible issues:</strong></p>";
    echo "<ul>";
    echo "<li>Database user lacks ALTER TABLE permissions</li>";
    echo "<li>Table is locked or in use</li>";
    echo "<li>MySQL version compatibility issues</li>";
    echo "</ul>";
    echo "</div>";
}

echo "</div>";

// Show final table structure
echo "<h3>📋 Final Users Table Structure:</h3>";
$result = $conn->query("DESCRIBE users");
if ($result) {
    $totalColumns = $result->num_rows;
    echo "<p><strong>Total columns:</strong> $totalColumns</p>";
    
    echo "<div style='max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; background: white;'>";
    echo "<table style='width: 100%; border-collapse: collapse; font-size: 12px;'>";
    echo "<tr style='background: #f1f1f1; position: sticky; top: 0;'>";
    echo "<th style='border: 1px solid #ddd; padding: 5px; text-align: left;'>Field</th>";
    echo "<th style='border: 1px solid #ddd; padding: 5px; text-align: left;'>Type</th>";
    echo "<th style='border: 1px solid #ddd; padding: 5px; text-align: left;'>Null</th>";
    echo "</tr>";
    
    $studentFieldsFound = 0;
    while ($row = $result->fetch_assoc()) {
        $isStudentField = in_array($row['Field'], ['title', 'cell_number', 'id_number', 'suburb', 'province', 'communication_method', 'owns_laptop', 'training_preference', 'job_field_interest', 'job_field_other', 'education_status', 'employment_status', 'heard_about_us', 'referral_person_name']);
        
        if ($isStudentField) {
            $studentFieldsFound++;
        }
        
        $highlight = $isStudentField ? 'background-color: #fff3cd; font-weight: bold;' : '';
        echo "<tr style='$highlight'>";
        echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['Field']) . "</td>";
        echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['Type']) . "</td>";
        echo "<td style='border: 1px solid #ddd; padding: 5px;'>" . htmlspecialchars($row['Null']) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    
    echo "<p><strong>🎯 Student fields found:</strong> $studentFieldsFound / 14</p>";
    if ($studentFieldsFound >= 14) {
        echo "<p style='color: #28a745; font-weight: bold;'>✅ All student registration fields are now available!</p>";
    }
}

$conn->close();

echo "<div style='background: #d1ecf1; border: 1px solid #b8daff; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<h3>🚀 Next Steps</h3>";
if ($successCount + $skippedCount >= 14) {
    echo "<p>✅ <strong>Success!</strong> You can now test the student registration verification.</p>";
    echo "<p>🔗 <a href='/verify-email.php?token=test' style='color: #007bff;'>Try verification again</a></p>";
} else {
    echo "<p>⚠️ <strong>Some columns failed to add.</strong> Check database permissions or contact your hosting provider.</p>";
}
echo "</div>";
?>

<style>
body { font-family: Arial, sans-serif; max-width: 1000px; margin: 0 auto; padding: 20px; line-height: 1.6; }
h1 { color: #28a745; }
h2 { color: #007bff; margin-top: 30px; }
h3 { color: #6f42c1; margin-top: 25px; }
</style>
