<?php
/**
 * Database Structure Analysis Tool
 * This script analyzes the existing database structure to understand 
 * current tables and plan AI grading system integration
 */

require_once '../config/db_connect.php';

// Start output buffering for clean HTML
ob_start();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Structure Analysis - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .table-section { margin-bottom: 2rem; }
        .column-info { font-size: 0.9em; }
        .suggested-columns { background-color: #e8f5e8; padding: 1rem; border-radius: 0.5rem; margin-top: 1rem; }
        .analysis-card { border-left: 4px solid #007bff; }
        pre { background: #f8f9fa; padding: 1rem; border-radius: 0.5rem; }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <h1 class="mb-4">🔍 Database Structure Analysis</h1>
        <p class="lead">Analyzing existing tables for AI Grading System integration</p>

        <?php
        
        class DatabaseAnalyzer {
            private $conn;
            
            public function __construct($connection) {
                $this->conn = $connection;
            }
            
            public function getAllTables() {
                $query = "SHOW TABLES";
                $result = $this->conn->query($query);
                $tables = [];
                
                if ($result) {
                    while ($row = $result->fetch_array()) {
                        $tables[] = $row[0];
                    }
                }
                
                return $tables;
            }
            
            public function getTableStructure($tableName) {
                $query = "DESCRIBE `$tableName`";
                $result = $this->conn->query($query);
                $columns = [];
                
                if ($result) {
                    while ($row = $result->fetch_assoc()) {
                        $columns[] = $row;
                    }
                }
                
                return $columns;
            }
            
            public function getTableRowCount($tableName) {
                $query = "SELECT COUNT(*) as count FROM `$tableName`";
                $result = $this->conn->query($query);
                $row = $result->fetch_assoc();
                return $row['count'];
            }
            
            public function getSampleData($tableName, $limit = 3) {
                $query = "SELECT * FROM `$tableName` LIMIT $limit";
                $result = $this->conn->query($query);
                $data = [];
                
                if ($result) {
                    while ($row = $result->fetch_assoc()) {
                        $data[] = $row;
                    }
                }
                
                return $data;
            }
            
            public function analyzeTableForAI($tableName, $columns) {
                $suggestions = [];
                
                switch (strtolower($tableName)) {
                    case 'assignments':
                    case 'assignment':
                        $suggestions = [
                            'memorandum_content' => 'LONGTEXT - Store extracted memorandum text',
                            'memorandum_file_path' => 'VARCHAR(500) - Path to original memorandum file',
                            'memorandum_key_points' => 'LONGTEXT JSON - Structured key points from memorandum',
                            'ai_grading_enabled' => 'TINYINT(1) DEFAULT 1 - Enable/disable AI grading',
                            'pass_mark' => 'DECIMAL(5,2) DEFAULT 50.00 - Minimum passing grade',
                            'max_attempts' => 'INT DEFAULT 3 - Maximum submission attempts',
                            'auto_feedback' => 'TINYINT(1) DEFAULT 1 - Enable automatic feedback',
                            'plagiarism_check' => 'TINYINT(1) DEFAULT 1 - Enable plagiarism detection'
                        ];
                        break;
                        
                    case 'quizzes':
                    case 'quiz':
                        $suggestions = [
                            'answer_key' => 'LONGTEXT JSON - Store correct answers and explanations',
                            'ai_grading_enabled' => 'TINYINT(1) DEFAULT 1 - Enable AI grading for essay questions',
                            'essay_questions' => 'LONGTEXT JSON - Questions requiring AI grading',
                            'auto_grade_mcq' => 'TINYINT(1) DEFAULT 1 - Auto-grade multiple choice',
                            'feedback_enabled' => 'TINYINT(1) DEFAULT 1 - Provide instant feedback'
                        ];
                        break;
                        
                    case 'submissions':
                    case 'assignment_submissions':
                        $suggestions = [
                            'extracted_text' => 'LONGTEXT - Extracted text from submitted file',
                            'ai_score' => 'DECIMAL(5,2) - AI-generated score (0-100)',
                            'ai_feedback' => 'LONGTEXT - Detailed AI feedback',
                            'similarity_score' => 'DECIMAL(5,2) - Similarity to memorandum',
                            'plagiarism_score' => 'DECIMAL(5,2) - Plagiarism detection score',
                            'keyword_match_score' => 'DECIMAL(5,2) - Keyword matching score',
                            'structure_score' => 'DECIMAL(5,2) - Document structure score',
                            'quality_score' => 'DECIMAL(5,2) - Writing quality score',
                            'processing_status' => 'ENUM("pending","processing","completed","failed") DEFAULT "pending"',
                            'processing_time' => 'DECIMAL(8,2) - Processing time in seconds',
                            'review_needed' => 'TINYINT(1) DEFAULT 0 - Flag for manual review',
                            'ai_confidence' => 'DECIMAL(5,2) - AI confidence in grading (0-100)'
                        ];
                        break;
                        
                    case 'videos':
                    case 'course_videos':
                        $suggestions = [
                            'min_watch_percentage' => 'DECIMAL(5,2) DEFAULT 80.00 - Minimum watch % for completion',
                            'skip_prevention' => 'TINYINT(1) DEFAULT 1 - Prevent video skipping',
                            'transcript' => 'LONGTEXT - Video transcript for AI analysis',
                            'key_moments' => 'LONGTEXT JSON - Important timestamps and topics'
                        ];
                        break;
                        
                    case 'progress':
                    case 'student_progress':
                    case 'course_progress':
                        $suggestions = [
                            'video_progress_percent' => 'DECIMAL(5,2) DEFAULT 0.00 - Video completion %',
                            'assignment_progress_percent' => 'DECIMAL(5,2) DEFAULT 0.00 - Assignment completion %',
                            'quiz_progress_percent' => 'DECIMAL(5,2) DEFAULT 0.00 - Quiz completion %',
                            'overall_progress_percent' => 'DECIMAL(5,2) DEFAULT 0.00 - Weighted overall progress',
                            'video_weight' => 'DECIMAL(3,2) DEFAULT 0.30 - Weight for video progress',
                            'assignment_weight' => 'DECIMAL(3,2) DEFAULT 0.50 - Weight for assignments',
                            'quiz_weight' => 'DECIMAL(3,2) DEFAULT 0.20 - Weight for quizzes',
                            'completion_criteria_met' => 'TINYINT(1) DEFAULT 0 - All criteria satisfied',
                            'certificate_eligible' => 'TINYINT(1) DEFAULT 0 - Eligible for certificate',
                            'last_activity' => 'DATETIME - Last learning activity timestamp'
                        ];
                        break;
                        
                    case 'certificates':
                        $suggestions = [
                            'verification_code' => 'VARCHAR(50) UNIQUE - Unique verification code',
                            'qr_code_path' => 'VARCHAR(500) - Path to QR code image',
                            'pdf_path' => 'VARCHAR(500) - Path to generated certificate PDF',
                            'final_score' => 'DECIMAL(5,2) - Final weighted score',
                            'completion_date' => 'DATETIME - Course completion date',
                            'certificate_data' => 'LONGTEXT JSON - Certificate metadata'
                        ];
                        break;
                }
                
                return $suggestions;
            }
        }
        
        // Initialize analyzer
        $analyzer = new DatabaseAnalyzer($conn);
        $tables = $analyzer->getAllTables();
        
        // Filter relevant tables
        $relevantTables = array_filter($tables, function($table) {
            $keywords = ['assignment', 'quiz', 'video', 'progress', 'course', 'submission', 'certificate'];
            foreach ($keywords as $keyword) {
                if (stripos($table, $keyword) !== false) {
                    return true;
                }
            }
            return false;
        });
        
        echo "<div class='row'>";
        echo "<div class='col-md-6'>";
        echo "<div class='card analysis-card'>";
        echo "<div class='card-body'>";
        echo "<h5 class='card-title'>📊 Database Overview</h5>";
        echo "<p><strong>Total Tables:</strong> " . count($tables) . "</p>";
        echo "<p><strong>Relevant Tables:</strong> " . count($relevantTables) . "</p>";
        echo "<p><strong>Analysis Date:</strong> " . date('Y-m-d H:i:s') . "</p>";
        echo "</div></div></div>";
        
        echo "<div class='col-md-6'>";
        echo "<div class='card'>";
        echo "<div class='card-body'>";
        echo "<h5 class='card-title'>🎯 AI Integration Status</h5>";
        echo "<p class='text-success'>✅ Ready for column additions</p>";
        echo "<p class='text-info'>📝 No new tables required</p>";
        echo "<p class='text-warning'>⚠️ Backup recommended before changes</p>";
        echo "</div></div></div>";
        echo "</div>";
        
        echo "<h2 class='mt-4'>📋 Table Analysis</h2>";
        
        foreach ($relevantTables as $table) {
            echo "<div class='table-section'>";
            echo "<div class='card'>";
            echo "<div class='card-header'>";
            echo "<h4>📁 Table: <code>$table</code></h4>";
            
            $rowCount = $analyzer->getTableRowCount($table);
            echo "<small class='text-muted'>Rows: $rowCount</small>";
            echo "</div>";
            
            echo "<div class='card-body'>";
            
            // Current structure
            echo "<h5>Current Structure:</h5>";
            $columns = $analyzer->getTableStructure($table);
            
            echo "<div class='table-responsive'>";
            echo "<table class='table table-sm'>";
            echo "<thead><tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr></thead>";
            echo "<tbody>";
            
            foreach ($columns as $column) {
                echo "<tr>";
                echo "<td><code>" . htmlspecialchars($column['Field'] ?? '') . "</code></td>";
                echo "<td>" . htmlspecialchars($column['Type'] ?? '') . "</td>";
                echo "<td>" . htmlspecialchars($column['Null'] ?? '') . "</td>";
                echo "<td>" . htmlspecialchars($column['Key'] ?? '') . "</td>";
                echo "<td>" . htmlspecialchars($column['Default'] ?? 'NULL') . "</td>";
                echo "<td>" . htmlspecialchars($column['Extra'] ?? '') . "</td>";
                echo "</tr>";
            }
            
            echo "</tbody></table></div>";
            
            // Sample data
            if ($rowCount > 0) {
                echo "<h6 class='mt-3'>Sample Data:</h6>";
                $sampleData = $analyzer->getSampleData($table, 2);
                
                if (!empty($sampleData)) {
                    echo "<pre class='column-info'>";
                    foreach ($sampleData as $i => $row) {
                        echo "<strong>Row " . ($i + 1) . ":</strong>\n";
                        foreach ($row as $key => $value) {
                            $displayValue = is_null($value) ? 'NULL' : 
                                           (strlen($value) > 50 ? substr($value, 0, 47) . '...' : $value);
                            echo sprintf("  %-20s: %s\n", $key, htmlspecialchars($displayValue));
                        }
                        echo "\n";
                    }
                    echo "</pre>";
                }
            }
            
            // AI Integration suggestions
            $suggestions = $analyzer->analyzeTableForAI($table, $columns);
            
            if (!empty($suggestions)) {
                echo "<div class='suggested-columns'>";
                echo "<h6>🤖 Suggested AI Columns:</h6>";
                echo "<div class='row'>";
                
                foreach ($suggestions as $columnName => $description) {
                    echo "<div class='col-md-6 mb-2'>";
                    echo "<strong><code>$columnName</code></strong><br>";
                    echo "<small class='text-muted'>$description</small>";
                    echo "</div>";
                }
                
                echo "</div>";
                
                // Generate ALTER TABLE statement
                echo "<h6 class='mt-3'>📝 SQL to Add Columns:</h6>";
                echo "<pre><code>";
                echo "ALTER TABLE `$table`\n";
                
                $alterStatements = [];
                foreach ($suggestions as $columnName => $description) {
                    $columnDef = explode(' - ', $description)[0];
                    $alterStatements[] = "ADD COLUMN `$columnName` $columnDef";
                }
                
                echo implode(",\n", $alterStatements) . ";\n";
                echo "</code></pre>";
                echo "</div>";
            }
            
            echo "</div></div></div>";
        }
        
        // Generate complete upgrade script
        echo "<div class='mt-4'>";
        echo "<div class='card border-success'>";
        echo "<div class='card-header bg-success text-white'>";
        echo "<h4>🚀 Complete AI Grading Database Upgrade Script</h4>";
        echo "</div>";
        echo "<div class='card-body'>";
        
        echo "<p>Copy and run this SQL script to add AI grading capabilities to your existing database:</p>";
        echo "<pre><code>";
        echo "-- AI Grading System Database Upgrade\n";
        echo "-- Generated: " . date('Y-m-d H:i:s') . "\n\n";
        echo "-- BACKUP YOUR DATABASE BEFORE RUNNING THIS SCRIPT!\n\n";
        
        foreach ($relevantTables as $table) {
            $suggestions = $analyzer->analyzeTableForAI($table, []);
            if (!empty($suggestions)) {
                echo "-- Upgrading table: $table\n";
                echo "ALTER TABLE `$table`\n";
                
                $alterStatements = [];
                foreach ($suggestions as $columnName => $description) {
                    $columnDef = explode(' - ', $description)[0];
                    $alterStatements[] = "ADD COLUMN `$columnName` $columnDef";
                }
                
                echo implode(",\n", $alterStatements) . ";\n\n";
            }
        }
        
        echo "-- Create indexes for better performance\n";
        foreach ($relevantTables as $table) {
            if (stripos($table, 'submission') !== false) {
                echo "CREATE INDEX idx_{$table}_ai_score ON `$table` (ai_score);\n";
                echo "CREATE INDEX idx_{$table}_processing_status ON `$table` (processing_status);\n";
            }
            if (stripos($table, 'progress') !== false) {
                echo "CREATE INDEX idx_{$table}_completion ON `$table` (completion_criteria_met);\n";
                echo "CREATE INDEX idx_{$table}_certificate_eligible ON `$table` (certificate_eligible);\n";
            }
        }
        
        echo "\n-- Script completed successfully!\n";
        echo "</code></pre>";
        
        echo "<div class='alert alert-warning'>";
        echo "<strong>⚠️ Important:</strong> Always backup your database before running this script!";
        echo "</div>";
        
        echo "</div></div></div>";
        
        ?>
        
        <div class="mt-4">
            <div class="card">
                <div class="card-body text-center">
                    <h5>🎯 Next Steps</h5>
                    <p>1. Backup your database</p>
                    <p>2. Run the upgrade script above</p>
                    <p>3. Implement AI grading features in teacher assignment/quiz pages</p>
                    <p>4. Test with sample assignments</p>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

<?php
ob_end_flush();
?>
