<?php
// Debug script for messaging system
// This script will analyze the structure of relevant tables and show relationships
// between students, courses, facilitators, and messages

// Start session to get current user if logged in
session_start();

// Include database connection
require_once '../config/db_connect.php';

// Output function for cleaner display
function outputSection($title, $content) {
    echo "<div style='margin: 20px 0; padding: 10px; border: 1px solid #ddd; border-radius: 5px;'>";
    echo "<h3 style='margin-top: 0;'>$title</h3>";
    echo $content;
    echo "</div>";
}

// HTML header
echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Messaging System Debug</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; }
        table { border-collapse: collapse; width: 100%; margin-bottom: 15px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        .success { background-color: #d4edda; padding: 10px; border-radius: 4px; }
        .error { background-color: #f8d7da; padding: 10px; border-radius: 4px; }
        .warning { background-color: #fff3cd; padding: 10px; border-radius: 4px; }
        h2 { margin-top: 30px; border-bottom: 1px solid #eee; padding-bottom: 10px; }
    </style>
</head>
<body>
    <h1>Messaging System Debug</h1>
    <p>This script analyzes the database structure and relationships for the messaging system.</p>
";

// 1. Database Tables Structure
$tablesToCheck = ['users', 'teachers', 'students', 'courses', 'enrollments', 'messages'];
$tableStructure = "";

foreach ($tablesToCheck as $table) {
    $query = "DESCRIBE $table";
    $result = $conn->query($query);
    
    if ($result && $result->num_rows > 0) {
        $tableStructure .= "<h4>Table: $table</h4>
        <table>
            <tr>
                <th>Field</th>
                <th>Type</th>
                <th>Null</th>
                <th>Key</th>
                <th>Default</th>
                <th>Extra</th>
            </tr>";
        
        while ($row = $result->fetch_assoc()) {
            $tableStructure .= "<tr>";
            $tableStructure .= "<td>{$row['Field']}</td>";
            $tableStructure .= "<td>{$row['Type']}</td>";
            $tableStructure .= "<td>{$row['Null']}</td>";
            $tableStructure .= "<td>{$row['Key']}</td>";
            $tableStructure .= "<td>" . ($row['Default'] === NULL ? 'NULL' : $row['Default']) . "</td>";
            $tableStructure .= "<td>{$row['Extra']}</td>";
            $tableStructure .= "</tr>";
        }
        
        $tableStructure .= "</table>";
    } else {
        $tableStructure .= "<p class='warning'>Table '$table' not found or empty.</p>";
    }
}

outputSection("1. Database Table Structures", $tableStructure);

// 2. Students and Their Enrollments
$studentEnrollments = "";
$query = "SELECT 
            s.student_id, 
            u.user_id, 
            u.first_name, 
            u.last_name, 
            u.email,
            COUNT(e.enrollment_id) as enrollment_count
          FROM 
            students s
            JOIN users u ON s.user_id = u.user_id
            LEFT JOIN enrollments e ON s.student_id = e.student_id
          GROUP BY 
            s.student_id
          ORDER BY 
            enrollment_count DESC
          LIMIT 10";

$result = $conn->query($query);

if ($result && $result->num_rows > 0) {
    $studentEnrollments .= "<table>
        <tr>
            <th>Student ID</th>
            <th>User ID</th>
            <th>Name</th>
            <th>Email</th>
            <th>Enrolled Courses</th>
        </tr>";
    
    while ($row = $result->fetch_assoc()) {
        $studentEnrollments .= "<tr>";
        $studentEnrollments .= "<td>{$row['student_id']}</td>";
        $studentEnrollments .= "<td>{$row['user_id']}</td>";
        $studentEnrollments .= "<td>{$row['first_name']} {$row['last_name']}</td>";
        $studentEnrollments .= "<td>{$row['email']}</td>";
        $studentEnrollments .= "<td>{$row['enrollment_count']}</td>";
        $studentEnrollments .= "</tr>";
    }
    
    $studentEnrollments .= "</table>";
    
    // Get detailed enrollments for first student
    if ($result->data_seek(0)) {
        $firstStudent = $result->fetch_assoc();
        $studentId = $firstStudent['student_id'];
        
        $query = "SELECT 
                    e.enrollment_id,
                    c.course_id,
                    c.title as course_title,
                    t.teacher_id,
                    u.first_name as teacher_first_name,
                    u.last_name as teacher_last_name,
                    e.enrolled_date,
                    e.status
                FROM 
                    enrollments e
                    JOIN courses c ON e.course_id = c.course_id
                    JOIN teachers t ON c.teacher_id = t.teacher_id
                    JOIN users u ON t.user_id = u.user_id
                WHERE 
                    e.student_id = $studentId";
                    
        $detailResult = $conn->query($query);
        
        if ($detailResult && $detailResult->num_rows > 0) {
            $studentEnrollments .= "<h4>Detailed enrollments for {$firstStudent['first_name']} {$firstStudent['last_name']}</h4>";
            $studentEnrollments .= "<table>
                <tr>
                    <th>Enrollment ID</th>
                    <th>Course</th>
                    <th>Teacher</th>
                    <th>Enrollment Date</th>
                    <th>Status</th>
                </tr>";
            
            while ($row = $detailResult->fetch_assoc()) {
                $studentEnrollments .= "<tr>";
                $studentEnrollments .= "<td>{$row['enrollment_id']}</td>";
                $studentEnrollments .= "<td>{$row['course_title']} (ID: {$row['course_id']})</td>";
                $studentEnrollments .= "<td>{$row['teacher_first_name']} {$row['teacher_last_name']} (ID: {$row['teacher_id']})</td>";
                $studentEnrollments .= "<td>{$row['enrollment_date']}</td>";
                $studentEnrollments .= "<td>{$row['status']}</td>";
                $studentEnrollments .= "</tr>";
            }
            
            $studentEnrollments .= "</table>";
        } else {
            $studentEnrollments .= "<p class='warning'>No detailed enrollments found for this student.</p>";
        }
    }
} else {
    $studentEnrollments .= "<p class='warning'>No students with enrollments found.</p>";
}

outputSection("2. Students and Their Enrollments", $studentEnrollments);

// 3. Teachers/Facilitators
$teachersInfo = "";
$query = "SELECT 
            t.teacher_id, 
            u.user_id, 
            u.first_name, 
            u.last_name, 
            u.email,
            COUNT(c.course_id) as course_count
          FROM 
            teachers t
            JOIN users u ON t.user_id = u.user_id
            LEFT JOIN courses c ON t.teacher_id = c.teacher_id
          GROUP BY 
            t.teacher_id
          ORDER BY 
            course_count DESC";

$result = $conn->query($query);

if ($result && $result->num_rows > 0) {
    $teachersInfo .= "<table>
        <tr>
            <th>Teacher ID</th>
            <th>User ID</th>
            <th>Name</th>
            <th>Email</th>
            <th>Courses Teaching</th>
        </tr>";
    
    while ($row = $result->fetch_assoc()) {
        $teachersInfo .= "<tr>";
        $teachersInfo .= "<td>{$row['teacher_id']}</td>";
        $teachersInfo .= "<td>{$row['user_id']}</td>";
        $teachersInfo .= "<td>{$row['first_name']} {$row['last_name']}</td>";
        $teachersInfo .= "<td>{$row['email']}</td>";
        $teachersInfo .= "<td>{$row['course_count']}</td>";
        $teachersInfo .= "</tr>";
    }
    
    $teachersInfo .= "</table>";
    
    // Get detailed courses for first teacher
    if ($result->data_seek(0)) {
        $firstTeacher = $result->fetch_assoc();
        $teacherId = $firstTeacher['teacher_id'];
        
        $query = "SELECT 
                    c.course_id,
                    c.title,
                    c.status,
                    c.price,
                    COUNT(e.enrollment_id) as student_count
                FROM 
                    courses c
                    LEFT JOIN enrollments e ON c.course_id = e.course_id
                WHERE 
                    c.teacher_id = $teacherId
                GROUP BY
                    c.course_id";
                    
        $detailResult = $conn->query($query);
        
        if ($detailResult && $detailResult->num_rows > 0) {
            $teachersInfo .= "<h4>Courses taught by {$firstTeacher['first_name']} {$firstTeacher['last_name']}</h4>";
            $teachersInfo .= "<table>
                <tr>
                    <th>Course ID</th>
                    <th>Title</th>
                    <th>Status</th>
                    <th>Price</th>
                    <th>Enrolled Students</th>
                </tr>";
            
            while ($row = $detailResult->fetch_assoc()) {
                $teachersInfo .= "<tr>";
                $teachersInfo .= "<td>{$row['course_id']}</td>";
                $teachersInfo .= "<td>{$row['title']}</td>";
                $teachersInfo .= "<td>{$row['status']}</td>";
                $teachersInfo .= "<td>{$row['price']}</td>";
                $teachersInfo .= "<td>{$row['student_count']}</td>";
                $teachersInfo .= "</tr>";
            }
            
            $teachersInfo .= "</table>";
        } else {
            $teachersInfo .= "<p class='warning'>No courses found for this teacher.</p>";
        }
    }
} else {
    $teachersInfo .= "<p class='warning'>No teachers found.</p>";
}

outputSection("3. Teachers/Facilitators", $teachersInfo);

// 4. Messages Table Analysis
$messagesInfo = "";
$query = "SELECT COUNT(*) as message_count FROM messages";
$result = $conn->query($query);

if ($result) {
    $row = $result->fetch_assoc();
    $messageCount = $row['message_count'];
    
    $messagesInfo .= "<p>Total messages in the system: <strong>$messageCount</strong></p>";
    
    if ($messageCount > 0) {
        // Get sample messages
        $query = "SELECT 
                    m.message_text,
                    m.message_id,
                    m.sender_id,
                    u_sender.first_name as sender_first_name,
                    u_sender.last_name as sender_last_name,
                    u_sender.user_role as sender_role,
                    m.recipient_id,
                    u_recipient.first_name as recipient_first_name,
                    u_recipient.last_name as recipient_last_name,
                    u_recipient.user_role as recipient_role,
                    LEFT(m.message, 50) as message_preview,
                    m.is_read,
                    m.created_at
                FROM 
                    messages m
                    JOIN users u_sender ON m.sender_id = u_sender.user_id
                    JOIN users u_recipient ON m.recipient_id = u_recipient.user_id
                ORDER BY
                    m.created_at DESC
                LIMIT 10";
                
        $result = $conn->query($query);
        
        if ($result && $result->num_rows > 0) {
            $messagesInfo .= "<h4>Sample Messages (Most Recent)</h4>";
            $messagesInfo .= "<table>
                <tr>
                    <th>Message ID</th>
                    <th>From</th>
                    <th>To</th>
                    <th>Message Preview</th>
                    <th>Read</th>
                    <th>Date</th>
                </tr>";
            
            while ($row = $result->fetch_assoc()) {
                $messagesInfo .= "<tr>";
                $messagesInfo .= "<td>{$row['message_id']}</td>";
                $messagesInfo .= "<td>{$row['sender_first_name']} {$row['sender_last_name']} ({$row['sender_role']})</td>";
                $messagesInfo .= "<td>{$row['recipient_first_name']} {$row['recipient_last_name']} ({$row['recipient_role']})</td>";
                $messagesInfo .= "<td>{$row['message_preview']}...</td>";
                $messagesInfo .= "<td>" . ($row['is_read'] ? 'Yes' : 'No') . "</td>";
                $messagesInfo .= "<td>{$row['created_at']}</td>";
                $messagesInfo .= "</tr>";
            }
            
            $messagesInfo .= "</table>";
            
            // Get message counts by user role
            $query = "SELECT 
                        u_sender.user_role as sender_role,
                        u_recipient.user_role as recipient_role,
                        COUNT(*) as message_count
                    FROM 
                        messages m
                        JOIN users u_sender ON m.sender_id = u_sender.user_id
                        JOIN users u_recipient ON m.recipient_id = u_recipient.user_id
                    GROUP BY
                        u_sender.user_role, u_recipient.user_role";
                        
            $result = $conn->query($query);
            
            if ($result && $result->num_rows > 0) {
                $messagesInfo .= "<h4>Message Counts by User Role</h4>";
                $messagesInfo .= "<table>
                    <tr>
                        <th>Sender Role</th>
                        <th>Recipient Role</th>
                        <th>Message Count</th>
                    </tr>";
                
                while ($row = $result->fetch_assoc()) {
                    $messagesInfo .= "<tr>";
                    $messagesInfo .= "<td>{$row['sender_role']}</td>";
                    $messagesInfo .= "<td>{$row['recipient_role']}</td>";
                    $messagesInfo .= "<td>{$row['message_count']}</td>";
                    $messagesInfo .= "</tr>";
                }
                
                $messagesInfo .= "</table>";
            }
        } else {
            $messagesInfo .= "<p class='warning'>Could not fetch sample messages.</p>";
        }
    } else {
        $messagesInfo .= "<p class='warning'>No messages found in the system.</p>";
    }
} else {
    $messagesInfo .= "<p class='error'>Error counting messages: " . $conn->error . "</p>";
}

outputSection("4. Messages Table Analysis", $messagesInfo);

// 5. Test Message Sending Capability
$testMsgResults = "";

// Get a student and teacher for testing
$query = "SELECT student_id, user_id FROM students LIMIT 1";
$studentResult = $conn->query($query);

$query = "SELECT teacher_id, user_id FROM teachers LIMIT 1";
$teacherResult = $conn->query($query);

if ($studentResult && $studentResult->num_rows > 0 && $teacherResult && $teacherResult->num_rows > 0) {
    $student = $studentResult->fetch_assoc();
    $teacher = $teacherResult->fetch_assoc();
    
    $studentUserId = $student['user_id'];
    $teacherUserId = $teacher['user_id'];
    
    $testMsgResults .= "<p>Selected test users:</p>";
    $testMsgResults .= "<ul>";
    $testMsgResults .= "<li>Student: User ID {$studentUserId}</li>";
    $testMsgResults .= "<li>Teacher: User ID {$teacherUserId}</li>";
    $testMsgResults .= "</ul>";
    
    // Check if they have existing messages
    $query = "SELECT COUNT(*) as msg_count FROM messages 
              WHERE (sender_id = $studentUserId AND recipient_id = $teacherUserId) 
              OR (sender_id = $teacherUserId AND recipient_id = $studentUserId)";
    $result = $conn->query($query);
    $row = $result->fetch_assoc();
    
    $testMsgResults .= "<p>Existing messages between them: {$row['msg_count']}</p>";
    
    // Test insert a message (but don't actually do it)
    $testMsgResults .= "<h4>Message Sending Test (Simulated)</h4>";
    $testMsgResults .= "<p>The following query would be executed to send a message:</p>";
    $testMsgResults .= "<pre>INSERT INTO messages (sender_id, recipient_id, message, is_read, created_at) VALUES ($studentUserId, $teacherUserId, 'Test message from debug script', 0, NOW())</pre>";
    
    // Check for any potential issues in message system
    $checks = [];
    
    // Check users table to ensure both users exist
    $query = "SELECT user_id, first_name, last_name, user_role, status FROM users WHERE user_id IN ($studentUserId, $teacherUserId)";
    $result = $conn->query($query);
    
    if ($result && $result->num_rows == 2) {
        $checks[] = "<span class='success'>✅ Both users exist in the users table.</span>";
        
        while ($row = $result->fetch_assoc()) {
            if ($row['status'] != 'active') {
                $checks[] = "<span class='error'>❌ User {$row['user_id']} ({$row['first_name']} {$row['last_name']}) has status '{$row['status']}', not 'active'.</span>";
            } else {
                $checks[] = "<span class='success'>✅ User {$row['user_id']} ({$row['first_name']} {$row['last_name']}) is active.</span>";
            }
        }
    } else {
        $checks[] = "<span class='error'>❌ Could not find both users in the users table.</span>";
    }
    
    // Check if student is enrolled in any of teacher's courses
    $query = "SELECT e.enrollment_id, c.course_id, c.title 
              FROM enrollments e
              JOIN courses c ON e.course_id = c.course_id
              JOIN teachers t ON c.teacher_id = t.teacher_id
              WHERE e.student_id = {$student['student_id']} AND t.teacher_id = {$teacher['teacher_id']}";
    $result = $conn->query($query);
    
    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $checks[] = "<span class='success'>✅ Student is enrolled in teacher's course: {$row['title']} (ID: {$row['course_id']})</span>";
    } else {
        $checks[] = "<span class='warning'>⚠️ Student is not enrolled in any of teacher's courses.</span>";
    }
    
    $testMsgResults .= "<h4>System Checks</h4>";
    $testMsgResults .= "<ul>";
    foreach ($checks as $check) {
        $testMsgResults .= "<li>$check</li>";
    }
    $testMsgResults .= "</ul>";
    
} else {
    $testMsgResults .= "<p class='error'>Could not find both a student and teacher for testing.</p>";
}

outputSection("5. Test Message Sending Capability", $testMsgResults);

// 6. Common Problems and Solutions
$troubleshooting = "
<h4>Common Issues and Solutions</h4>
<ol>
    <li><strong>No teachers showing in the messaging page</strong>
        <ul>
            <li>Check if there are any teachers in the teachers table</li>
            <li>Verify teachers are linked to users with user_role = 'teacher'</li>
            <li>Ensure teachers have active courses assigned to them</li>
        </ul>
    </li>
    <li><strong>Students can't message teachers</strong>
        <ul>
            <li>Check if the messaging system requires course enrollment</li>
            <li>Verify the send_message.php API is properly querying teachers</li>
            <li>Ensure the student has valid session and authentication</li>
        </ul>
    </li>
    <li><strong>Messages not showing up after being sent</strong>
        <ul>
            <li>Check if the messages are being stored in the database</li>
            <li>Verify the UI is correctly retrieving and displaying messages</li>
            <li>Check for JavaScript errors that might prevent message display</li>
        </ul>
    </li>
</ol>";

outputSection("6. Common Problems and Solutions", $troubleshooting);

// HTML footer
echo "</body></html>";
?>
