<?php
/**
 * Manual Payment Verification Script
 * For debugging PayFast notification issues
 */

header('Content-Type: application/json');
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../config/db_connect.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    $payment_reference = $_POST['payment_reference'] ?? '';
    $user_id = intval($_POST['user_id'] ?? 0);
    
    if (empty($payment_reference) || !$user_id) {
        throw new Exception('Missing payment reference or user ID');
    }
    
    // Check if this is a valid accreditation request for this user
    $stmt = $conn->prepare("
        SELECT request_id, payment_status, status 
        FROM accreditation_requests 
        WHERE payment_reference = ? AND student_id = ?
    ");
    $stmt->bind_param("si", $payment_reference, $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if (!$row = $result->fetch_assoc()) {
        throw new Exception('Payment reference not found or unauthorized');
    }
    
    // If payment is already completed, no need to update
    if ($row['payment_status'] === 'completed') {
        echo json_encode([
            'success' => true, 
            'message' => 'Payment already completed',
            'status' => $row['status']
        ]);
        exit;
    }
    
    // 🔧 MANUAL VERIFICATION: Since PayFast completed but notification failed,
    // manually update the payment status to completed
    $update_stmt = $conn->prepare("
        UPDATE accreditation_requests 
        SET payment_status = 'completed',
            payment_date = NOW(),
            status = 'under_review',
            updated_at = NOW()
        WHERE payment_reference = ? AND student_id = ?
    ");
    $update_stmt->bind_param("si", $payment_reference, $user_id);
    
    if ($update_stmt->execute() && $update_stmt->affected_rows > 0) {
        error_log("✅ Manual payment verification successful: $payment_reference for user $user_id");
        
        echo json_encode([
            'success' => true, 
            'message' => 'Payment status updated to completed',
            'status' => 'under_review'
        ]);
    } else {
        throw new Exception('Failed to update payment status');
    }
    
} catch (Exception $e) {
    error_log("❌ Manual payment verification failed: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
