<?php
/**
 * Database AI Grading Upgrade Script
 * This script will add all necessary AI grading columns to your existing tables
 */

require_once '../config/db_connect.php';

echo "<h2>🚀 AI Grading Database Upgrade</h2>";
echo "<hr>";

$errors = [];
$success = [];

try {
    // First, let's check what columns exist in assignments table
    echo "<h4>📊 Current Assignments Table Structure:</h4>";
    $result = $conn->query("DESCRIBE assignments");
    if ($result) {
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>";
        echo "<tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['Field']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Null']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Key']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Default'] ?? 'NULL') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    
    // Check which AI grading columns are missing
    $existing_columns = [];
    $result = $conn->query("SHOW COLUMNS FROM assignments");
    while ($row = $result->fetch_assoc()) {
        $existing_columns[] = $row['Field'];
    }
    
    // Define required AI grading columns
    $ai_columns = [
        'memorandum_content' => 'LONGTEXT',
        'memorandum_file_path' => 'VARCHAR(500)',
        'memorandum_key_points' => 'LONGTEXT',
        'ai_grading_enabled' => 'TINYINT(1) DEFAULT 1',
        'pass_mark' => 'DECIMAL(5,2) DEFAULT 50.00',
        'max_attempts' => 'INT DEFAULT 3',
        'auto_feedback' => 'TINYINT(1) DEFAULT 1',
        'plagiarism_check' => 'TINYINT(1) DEFAULT 1'
    ];
    
    echo "<h4>🔧 Adding Missing AI Grading Columns:</h4>";
    
    foreach ($ai_columns as $column => $definition) {
        if (!in_array($column, $existing_columns)) {
            $sql = "ALTER TABLE assignments ADD COLUMN `$column` $definition";
            echo "<p>Adding column <strong>$column</strong>: ";
            
            if ($conn->query($sql)) {
                echo "<span style='color: green;'>✅ SUCCESS</span></p>";
                $success[] = "Added column: $column";
            } else {
                echo "<span style='color: red;'>❌ FAILED: " . $conn->error . "</span></p>";
                $errors[] = "Failed to add column $column: " . $conn->error;
            }
        } else {
            echo "<p>Column <strong>$column</strong>: <span style='color: blue;'>✓ Already exists</span></p>";
        }
    }
    
    // Now check assignment_submissions table
    echo "<hr><h4>📊 Assignment Submissions Table:</h4>";
    
    $submission_check = $conn->query("SHOW TABLES LIKE 'assignment_submissions'");
    if ($submission_check->num_rows > 0) {
        // Check existing columns
        $existing_sub_columns = [];
        $result = $conn->query("SHOW COLUMNS FROM assignment_submissions");
        while ($row = $result->fetch_assoc()) {
            $existing_sub_columns[] = $row['Field'];
        }
        
        // Define required submission AI columns
        $submission_ai_columns = [
            'extracted_text' => 'LONGTEXT',
            'ai_score' => 'DECIMAL(5,2)',
            'ai_feedback' => 'LONGTEXT',
            'similarity_score' => 'DECIMAL(5,2)',
            'plagiarism_score' => 'DECIMAL(5,2)',
            'keyword_match_score' => 'DECIMAL(5,2)',
            'structure_score' => 'DECIMAL(5,2)',
            'quality_score' => 'DECIMAL(5,2)',
            'processing_status' => 'ENUM("pending","processing","completed","failed") DEFAULT "pending"',
            'processing_time' => 'DECIMAL(8,2)',
            'review_needed' => 'TINYINT(1) DEFAULT 0',
            'ai_confidence' => 'DECIMAL(5,2)'
        ];
        
        foreach ($submission_ai_columns as $column => $definition) {
            if (!in_array($column, $existing_sub_columns)) {
                $sql = "ALTER TABLE assignment_submissions ADD COLUMN `$column` $definition";
                echo "<p>Adding submission column <strong>$column</strong>: ";
                
                if ($conn->query($sql)) {
                    echo "<span style='color: green;'>✅ SUCCESS</span></p>";
                    $success[] = "Added submission column: $column";
                } else {
                    echo "<span style='color: red;'>❌ FAILED: " . $conn->error . "</span></p>";
                    $errors[] = "Failed to add submission column $column: " . $conn->error;
                }
            } else {
                echo "<p>Submission column <strong>$column</strong>: <span style='color: blue;'>✓ Already exists</span></p>";
            }
        }
    } else {
        echo "<p style='color: orange;'>⚠️ assignment_submissions table doesn't exist yet - will be created when first assignment is made</p>";
    }
    
    // Create uploads directory structure
    echo "<hr><h4>📁 Creating Upload Directories:</h4>";
    
    $directories = [
        '../uploads/memorandums/',
        '../uploads/assignments/',
        '../uploads/submissions/'
    ];
    
    foreach ($directories as $dir) {
        if (!is_dir($dir)) {
            if (mkdir($dir, 0755, true)) {
                echo "<p>Created directory: <strong>$dir</strong> ✅</p>";
                $success[] = "Created directory: $dir";
            } else {
                echo "<p style='color: red;'>Failed to create directory: <strong>$dir</strong> ❌</p>";
                $errors[] = "Failed to create directory: $dir";
            }
        } else {
            echo "<p>Directory exists: <strong>$dir</strong> ✓</p>";
        }
    }
    
    // Summary
    echo "<hr><h4>📋 Upgrade Summary:</h4>";
    echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h5 style='color: #155724;'>✅ Successful Operations (" . count($success) . "):</h5>";
    if (empty($success)) {
        echo "<p>No operations needed - everything was already configured!</p>";
    } else {
        echo "<ul>";
        foreach ($success as $item) {
            echo "<li>$item</li>";
        }
        echo "</ul>";
    }
    echo "</div>";
    
    if (!empty($errors)) {
        echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
        echo "<h5 style='color: #721c24;'>❌ Errors (" . count($errors) . "):</h5>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li>$error</li>";
        }
        echo "</ul>";
        echo "</div>";
    }
    
    if (empty($errors)) {
        echo "<div style='background: #d1ecf1; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
        echo "<h5 style='color: #0c5460;'>🎉 Database is now ready for AI Grading!</h5>";
        echo "<p>You can now create assignments with AI grading enabled.</p>";
        echo "</div>";
    }
    
    // Test database connection and table integrity
    echo "<hr><h4>🔍 Database Integrity Check:</h4>";
    
    $test_query = "SELECT assignment_id, title, ai_grading_enabled, memorandum_content FROM assignments LIMIT 1";
    $test_result = $conn->query($test_query);
    
    if ($test_result !== false) {
        echo "<p style='color: green;'>✅ AI grading columns are accessible and working!</p>";
    } else {
        echo "<p style='color: red;'>❌ Error accessing AI grading columns: " . $conn->error . "</p>";
    }
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px;'>";
    echo "<h5 style='color: #721c24;'>💥 Critical Error:</h5>";
    echo "<p>" . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "<hr>";
echo "<p><a href='../teacher/create-assignment.php' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>← Back to Create Assignment</a></p>";
?>
