<?php
// Ultra simplified test page for messaging system

// Get database connection
require_once '../config/db_connect.php';

// Initialize variables
$recipientId = isset($_GET['recipient']) ? intval($_GET['recipient']) : 2; // Default recipient ID = 2
$senderId = isset($_GET['sender']) ? intval($_GET['sender']) : 1;  // Default sender ID = 1
$courseId = isset($_GET['course']) ? intval($_GET['course']) : null;
$successMessage = '';
$errorMessage = '';
$senderDetails = [];
$recipientDetails = [];
$messages = [];

// Get user details for display
if ($conn) {
    // Get sender details
    $stmt = $conn->prepare("SELECT user_id, first_name, last_name, profile_picture FROM users WHERE user_id = ?");
    $stmt->bind_param("i", $senderId);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $senderDetails = $row;
    }
    
    // Get recipient details
    $stmt = $conn->prepare("SELECT user_id, first_name, last_name, profile_picture FROM users WHERE user_id = ?");
    $stmt->bind_param("i", $recipientId);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $recipientDetails = $row;
    }
    
    // Get existing messages
    $stmt = $conn->prepare("SELECT m.*, 
                          s.first_name AS sender_first_name, 
                          s.last_name AS sender_last_name,
                          s.profile_picture AS sender_profile_picture
                          FROM messages m
                          JOIN users s ON m.sender_id = s.user_id
                          WHERE (m.sender_id = ? AND m.recipient_id = ?)
                          OR (m.sender_id = ? AND m.recipient_id = ?)
                          ORDER BY m.sent_at ASC");
    $stmt->bind_param("iiii", $senderId, $recipientId, $recipientId, $senderId);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $messages[] = $row;
    }
}

// Handle message submission
if (isset($_POST['submit'])) {
    $message = trim($_POST['message']);
    
    if (!empty($message) && $conn) {
        $stmt = $conn->prepare("INSERT INTO messages (sender_id, recipient_id, message_text, is_read, sent_at) VALUES (?, ?, ?, 0, NOW())");
        $stmt->bind_param("iis", $senderId, $recipientId, $message);
        
        if ($stmt->execute()) {
            $successMessage = "Message sent successfully!";
            
            // Get the new message details
            $messageId = $conn->insert_id;
            $stmt = $conn->prepare("SELECT m.*, 
                                  s.first_name AS sender_first_name, 
                                  s.last_name AS sender_last_name,
                                  s.profile_picture AS sender_profile_picture
                                  FROM messages m
                                  JOIN users s ON m.sender_id = s.user_id
                                  WHERE m.message_id = ?");
            $stmt->bind_param("i", $messageId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $messages[] = $row;
            }
            
            // Clear the message input
            $_POST['message'] = '';
        } else {
            $errorMessage = "Error sending message: " . $stmt->error;
        }
    } else {
        $errorMessage = "Please enter a message.";
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Messaging System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <style>
        .messages-container {
            height: 400px;
            overflow-y: auto;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .message {
            margin-bottom: 15px;
            display: flex;
            align-items: flex-start;
        }
        .message-sent {
            flex-direction: row-reverse;
        }
        .message-bubble {
            padding: 10px 15px;
            border-radius: 15px;
            max-width: 70%;
            word-wrap: break-word;
        }
        .message-sent .message-bubble {
            background: #0d6efd;
            color: white;
            border-bottom-right-radius: 0;
        }
        .message-received .message-bubble {
            background: #e9ecef;
            color: #212529;
            border-bottom-left-radius: 0;
        }
        .message-time {
            font-size: 0.75rem;
            color: #6c757d;
            margin-top: 5px;
        }
        .avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            margin: 0 10px;
            object-fit: cover;
        }
        .avatar-placeholder {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            margin: 0 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #6c757d;
            color: white;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <div class="container py-4">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            Test Messaging System 
                            (Sender: <?= htmlspecialchars($senderDetails['first_name'] ?? 'User') . ' ' . htmlspecialchars($senderDetails['last_name'] ?? $senderId) ?>)
                        </h5>
                        <div>
                            <a href="../student/messages.php" class="btn btn-sm btn-light">Back to Real Messages</a>
                        </div>
                    </div>
                    <div class="card-body">
                        <h6>Conversation with: <?= htmlspecialchars($recipientDetails['first_name'] ?? 'User') . ' ' . htmlspecialchars($recipientDetails['last_name'] ?? $recipientId) ?></h6>
                        
                        <?php if ($successMessage): ?>
                            <div class="alert alert-success"><?= $successMessage ?></div>
                        <?php endif; ?>
                        
                        <?php if ($errorMessage): ?>
                            <div class="alert alert-danger"><?= $errorMessage ?></div>
                        <?php endif; ?>
                        
                        <div class="messages-container">
                            <?php if (empty($messages)): ?>
                                <div class="text-center text-muted py-5">
                                    <i class="bx bx-message-dots bx-lg"></i>
                                    <p>No messages yet. Start the conversation!</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($messages as $message): ?>
                                    <?php $isCurrentUser = ($message['sender_id'] == $senderId); ?>
                                    <div class="message <?= $isCurrentUser ? 'message-sent' : 'message-received' ?>">
                                        <?php if (!$isCurrentUser): ?>
                                            <?php if (!empty($message['sender_profile_picture'])): ?>
                                                <img src="<?= htmlspecialchars($message['sender_profile_picture']) ?>" alt="" class="avatar">
                                            <?php else: ?>
                                                <div class="avatar-placeholder">
                                                    <?= strtoupper(substr($message['sender_first_name'] ?? 'U', 0, 1)) ?>
                                                </div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                        
                                        <div>
                                            <div class="message-bubble">
                                                <?= nl2br(htmlspecialchars($message['message_text'])) ?>
                                            </div>
                                            <div class="message-time">
                                                <?= date('h:i A', strtotime($message['sent_at'])) ?>
                                                <?php if ($isCurrentUser): ?>
                                                    <?php if ($message['is_read']): ?>
                                                        <i class="bx bx-check-double text-primary"></i>
                                                    <?php else: ?>
                                                        <i class="bx bx-check"></i>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        
                                        <?php if ($isCurrentUser): ?>
                                            <?php if (!empty($message['sender_profile_picture'])): ?>
                                                <img src="<?= htmlspecialchars($message['sender_profile_picture']) ?>" alt="" class="avatar">
                                            <?php else: ?>
                                                <div class="avatar-placeholder">
                                                    <?= strtoupper(substr($message['sender_first_name'] ?? 'U', 0, 1)) ?>
                                                </div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        
                        <form method="post" class="mt-3">
                            <div class="input-group">
                                <textarea class="form-control" name="message" placeholder="Type your message..." required><?= $_POST['message'] ?? '' ?></textarea>
                                <button type="submit" name="submit" class="btn btn-primary">
                                    <i class="bx bx-send"></i> Send
                                </button>
                            </div>
                        </form>
                    </div>
                    <div class="card-footer">
                        <form method="get" class="row g-3">
                            <div class="col-md-4">
                                <label for="sender" class="form-label">Sender ID</label>
                                <input type="number" class="form-control" id="sender" name="sender" value="<?= $senderId ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="recipient" class="form-label">Recipient ID</label>
                                <input type="number" class="form-control" id="recipient" name="recipient" value="<?= $recipientId ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="course" class="form-label">Course ID (optional)</label>
                                <input type="number" class="form-control" id="course" name="course" value="<?= $courseId ?>">
                            </div>
                            <div class="col-12">
                                <button type="submit" class="btn btn-secondary">Change Users</button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Auto-scroll to bottom of messages
        window.onload = function() {
            const messagesContainer = document.querySelector('.messages-container');
            if (messagesContainer) {
                messagesContainer.scrollTop = messagesContainer.scrollHeight;
            }
        };
    </script>
</body>
</html>
