<?php
require_once 'auth/functions.php';
require_once 'config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but admin uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send password reset email notification to user
 */
function sendPasswordResetEmailNotification($email, $token, $user_name) {
    global $conn;
    
    // Get site config variables (same as admin courses.php pattern)
    global $site_config;
    $site_url = $site_config['site_url'] ?? 'https://panaditeacademy.com';
    $site_name = $site_config['site_name'] ?? 'Panadite Academy';
    
    try {
        // Create email content (exact structure like admin courses.php)
        $reset_link = $site_url . "/reset-password.php?token=" . $token;
        $subject = "🔐 Password Reset Request - " . $site_name;
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#276BE4; color:white; padding:20px; text-align:center;'>
                <h1>🔐 Password Reset</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hello, " . htmlspecialchars($user_name) . "! 👋</h2>
                <p>We received a request to reset your password for your " . htmlspecialchars($site_name) . " account.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Reset Instructions:</h3>
                    <p><strong>Security Notice:</strong> This link will expire in 1 hour</p>
                    <p><strong>Time Requested:</strong> " . date('M j, Y \a\t g:i A') . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='" . $reset_link . "' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Reset My Password
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending password reset email: " . $e->getMessage());
        return false;
    }
}

// Initialize variables
$message = '';
$error = '';
$success = false;

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    
    // Basic validation
    if (empty($email)) {
        $error = 'Please enter your email address.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        // Check if email exists in database
        $stmt = $conn->prepare("SELECT user_id, first_name FROM users WHERE email = ?");
        if (!$stmt) {
            $error = 'Database error: ' . $conn->error . '. Please contact support.';
        } else {
            $stmt->bind_param("s", $email);
            if (!$stmt->execute()) {
                $error = 'Database query failed. Please try again.';
            } else {
                $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $user = $result->fetch_assoc();
            
            // Generate secure token
            $token = bin2hex(random_bytes(32));
            $expires_at = date('Y-m-d H:i:s', strtotime('+1 hour'));
            
            // Delete any existing tokens for this user
            $delete_stmt = $conn->prepare("DELETE FROM password_reset_tokens WHERE user_id = ?");
            if ($delete_stmt) {
                $delete_stmt->bind_param("i", $user['user_id']);
                $delete_stmt->execute();
            }
            
            // Insert new token
            $insert_stmt = $conn->prepare("INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?, ?, ?)");
            if ($insert_stmt) {
                $insert_stmt->bind_param("iss", $user['user_id'], $token, $expires_at);
            
            if ($insert_stmt->execute()) {
                // Send password reset email (exact same pattern as admin courses.php)
                global $debug_info;
                $email_sent = sendPasswordResetEmailNotification($email, $token, $user['first_name']);
                
                if ($email_sent) {
                    $success = true;
                    $message = 'Password reset instructions have been sent to your email address.';
                } else {
                    $error = 'Failed to send password reset email. Please try again later.';
                }
            } else {
                $error = 'Failed to generate password reset token. Please try again.';
            }
                } else {
                    $error = 'Failed to prepare password reset token statement.';
                }
            } else {
                // For security, always show success message even if email doesn't exist
                $success = true;
                $message = 'If an account with that email exists, password reset instructions have been sent.';
            }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Password - <?php echo $site_name; ?></title>
    
    <!-- Bootstrap 5.2.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Animate.css -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    
    <style>
        :root {
            --primary-color: #276BE4;
            --secondary-color: #FFA915;
            --success-color: #28a745;
            --danger-color: #dc3545;
        }

        body {
            background: whitesmoke;
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 0;
            position: relative;
            overflow-x: hidden;
        }

        /* Subtle animated background elements */
        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: 
                radial-gradient(circle at 20% 20%, rgba(39, 107, 228, 0.03) 0%, transparent 50%),
                radial-gradient(circle at 80% 80%, rgba(255, 169, 21, 0.03) 0%, transparent 50%),
                radial-gradient(circle at 40% 60%, rgba(39, 107, 228, 0.02) 0%, transparent 50%);
            animation: gentleFloat 20s ease-in-out infinite;
            z-index: -1;
        }

        body::after {
            content: '';
            position: fixed;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: 
                radial-gradient(circle at 60% 40%, rgba(39, 107, 228, 0.02) 0%, transparent 40%),
                radial-gradient(circle at 30% 70%, rgba(255, 169, 21, 0.02) 0%, transparent 40%);
            animation: gentleRotate 30s linear infinite;
            z-index: -1;
        }

        /* Floating particles */
        .floating-particles {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: -1;
        }

        .floating-particles::before,
        .floating-particles::after {
            content: '';
            position: absolute;
            border-radius: 50%;
            background: rgba(39, 107, 228, 0.05);
        }

        .floating-particles::before {
            width: 4px;
            height: 4px;
            top: 20%;
            left: 10%;
            animation: floatUp 15s infinite linear;
        }

        .floating-particles::after {
            width: 2px;
            height: 2px;
            top: 60%;
            right: 20%;
            background: rgba(255, 169, 21, 0.05);
            animation: floatUp 12s infinite linear reverse;
        }

        .main-content {
            min-height: calc(100vh - 200px);
            display: flex;
            align-items: center;
            padding: 40px 0;
        }

        .forgot-password-container {
            max-width: 500px;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.15);
            overflow: hidden;
            backdrop-filter: blur(10px);
            position: relative;
        }

        .forgot-password-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
        }

        .header {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            color: white;
            padding: 40px 30px;
            text-align: center;
            position: relative;
        }

        .header i {
            font-size: 48px;
            margin-bottom: 15px;
            opacity: 0.9;
        }

        .header h1 {
            margin: 0;
            font-size: 28px;
            font-weight: 700;
        }

        .header p {
            margin: 10px 0 0 0;
            opacity: 0.9;
            font-size: 16px;
        }

        .form-section {
            padding: 40px 30px;
        }

        .form-floating {
            margin-bottom: 20px;
        }

        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 12px 16px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.9);
        }

        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(39, 107, 228, 0.25);
            background: white;
        }

        .form-floating label {
            color: #6c757d;
            font-weight: 500;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            border-radius: 12px;
            padding: 14px 30px;
            font-size: 16px;
            font-weight: 600;
            width: 100%;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }

        .btn-primary:active {
            transform: translateY(0);
        }

        .alert {
            border-radius: 12px;
            border: none;
            padding: 16px 20px;
            margin-bottom: 25px;
            font-weight: 500;
            position: relative;
            overflow: hidden;
        }

        .alert::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
            background: currentColor;
        }

        .alert-success {
            background: linear-gradient(135deg, #d4edda, #c3e6cb);
            color: #155724;
        }

        .alert-danger {
            background: linear-gradient(135deg, #f8d7da, #f5c6cb);
            color: #721c24;
        }

        .success-animation {
            text-align: center;
            padding: 20px;
        }

        .success-animation i {
            font-size: 64px;
            color: var(--success-color);
            margin-bottom: 20px;
            animation: pulse 2s infinite;
        }

        .back-links {
            text-align: center;
            padding: 20px 30px 30px;
            background: #f8f9fa;
            border-top: 1px solid #e9ecef;
        }

        .back-links a {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
            margin: 0 15px;
            transition: all 0.3s ease;
        }

        .back-links a:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }

        @keyframes gentleFloat {
            0%, 100% { transform: translate(0, 0) scale(1); }
            25% { transform: translate(10px, -5px) scale(1.02); }
            50% { transform: translate(-5px, 10px) scale(0.98); }
            75% { transform: translate(8px, -8px) scale(1.01); }
        }

        @keyframes gentleRotate {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        @keyframes floatUp {
            0% {
                transform: translateY(100vh) translateX(0);
                opacity: 0;
            }
            10% {
                opacity: 0.3;
            }
            90% {
                opacity: 0.1;
            }
            100% {
                transform: translateY(-100px) translateX(30px);
                opacity: 0;
            }
        }

        @keyframes gentleBob {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-3px); }
        }

        @media (max-width: 768px) {
            .main-content {
                padding: 20px 0;
            }
            
            .forgot-password-container {
                margin: 0 15px;
            }
            
            .header {
                padding: 30px 20px;
            }
            
            .form-section {
                padding: 30px 20px;
            }
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/includes/navbar.php'; ?>
    
    <!-- Floating particles for subtle animation -->
    <div class="floating-particles"></div>
    
    <div class="main-content">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-lg-6 col-md-8">
                    <div class="forgot-password-container animate__animated animate__fadeInUp">
                        
                        <?php if ($success): ?>
                        <!-- Success State -->
                        <div class="header">
                            <div class="success-animation">
                                <i class="fas fa-check-circle"></i>
                            </div>
                            <h1>Email Sent!</h1>
                            <p>Check your inbox for password reset instructions</p>
                        </div>
                        
                        <div class="form-section">
                            <div class="alert alert-success animate__animated animate__fadeIn">
                                <i class="fas fa-envelope me-2"></i>
                                <?php echo htmlspecialchars($message); ?>
                            </div>
                            
                            <div class="text-center">
                                <p class="text-muted mb-3">
                                    <i class="fas fa-clock me-1"></i>
                                    The reset link will expire in 1 hour for security.
                                </p>
                                <p class="text-muted small">
                                    Didn't receive the email? Check your spam folder or try again.
                                </p>
                            </div>
                        </div>
                        
                        <?php else: ?>
                        <!-- Form State -->
                        <div class="header">
                            <i class="fas fa-key"></i>
                            <h1>Forgot Password?</h1>
                            <p>Enter your email to receive password reset instructions</p>
                        </div>
                        
                        <div class="form-section">
                            <?php if ($error): ?>
                            <div class="alert alert-danger animate__animated animate__shakeX">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                <?php echo htmlspecialchars($error); ?>
                            </div>
                            <?php endif; ?>
                            
                            <form method="POST" class="animate__animated animate__fadeIn">
                                <div class="form-floating">
                                    <input type="email" 
                                           class="form-control" 
                                           id="email" 
                                           name="email" 
                                           placeholder="Enter your email address"
                                           value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>"
                                           required>
                                    <label for="email">
                                        <i class="fas fa-envelope me-2"></i>Email Address
                                    </label>
                                </div>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-paper-plane me-2"></i>
                                    Send Reset Instructions
                                </button>
                            </form>
                            
                            <div class="text-center mt-4">
                                <p class="text-muted small">
                                    <i class="fas fa-info-circle me-1"></i>
                                    We'll send you a secure link to reset your password
                                </p>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <div class="back-links">
                            <a href="auth/login.php">
                                <i class="fas fa-arrow-left me-1"></i>Back to Login
                            </a>
                            <a href="index.php">
                                <i class="fas fa-home me-1"></i>Home
                            </a>
                            <?php if (!$success): ?>
                            <a href="auth/register.php">
                                <i class="fas fa-user-plus me-1"></i>Create Account
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include __DIR__ . '/includes/footer.php'; ?>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Add smooth animations
        document.addEventListener('DOMContentLoaded', function() {
            // Auto-focus email field if form is visible
            const emailInput = document.getElementById('email');
            if (emailInput && !<?php echo $success ? 'true' : 'false'; ?>) {
                setTimeout(() => emailInput.focus(), 500);
            }
            
            // Add loading state to submit button
            const form = document.querySelector('form');
            const submitBtn = document.querySelector('.btn-primary');
            
            if (form && submitBtn) {
                form.addEventListener('submit', function() {
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Sending...';
                    submitBtn.disabled = true;
                });
            }
        });
    </script>
</body>
</html>
