<?php
/**
 * Panadite Academy - Authentication Manager
 * Handles user authentication, login, logout operations
 */

require_once __DIR__ . '/SessionManager.php';

class AuthManager {
    
    private $sessionManager;
    private $conn;
    
    public function __construct($database_connection = null) {
        $this->sessionManager = SessionManager::getInstance();
        $this->conn = $database_connection;
    }
    
    /**
     * Authenticate user with username/email and password
     */
    public function authenticate($username, $password) {
        try {
            // Initialize session
            $this->sessionManager->init();
            
            // Validate input
            if (empty($username) || empty($password)) {
                throw new Exception('Username and password are required');
            }
            
            // Check for emergency credentials first
            if ($this->checkEmergencyCredentials($username, $password)) {
                return $this->createEmergencySession($username);
            }
            
            // Database authentication
            if (!$this->conn) {
                throw new Exception('Database connection not available');
            }
            
            return $this->authenticateWithDatabase($username, $password);
            
        } catch (Exception $e) {
            error_log('AuthManager Error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Check emergency/fallback credentials
     */
    private function checkEmergencyCredentials($username, $password) {
        $emergencyCredentials = [
            'admin' => 'admin123',
            'teacher' => 'teacher123', 
            'student' => 'student123'
        ];
        
        return isset($emergencyCredentials[$username]) && 
               $emergencyCredentials[$username] === $password;
    }
    
    /**
     * Create emergency session for fallback access
     */
    private function createEmergencySession($username) {
        $userData = [
            'user_id' => 999,
            'username' => $username,
            'email' => $username . '@emergency.local',
            'first_name' => ucfirst($username),
            'last_name' => 'Emergency',
            'user_role' => $username
        ];
        
        $this->sessionManager->createUserSession($userData);
        
        error_log('AuthManager: Emergency login successful for: ' . $username);
        
        return [
            'success' => true,
            'user_data' => $userData,
            'message' => 'Emergency login successful',
            'redirect_url' => $this->sessionManager->getRoleBasedRedirect()
        ];
    }
    
    /**
     * Authenticate with database
     */
    private function authenticateWithDatabase($username, $password) {
        try {
            // Prepare query - allow login with username or email
            $stmt = $this->conn->prepare("
                SELECT user_id, username, email, password, first_name, last_name, user_role, status 
                FROM users 
                WHERE (username = ? OR email = ?) AND status = 'active'
            ");
            
            if (!$stmt) {
                throw new Exception('Database query preparation failed: ' . $this->conn->error);
            }
            
            $stmt->bind_param("ss", $username, $username);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows === 0) {
                throw new Exception('Invalid username/email or password');
            }
            
            $user = $result->fetch_assoc();
            $stmt->close();
            
            // Verify password
            if (!password_verify($password, $user['password'])) {
                // Also try direct comparison for legacy passwords
                if ($user['password'] !== $password) {
                    throw new Exception('Invalid username/email or password');
                }
            }
            
            // Create session
            $this->sessionManager->createUserSession($user);
            
            error_log('AuthManager: Database login successful for: ' . $user['username'] . ' (ID: ' . $user['user_id'] . ')');
            
            return [
                'success' => true,
                'user_data' => $user,
                'message' => 'Login successful',
                'redirect_url' => $this->sessionManager->getRoleBasedRedirect()
            ];
            
        } catch (Exception $e) {
            error_log('AuthManager Database Error: ' . $e->getMessage());
            throw new Exception('Authentication failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Logout user
     */
    public function logout() {
        try {
            $userData = $this->sessionManager->getUserData();
            if ($userData) {
                error_log('AuthManager: Logout for user: ' . $userData['username']);
            }
            
            $this->sessionManager->destroy();
            
            return [
                'success' => true,
                'message' => 'Logout successful',
                'redirect_url' => $this->getBaseUrl() . '/auth/login.php'
            ];
            
        } catch (Exception $e) {
            error_log('AuthManager Logout Error: ' . $e->getMessage());
            // Even if there's an error, try to destroy session
            $this->sessionManager->destroy();
            return [
                'success' => true,
                'message' => 'Logout completed',
                'redirect_url' => $this->getBaseUrl() . '/auth/login.php'
            ];
        }
    }
    
    /**
     * Check if user is logged in
     */
    public function isLoggedIn() {
        return $this->sessionManager->isLoggedIn();
    }
    
    /**
     * Get current user data
     */
    public function getCurrentUser() {
        return $this->sessionManager->getUserData();
    }
    
    /**
     * Get user role
     */
    public function getUserRole() {
        return $this->sessionManager->getUserRole();
    }
    
    /**
     * Check if user has specific role
     */
    public function hasRole($role) {
        return $this->getUserRole() === $role;
    }
    
    /**
     * Check if user has any of the specified roles
     */
    public function hasAnyRole($roles) {
        $userRole = $this->getUserRole();
        return in_array($userRole, (array)$roles);
    }
    
    /**
     * Require login - redirect to login page if not logged in
     */
    public function requireLogin($allowedRoles = null) {
        if (!$this->isLoggedIn()) {
            $this->redirectToLogin();
        }
        
        if ($allowedRoles && !$this->hasAnyRole($allowedRoles)) {
            $this->redirectToUnauthorized();
        }
    }
    
    /**
     * Redirect to login page
     */
    public function redirectToLogin($returnUrl = null) {
        $loginUrl = $this->getBaseUrl() . '/auth/login.php';
        
        if ($returnUrl) {
            $loginUrl .= '?redirect=' . urlencode($returnUrl);
        } else {
            // Store current page as return URL
            $currentUrl = $_SERVER['REQUEST_URI'];
            if (!strpos($currentUrl, 'login.php') && !strpos($currentUrl, 'logout.php')) {
                $loginUrl .= '?redirect=' . urlencode($currentUrl);
            }
        }
        
        header('Location: ' . $loginUrl);
        exit();
    }
    
    /**
     * Redirect to unauthorized page
     */
    public function redirectToUnauthorized() {
        header('Location: ' . $this->getBaseUrl() . '/auth/unauthorized.php');
        exit();
    }
    
    /**
     * Redirect to role-based dashboard
     */
    public function redirectToDashboard() {
        header('Location: ' . $this->sessionManager->getRoleBasedRedirect());
        exit();
    }
    
    /**
     * Change user password
     */
    public function changePassword($currentPassword, $newPassword) {
        if (!$this->isLoggedIn()) {
            throw new Exception('User not logged in');
        }
        
        if (!$this->conn) {
            throw new Exception('Database connection not available');
        }
        
        $userId = $this->sessionManager->getUserId();
        
        // Verify current password
        $stmt = $this->conn->prepare("SELECT password FROM users WHERE user_id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $stmt->close();
        
        if (!$user || !password_verify($currentPassword, $user['password'])) {
            throw new Exception('Current password is incorrect');
        }
        
        // Update password
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $this->conn->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE user_id = ?");
        $stmt->bind_param("si", $hashedPassword, $userId);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to update password');
        }
        
        $stmt->close();
        
        error_log('AuthManager: Password changed for user ID: ' . $userId);
        return true;
    }
    
    /**
     * Get base URL
     */
    private function getBaseUrl() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $path = dirname($_SERVER['SCRIPT_NAME']);
        
        // Clean up path to get root directory
        $path = rtrim($path, '/');
        if (strpos($path, '/auth') !== false) {
            $path = dirname($path);
        }
        if (strpos($path, '/includes') !== false) {
            $path = dirname($path);
        }
        
        return $protocol . '://' . $host . $path;
    }
    
    /**
     * Set flash message
     */
    public function setFlash($type, $message) {
        $this->sessionManager->setFlash($type, $message);
    }
    
    /**
     * Get flash message
     */
    public function getFlash($type) {
        return $this->sessionManager->getFlash($type);
    }
    
    /**
     * Get debug information
     */
    public function getDebugInfo() {
        return [
            'session_info' => $this->sessionManager->getDebugInfo(),
            'current_user' => $this->getCurrentUser(),
            'is_logged_in' => $this->isLoggedIn(),
            'user_role' => $this->getUserRole()
        ];
    }
}
