<?php
/**
 * Real-Time Notification Service for Teachers
 * Handles creating, tracking, and delivering notifications based on database events
 */

class NotificationService {
    private $conn;
    
    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }
    
    /**
     * Create a new notification
     */
    public function createNotification($user_id, $type, $title, $content, $related_to_type = null, $related_to_id = null, $sender_id = null) {
        $stmt = $this->conn->prepare(
            "INSERT INTO notifications (user_id, type, title, content, related_to_type, related_to_id, sender_id, is_read, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, 0, NOW())"
        );
        
        if ($stmt) {
            $stmt->bind_param("issssii", $user_id, $type, $title, $content, $related_to_type, $related_to_id, $sender_id);
            $result = $stmt->execute();
            $stmt->close();
            return $result;
        }
        
        return false;
    }
    
    /**
     * 1. STUDENT ENROLLMENT NOTIFICATION
     * Triggers when a student enrolls in a teacher's course
     */
    public function notifyTeacherOfEnrollment($teacher_id, $student_id, $course_id) {
        // Get student and course details
        $student = $this->getUserById($student_id);
        $course = $this->getCourseById($course_id);
        
        if ($student && $course) {
            $title = "New Student Enrollment";
            $content = "{$student['first_name']} {$student['last_name']} enrolled in \"{$course['title']}\"";
            
            return $this->createNotification(
                $teacher_id,
                'student_enrollment', 
                $title,
                $content,
                'enrollment',
                null,
                $student_id
            );
        }
        
        return false;
    }
    
    /**
     * 2. COURSE COMPLETION NOTIFICATION
     * Triggers when a student completes a teacher's course
     */
    public function notifyTeacherOfCompletion($teacher_id, $student_id, $course_id, $completion_percentage = 100) {
        $student = $this->getUserById($student_id);
        $course = $this->getCourseById($course_id);
        
        if ($student && $course) {
            $title = "Course Completed";
            $content = "{$student['first_name']} {$student['last_name']} completed \"{$course['title']}\" with {$completion_percentage}% score";
            
            return $this->createNotification(
                $teacher_id,
                'course_completion',
                $title,
                $content,
                'completion',
                $course_id,
                $student_id
            );
        }
        
        return false;
    }
    
    /**
     * 3. WITHDRAWAL REQUEST NOTIFICATION
     * Triggers when teacher's withdrawal request status changes
     */
    public function notifyTeacherOfWithdrawalUpdate($teacher_id, $withdrawal_id, $status, $amount) {
        $status_messages = [
            'pending' => 'Your withdrawal request of R' . number_format($amount, 2) . ' is being processed',
            'approved' => 'Great news! Your withdrawal request of R' . number_format($amount, 2) . ' has been approved',
            'completed' => 'Your withdrawal of R' . number_format($amount, 2) . ' has been completed successfully',
            'rejected' => 'Your withdrawal request of R' . number_format($amount, 2) . ' was not approved'
        ];
        
        $title = 'Withdrawal Update';
        $content = $status_messages[$status] ?? 'Withdrawal status updated';
        
        return $this->createNotification(
            $teacher_id,
            'withdrawal_update',
            $title,
            $content,
            'withdrawal',
            $withdrawal_id
        );
    }
    
    /**
     * 4. REFERRAL NOTIFICATION
     * Triggers when teacher earns referral commission
     */
    public function notifyTeacherOfReferralReward($teacher_id, $referral_id, $commission_amount, $referred_user_name = null) {
        $title = 'Referral Commission Earned';
        
        if ($referred_user_name) {
            $content = "You earned R" . number_format($commission_amount, 2) . " commission from {$referred_user_name}'s enrollment";
        } else {
            $content = "You earned R" . number_format($commission_amount, 2) . " referral commission";
        }
        
        return $this->createNotification(
            $teacher_id,
            'referral_reward',
            $title,
            $content,
            'referral',
            $referral_id
        );
    }
    
    /**
     * 5. COURSE CREATION NOTIFICATION
     * Triggers when teacher successfully creates a new course
     */
    public function notifyTeacherOfCourseCreation($teacher_id, $course_id) {
        $course = $this->getCourseById($course_id);
        
        if ($course) {
            $title = 'Course Created Successfully';
            $content = "Your course \"{$course['title']}\" has been created successfully";
            
            return $this->createNotification(
                $teacher_id,
                'course_creation',
                $title,
                $content,
                'course',
                $course_id
            );
        }
        
        return false;
    }
    
    /**
     * 6. VIDEO UPLOAD NOTIFICATION
     * Triggers when teacher uploads a new video
     */
    public function notifyTeacherOfVideoUpload($teacher_id, $video_id, $course_id) {
        $video = $this->getVideoById($video_id);
        $course = $this->getCourseById($course_id);
        
        if ($video && $course) {
            $title = 'Video Uploaded Successfully';
            $content = "Video \"{$video['title']}\" uploaded to course \"{$course['title']}\"";
            
            return $this->createNotification(
                $teacher_id,
                'video_upload',
                $title,
                $content,
                'video',
                $video_id
            );
        }
        
        return false;
    }
    
    /**
     * Get teacher's notifications (for the notifications page)
     */
    public function getTeacherNotifications($teacher_id, $limit = 20, $unread_only = false) {
        $sql = "SELECT n.*, u.first_name as sender_first_name, u.last_name as sender_last_name 
                FROM notifications n 
                LEFT JOIN users u ON n.sender_id = u.user_id 
                WHERE n.user_id = ?";
        
        if ($unread_only) {
            $sql .= " AND n.is_read = 0";
        }
        
        $sql .= " ORDER BY n.created_at DESC LIMIT ?";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ii", $teacher_id, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $notifications = [];
        while ($row = $result->fetch_assoc()) {
            // Format time ago
            $row['time_ago'] = $this->timeAgo($row['created_at']);
            $notifications[] = $row;
        }
        
        $stmt->close();
        return $notifications;
    }
    
    /**
     * Mark notification as read
     */
    public function markAsRead($notification_id, $user_id) {
        $stmt = $this->conn->prepare("UPDATE notifications SET is_read = 1 WHERE notification_id = ? AND user_id = ?");
        $stmt->bind_param("ii", $notification_id, $user_id);
        $result = $stmt->execute();
        $stmt->close();
        return $result;
    }
    
    /**
     * Mark all notifications as read for a user
     */
    public function markAllAsRead($user_id) {
        $stmt = $this->conn->prepare("UPDATE notifications SET is_read = 1 WHERE user_id = ? AND is_read = 0");
        $stmt->bind_param("i", $user_id);
        $result = $stmt->execute();
        $stmt->close();
        return $result;
    }
    
    /**
     * Get unread notification count
     */
    public function getUnreadCount($user_id) {
        $stmt = $this->conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        return $row['count'];
    }
    
    // Helper methods
    private function getUserById($user_id) {
        $stmt = $this->conn->prepare("SELECT user_id, first_name, last_name, email FROM users WHERE user_id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $stmt->close();
        return $user;
    }
    
    private function getCourseById($course_id) {
        $stmt = $this->conn->prepare("SELECT course_id, title, teacher_id FROM courses WHERE course_id = ?");
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        $stmt->close();
        return $course;
    }
    
    private function getVideoById($video_id) {
        $stmt = $this->conn->prepare("SELECT video_id, title, course_id, teacher_id FROM videos WHERE video_id = ?");
        $stmt->bind_param("i", $video_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $video = $result->fetch_assoc();
        $stmt->close();
        return $video;
    }
    
    private function timeAgo($datetime) {
        $time = time() - strtotime($datetime);
        
        if ($time < 60) return 'Just now';
        if ($time < 3600) return floor($time/60) . ' min ago';
        if ($time < 86400) return floor($time/3600) . ' hr ago';
        if ($time < 2592000) return floor($time/86400) . ' days ago';
        if ($time < 31104000) return floor($time/2592000) . ' months ago';
        return floor($time/31104000) . ' years ago';
    }
}

/**
 * Global notification helper functions
 * These can be called from anywhere in the application
 */

// Initialize notification service
function getNotificationService() {
    global $conn;
    return new NotificationService($conn);
}

// Quick notification creation functions
function notifyStudentEnrollment($teacher_id, $student_id, $course_id) {
    return getNotificationService()->notifyTeacherOfEnrollment($teacher_id, $student_id, $course_id);
}

function notifyCourseCompletion($teacher_id, $student_id, $course_id, $completion_percentage = 100) {
    return getNotificationService()->notifyTeacherOfCompletion($teacher_id, $student_id, $course_id, $completion_percentage);
}

function notifyWithdrawalUpdate($teacher_id, $withdrawal_id, $status, $amount) {
    return getNotificationService()->notifyTeacherOfWithdrawalUpdate($teacher_id, $withdrawal_id, $status, $amount);
}

function notifyReferralReward($teacher_id, $referral_id, $commission_amount, $referred_user_name = null) {
    return getNotificationService()->notifyTeacherOfReferralReward($teacher_id, $referral_id, $commission_amount, $referred_user_name);
}

function notifyCourseCreation($teacher_id, $course_id) {
    return getNotificationService()->notifyTeacherOfCourseCreation($teacher_id, $course_id);
}

function notifyVideoUpload($teacher_id, $video_id, $course_id) {
    return getNotificationService()->notifyTeacherOfVideoUpload($teacher_id, $video_id, $course_id);
}
?>
