<?php
/**
 * Panadite Academy - Session Manager
 * Handles all session operations securely and consistently
 */

class SessionManager {
    
    private static $instance = null;
    private static $isInitialized = false;
    
    // Session configuration
    const SESSION_LIFETIME = 7200; // 2 hours
    const SESSION_REGENERATE_INTERVAL = 300; // 5 minutes
    
    /**
     * Get singleton instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize session with secure settings
     */
    public function init() {
        if (self::$isInitialized) {
            return true;
        }
        
        // Don't start session if already started
        if (session_status() === PHP_SESSION_ACTIVE) {
            self::$isInitialized = true;
            return true;
        }
        
        try {
            // Configure session settings for security
            ini_set('session.cookie_httponly', 1);
            ini_set('session.use_only_cookies', 1);
            ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) ? 1 : 0);
            ini_set('session.cookie_samesite', 'Lax');
            ini_set('session.gc_maxlifetime', self::SESSION_LIFETIME);
            ini_set('session.cookie_lifetime', self::SESSION_LIFETIME);
            
            // Set session save path to a writable directory
            $temp_dir = sys_get_temp_dir();
            if (is_dir('/tmp') && is_writable('/tmp')) {
                $temp_dir = '/tmp';
            }
            session_save_path($temp_dir);
            
            // Start session
            if (!session_start()) {
                throw new Exception('Failed to start session');
            }
            
            // Initialize session security
            $this->initializeSecurity();
            
            self::$isInitialized = true;
            error_log('SessionManager: Session initialized successfully');
            return true;
            
        } catch (Exception $e) {
            error_log('SessionManager Error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Initialize session security measures
     */
    private function initializeSecurity() {
        // Check if session needs regeneration
        if (!isset($_SESSION['created_at'])) {
            $_SESSION['created_at'] = time();
            $_SESSION['last_regeneration'] = time();
            $this->regenerateId();
        } else {
            // Regenerate session ID periodically for security
            if (time() - $_SESSION['last_regeneration'] > self::SESSION_REGENERATE_INTERVAL) {
                $_SESSION['last_regeneration'] = time();
                $this->regenerateId();
            }
            
            // Check session lifetime
            if (time() - $_SESSION['created_at'] > self::SESSION_LIFETIME) {
                $this->destroy();
                throw new Exception('Session expired');
            }
        }
        
        // Store user agent and IP for security validation
        if (!isset($_SESSION['user_agent'])) {
            $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
            $_SESSION['ip_address'] = $_SERVER['REMOTE_ADDR'] ?? '';
        } else {
            // Validate session integrity
            $current_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
            $current_ip = $_SERVER['REMOTE_ADDR'] ?? '';
            
            if ($_SESSION['user_agent'] !== $current_agent || $_SESSION['ip_address'] !== $current_ip) {
                error_log('SessionManager: Security breach detected - session destroyed');
                $this->destroy();
                throw new Exception('Session security violation');
            }
        }
    }
    
    /**
     * Create user session after successful login
     */
    public function createUserSession($userData) {
        $this->init();
        
        // Clear any existing session data
        $this->clearUserData();
        
        // Regenerate session ID for security
        $this->regenerateId();
        
        // Set user session data
        $_SESSION['user_id'] = $userData['user_id'];
        $_SESSION['username'] = $userData['username'];
        $_SESSION['email'] = $userData['email'] ?? '';
        $_SESSION['first_name'] = $userData['first_name'] ?? '';
        $_SESSION['last_name'] = $userData['last_name'] ?? '';
        $_SESSION['user_role'] = $userData['user_role'];
        $_SESSION['is_logged_in'] = true;
        $_SESSION['login_time'] = time();
        $_SESSION['last_activity'] = time();
        
        error_log('SessionManager: User session created for: ' . $userData['username'] . ' (Role: ' . $userData['user_role'] . ')');
        
        return true;
    }
    
    /**
     * Check if user is logged in
     */
    public function isLoggedIn() {
        $this->init();
        
        if (!isset($_SESSION['is_logged_in']) || !$_SESSION['is_logged_in']) {
            return false;
        }
        
        // Check session timeout
        if (isset($_SESSION['last_activity'])) {
            if (time() - $_SESSION['last_activity'] > self::SESSION_LIFETIME) {
                $this->destroy();
                return false;
            }
            $_SESSION['last_activity'] = time();
        }
        
        return true;
    }
    
    /**
     * Get current user data
     */
    public function getUserData() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'user_id' => $_SESSION['user_id'] ?? null,
            'username' => $_SESSION['username'] ?? '',
            'email' => $_SESSION['email'] ?? '',
            'first_name' => $_SESSION['first_name'] ?? '',
            'last_name' => $_SESSION['last_name'] ?? '',
            'user_role' => $_SESSION['user_role'] ?? '',
            'login_time' => $_SESSION['login_time'] ?? null,
            'last_activity' => $_SESSION['last_activity'] ?? null
        ];
    }
    
    /**
     * Get user role
     */
    public function getUserRole() {
        return $_SESSION['user_role'] ?? null;
    }
    
    /**
     * Get user ID
     */
    public function getUserId() {
        return $_SESSION['user_id'] ?? null;
    }
    
    /**
     * Update user activity timestamp
     */
    public function updateActivity() {
        if ($this->isLoggedIn()) {
            $_SESSION['last_activity'] = time();
        }
    }
    
    /**
     * Clear user session data (but keep session alive)
     */
    public function clearUserData() {
        // Keep session security data but clear user data
        $keep_keys = ['created_at', 'last_regeneration', 'user_agent', 'ip_address'];
        $session_backup = [];
        
        foreach ($keep_keys as $key) {
            if (isset($_SESSION[$key])) {
                $session_backup[$key] = $_SESSION[$key];
            }
        }
        
        $_SESSION = $session_backup;
    }
    
    /**
     * Destroy session completely
     */
    public function destroy() {
        if (session_status() === PHP_SESSION_ACTIVE) {
            $_SESSION = [];
            
            // Delete session cookie
            if (ini_get("session.use_cookies")) {
                $params = session_get_cookie_params();
                setcookie(session_name(), '', time() - 42000,
                    $params["path"], $params["domain"],
                    $params["secure"], $params["httponly"]
                );
            }
            
            session_destroy();
        }
        
        self::$isInitialized = false;
        error_log('SessionManager: Session destroyed');
    }
    
    /**
     * Regenerate session ID
     */
    public function regenerateId() {
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_regenerate_id(true);
            error_log('SessionManager: Session ID regenerated');
        }
    }
    
    /**
     * Get redirect URL based on user role
     */
    public function getRoleBasedRedirect() {
        $role = $this->getUserRole();
        $base_url = $this->getBaseUrl();
        
        switch ($role) {
            case 'admin':
                return $base_url . '/admin/dashboard.php';
            case 'teacher':
                return $base_url . '/teacher/dashboard.php';
            case 'student':
                return $base_url . '/student/dashboard.php';
            default:
                return $base_url . '/auth/login.php';
        }
    }
    
    /**
     * Get base URL
     */
    private function getBaseUrl() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $path = dirname($_SERVER['SCRIPT_NAME']);
        
        // Clean up path
        $path = rtrim($path, '/');
        if (strpos($path, '/auth') !== false) {
            $path = dirname($path);
        }
        if (strpos($path, '/includes') !== false) {
            $path = dirname($path);
        }
        
        return $protocol . '://' . $host . $path;
    }
    
    /**
     * Set flash message
     */
    public function setFlash($type, $message) {
        $_SESSION['flash'][$type] = $message;
    }
    
    /**
     * Get and clear flash message
     */
    public function getFlash($type) {
        if (isset($_SESSION['flash'][$type])) {
            $message = $_SESSION['flash'][$type];
            unset($_SESSION['flash'][$type]);
            return $message;
        }
        return null;
    }
    
    /**
     * Debug session information
     */
    public function getDebugInfo() {
        return [
            'session_status' => session_status(),
            'session_id' => session_id(),
            'is_logged_in' => $this->isLoggedIn(),
            'user_data' => $this->getUserData(),
            'session_data' => $_SESSION ?? []
        ];
    }
}
