<?php
/**
 * Clean AI Grading Service - Fixed Implementation
 */

class AIGradingService {
    private $conn;
    private $debug;
    
    public function __construct($connection, $debug = false) {
        $this->conn = $connection;
        $this->debug = $debug;
    }
    
    /**
     * Main grading function
     */
    public function gradeAssignment($studentText, $memorandumData, $assignmentId = null) {
        try {
            // Debug parameters
            if ($this->debug) {
                error_log("DEBUG: gradeAssignment called");
                error_log("DEBUG: studentText type: " . gettype($studentText));
                error_log("DEBUG: memorandumData type: " . gettype($memorandumData));
                error_log("DEBUG: studentText length: " . strlen($studentText));
                error_log("DEBUG: memorandumData length: " . strlen($memorandumData));
            }
            
            // Validate input
            if (empty($studentText) || trim($studentText) === '') {
                throw new Exception('Empty submission text provided');
            }
            
            if (empty($memorandumData) || trim($memorandumData) === '') {
                throw new Exception('No memorandum content available for grading');
            }
            
            // Convert memorandum to plain text
            $memorandumText = $this->extractMemorandumText($memorandumData);
            
            // Debug extracted memorandum
            if ($this->debug) {
                error_log("DEBUG: Extracted memorandum length: " . strlen($memorandumText));
                error_log("DEBUG: Memorandum preview: " . substr($memorandumText, 0, 200));
            }
            
            // Calculate similarity between student text and memorandum
            $similarity = $this->calculateSimilarity($studentText, $memorandumText);
            
            // Calculate keyword coverage
            $keywordCoverage = $this->calculateKeywordCoverage($studentText, $memorandumText);
            
            // Calculate quality score
            $qualityScore = $this->calculateQualityScore($studentText);
            
            // Calculate final grade (weighted average)
            $finalGrade = ($similarity * 0.4) + ($keywordCoverage * 0.35) + ($qualityScore * 0.25);
            
            // Apply realistic scoring adjustments
            $finalGrade = $this->applyRealisticScoring($finalGrade, strlen($studentText), $keywordCoverage);
            
            // Debug final calculations
            if ($this->debug) {
                error_log("DEBUG: Similarity: $similarity");
                error_log("DEBUG: Keyword Coverage: $keywordCoverage");
                error_log("DEBUG: Quality Score: $qualityScore");
                error_log("DEBUG: Final Grade: $finalGrade");
            }
            
            return [
                'ai_score' => $finalGrade,
                'ai_feedback' => $this->generateFeedback($similarity, $keywordCoverage, $qualityScore),
                'similarity_score' => round($similarity, 2),
                'plagiarism_score' => round(max(0, 100 - $similarity), 2),
                'keyword_match_score' => round($keywordCoverage, 2),
                'structure_score' => round($qualityScore, 2),
                'quality_score' => round($qualityScore, 2),
                'review_needed' => ($finalGrade < 50 || $similarity < 30) ? 1 : 0,
                'ai_confidence' => $this->calculateConfidence([
                    'similarity' => $similarity,
                    'keyword_coverage' => $keywordCoverage,
                    'quality' => $qualityScore
                ])
            ];
            
        } catch (Exception $e) {
            if ($this->debug) {
                error_log('AI Grading error: ' . $e->getMessage());
            }
            throw $e;
        }
    }
    
    /**
     * Extract memorandum text from various formats
     */
    private function extractMemorandumText($memorandumData) {
        if (is_string($memorandumData)) {
            // Try to decode as JSON first
            $decodedData = json_decode($memorandumData, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decodedData)) {
                return $decodedData['full_text'] ?? $decodedData['content'] ?? $memorandumData;
            } else {
                return $memorandumData;
            }
        } else if (is_array($memorandumData)) {
            return $memorandumData['full_text'] ?? $memorandumData['content'] ?? (string)$memorandumData;
        } else {
            return (string)$memorandumData;
        }
    }
    
    /**
     * Process text and extract meaningful words
     */
    public function processText($text) {
        if (empty($text)) {
            return [];
        }
        
        // Clean and normalize text
        $text = strtolower($text);
        $text = preg_replace('/[^\w\s]/', ' ', $text);
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Split into words and filter out common stop words
        $words = explode(' ', trim($text));
        $stopWords = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'will', 'would', 'could', 'should', 'may', 'might', 'can', 'this', 'that', 'these', 'those'];
        
        $meaningfulWords = array_filter($words, function($word) use ($stopWords) {
            return strlen($word) > 2 && !in_array($word, $stopWords);
        });
        
        return array_values(array_unique($meaningfulWords));
    }
    
    /**
     * Calculate similarity between two texts
     */
    private function calculateSimilarity($text1, $text2) {
        $words1 = $this->processText($text1);
        $words2 = $this->processText($text2);
        
        if (empty($words1) || empty($words2)) {
            return 0;
        }
        
        $intersection = array_intersect($words1, $words2);
        $union = array_unique(array_merge($words1, $words2));
        
        if (empty($union)) {
            return 0;
        }
        
        $similarity = (count($intersection) / count($union)) * 100;
        
        // Apply conservative similarity scoring
        return min($similarity * 0.8, 65); // Cap at 65% to avoid unrealistic scores
    }
    
    /**
     * Calculate keyword coverage
     */
    private function calculateKeywordCoverage($studentText, $memorandumText) {
        $studentWords = $this->processText($studentText);
        $memorandumWords = $this->processText($memorandumText);
        
        if (empty($memorandumWords)) {
            return 0;
        }
        
        // Find important keywords (longer words are more important)
        $importantWords = array_filter($memorandumWords, function($word) {
            return strlen($word) > 4;
        });
        
        if (empty($importantWords)) {
            $importantWords = $memorandumWords;
        }
        
        $matchedWords = array_intersect($studentWords, $importantWords);
        $coverage = (count($matchedWords) / count($importantWords)) * 100;
        
        // Conservative keyword coverage
        return min($coverage * 0.75, 70); // Cap at 70%
    }
    
    /**
     * Calculate quality score based on structure and length
     */
    private function calculateQualityScore($text) {
        $score = 0;
        $length = strlen($text);
        
        // Length scoring (realistic expectations)
        if ($length > 2000) {
            $score += 25;
        } elseif ($length > 1000) {
            $score += 20;
        } elseif ($length > 500) {
            $score += 15;
        } elseif ($length > 200) {
            $score += 10;
        } else {
            $score += 5; // Very short submissions get low scores
        }
        
        // Sentence structure
        $sentences = preg_split('/[.!?]+/', $text);
        $sentenceCount = count(array_filter($sentences, 'trim'));
        
        if ($sentenceCount > 10) {
            $score += 20;
        } elseif ($sentenceCount > 5) {
            $score += 15;
        } else {
            $score += 8;
        }
        
        // Vocabulary diversity
        $words = $this->processText($text);
        $uniqueWords = array_unique($words);
        $diversity = empty($words) ? 0 : (count($uniqueWords) / count($words));
        
        $score += min($diversity * 30, 25);
        
        // Professional terminology bonus
        $professionalTerms = ['analysis', 'implementation', 'strategy', 'methodology', 'framework', 'approach', 'solution', 'evaluation', 'assessment', 'development'];
        $termCount = 0;
        foreach ($professionalTerms as $term) {
            if (stripos($text, $term) !== false) {
                $termCount++;
            }
        }
        
        $score += min($termCount * 3, 15);
        
        return min($score, 80); // Cap quality score at 80
    }
    
    /**
     * Apply realistic scoring adjustments
     */
    private function applyRealisticScoring($baseScore, $textLength, $keywordCoverage) {
        // Conservative scoring - avoid perfect scores
        $adjustedScore = $baseScore * 0.85;
        
        // Penalty for very short content
        if ($textLength < 500) {
            $adjustedScore *= 0.7;
        }
        
        // Bonus for comprehensive coverage
        if ($keywordCoverage > 40 && $textLength > 1500) {
            $adjustedScore *= 1.1;
        }
        
        // Ensure realistic range (20-75%)
        $adjustedScore = max(20, min($adjustedScore, 75));
        
        return round($adjustedScore, 2);
    }
    
    /**
     * Generate educational feedback
     */
    private function generateFeedback($similarity, $keywordCoverage, $qualityScore) {
        $feedback = [];
        
        // Similarity feedback
        if ($similarity < 25) {
            $feedback[] = "Your response shows limited alignment with the expected content. Review the assignment requirements and key concepts.";
        } elseif ($similarity < 40) {
            $feedback[] = "Your response partially addresses the assignment requirements. Consider including more relevant details and key terms.";
        } else {
            $feedback[] = "Your response demonstrates good understanding of the assignment requirements.";
        }
        
        // Keyword coverage feedback
        if ($keywordCoverage < 30) {
            $feedback[] = "Include more subject-specific terminology and key concepts from the course material.";
        } elseif ($keywordCoverage < 50) {
            $feedback[] = "Good use of relevant terminology. Consider incorporating additional key concepts for comprehensive coverage.";
        } else {
            $feedback[] = "Excellent use of subject-specific terminology and key concepts.";
        }
        
        // Quality feedback
        if ($qualityScore < 40) {
            $feedback[] = "Expand your response with more detailed analysis and examples. Ensure proper structure and organization.";
        } elseif ($qualityScore < 60) {
            $feedback[] = "Your response shows good structure. Consider adding more depth to your analysis.";
        } else {
            $feedback[] = "Well-structured response with good depth of analysis.";
        }
        
        // Additional suggestions
        $feedback[] = "For improvement, focus on: comprehensive coverage of topics, clear explanations, and practical examples.";
        
        return implode(" ", $feedback);
    }
    
    /**
     * Calculate confidence score
     */
    private function calculateConfidence($scores) {
        $avgScore = ($scores['similarity'] + $scores['keyword_coverage'] + $scores['quality']) / 3;
        
        // Higher confidence for more balanced scores
        $balance = 100 - (max($scores) - min($scores));
        $confidence = ($avgScore * 0.7) + ($balance * 0.3);
        
        return round(min($confidence, 95), 2); // Cap confidence at 95%
    }
}
?>
