<?php
class PHPAIGradingService {
    private $conn;
    private $debug;
    
    public function __construct($database_connection, $debug_mode = false) {
        $this->conn = $database_connection;
        $this->debug = $debug_mode;
    }
    
    public function gradeAssignment($studentText, $memorandumText, $assignmentId) {
        if (empty($memorandumText)) {
            throw new Exception("No memorandum content provided for assignment");
        }
        
        return $this->calculateRealisticScore($studentText, $memorandumText);
    }
    
    private function calculateRealisticScore($studentText, $memorandumText) {
        $studentWords = str_word_count(strtolower($studentText), 1);
        $memoWords = str_word_count(strtolower($memorandumText), 1);
        
        // Keyword overlap (conservative)
        $commonWords = array_intersect($studentWords, $memoWords);
        $keywordScore = min((count($commonWords) / max(count($memoWords), 1)) * 60, 25);
        
        // Content length scoring
        $textLength = strlen($studentText);
        if ($textLength < 400) {
            $lengthScore = ($textLength / 400) * 15;
        } elseif ($textLength < 1000) {
            $lengthScore = 15 + (($textLength - 400) / 600) * 15;
        } else {
            $lengthScore = 30;
        }
        
        // Domain terminology
        $cyberTerms = [
            'basic' => ['security', 'cyber', 'firewall', 'password', 'network', 'attack'],
            'intermediate' => ['threat', 'vulnerability', 'encryption', 'malware', 'authentication', 'intrusion'],
            'advanced' => ['cryptography', 'forensics', 'penetration', 'apt', 'siem', 'nist', 'pki', 'segmentation']
        ];
        
        $basicCount = $intCount = $advCount = 0;
        foreach ($studentWords as $word) {
            if (in_array($word, $cyberTerms['basic'])) $basicCount++;
            if (in_array($word, $cyberTerms['intermediate'])) $intCount++;
            if (in_array($word, $cyberTerms['advanced'])) $advCount++;
        }
        
        $domainScore = min(($basicCount * 1 + $intCount * 2 + $advCount * 3), 20);
        
        // Quality scoring
        $uniqueWords = array_unique($studentWords);
        $diversityRatio = count($uniqueWords) / max(count($studentWords), 1);
        $qualityScore = $diversityRatio * 15;
        
        // Structure penalty
        $structurePenalty = 0;
        if ($textLength < 300) $structurePenalty = -10;
        elseif ($textLength < 600) $structurePenalty = -5;
        
        $baseScore = $keywordScore + $lengthScore + $domainScore + $qualityScore + $structurePenalty;
        
        // Realistic caps
        if ($textLength < 500) $maxScore = 35;
        elseif ($textLength < 800) $maxScore = 55;
        elseif ($textLength < 1200) $maxScore = 75;
        else $maxScore = 90;
        
        $finalScore = min(max($baseScore, 10), $maxScore);
        
        return [
            'ai_score' => round($finalScore, 1),
            'similarity_score' => round(min($keywordScore * 1.5, 40), 1),
            'keyword_match_score' => round(min(($basicCount + $intCount * 2 + $advCount * 3) * 3, 100), 1),
            'quality_score' => round(min($qualityScore * 2 + $lengthScore * 0.8, 80), 1),
            'structure_score' => round(max(50 + $structurePenalty, 20), 1),
            'plagiarism_score' => 5,
            'ai_confidence' => 85,
            'review_needed' => 0,
            'ai_feedback' => $this->generateFeedback($finalScore, $textLength, $basicCount + $intCount + $advCount)
        ];
    }
    
    private function generateFeedback($score, $length, $termCount) {
        if ($score >= 70) {
            return "Excellent analysis demonstrating comprehensive understanding of cyber security concepts. Your submission shows strong knowledge of key areas and technical depth.";
        } elseif ($score >= 50) {
            return "Good understanding of cyber security fundamentals with room for improvement. Consider expanding on technical details and providing more specific examples.";
        } elseif ($score >= 30) {
            return "Basic understanding demonstrated but needs significant improvement. Include more technical terminology, expand content length, and provide detailed analysis.";
        } else {
            return "Limited understanding shown. Please review course materials and provide more comprehensive analysis with relevant cyber security concepts and terminology.";
        }
    }
}
?>