<?php
/**
 * API Session Fix
 * 
 * This file solves the common PHP session issue where API endpoints 
 * cannot access the same session as the main application.
 */

// Error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start with a debug log
error_log('API Session Fix: Starting session repair');

// 1. First ensure correct session parameters
ini_set('session.cookie_path', '/');       // Ensure cookie is available across entire domain
ini_set('session.use_cookies', 1);         // Use cookies to store session ID
ini_set('session.use_only_cookies', 1);    // Only use cookies, not URL parameters
ini_set('session.use_strict_mode', 1);     // Only use session IDs created by PHP
ini_set('session.use_trans_sid', 0);       // Don't pass session ID via URL
ini_set('session.cookie_httponly', 1);     // Protect from XSS
ini_set('session.cookie_samesite', 'Lax'); // Prevent CSRF
ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on'); // HTTPS only if available

// 2. Get the existing cookie
$cookie_name = 'PHPSESSID'; // Default PHP session cookie name
$existing_session_id = $_COOKIE[$cookie_name] ?? null;

// Log session cookie info
error_log('Cookie before fix: ' . json_encode($_COOKIE));

// 3. Special handling if we have a cookie but no active session
if ($existing_session_id && session_status() == PHP_SESSION_NONE) {
    // Try to resume the existing session
    session_id($existing_session_id);
    error_log('API Session Fix: Using existing session ID: ' . $existing_session_id);
}

// 4. Session garbage collection settings
ini_set('session.gc_maxlifetime', 7200); // 2 hours
ini_set('session.gc_probability', 1);
ini_set('session.gc_divisor', 100);      // 1% chance of collection

// 5. Set session save path if needed
$session_save_path = session_save_path();
if (empty($session_save_path) || !is_writable($session_save_path)) {
    $temp_dir = sys_get_temp_dir();
    if (is_dir($temp_dir) && is_writable($temp_dir)) {
        session_save_path($temp_dir);
        error_log('API Session Fix: Changed session save path to: ' . $temp_dir);
    }
}

// 6. Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
    error_log('API Session Fix: Session started');
}

// 7. Debug info
$debug = [
    'session_id' => session_id(),
    'session_name' => session_name(),
    'session_save_path' => session_save_path(),
    'cookie_path' => ini_get('session.cookie_path'),
    'session_status' => session_status(),
    'user_id' => $_SESSION['user_id'] ?? null,
    'user_role' => $_SESSION['user_role'] ?? null
];

error_log('API Session Fix: ' . json_encode($debug));

// Return debug info for potential use
return $debug;
?>
