<?php
/**
 * Course Completion Certificate Automation
 * This file should be included/called whenever a student completes a course
 * Also handles API requests for certificate generation triggering
 */

require_once __DIR__ . '/../config/db_connect.php';
require_once __DIR__ . '/../admin/auto_certificate_generator.php';
require_once __DIR__ . '/../auth/functions.php';

// Handle API requests for certificate generation
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Enable CORS if needed
    header('Content-Type: application/json');
    
    try {
        // Check authentication using the correct session variable names
        require_once __DIR__ . '/../auth/functions.php';
        
        // The session uses 'user_role' not 'role'
        if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role'])) {
            throw new Exception('User not logged in');
        }
        
        if ($_SESSION['user_role'] !== 'student') {
            throw new Exception('Access denied - not a student');
        }
        
        $student_id = $_SESSION['user_id'];
        
        // Get POST data
        $input = json_decode(file_get_contents('php://input'), true);
        $action = $input['action'] ?? '';
        $course_id = intval($input['course_id'] ?? 0);
        
        if (!$course_id) {
            throw new Exception('Course ID is required');
        }
        
        switch ($action) {
            case 'check_completion':
                // Check if student is eligible and trigger certificate generation
                $completionResult = autoCheckCourseCompletion($student_id, $course_id);
                
                if ($completionResult['success']) {
                    echo json_encode([
                        'success' => true,
                        'message' => 'Certificate generated successfully!',
                        'data' => $completionResult
                    ]);
                } else {
                    // Even if not "completed" in traditional sense, check if eligible for certificate
                    require_once __DIR__ . '/course_progress_service_quiz_only.php';
                    $progressService = new QuizOnlyProgressService($conn);
                    $eligibility = $progressService->isCertificateEligible($student_id, $course_id);
                    
                    if ($eligibility['eligible']) {
                        // Student is eligible, force certificate generation
                        $certResult = onCourseCompletion($student_id, $course_id);
                        echo json_encode([
                            'success' => $certResult['success'],
                            'message' => $certResult['success'] ? 'Certificate generated successfully!' : $certResult['message']
                        ]);
                    } else {
                        echo json_encode([
                            'success' => false,
                            'message' => 'Student not yet eligible for certificate. Current: ' . $eligibility['current_percentage'] . '%, Required: ' . $eligibility['minimum_required'] . '%'
                        ]);
                    }
                }
                break;
                
            default:
                throw new Exception('Invalid action specified');
        }
        
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
    exit;
}

/**
 * Function to call when student completes a course
 * This triggers automatic certificate generation and email delivery
 */
function onCourseCompletion($student_id, $course_id, $completion_date = null) {
    global $conn;
    
    try {
        // Initialize the certificate generator
        $certGenerator = new AutoCertificateGenerator($conn);
        
        // Generate certificate automatically
        $result = $certGenerator->generateCertificateOnCompletion($student_id, $course_id, $completion_date);
        
        // Log the result
        error_log("Certificate automation for student {$student_id}, course {$course_id}: " . 
                 ($result['success'] ? 'SUCCESS' : 'FAILED - ' . $result['message']));
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Certificate automation error: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Update course enrollment status and trigger certificate generation
 */
function markCourseCompleted($student_id, $course_id, $completion_date = null) {
    global $conn;
    
    try {
        $completion_date = $completion_date ?: date('Y-m-d H:i:s');
        
        // Update course enrollment status
        $updateQuery = "
            UPDATE course_enrollments 
            SET completion_status = 'completed', 
                completion_date = ?, 
                updated_at = NOW() 
            WHERE user_id = ? AND course_id = ?
        ";
        
        $stmt = $conn->prepare($updateQuery);
        $stmt->bind_param("sii", $completion_date, $student_id, $course_id);
        
        if ($stmt->execute()) {
            // Trigger automatic certificate generation
            $certResult = onCourseCompletion($student_id, $course_id, $completion_date);
            
            return [
                'success' => true,
                'message' => 'Course marked as completed',
                'certificate' => $certResult
            ];
        } else {
            return ['success' => false, 'message' => 'Failed to update course status'];
        }
        
    } catch (Exception $e) {
        error_log("Course completion error: " . $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Check if student has completed all course requirements
 * QUIZ-ONLY SYSTEM: Only quizzes count for course completion
 * Assignments and other activities are excluded from completion calculations
 */
function checkCourseCompletion($student_id, $course_id) {
    global $conn;
    
    try {
        // QUIZ-ONLY COMPLETION: Only check quiz performance for course completion
        // Videos, assignments, and other activities are NOT required for completion
        
        // Get course quiz requirements (with flexible table structure)
        $courseQuery = "
            SELECT c.course_id, 
                   COALESCE(c.course_title, c.title) as course_title,
                   COUNT(DISTINCT q.quiz_id) as total_quizzes,
                   AVG(q.passing_score) as avg_passing_score
            FROM courses c
            LEFT JOIN quizzes q ON c.course_id = q.course_id
            WHERE c.course_id = ?
            GROUP BY c.course_id
        ";
        
        $stmt = $conn->prepare($courseQuery);
        if (!$stmt) {
            error_log("Course query prepare failed: " . $conn->error);
            return ['completed' => false, 'message' => 'Database error: ' . $conn->error];
        }
        
        $stmt->bind_param("i", $course_id);
        if (!$stmt->execute()) {
            error_log("Course query execute failed: " . $stmt->error);
            return ['completed' => false, 'message' => 'Query execution failed'];
        }
        
        $course = $stmt->get_result()->fetch_assoc();
        
        if (!$course) {
            return ['completed' => false, 'message' => 'Course not found'];
        }
        
        // If no quizzes exist, course is automatically completed
        if ($course['total_quizzes'] == 0) {
            return [
                'completed' => true,
                'progress' => [
                    'quizzes_passed' => 0,
                    'total_quizzes' => 0,
                    'completion_method' => 'no_quizzes_required'
                ]
            ];
        }
        
        // Check student quiz performance - QUIZ-ONLY SYSTEM
        $progressQuery = "
            SELECT 
                COUNT(DISTINCT qa.quiz_id) as quizzes_attempted,
                COUNT(DISTINCT CASE WHEN qa.score >= q.passing_score THEN qa.quiz_id END) as quizzes_passed,
                AVG(qa.score) as average_score,
                MIN(qa.score) as lowest_score,
                MAX(qa.score) as highest_score
            FROM quiz_attempts qa
            JOIN quizzes q ON qa.quiz_id = q.quiz_id
            WHERE qa.student_id = ? AND q.course_id = ?
            AND qa.attempt_id IN (
                SELECT MAX(attempt_id) 
                FROM quiz_attempts 
                WHERE student_id = ? AND quiz_id = q.quiz_id
                GROUP BY quiz_id
            )
        ";
        
        $stmt = $conn->prepare($progressQuery);
        $stmt->bind_param("iii", $student_id, $course_id, $student_id);
        $stmt->execute();
        $progress = $stmt->get_result()->fetch_assoc();
        
        // QUIZ-ONLY COMPLETION CRITERIA:
        // Student must pass ALL quizzes in the course to complete
        $quizzes_completed = ($progress['quizzes_passed'] >= $course['total_quizzes']);
        
        return [
            'completed' => $quizzes_completed,
            'progress' => [
                'quizzes_attempted' => $progress['quizzes_attempted'],
                'quizzes_passed' => $progress['quizzes_passed'],
                'total_quizzes' => $course['total_quizzes'],
                'average_score' => number_format($progress['average_score'] ?? 0, 1),
                'completion_method' => 'quiz_only_system',
                'completion_percentage' => $course['total_quizzes'] > 0 ? 
                    round(($progress['quizzes_passed'] / $course['total_quizzes']) * 100, 1) : 100
            ]
        ];
        
    } catch (Exception $e) {
        error_log("Quiz-only completion check error: " . $e->getMessage());
        return ['completed' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Automatically check and process course completion
 * Call this function whenever student completes a video, quiz, or other course activity
 */
function autoCheckCourseCompletion($student_id, $course_id) {
    $completionCheck = checkCourseCompletion($student_id, $course_id);
    
    if ($completionCheck['completed']) {
        // Check if already marked as completed
        global $conn;
        $checkQuery = "SELECT completion_status FROM course_enrollments WHERE user_id = ? AND course_id = ?";
        $stmt = $conn->prepare($checkQuery);
        $stmt->bind_param("ii", $student_id, $course_id);
        $stmt->execute();
        $enrollment = $stmt->get_result()->fetch_assoc();
        
        if ($enrollment && $enrollment['completion_status'] !== 'completed') {
            // Mark as completed and generate certificate
            return markCourseCompleted($student_id, $course_id);
        }
    }
    
    return ['success' => false, 'message' => 'Course not yet completed'];
}
?>
