<?php
/**
 * Professional PDF Certificate Generator for Panadite Academy
 * Uses TCPDF library for high-quality PDF generation
 */

// Include TCPDF library
require_once __DIR__ . '/../vendor/tcpdf/tcpdf.php';

class CertificatePDFGenerator {
    
    private $conn;
    private $academy_name = "PANADITE ACADEMY";
    private $academy_tagline = "Excellence in Digital Education";
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
    }
    
    /**
     * Generate a professional certificate PDF
     */
    public function generateCertificate($user_id, $course_id, $download = false) {
        
        // Validate course completion first
        if (!$this->isStudentEligibleForCertificate($user_id, $course_id)) {
            throw new Exception("Student has not completed this course yet.");
        }
        
        // Get student and course data
        $student_data = $this->getStudentData($user_id);
        $course_data = $this->getCourseData($course_id);
        $completion_data = $this->getCompletionData($user_id, $course_id);
        
        if (!$student_data || !$course_data) {
            throw new Exception("Invalid student or course data.");
        }
        
        // Create new PDF document
        $pdf = new TCPDF('L', 'mm', 'A4', true, 'UTF-8', false); // Landscape orientation
        
        // Set document information
        $pdf->SetCreator('Panadite Academy');
        $pdf->SetAuthor('Panadite Academy');
        $pdf->SetTitle('Certificate of Completion - ' . $course_data['title']);
        $pdf->SetSubject('Course Completion Certificate');
        $pdf->SetKeywords('Certificate, Course, Completion, Education');
        
        // Remove header and footer
        $pdf->setPrintHeader(false);
        $pdf->setPrintFooter(false);
        
        // Set margins
        $pdf->SetMargins(15, 15, 15);
        $pdf->SetAutoPageBreak(false, 0);
        
        // Add a page
        $pdf->AddPage();
        
        // Generate certificate content
        $this->generateCertificateContent($pdf, $student_data, $course_data, $completion_data);
        
        // Generate filename
        $safe_course_title = preg_replace('/[^A-Za-z0-9\-_]/', '_', $course_data['title']);
        $safe_student_name = preg_replace('/[^A-Za-z0-9\-_]/', '_', $student_data['first_name'] . '_' . $student_data['last_name']);
        $filename = "Certificate_{$safe_course_title}_{$safe_student_name}_" . date('Y-m-d') . ".pdf";
        
        // Save certificate record to database
        $this->saveCertificateRecord($user_id, $course_id, $filename);
        
        if ($download) {
            // Force download
            $pdf->Output($filename, 'D');
            exit;
        } else {
            // Return PDF string for further processing
            return $pdf->Output($filename, 'S');
        }
    }
    
    /**
     * Generate the actual certificate content
     */
    private function generateCertificateContent($pdf, $student_data, $course_data, $completion_data) {
        
        // Set background color (light cream)
        $pdf->SetFillColor(255, 253, 250);
        $pdf->Rect(0, 0, 297, 210, 'F');
        
        // Add decorative border
        $pdf->SetLineWidth(3);
        $pdf->SetDrawColor(70, 115, 223); // Panadite blue
        $pdf->Rect(10, 10, 277, 190, 'D');
        
        // Inner border
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(200, 200, 200);
        $pdf->Rect(15, 15, 267, 180, 'D');
        
        // Academy Logo Area (if logo exists)
        $logo_y = 25;
        $pdf->SetXY(148.5 - 15, $logo_y); // Center horizontally
        
        // Academy Name - Large Header
        $pdf->SetFont('helvetica', 'B', 32);
        $pdf->SetTextColor(70, 115, 223); // Panadite blue
        $pdf->Cell(0, 15, $this->academy_name, 0, 1, 'C');
        
        // Academy Tagline
        $pdf->SetFont('helvetica', 'I', 14);
        $pdf->SetTextColor(100, 100, 100);
        $pdf->Cell(0, 8, $this->academy_tagline, 0, 1, 'C');
        
        // Certificate Title
        $pdf->Ln(10);
        $pdf->SetFont('helvetica', 'B', 28);
        $pdf->SetTextColor(50, 50, 50);
        $pdf->Cell(0, 15, 'CERTIFICATE OF COMPLETION', 0, 1, 'C');
        
        // Decorative line
        $pdf->SetLineWidth(2);
        $pdf->SetDrawColor(70, 115, 223);
        $pdf->Line(80, $pdf->GetY() + 5, 217, $pdf->GetY() + 5);
        
        // "This is to certify that" text
        $pdf->Ln(15);
        $pdf->SetFont('helvetica', '', 16);
        $pdf->SetTextColor(80, 80, 80);
        $pdf->Cell(0, 8, 'This is to certify that', 0, 1, 'C');
        
        // Student Name - Main Focus
        $pdf->Ln(8);
        $pdf->SetFont('helvetica', 'B', 36);
        $pdf->SetTextColor(0, 0, 0);
        $student_name = trim($student_data['first_name'] . ' ' . $student_data['last_name']);
        $pdf->Cell(0, 15, $student_name, 0, 1, 'C');
        
        // Add underline under name
        $name_width = $pdf->GetStringWidth($student_name);
        $start_x = (297 - $name_width) / 2;
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(0, 0, 0);
        $pdf->Line($start_x, $pdf->GetY() - 2, $start_x + $name_width, $pdf->GetY() - 2);
        
        // "Has successfully completed" text
        $pdf->Ln(8);
        $pdf->SetFont('helvetica', '', 16);
        $pdf->SetTextColor(80, 80, 80);
        $pdf->Cell(0, 8, 'has successfully completed the course', 0, 1, 'C');
        
        // Course Title
        $pdf->Ln(8);
        $pdf->SetFont('helvetica', 'B', 24);
        $pdf->SetTextColor(70, 115, 223);
        $course_title = $course_data['title'];
        $pdf->Cell(0, 12, $course_title, 0, 1, 'C');
        
        // Course description (if available)
        if (!empty($course_data['description'])) {
            $pdf->Ln(5);
            $pdf->SetFont('helvetica', 'I', 12);
            $pdf->SetTextColor(100, 100, 100);
            $description = substr($course_data['description'], 0, 150);
            if (strlen($course_data['description']) > 150) $description .= '...';
            $pdf->Cell(0, 6, $description, 0, 1, 'C');
        }
        
        // Completion Date
        $pdf->Ln(15);
        $pdf->SetFont('helvetica', '', 14);
        $pdf->SetTextColor(80, 80, 80);
        $completion_date = date('F j, Y', strtotime($completion_data['completion_date']));
        $pdf->Cell(0, 8, "Date of Completion: {$completion_date}", 0, 1, 'C');
        
        // Footer Section - Signatures/Seals
        $footer_y = 160;
        $pdf->SetXY(50, $footer_y);
        
        // Academy Seal/Signature Area
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->Cell(80, 6, '', 0, 1, 'C'); // Placeholder for signature
        $pdf->SetX(50);
        $pdf->Cell(80, 6, 'Panadite Academy', 0, 0, 'C');
        
        // Date issued
        $pdf->SetXY(167, $footer_y);
        $pdf->Cell(80, 6, '', 0, 1, 'C'); // Placeholder for date
        $pdf->SetX(167);
        $pdf->Cell(80, 6, 'Date Issued: ' . date('F j, Y'), 0, 0, 'C');
        
        // Certificate ID (unique identifier)
        $certificate_id = 'CERT-' . strtoupper(substr(md5($student_data['user_id'] . $course_data['course_id'] . time()), 0, 8));
        $pdf->SetXY(15, 185);
        $pdf->SetFont('helvetica', '', 10);
        $pdf->SetTextColor(150, 150, 150);
        $pdf->Cell(0, 6, "Certificate ID: {$certificate_id}", 0, 1, 'L');
        
        // Website URL
        $pdf->SetXY(15, 190);
        $pdf->Cell(0, 6, "Verify at: panaditeacademy.com", 0, 1, 'L');
        
        return $certificate_id;
    }
    
    /**
     * Check if student is eligible for certificate (course completed)
     */
    private function isStudentEligibleForCertificate($user_id, $course_id) {
        
        // Method 1: Check video progress completion
        $video_completion_query = "
            SELECT 
                COUNT(v.video_id) as total_videos,
                COUNT(vp.video_id) as watched_videos,
                (COUNT(vp.video_id) / COUNT(v.video_id)) * 100 as completion_percentage
            FROM videos v
            LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.user_id = ?
            WHERE v.course_id = ?
            GROUP BY v.course_id
        ";
        
        $stmt = $this->conn->prepare($video_completion_query);
        if ($stmt) {
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($row = $result->fetch_assoc()) {
                $completion_percentage = floatval($row['completion_percentage']);
                
                // Student must complete at least 80% of videos
                if ($completion_percentage >= 80.0) {
                    return true;
                }
            }
            $stmt->close();
        }
        
        // Method 2: Check enrollment and course access
        $enrollment_query = "
            SELECT e.enrollment_id, e.enrollment_date, c.title
            FROM enrollments e
            JOIN courses c ON e.course_id = c.course_id
            WHERE e.user_id = ? AND e.course_id = ?
        ";
        
        $stmt = $this->conn->prepare($enrollment_query);
        if ($stmt) {
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                // Student is enrolled, check if course has videos or content
                $content_query = "SELECT COUNT(*) as video_count FROM videos WHERE course_id = ?";
                $content_stmt = $this->conn->prepare($content_query);
                if ($content_stmt) {
                    $content_stmt->bind_param("i", $course_id);
                    $content_stmt->execute();
                    $content_result = $content_stmt->get_result();
                    $content_row = $content_result->fetch_assoc();
                    
                    // If course has no videos, consider it complete upon enrollment
                    if ($content_row['video_count'] == 0) {
                        return true;
                    }
                }
            }
            $stmt->close();
        }
        
        return false;
    }
    
    /**
     * Get student data
     */
    private function getStudentData($user_id) {
        $query = "SELECT user_id, first_name, last_name, email FROM users WHERE user_id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        return null;
    }
    
    /**
     * Get course data
     */
    private function getCourseData($course_id) {
        $query = "SELECT course_id, title, description FROM courses WHERE course_id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        return null;
    }
    
    /**
     * Get completion data
     */
    private function getCompletionData($user_id, $course_id) {
        // Get enrollment date as completion date for now
        $query = "SELECT enrollment_date as completion_date FROM enrollments WHERE user_id = ? AND course_id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $data = $result->fetch_assoc();
            
            // If no enrollment date, use current date
            if (!$data) {
                return ['completion_date' => date('Y-m-d H:i:s')];
            }
            
            return $data;
        }
        return ['completion_date' => date('Y-m-d H:i:s')];
    }
    
    /**
     * Save certificate record to database
     */
    private function saveCertificateRecord($user_id, $course_id, $filename) {
        // Create certificates table if it doesn't exist
        $create_table = "
            CREATE TABLE IF NOT EXISTS student_certificates (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                course_id INT NOT NULL,
                certificate_id VARCHAR(50) NOT NULL,
                filename VARCHAR(255) NOT NULL,
                issued_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                status ENUM('active', 'revoked') DEFAULT 'active',
                UNIQUE KEY unique_user_course (user_id, course_id),
                FOREIGN KEY (user_id) REFERENCES users(user_id),
                FOREIGN KEY (course_id) REFERENCES courses(course_id)
            )
        ";
        $this->conn->query($create_table);
        
        // Generate certificate ID
        $certificate_id = 'CERT-' . strtoupper(substr(md5($user_id . $course_id . time()), 0, 8));
        
        // Insert or update certificate record
        $query = "
            INSERT INTO student_certificates (user_id, course_id, certificate_id, filename)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                certificate_id = VALUES(certificate_id),
                filename = VALUES(filename),
                issued_date = CURRENT_TIMESTAMP
        ";
        
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("iiss", $user_id, $course_id, $certificate_id, $filename);
            $stmt->execute();
            $stmt->close();
        }
        
        return $certificate_id;
    }
}

?>
