<?php
/**
 * DatabaseBackup Class
 * Handles database backup and export functionalities
 */
class DatabaseBackup {
    private $conn;
    private $tables = [];
    private $backupDir;
    private $format;
    private $filename;
    private $backupData = [];
    
    /**
     * Constructor
     * @param array $tables Tables to backup (empty for all tables)
     * @param string $format Export format (sql, csv, excel, pdf)
     */
    public function __construct($tables = [], $format = 'sql') {
        global $conn;
        $this->conn = $conn;
        $this->tables = !empty($tables) ? $tables : $this->getAllTables();
        $this->format = $format;
        
        // Create backup directory if it doesn't exist
        $this->backupDir = dirname(dirname(__DIR__)) . '/backups';
        if (!file_exists($this->backupDir)) {
            if (!mkdir($this->backupDir, 0755, true)) {
                throw new Exception("Failed to create backup directory at {$this->backupDir}. Please check permissions.");
            }
        }
        
        // Check if directory is writable
        if (!is_writable($this->backupDir)) {
            throw new Exception("Backup directory is not writable. Please check permissions.");
        }
        
        // Generate timestamp for filename
        $timestamp = date('Y-m-d_H-i-s');
        $tableStr = count($this->tables) > 3 
            ? 'full_backup' 
            : implode('_', array_slice($this->tables, 0, 3));
        
        $this->filename = "{$tableStr}_{$timestamp}";
    }
    
    /**
     * Get all tables in the database
     * @return array List of table names
     */
    private function getAllTables() {
        $tables = [];
        $result = $this->conn->query("SHOW TABLES");
        
        while ($row = $result->fetch_row()) {
            $tables[] = $row[0];
        }
        
        return $tables;
    }
    
    /**
     * Get table structure and data
     * @param string $table Table name
     * @return array Table structure and data
     */
    private function getTableData($table) {
        $tableData = [
            'name' => $table,
            'structure' => '',
            'data' => []
        ];
        
        // Get table structure
        $result = $this->conn->query("SHOW CREATE TABLE `$table`");
        $row = $result->fetch_row();
        $tableData['structure'] = $row[1];
        
        // Get table data
        $result = $this->conn->query("SELECT * FROM `$table`");
        
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $tableData['data'][] = $row;
            }
        }
        
        return $tableData;
    }
    
    /**
     * Create SQL backup
     * @return string Path to the backup file
     */
    public function createSqlBackup() {
        $output = "-- Database Backup - Generated on " . date("Y-m-d H:i:s") . "\n";
        $output .= "-- Tables: " . implode(', ', $this->tables) . "\n\n";
        
        foreach ($this->tables as $table) {
            $tableData = $this->getTableData($table);
            
            $output .= "-- Table structure for table `$table`\n";
            $output .= "DROP TABLE IF EXISTS `$table`;\n";
            $output .= $tableData['structure'] . ";\n\n";
            
            if (!empty($tableData['data'])) {
                $output .= "-- Data for table `$table`\n";
                
                foreach ($tableData['data'] as $row) {
                    $columns = array_map([$this->conn, 'real_escape_string'], array_keys($row));
                    $values = array_map(function($value) {
                        if (is_null($value)) return 'NULL';
                        return "'" . $this->conn->real_escape_string($value) . "'";
                    }, array_values($row));
                    
                    $output .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $values) . ");\n";
                }
                
                $output .= "\n";
            }
        }
        
        $filePath = "{$this->backupDir}/{$this->filename}.sql";
        
        // Try to write the file with error handling
        $result = @file_put_contents($filePath, $output);
        if ($result === false) {
            throw new Exception("Failed to write backup file to {$filePath}. Error: " . error_get_last()['message']);
        }
        
        // Verify file was created
        if (!file_exists($filePath)) {
            throw new Exception("Backup file was not created at {$filePath}");
        }
        
        return $filePath;
    }
    
    /**
     * Create CSV backup
     * @return string Path to the zip file containing CSVs
     */
    public function createCsvBackup() {
        $zipFile = "{$this->backupDir}/{$this->filename}.zip";
        $zip = new ZipArchive();
        
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive')) {
            throw new Exception("ZipArchive extension is not available. Cannot create CSV backup.");
        }
        
        // Try to create the zip file
        $result = $zip->open($zipFile, ZipArchive::CREATE);
        if ($result !== TRUE) {
            throw new Exception("Cannot create ZIP archive. Error code: $result. Please check directory permissions.");
        }
        
        // Track if we've added any files to the zip
        $filesAdded = 0;
        
        // Add each table as a CSV file
        foreach ($this->tables as $table) {
            $tableData = $this->getTableData($table);
            $output = '';
            
            if (!empty($tableData['data'])) {
                // Add header row
                $output .= implode(',', array_map(function($col) {
                    return '"' . str_replace('"', '""', $col) . '"';
                }, array_keys($tableData['data'][0]))) . "\n";
                
                // Add data rows
                foreach ($tableData['data'] as $row) {
                    $output .= implode(',', array_map(function($val) {
                        return '"' . str_replace('"', '""', $val) . '"';
                    }, $row)) . "\n";
                }
                
                // Add the CSV data to the zip file
                if ($zip->addFromString("{$table}.csv", $output)) {
                    $filesAdded++;
                } else {
                    error_log("Failed to add {$table}.csv to the zip archive");
                }
            } else {
                // Add an empty CSV with just headers if there's no data
                $result = $this->conn->query("SHOW COLUMNS FROM `$table`");
                if ($result && $result->num_rows > 0) {
                    $columns = [];
                    while ($row = $result->fetch_assoc()) {
                        $columns[] = $row['Field'];
                    }
                    
                    $output = implode(',', array_map(function($col) {
                        return '"' . str_replace('"', '""', $col) . '"';
                    }, $columns)) . "\n";
                    
                    if ($zip->addFromString("{$table}.csv", $output)) {
                        $filesAdded++;
                    }
                }
            }
        }
        
        // Close the zip file
        if (!$zip->close()) {
            throw new Exception("Failed to finalize the ZIP archive. Please check disk space and permissions.");
        }
        
        // Verify the zip file was created successfully
        if (!file_exists($zipFile)) {
            throw new Exception("ZIP archive was not created at {$zipFile}");
        }
        
        // Check if we added any files
        if ($filesAdded === 0) {
            throw new Exception("No table data was added to the backup. The ZIP file may be empty.");
        }
        
        return $zipFile;
    }
    
    /**
     * Create Excel backup using PhpSpreadsheet library
     * Note: This is a placeholder for the actual implementation
     * In a real implementation, you would include PhpSpreadsheet via composer
     * @return string Path to the excel file
     */
    public function createExcelBackup() {
        // Log that we're using a placeholder implementation
        error_log("Excel backup requested, but using CSV backup as a placeholder (PhpSpreadsheet not implemented)");
        
        try {
            // For now, return CSV backup
            return $this->createCsvBackup();
        } catch (Exception $e) {
            // Rewrap the exception to provide more context
            throw new Exception("Excel backup failed (using CSV fallback): " . $e->getMessage(), 0, $e);
        }
    }
    
    /**
     * Create PDF backup
     * Note: This is a placeholder for the actual implementation
     * In a real implementation, you would include a PDF library like TCPDF or FPDF
     * @return string Path to the pdf file
     */
    public function createPdfBackup() {
        // Placeholder implementation - would require TCPDF or FPDF library
        
        // For now, just create a simple text file with table data
        $output = "Database Backup - Generated on " . date("Y-m-d H:i:s") . "\n";
        $output .= "Tables: " . implode(', ', $this->tables) . "\n\n";
        
        foreach ($this->tables as $table) {
            $tableData = $this->getTableData($table);
            
            $output .= "Table: $table\n";
            $output .= "Records: " . count($tableData['data']) . "\n\n";
            
            if (!empty($tableData['data'])) {
                // Add header row
                $output .= implode("\t", array_keys($tableData['data'][0])) . "\n";
                $output .= str_repeat('-', 80) . "\n";
                
                // Add data rows (limit to first 100 for PDF readability)
                $counter = 0;
                foreach ($tableData['data'] as $row) {
                    $output .= implode("\t", $row) . "\n";
                    $counter++;
                    if ($counter >= 100) {
                        $output .= "... (more records not shown) ...\n";
                        break;
                    }
                }
                
                $output .= "\n\n";
            }
        }
        
        $filePath = "{$this->backupDir}/{$this->filename}.txt";
        
        // Try to write the file with error handling
        $result = @file_put_contents($filePath, $output);
        if ($result === false) {
            throw new Exception("Failed to write PDF backup file to {$filePath}. Error: " . error_get_last()['message']);
        }
        
        // Verify file was created
        if (!file_exists($filePath)) {
            throw new Exception("PDF backup file was not created at {$filePath}");
        }
        
        return $filePath;
    }
    
    /**
     * Create backup based on specified format
     * @return string Path to the backup file
     */
    public function createBackup() {
        error_log("Starting backup creation with format: {$this->format}");
        
        try {
            $filePath = '';
            $startTime = microtime(true);
            
            switch ($this->format) {
                case 'excel':
                    $filePath = $this->createExcelBackup();
                    break;
                case 'pdf':
                    $filePath = $this->createPdfBackup();
                    break;
                case 'csv':
                    $filePath = $this->createCsvBackup();
                    break;
                case 'sql':
                default:
                    $filePath = $this->createSqlBackup();
                    break;
            }
            
            $endTime = microtime(true);
            $executionTime = round($endTime - $startTime, 2);
            
            // Validate the returned file path
            if (empty($filePath)) {
                throw new Exception("Empty file path returned from {$this->format} backup method");
            }
            
            if (!file_exists($filePath)) {
                throw new Exception("Backup file was not created at the expected location: {$filePath}");
            }
            
            // Check file size
            $fileSize = filesize($filePath);
            if ($fileSize === 0) {
                throw new Exception("Backup file is empty (0 bytes): {$filePath}");
            }
            
            error_log("Backup successfully created at {$filePath} ({$this->formatBytes($fileSize)}) in {$executionTime} seconds");
            return $filePath;
            
        } catch (Exception $e) {
            error_log("Backup creation failed: " . $e->getMessage());
            throw $e; // Re-throw to be caught by the caller
        }
    }
    
    /**
     * Get estimated table size
     * @param string $table Table name
     * @return string Formatted table size
     */
    public static function getTableSize($table) {
        global $conn;
        $result = $conn->query("SHOW TABLE STATUS LIKE '$table'");
        $row = $result->fetch_assoc();
        
        // Data length + Index length
        $sizeBytes = $row['Data_length'] + $row['Index_length'];
        
        return self::formatBytes($sizeBytes);
    }
    
    /**
     * Format bytes to human-readable format
     * @param int $bytes Size in bytes
     * @param int $precision Number of decimal places
     * @return string Formatted size
     */
    public static function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= (1 << (10 * $pow));
        
        return round($bytes, $precision) . ' ' . $units[$pow];
    }
}
