<?php
session_start();
require_once 'includes/db.php'; // Use the correct database connection
require_once 'config/functions.php';

// Check if database connection is working - ADDED FROM INDEX.PHP
$db_available = !isset($GLOBALS['db_connection_failed']) || !$GLOBALS['db_connection_failed'];

// Get current user if logged in and if DB is available
$current_user = $db_available ? getCurrentUser() : null;

// Debug info
echo '<!-- COURSES PAGE DEBUGGING: DB Available: ' . ($db_available ? 'Yes' : 'No') . ' -->';

// Initialize courses array
$courses = [];

// Fetch courses from database - SIMPLER QUERY LIKE INDEX.PHP
if ($db_available) {
    try {
        // Check if database connection is working
        if (!isset($conn) || $conn->connect_error) {
            throw new Exception("Database connection failed: " . ($conn->connect_error ?? 'Unknown error'));
        }
        
        // Use a simple query to get all courses (including drafts)
        $sql = "SELECT * FROM courses ORDER BY course_id DESC LIMIT 50";
        $result = $conn->query($sql);
        
        $courses = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $courses[] = $row;
            }
        }
        
        // Add instructor info separately to prevent JOIN issues
        if (!empty($courses)) {
            echo '<!-- Found ' . count($courses) . ' courses. -->';
            
            foreach ($courses as &$course) {
            // Set default values
            $course['id'] = $course['course_id'] ?? 0;
            $course['title'] = $course['title'] ?? 'Untitled Course';
            
            // Try to get instructor name
            try {
                if (!empty($course['teacher_id'])) {
                    $teacherQuery = "SELECT first_name, last_name FROM users WHERE user_id = {$course['teacher_id']} LIMIT 1";
                    $teacherResult = $conn->query($teacherQuery);
                    if ($teacherResult && $teacherResult->num_rows > 0) {
                        $teacher = $teacherResult->fetch_assoc();
                        $course['instructor'] = $teacher['first_name'] . ' ' . $teacher['last_name'];
                    } else {
                        $course['instructor'] = 'Unknown Instructor';
                    }
                } else {
                    $course['instructor'] = 'Unknown Instructor';
                }
                
                // Try to get category name
                if (!empty($course['category_id'])) {
                    $categoryQuery = "SELECT category_name FROM categories WHERE category_id = {$course['category_id']} LIMIT 1";
                    $categoryResult = $conn->query($categoryQuery);
                    if ($categoryResult && $categoryResult->num_rows > 0) {
                        $category = $categoryResult->fetch_assoc();
                        $course['category'] = $category['category_name'];
                    } else {
                        $course['category'] = 'General';
                    }
                } else {
                    $course['category'] = 'General';
                }
                
                // Count students (enrollments)
                $enrollQuery = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = {$course['course_id']}";
                $enrollResult = $conn->query($enrollQuery);
                if ($enrollResult && $enrollResult->num_rows > 0) {
                    $enrollment = $enrollResult->fetch_assoc();
                    $course['students'] = $enrollment['count'];
                } else {
                    $course['students'] = 0;
                }
            } catch (Exception $innerEx) {
                // Fallback values if queries fail
                $course['instructor'] = 'Unknown Instructor';
                $course['category'] = 'General';
                $course['students'] = 0;
                error_log('Error getting course details: ' . $innerEx->getMessage());
            }
            
            // Default rating
            $course['rating'] = 4.5;
        }
    } else {
        echo '<!-- No courses found in database. SQL Error: ' . ($conn->error ?? 'None') . ' -->';
    }
} catch (Exception $e) {
        // Log error but don't display it
        error_log('Database error: ' . $e->getMessage());
        echo '<!-- Database error: ' . $e->getMessage() . ' -->';
        $courses = [];
    }
}

// Initialize featured courses array
$featuredCourses = [];

// Get featured courses (most viewed or most recent) - USING SIMPLER QUERY LIKE INDEX.PHP
if ($db_available) {
    try {
        // SIMPLER QUERY - just get basic courses, order by views/date
        $featuredSql = "SELECT * FROM courses ORDER BY views DESC, created_at DESC LIMIT 3";
        
        $featuredResult = $conn->query($featuredSql);
        $featuredCourses = [];
        
        if ($featuredResult && $featuredResult->num_rows > 0) {
            while ($row = $featuredResult->fetch_assoc()) {
                $featuredCourses[] = $row;
            }
            echo '<!-- Found ' . count($featuredCourses) . ' featured courses. -->';
        } else {
            echo '<!-- No featured courses found. SQL Error: ' . $conn->error . ' -->';
        }
        
        // Process each course to add additional info
        if (!empty($featuredCourses)) {
            foreach ($featuredCourses as &$course) {
                // Set default values
                $course['id'] = $course['course_id'] ?? 0;
                $course['title'] = $course['title'] ?? 'Untitled Course';
                $course['description'] = $course['description'] ?? '';
                
                // Try to get instructor name
                try {
                    if (!empty($course['teacher_id'])) {
                        $teacherQuery = "SELECT first_name, last_name FROM users WHERE user_id = {$course['teacher_id']} LIMIT 1";
                        $teacherResult = $conn->query($teacherQuery);
                        if ($teacherResult && $teacherResult->num_rows > 0) {
                            $teacher = $teacherResult->fetch_assoc();
                            $course['instructor'] = $teacher['first_name'] . ' ' . $teacher['last_name'];
                        } else {
                            $course['instructor'] = 'Unknown Instructor';
                        }
                    } else {
                        $course['instructor'] = 'Unknown Instructor';
                    }
                    
                    // Try to get category name
                    if (!empty($course['category_id'])) {
                        $categoryQuery = "SELECT category_name FROM categories WHERE category_id = {$course['category_id']} LIMIT 1";
                        $categoryResult = $conn->query($categoryQuery);
                        if ($categoryResult && $categoryResult->num_rows > 0) {
                            $category = $categoryResult->fetch_assoc();
                            $course['category'] = $category['category_name'];
                        } else {
                            $course['category'] = 'General';
                        }
                    } else {
                        $course['category'] = 'General';
                    }
                    
                    // Count students (enrollments)
                    $enrollQuery = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = {$course['course_id']}";
                    $enrollResult = $conn->query($enrollQuery);
                    if ($enrollResult && $enrollResult->num_rows > 0) {
                        $enrollment = $enrollResult->fetch_assoc();
                        $course['students'] = $enrollment['count'];
                    } else {
                        $course['students'] = 0;
                    }
                } catch (Exception $innerEx) {
                    // Fallback values if queries fail
                    $course['instructor'] = 'Unknown Instructor';
                    $course['category'] = 'General';
                    $course['students'] = 0;
                    error_log('Error getting featured course details: ' . $innerEx->getMessage());
                }
                
                // Default rating
                $course['rating'] = 4.5;
            }
        } else {
            echo '<!-- No featured courses found. SQL Error: ' . ($conn->error ?? 'None') . ' -->';
        }

    } catch (Exception $e) {
        // Log error but don't display it
        error_log('Database error in featured courses: ' . $e->getMessage());
        echo '<!-- Database error in featured courses: ' . $e->getMessage() . ' -->';
        $featuredCourses = [];
    }
}

// If no courses found, use empty array instead of demo data
if (empty($courses)) {
    $courses = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Courses | Panadite Academy</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">

    <style>
        /* Modern Courses Page Styles */
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: #f8fafc;
        }

        /* Hero Section */
        .courses-hero {
            background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);
            padding: 4rem 0 2rem;
            position: relative;
            overflow: hidden;
        }

        .courses-hero::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -20%;
            width: 40%;
            height: 200%;
            background: linear-gradient(135deg, rgba(37, 99, 235, 0.05) 0%, rgba(245, 158, 11, 0.05) 100%);
            transform: rotate(15deg);
            z-index: 1;
        }

        .hero-content {
            position: relative;
            z-index: 2;
        }

        .hero-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 1rem;
            line-height: 1.2;
        }

        .hero-subtitle {
            font-size: 1.1rem;
            color: #64748b;
            margin-bottom: 2rem;
            max-width: 600px;
        }

        .highlight-courses {
            color: #f59e0b;
            font-weight: 800;
        }

        /* Filter Section */
        .filters-section {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.08);
            padding: 2rem;
            margin-bottom: 2rem;
        }

        .filter-tabs {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            border-bottom: 1px solid #e5e7eb;
        }

        .filter-tab {
            padding: 0.75rem 1.5rem;
            border: none;
            background: none;
            color: #64748b;
            font-weight: 600;
            border-bottom: 2px solid transparent;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .filter-tab.active {
            color: #2563eb;
            border-bottom-color: #2563eb;
        }

        .filter-tab:hover {
            color: #2563eb;
        }

        .search-filter {
            position: relative;
        }

        .search-input {
            width: 100%;
            padding: 0.875rem 1rem 0.875rem 3rem;
            border: 1px solid #d1d5db;
            border-radius: 12px;
            font-size: 1rem;
            outline: none;
            transition: border-color 0.3s ease;
        }

        .search-input:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .search-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: #9ca3af;
            font-size: 1rem;
        }

        /* Course Cards */
        .courses-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 2rem;
        }

        .course-card {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            border: none;
        }

        .course-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12);
        }

        .course-thumbnail {
            position: relative;
            height: 200px;
            background: linear-gradient(135deg, #2563eb 0%, #3b82f6 100%);
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .course-thumbnail i {
            font-size: 3rem;
            color: rgba(255, 255, 255, 0.8);
        }

        .course-level-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: rgba(255, 255, 255, 0.9);
            color: #2563eb;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }

        .course-content {
            padding: 1.5rem;
        }

        .course-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 0.5rem;
            line-height: 1.4;
        }

        .course-instructor {
            color: #64748b;
            font-size: 0.9rem;
            margin-bottom: 1rem;
        }

        .course-meta {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1rem;
            font-size: 0.85rem;
            color: #64748b;
        }

        .course-meta-item {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }

        .course-price {
            font-size: 1.25rem;
            font-weight: 700;
            color: #f59e0b;
            margin-bottom: 1rem;
        }

        .course-cta {
            width: 100%;
            background: #2563eb;
            color: white;
            border: none;
            padding: 0.75rem 1rem;
            border-radius: 8px;
            font-weight: 600;
            transition: background 0.3s ease;
        }

        .course-cta:hover {
            background: #1d4ed8;
            color: white;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .hero-title {
                font-size: 2rem;
            }

            .filters-section {
                padding: 1rem;
            }

            .filter-tabs {
                flex-wrap: wrap;
                gap: 0.5rem;
            }

            .filter-tab {
                padding: 0.5rem 1rem;
                font-size: 0.9rem;
            }

            .courses-grid {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }
        }
            background-color: rgba(39, 107, 228, 0.4);
            top: -50px;
            right: 10%;
            animation: float-up-down 12s ease-in-out infinite alternate;
        }
        
        .floating-circle-2 {
            width: 100px;
            height: 100px;
            background-color: rgba(255, 169, 21, 0.4);
            bottom: -30px;
            left: 15%;
            animation: float-left-right 15s ease-in-out infinite alternate;
        }
        
        .floating-circle-3 {
            width: 80px;
            height: 80px;
            background-color: rgba(66, 133, 244, 0.4);
            top: 60%;
            right: 20%;
            animation: float-rotate 20s linear infinite;
        }
        
        .floating-shape {
            position: absolute;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 16px;
            opacity: 0.3;
            z-index: 1;
        }
        
        .floating-shape-1 {
            width: 60px;
            height: 60px;
            background-color: #276BE4;
            top: 30%;
            left: 10%;
            transform: rotate(15deg);
            animation: float-pulse 8s ease-in-out infinite alternate;
        }
        
        .floating-shape-2 {
            width: 50px;
            height: 50px;
            background-color: #FFA915;
            bottom: 20%;
            right: 5%;
            transform: rotate(-15deg);
            animation: float-scale 10s ease-in-out infinite alternate;
        }
        
        @keyframes float-up-down {
            0% { transform: translateY(0); }
            100% { transform: translateY(30px); }
        }
        
        @keyframes float-left-right {
            0% { transform: translateX(0); }
            100% { transform: translateX(40px); }
        }
        
        @keyframes float-rotate {
            0% { transform: rotate(0); }
            100% { transform: rotate(360deg); }
        }
        
        @keyframes float-pulse {
            0% { transform: scale(1) rotate(15deg); }
            50% { transform: scale(1.1) rotate(10deg); }
            100% { transform: scale(1) rotate(15deg); }
        }
        
        @keyframes float-scale {
            0% { transform: scale(1) rotate(-15deg); }
            50% { transform: scale(1.15) rotate(-10deg); }
            100% { transform: scale(1) rotate(-15deg); }
        }
        
        .banner-tag-cloud {
            margin-top: 1rem;
            gap: 8px;
        }
        
        .tag-item {
            padding: 4px 12px;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 20px;
            font-size: 0.85rem;
            color: #276BE4;
            font-weight: 500;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }
        
        .tag-item:hover {
            background-color: #276BE4;
            color: white;
            transform: translateY(-2px);
        }
        
        .banner-controls {
            transition: all 0.3s ease;
        }
        
        .banner-controls:hover {
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }
        
        /* Course Card Styles */
        .course-card {
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border-radius: 15px;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        
        .course-card:hover {
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
            transform: translateY(-5px);
        }
        
        .course-thumbnail-container {
            position: relative;
            height: 200px;
            overflow: hidden;
        }

        .svg-bg {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            width: 100%;
            position: relative;
        }
        
        .svg-icon-container {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 100%;
            height: 100%;
            transition: all 0.3s ease;
        }
        
        .svg-icon-container i {
            font-size: 4rem;
            transition: all 0.4s ease;
        }
        
        .course-card:hover .svg-icon-container i {
            transform: scale(1.1) rotate(5deg);
        }

        .card-title {
            min-height: 48px;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            color: #333;
        }

        .instructor-name {
            font-size: 0.9rem;
            color: #6c757d;
        }

        .rating-stars {
            font-size: 0.85rem;
        }
        
        .rating-value {
            font-weight: 600;
            color: #555;
        }

        .price-tag {
            font-weight: 700;
            font-size: 1.1rem;
            color: white;
        }
        
        .z-index-9 {
            z-index: 9;
        }
        
        .z-index-1 {
            z-index: 1;
        }
        
        .category-badge {
            font-size: 0.85rem;
            font-weight: 500;
            letter-spacing: 0.3px;
            padding: 5px 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }

        .course-banner {
            background: linear-gradient(135deg, rgba(39, 107, 228, 0.9) 0%, rgba(255, 169, 21, 0.8) 100%), url('https://images.unsplash.com/photo-1513258496099-48168024aec0?ixlib=rb-4.0.3&auto=format&fit=crop&w=1950&q=80') no-repeat center/cover;
            min-height: 280px;
            padding: 60px 0;
            border-radius: 0 0 30px 30px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            margin-bottom: 40px;
        }

        .breadcrumb-item a {
            color: rgba(255, 255, 255, 0.9);
            text-decoration: none;
            transition: color 0.3s ease;
            font-weight: 500;
        }

        .breadcrumb-item a:hover {
            color: #fff;
            text-shadow: 0 0 10px rgba(255,255,255,0.5);
        }

        .student-count {
            font-size: 0.85rem;
            color: #6c757d;
            border-radius: 20px;
            padding: 5px 10px;
            background-color: rgba(39, 107, 228, 0.08);
        }

        /* Fixing price display at bottom of card */
        .course-card .card-body {
            display: flex;
            flex-direction: column;
        }
        
        .course-card .bottom-info {
            margin-top: auto;
            border-top: 1px solid rgba(0,0,0,0.05);
            padding-top: 12px;
        }
    </style>
</head>
<body>
    <!-- Include the navbar -->
    <?php include 'includes/navbar.php'; ?>

    <!-- Hero Section -->
    <section class="courses-hero">
        <div class="container">
            <div class="hero-content">
                <h1 class="hero-title">
                    Explore Our <span class="highlight-courses">Courses</span>
                </h1>
                <p class="hero-subtitle">
                    Discover world-class courses designed by industry experts to help you master in-demand skills and advance your career.
                </p>
            </div>
        </div>
    </section>

    <!-- Main Content -->
    <section class="py-5">
        <div class="container">
            <!-- Filters Section -->
            <div class="filters-section">
                <div class="row">
                    <div class="col-lg-8">
                        <div class="filter-tabs">
                            <button class="filter-tab active" onclick="filterCourses('all')">All Courses</button>
                            <button class="filter-tab" onclick="filterCourses('beginner')">Beginner</button>
                            <button class="filter-tab" onclick="filterCourses('intermediate')">Intermediate</button>
                            <button class="filter-tab" onclick="filterCourses('advanced')">Advanced</button>
                        </div>
                    </div>
                    <div class="col-lg-4">
                        <div class="search-filter">
                            <i class="fas fa-search search-icon"></i>
                            <input type="text" class="search-input" placeholder="Search courses..." onkeyup="searchCourses(this.value)">
                        </div>
                    </div>
                </div>
            </div>

            <!-- Courses Grid -->
            <div class="courses-grid" id="coursesGrid">
                <?php foreach ($courses as $course): ?>
                    <?php
                    // Determine course level
                    $level = 'Beginner';
                    if (strpos(strtolower($course['title']), 'advanced') !== false) {
                        $level = 'Advanced';
                    } elseif (strpos(strtolower($course['title']), 'intermediate') !== false) {
                        $level = 'Intermediate';
                    }

                    // Determine icon based on course category/title
                    $icon = 'fa-graduation-cap';
                    $title_lower = strtolower($course['title']);
                    if (strpos($title_lower, 'web') !== false || strpos($title_lower, 'html') !== false || strpos($title_lower, 'css') !== false) {
                        $icon = 'fa-code';
                    } elseif (strpos($title_lower, 'business') !== false || strpos($title_lower, 'marketing') !== false) {
                        $icon = 'fa-chart-line';
                    } elseif (strpos($title_lower, 'design') !== false || strpos($title_lower, 'ui') !== false || strpos($title_lower, 'ux') !== false) {
                        $icon = 'fa-paint-brush';
                    } elseif (strpos($title_lower, 'data') !== false || strpos($title_lower, 'analytics') !== false) {
                        $icon = 'fa-database';
                    }

                    // Calculate price display
                    $price = isset($course['price']) ? $course['price'] : '0';
                    $price_display = $price == '0' || $price == '' ? 'Free' : '$' . number_format($price, 0);
                    ?>
                    
                    <div class="course-card" data-level="<?php echo strtolower($level); ?>" data-title="<?php echo strtolower($course['title']); ?>">
                        <div class="course-thumbnail">
                            <i class="fas <?php echo $icon; ?>"></i>
                            <div class="course-level-badge"><?php echo $level; ?></div>
                        </div>
                        <div class="course-content">
                            <h3 class="course-title"><?php echo htmlspecialchars($course['title']); ?></h3>
                            <div class="course-instructor">
                                <i class="fas fa-user-circle me-1"></i>
                                <?php echo htmlspecialchars($course['instructor']); ?>
                            </div>
                            <div class="course-meta">
                                <div class="course-meta-item">
                                    <i class="fas fa-clock"></i>
                                    <span><?php echo isset($course['duration']) ? $course['duration'] : '6 weeks'; ?></span>
                                </div>
                                <div class="course-meta-item">
                                    <i class="fas fa-users"></i>
                                    <span><?php echo isset($course['enrolled']) ? $course['enrolled'] : rand(50, 300); ?> students</span>
                                </div>
                            </div>
                            <div class="course-price"><?php echo $price_display; ?></div>
                            <button class="course-cta" onclick="window.location.href='/course.php?id=<?php echo $course['course_id']; ?>'">
                                View Course
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <?php if (empty($courses)): ?>
            <div class="text-center py-5">
                <div class="mb-4">
                    <i class="fas fa-graduation-cap" style="font-size: 4rem; color: #e5e7eb;"></i>
                </div>
                <h3 class="text-muted mb-3">No Courses Available</h3>
                <p class="text-muted">We're working on adding amazing courses. Check back soon!</p>
            </div>
            <?php endif; ?>

            <!-- Load More Button -->
            <?php if (count($courses) >= 12): ?>
            <div class="text-center mt-5">
                <button class="btn btn-outline-primary px-4 py-2" onclick="loadMoreCourses()">
                    <i class="fas fa-plus me-2"></i>Load More Courses
                </button>
            </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Include Footer -->
    <?php include 'includes/footer.php'; ?>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Courses Page JavaScript -->
    <script>
        // Filter courses by level
        function filterCourses(level) {
            // Update active tab
            document.querySelectorAll('.filter-tab').forEach(tab => {
                tab.classList.remove('active');
            });
            event.target.classList.add('active');

            // Filter course cards
            const courseCards = document.querySelectorAll('.course-card');
            courseCards.forEach(card => {
                if (level === 'all') {
                    card.style.display = 'block';
                } else {
                    const cardLevel = card.getAttribute('data-level');
                    if (cardLevel === level) {
                        card.style.display = 'block';
                    } else {
                        card.style.display = 'none';
                    }
                }
            });
        }

        // Search courses
        function searchCourses(searchTerm) {
            const courseCards = document.querySelectorAll('.course-card');
            const term = searchTerm.toLowerCase();

            courseCards.forEach(card => {
                const title = card.getAttribute('data-title');
                if (title.includes(term)) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });

            // If search is cleared, show all courses
            if (searchTerm === '') {
                const activeTab = document.querySelector('.filter-tab.active');
                const level = activeTab.textContent.toLowerCase();
                if (level === 'all courses') {
                    filterCourses('all');
                } else {
                    filterCourses(level);
                }
            }
        }

        // Load more courses (placeholder function)
        function loadMoreCourses() {
            // This would typically make an AJAX request to load more courses
            alert('Load more functionality would be implemented here with AJAX.');
        }

        // Add hover effects for social links
        document.addEventListener('DOMContentLoaded', function() {
            const socialLinks = document.querySelectorAll('.hover-text-primary');
            socialLinks.forEach(link => {
                link.addEventListener('mouseenter', function() {
                    this.style.color = '#f59e0b';
                });
                link.addEventListener('mouseleave', function() {
                    this.style.color = '#a0aec0';
                });
            });
        });
    </script>

</body>
</html>



            link.addEventListener('mouseenter', function() {
                this.style.color = '#276BE4';
                this.style.transition = 'color 0.3s ease';
            });
            
            link.addEventListener('mouseleave', function() {
                this.style.color = '#212529';
            });
        });

        // Add hover effects for social icons
        document.querySelectorAll('.social-icon').forEach(icon => {
            icon.addEventListener('mouseenter', function() {
                const circle = this.querySelector('.rounded-circle');
                circle.style.backgroundColor = '#276BE4';
                circle.querySelector('i').style.color = '#fff';
            });
            
            icon.addEventListener('mouseleave', function() {
                const circle = this.querySelector('.rounded-circle');
                circle.style.backgroundColor = '#f8f9fa';
                circle.querySelector('i').style.color = '#276BE4';
            });
        });
    </script>
</body>
</html>