<?php
/**
 * Comprehensive Grade Calculation System
 * 
 * This system ensures accurate grading for students across multiple courses
 * with varying numbers of assignments and quizzes, while maintaining 100% accuracy.
 */

class GradeCalculator {
    private $conn;
    
    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }
    
    /**
     * Calculate final grade for a specific student in a specific course
     * 
     * @param int $student_id
     * @param int $course_id
     * @return array Grade breakdown and final percentage
     */
    public function calculateCourseGrade($student_id, $course_id) {
        // Get course weighting configuration
        $course_weights = $this->getCourseWeights($course_id);
        
        // Get assignment and quiz grades
        $assignment_grade = $this->calculateAssignmentGrade($student_id, $course_id);
        $quiz_grade = $this->calculateQuizGrade($student_id, $course_id);
        
        // Calculate weighted final grade
        $final_grade = 0;
        $grade_breakdown = [];
        
        if ($assignment_grade['has_grades'] && $quiz_grade['has_grades']) {
            // Both assignments and quizzes exist
            $final_grade = ($assignment_grade['average'] * $course_weights['assignment_weight'] / 100) + 
                          ($quiz_grade['average'] * $course_weights['quiz_weight'] / 100);
            
            $grade_breakdown = [
                'assignment_component' => round($assignment_grade['average'] * $course_weights['assignment_weight'] / 100, 2),
                'quiz_component' => round($quiz_grade['average'] * $course_weights['quiz_weight'] / 100, 2),
                'assignment_weight_used' => $course_weights['assignment_weight'],
                'quiz_weight_used' => $course_weights['quiz_weight']
            ];
            
        } elseif ($assignment_grade['has_grades'] && !$quiz_grade['has_grades']) {
            // Only assignments exist - use 100% assignment weight
            $final_grade = $assignment_grade['average'];
            
            $grade_breakdown = [
                'assignment_component' => round($assignment_grade['average'], 2),
                'quiz_component' => 0,
                'assignment_weight_used' => 100,
                'quiz_weight_used' => 0,
                'note' => 'No quizzes found - using 100% assignment weight'
            ];
            
        } elseif (!$assignment_grade['has_grades'] && $quiz_grade['has_grades']) {
            // Only quizzes exist - use 100% quiz weight
            $final_grade = $quiz_grade['average'];
            
            $grade_breakdown = [
                'assignment_component' => 0,
                'quiz_component' => round($quiz_grade['average'], 2),
                'assignment_weight_used' => 0,
                'quiz_weight_used' => 100,
                'note' => 'No assignments found - using 100% quiz weight'
            ];
            
        } else {
            // No grades available
            $final_grade = 0;
            $grade_breakdown = [
                'assignment_component' => 0,
                'quiz_component' => 0,
                'assignment_weight_used' => 0,
                'quiz_weight_used' => 0,
                'note' => 'No grades available'
            ];
        }
        
        return [
            'final_grade' => round($final_grade, 2),
            'assignment_data' => $assignment_grade,
            'quiz_data' => $quiz_grade,
            'course_weights' => $course_weights,
            'breakdown' => $grade_breakdown,
            'calculation_method' => $this->getCalculationMethod($assignment_grade['has_grades'], $quiz_grade['has_grades']),
            'total_weight_verification' => $grade_breakdown['assignment_weight_used'] + $grade_breakdown['quiz_weight_used']
        ];
    }
    
    /**
     * Calculate assignment average for a student in a course
     * QUIZ-ONLY SYSTEM: Always returns empty/zero - assignments ignored
     */
    private function calculateAssignmentGrade($student_id, $course_id) {
        // QUIZ-ONLY CERTIFICATE SYSTEM: Assignments are completely ignored
        // This ensures only quizzes contribute to final grades and certificate eligibility
        
        return [
            'assignments' => [],
            'count' => 0,
            'graded_count' => 0,
            'average' => 0,
            'has_grades' => false,
            'system_note' => 'Quiz-only system - assignments ignored for certificate calculation'
        ];
    }
    
    /**
     * Calculate quiz average for a student in a course
     */
    private function calculateQuizGrade($student_id, $course_id) {
        $query = "
            SELECT q.quiz_id, q.title, q.contributes_to_final_grade,
                   qa.attempt_id, qa.score, qa.pass_status
            FROM quizzes q
            LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
            WHERE q.course_id = ? AND q.contributes_to_final_grade = 1 AND q.is_published = 1
            ORDER BY q.created_at
        ";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ii", $student_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $quizzes = [];
        $total_score = 0;
        $completed_quizzes = 0;
        
        while ($row = $result->fetch_assoc()) {
            $quizzes[] = $row;
            
            if ($row['score'] !== null && $row['score'] >= 0) {
                $total_score += floatval($row['score']);
                $completed_quizzes++;
            }
        }
        
        $average = $completed_quizzes > 0 ? ($total_score / $completed_quizzes) : 0;
        
        return [
            'quizzes' => $quizzes,
            'count' => count($quizzes),
            'completed_count' => $completed_quizzes,
            'average' => round($average, 2),
            'has_grades' => $completed_quizzes > 0
        ];
    }
    
    /**
     * Get course weight configuration
     */
    private function getCourseWeights($course_id) {
        $query = "SELECT assignment_weight, quiz_weight FROM courses WHERE course_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        
        return [
            'assignment_weight' => floatval($result['assignment_weight'] ?? 70),
            'quiz_weight' => floatval($result['quiz_weight'] ?? 30)
        ];
    }
    
    /**
     * Get calculation method description
     */
    private function getCalculationMethod($has_assignments, $has_quizzes) {
        if ($has_assignments && $has_quizzes) {
            return 'Weighted average using course-specific assignment and quiz weights';
        } elseif ($has_assignments) {
            return 'Assignment-only grading (100% assignment weight)';
        } elseif ($has_quizzes) {
            return 'Quiz-only grading (100% quiz weight)';
        } else {
            return 'No grades available';
        }
    }
    
    /**
     * Calculate overall GPA across all enrolled courses for a student
     */
    public function calculateOverallGPA($student_id) {
        // Get all enrolled courses for the student
        $query = "
            SELECT DISTINCT c.course_id, c.title, c.credit_hours
            FROM courses c
            JOIN enrollments e ON c.course_id = e.course_id
            WHERE e.student_id = ?
        ";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $total_grade_points = 0;
        $total_credit_hours = 0;
        $course_grades = [];
        
        while ($course = $result->fetch_assoc()) {
            $grade_data = $this->calculateCourseGrade($student_id, $course['course_id']);
            $credit_hours = floatval($course['credit_hours'] ?? 3); // Default 3 credits
            
            $course_grades[] = [
                'course_id' => $course['course_id'],
                'title' => $course['title'],
                'final_grade' => $grade_data['final_grade'],
                'credit_hours' => $credit_hours,
                'grade_points' => ($grade_data['final_grade'] / 100) * 4.0 * $credit_hours, // Convert to 4.0 scale
                'breakdown' => $grade_data['breakdown']
            ];
            
            $total_grade_points += ($grade_data['final_grade'] / 100) * 4.0 * $credit_hours;
            $total_credit_hours += $credit_hours;
        }
        
        $gpa = $total_credit_hours > 0 ? ($total_grade_points / $total_credit_hours) : 0;
        
        return [
            'gpa' => round($gpa, 3),
            'total_credit_hours' => $total_credit_hours,
            'course_grades' => $course_grades,
            'calculation_summary' => [
                'total_grade_points' => round($total_grade_points, 2),
                'total_credit_hours' => $total_credit_hours,
                'courses_count' => count($course_grades)
            ]
        ];
    }
    
    /**
     * Generate detailed grade report for a student
     */
    public function generateGradeReport($student_id) {
        $overall_data = $this->calculateOverallGPA($student_id);
        
        $report = [
            'student_id' => $student_id,
            'generated_at' => date('Y-m-d H:i:s'),
            'overall_gpa' => $overall_data['gpa'],
            'total_credit_hours' => $overall_data['total_credit_hours'],
            'courses' => []
        ];
        
        foreach ($overall_data['course_grades'] as $course) {
            $detailed_grade = $this->calculateCourseGrade($student_id, $course['course_id']);
            
            $report['courses'][] = [
                'course_info' => $course,
                'detailed_breakdown' => $detailed_grade,
                'verification' => [
                    'weight_sum' => $detailed_grade['total_weight_verification'],
                    'is_valid' => abs($detailed_grade['total_weight_verification'] - 100) < 0.01
                ]
            ];
        }
        
        return $report;
    }
}

/**
 * Helper function to get a configured grade calculator instance
 */
function getGradeCalculator($db_connection = null) {
    global $conn;
    $connection = $db_connection ?? $conn;
    return new GradeCalculator($connection);
}

/**
 * Quick function to calculate a student's grade in a specific course
 */
function calculateStudentCourseGrade($student_id, $course_id, $db_connection = null) {
    $calculator = getGradeCalculator($db_connection);
    return $calculator->calculateCourseGrade($student_id, $course_id);
}

/**
 * Quick function to calculate a student's overall GPA
 */
function calculateStudentGPA($student_id, $db_connection = null) {
    $calculator = getGradeCalculator($db_connection);
    return $calculator->calculateOverallGPA($student_id);
}
?>
