<?php
/**
 * Centralized ID Encryption System for Panadite Academy
 * 
 * This utility provides secure encryption and decryption of database IDs
 * to prevent exposure of raw database IDs in URLs across the entire platform.
 * 
 * Usage:
 * - Include this file in any page that needs ID encryption
 * - Use encryptId($id) to encrypt database IDs for URLs
 * - Use decryptId($encrypted_id) to decrypt IDs for database queries
 * 
 * Compatible with: Student, Teacher, Admin, and Landing pages
 * 
 * @author Panadite Academy Development Team
 * @version 1.0
 */

class IdEncryption {
    
    // Encryption configuration
    private static $encryption_method = 'AES-256-CBC';
    private static $secret_key = 'PanaditeAcademy2024SecureKeyForIdEncryption!@#$%^&*()';
    private static $secret_iv = 'PanaditeIVSecure2024!@#$%^&*()_+{}|:<>?';
    
    /**
     * Initialize encryption keys
     */
    private static function getKey() {
        return hash('sha256', self::$secret_key);
    }
    
    private static function getIV() {
        return substr(hash('sha256', self::$secret_iv), 0, 16);
    }
    
    /**
     * Encrypt a database ID
     * 
     * @param int $id The database ID to encrypt
     * @return string The encrypted ID (URL-safe)
     */
    public static function encrypt($id) {
        if (empty($id) || !is_numeric($id)) {
            return '';
        }
        
        $encrypted = openssl_encrypt(
            $id, 
            self::$encryption_method, 
            self::getKey(), 
            0, 
            self::getIV()
        );
        
        // Make URL-safe by replacing characters that might cause issues
        $url_safe = base64_encode($encrypted);
        $url_safe = str_replace(['+', '/', '='], ['-', '_', ''], $url_safe);
        
        return $url_safe;
    }
    
    /**
     * Decrypt an encrypted ID
     * 
     * @param string $encrypted_id The encrypted ID to decrypt
     * @return int|false The original database ID or false on failure
     */
    public static function decrypt($encrypted_id) {
        if (empty($encrypted_id)) {
            return false;
        }
        
        // Restore from URL-safe format
        $encrypted_id = str_replace(['-', '_'], ['+', '/'], $encrypted_id);
        
        // Add padding if needed
        $padding = 4 - (strlen($encrypted_id) % 4);
        if ($padding !== 4) {
            $encrypted_id .= str_repeat('=', $padding);
        }
        
        $encrypted = base64_decode($encrypted_id);
        
        if ($encrypted === false) {
            return false;
        }
        
        $decrypted = openssl_decrypt(
            $encrypted, 
            self::$encryption_method, 
            self::getKey(), 
            0, 
            self::getIV()
        );
        
        return is_numeric($decrypted) ? (int)$decrypted : false;
    }
    
    /**
     * Validate if an encrypted ID is valid
     * 
     * @param string $encrypted_id The encrypted ID to validate
     * @return bool True if valid, false otherwise
     */
    public static function isValid($encrypted_id) {
        return self::decrypt($encrypted_id) !== false;
    }
    
    /**
     * Generate a secure URL with encrypted ID
     * 
     * @param string $base_url The base URL (e.g., 'course-details.php')
     * @param int $id The database ID to encrypt
     * @param array $additional_params Additional URL parameters
     * @return string The complete URL with encrypted ID
     */
    public static function generateUrl($base_url, $id, $additional_params = []) {
        $encrypted_id = self::encrypt($id);
        
        if (empty($encrypted_id)) {
            return $base_url;
        }
        
        $params = ['id' => $encrypted_id];
        
        if (!empty($additional_params)) {
            $params = array_merge($params, $additional_params);
        }
        
        $query_string = http_build_query($params);
        
        return $base_url . '?' . $query_string;
    }
    
    /**
     * Extract and decrypt ID from URL parameters
     * 
     * @param string $param_name The parameter name (default: 'id')
     * @return int|false The decrypted ID or false on failure
     */
    public static function getIdFromUrl($param_name = 'id') {
        if (!isset($_GET[$param_name])) {
            return false;
        }
        
        return self::decrypt($_GET[$param_name]);
    }
    
    /**
     * Safely get an encrypted ID from POST data
     * 
     * @param string $param_name The parameter name (default: 'id')
     * @return int|false The decrypted ID or false on failure
     */
    public static function getIdFromPost($param_name = 'id') {
        if (!isset($_POST[$param_name])) {
            return false;
        }
        
        return self::decrypt($_POST[$param_name]);
    }
}

// Convenience functions for easy usage
function encryptId($id) {
    return IdEncryption::encrypt($id);
}

function decryptId($encrypted_id) {
    return IdEncryption::decrypt($encrypted_id);
}

function generateSecureUrl($base_url, $id, $additional_params = []) {
    return IdEncryption::generateUrl($base_url, $id, $additional_params);
}

function getSecureId($param_name = 'id') {
    return IdEncryption::getIdFromUrl($param_name);
}

function getSecurePostId($param_name = 'id') {
    return IdEncryption::getIdFromPost($param_name);
}

function isValidEncryptedId($encrypted_id) {
    return IdEncryption::isValid($encrypted_id);
}

/**
 * Debug function for development (remove in production)
 */
function debugEncryption($id) {
    echo "<div style='background: #f0f8ff; padding: 10px; margin: 10px; border: 1px solid #ccc; border-radius: 5px;'>";
    echo "<h4>ID Encryption Debug</h4>";
    echo "<strong>Original ID:</strong> $id<br>";
    
    $encrypted = encryptId($id);
    echo "<strong>Encrypted:</strong> $encrypted<br>";
    
    $decrypted = decryptId($encrypted);
    echo "<strong>Decrypted:</strong> $decrypted<br>";
    
    echo "<strong>Valid:</strong> " . (isValidEncryptedId($encrypted) ? 'Yes' : 'No') . "<br>";
    echo "<strong>URL Example:</strong> course-details.php?id=$encrypted<br>";
    echo "</div>";
}

?>
