<?php
// Include the proper database connection file
require_once __DIR__ . '/db_connection.php';
require_once __DIR__ . '/id_encryption.php';

// Include comprehensive SEO optimization
require_once __DIR__ . '/seo_header.php';

// Initialize categories array
$categories = [];

// Use the existing database connection from db_connection.php
if ($conn && !$conn->connect_error) {
    // Get categories with course counts - using EXACT query from your debug that works  
    $query = "SELECT cat.category_id, cat.category_name, COUNT(c.course_id) as course_count 
              FROM categories cat 
              LEFT JOIN courses c ON cat.category_id = c.category_id 
              WHERE c.is_published = 1 
              GROUP BY cat.category_id, cat.category_name 
              ORDER BY cat.category_name ASC";
    
    $result = $conn->query($query);
    
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $categories[] = $row;
        }
        $result->free();
    }
}
?>

<!-- Clean Navbar - Dynamic Categories Loaded -->
<nav id="main-navbar" class="navbar navbar-expand-lg navbar-light sticky-top bg-white border-bottom">
    <div class="container-fluid px-4">
        <!-- Logo -->
        <a class="navbar-brand" href="/index.php">
            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 45px; width: auto;">
        </a>

        <!-- Navigation Links (Desktop Only) -->
        <div class="d-none d-lg-flex align-items-center" style="gap: 0;">
            <a href="/index.php" style="color: #1c1d1f; font-size: 14px; padding: 10px 16px; text-decoration: none; line-height: 1.5;">Home</a>
            
            <!-- Courses - Direct Link -->
            <a href="/courses.php" style="color: #1c1d1f; font-size: 14px; padding: 10px 16px; text-decoration: none; line-height: 1.5;">Courses</a>
            
            <!-- Categories - Dynamic Dropdown -->
            <div class="dropdown d-flex align-items-center navbar-dropdown mega-dropdown-wrapper" style="position: static;">
                <a href="#" class="dropdown-toggle" style="color: #1c1d1f; font-size: 14px; padding: 10px 16px; text-decoration: none; line-height: 1.5; display: flex; align-items: center;">Categories</a>
                <div class="dropdown-menu mega-dropdown">
                    <div class="container-fluid">
                        <!-- Flex Layout Container -->
                        <div style="display: flex; flex-direction: column; gap: 0; max-width: 100%;">
                            <?php if (!empty($categories)): ?>
                                <!-- Intelligent Auto-Categorization System -->
                                <?php
                                // Define intelligent category mapping with keywords for auto-detection
                                $category_themes = [
                                    'Technology' => [
                                        'icon' => 'fas fa-code',
                                        'keywords' => ['programming', 'cyber', 'security', 'software', 'coding', 'development', 'web', 'mobile', 'app', 'tech', 'computer', 'data', 'ai', 'machine learning', 'blockchain', 'cloud'],
                                        'categories' => []
                                    ],
                                    'Business & Entrepreneurship' => [
                                        'icon' => 'fas fa-briefcase',
                                        'keywords' => ['business', 'entrepreneur', 'venture', 'startup', 'management', 'marketing', 'finance', 'sales', 'leadership', 'strategy', 'commerce'],
                                        'categories' => []
                                    ],
                                    'Professional Development' => [
                                        'icon' => 'fas fa-graduation-cap',
                                        'keywords' => ['learning', 'development', 'facilitation', 'assessment', 'skills', 'training', 'coaching', 'certification', 'professional'],
                                        'categories' => []
                                    ],
                                    'Healthcare & Medical' => [
                                        'icon' => 'fas fa-heartbeat',
                                        'keywords' => ['health', 'medical', 'nursing', 'pharmacy', 'medicine', 'healthcare', 'clinical', 'therapy', 'rehabilitation'],
                                        'categories' => []
                                    ],
                                    'Hospitality & Tourism' => [
                                        'icon' => 'fas fa-hotel',
                                        'keywords' => ['hospitality', 'tourism', 'hotel', 'restaurant', 'culinary', 'travel', 'service', 'catering', 'event'],
                                        'categories' => []
                                    ],
                                    'Creative & Arts' => [
                                        'icon' => 'fas fa-palette',
                                        'keywords' => ['design', 'art', 'creative', 'photography', 'video', 'music', 'writing', 'media', 'graphic', 'animation'],
                                        'categories' => []
                                    ]
                                ];
                                
                                // Auto-categorize using intelligent keyword matching
                                foreach ($categories as $category) {
                                    $cat_name = strtolower($category['category_name']);
                                    $categorized = false;
                                    
                                    // Check each theme for keyword matches
                                    foreach ($category_themes as $theme_name => $theme_data) {
                                        foreach ($theme_data['keywords'] as $keyword) {
                                            if (strpos($cat_name, $keyword) !== false) {
                                                $category_themes[$theme_name]['categories'][] = $category;
                                                $categorized = true;
                                                break 2; // Exit both loops once categorized
                                            }
                                        }
                                    }
                                    
                                    // If no match found, add to "Other Categories"
                                    if (!$categorized) {
                                        if (!isset($category_themes['Other Categories'])) {
                                            $category_themes['Other Categories'] = [
                                                'icon' => 'fas fa-th-large',
                                                'keywords' => [],
                                                'categories' => []
                                            ];
                                        }
                                        $category_themes['Other Categories']['categories'][] = $category;
                                    }
                                }
                                
                                // Remove empty themes
                                $category_themes = array_filter($category_themes, function($theme) {
                                    return !empty($theme['categories']);
                                });
                                ?>
                                
                                <!-- Flex Categories Container - All in 1 Row -->
                                <div style="display: flex; flex-wrap: wrap; gap: 16px; margin-bottom: 20px; justify-content: flex-start;">
                                    <?php foreach ($category_themes as $theme_name => $theme_data): ?>
                                        <div style="flex: 0 0 calc(33.333% - 11px); min-width: 280px; max-width: 320px;">
                                            <!-- Theme Header -->
                                            <div class="theme-header" style="background: linear-gradient(135deg, rgba(39,107,228,0.1), rgba(39,107,228,0.05)); 
                                                                              border-radius: 12px; padding: 12px 16px; margin-bottom: 12px; 
                                                                              border-left: 4px solid #276BE4;">
                                                <h6 class="dropdown-header" style="color: #276BE4; font-weight: 700; margin: 0; font-size: 14px;">
                                                    <i class="<?php echo $theme_data['icon']; ?> me-2" style="color: #FFA915;"></i><?php echo htmlspecialchars($theme_name); ?>
                                                </h6>
                                            </div>
                                            
                                            <!-- Category Items -->
                                            <?php foreach ($theme_data['categories'] as $category): 
                                                // Encrypt category ID
                                                $category_id = $category['category_id'] ?? 0;
                                                $encrypted_category_id = class_exists('IdEncryption') ? IdEncryption::encrypt($category_id) : $category_id;
                                            ?>
                                                <a class="dropdown-item d-flex justify-content-between align-items-center modern-category-item" 
                                                   href="/category.php?category_id=<?php echo urlencode($encrypted_category_id); ?>"
                                                   style="padding: 10px 16px; border-radius: 8px; margin-bottom: 6px; 
                                                          transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
                                                          border: 1px solid transparent; background: rgba(248,249,250,0.6);
                                                          display: block;">
                                                    <span style="font-weight: 500; color: #2d3436; font-size: 14px;">
                                                        <?php echo htmlspecialchars($category['category_name']); ?>
                                                    </span>
                                                    <span class="course-count-badge" style="background: linear-gradient(135deg, #276BE4, #1e5bcc); 
                                                                                             color: white; padding: 2px 8px; border-radius: 12px; 
                                                                                             font-size: 11px; font-weight: 600; min-width: 24px; 
                                                                                             text-align: center; float: right;">
                                                        <?php echo $category['course_count']; ?>
                                                    </span>
                                                </a>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endforeach; ?>
                                    
                                    <!-- Ready to Start Learning Section - As Third Column -->
                                    <div style="flex: 0 0 calc(33.333% - 11px); min-width: 280px; max-width: 320px;">
                                        <!-- Enrollment Banner -->
                                        <div class="enrollment-banner" 
                                             style="background: linear-gradient(135deg, rgba(39,107,228,0.95), rgba(30,91,204,0.9)), 
                                                               url('/landing_images/images (3).jpeg') right center/150px auto no-repeat;
                                                    border-radius: 16px; padding: 20px; color: white; text-align: center;
                                                    box-shadow: 0 8px 32px rgba(39,107,228,0.3); position: relative; overflow: hidden;
                                                    height: 100%; display: flex; flex-direction: column; justify-content: center;">
                                            
                                            <!-- Content -->
                                            <div style="position: relative; z-index: 2;">
                                                <div style="margin-bottom: 12px;">
                                                    <i class="fas fa-graduation-cap" style="font-size: 24px; color: #FFA915;"></i>
                                                </div>
                                                <h6 style="margin: 0 0 10px 0; font-weight: 700; font-size: 16px;">Ready to Start Learning?</h6>
                                                <p style="margin-bottom: 16px; font-size: 12px; opacity: 0.9; line-height: 1.3;">
                                                    Join thousands advancing their careers
                                                </p>
                                                <div style="display: flex; flex-direction: column; gap: 8px;">
                                                    <a href="/courses.php" class="btn" 
                                                       style="background: linear-gradient(135deg, #FFA915, #ff9500); color: white; 
                                                              border: none; padding: 8px 12px; border-radius: 8px; font-weight: 600;
                                                              text-decoration: none; font-size: 12px; transition: all 0.3s ease;">
                                                        <i class="fas fa-th-large me-1"></i>Browse Courses
                                                    </a>
                                                    <a href="/auth/register.php" class="btn" 
                                                       style="background: rgba(255,255,255,0.2); color: white; border: 2px solid white;
                                                              padding: 8px 12px; border-radius: 8px; font-weight: 600;
                                                              text-decoration: none; font-size: 12px; transition: all 0.3s ease;">
                                                        <i class="fas fa-user-plus me-1"></i>Register to Enroll
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php else: ?>
                                <!-- Fallback when no categories -->
                                <div class="col-md-12">
                                    <div class="text-center py-3">
                                        <p class="text-muted mb-2">No categories available</p>
                                        <a href="/courses.php" class="btn btn-sm btn-primary">Browse All Courses</a>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Teach - Mega Dropdown -->
            <div class="dropdown d-flex align-items-center navbar-dropdown mega-dropdown-wrapper" style="position: static;">
                <a href="#" class="dropdown-toggle" style="color: #1c1d1f; font-size: 14px; padding: 10px 16px; text-decoration: none; line-height: 1.5; display: flex; align-items: center;">Teach</a>
                <div class="dropdown-menu mega-dropdown">
                    <div class="container-fluid">
                        <div class="row">
                            <div class="col-md-4">
                                <h6 class="dropdown-header">What You Can Do</h6>
                                <a class="dropdown-item" href="#">
                                    <i class="fas fa-money-bill-wave me-2" style="color: #28a745;"></i>
                                    <span><strong>Earn Money</strong><br><small class="text-muted">Generate income from your expertise</small></span>
                                </a>
                                <a class="dropdown-item" href="#">
                                    <i class="fas fa-upload me-2" style="color: #007bff;"></i>
                                    <span><strong>Upload Courses</strong><br><small class="text-muted">Create and share your knowledge</small></span>
                                </a>
                                <a class="dropdown-item" href="#">
                                    <i class="fas fa-users me-2" style="color: #6f42c1;"></i>
                                    <span><strong>Build Your Audience</strong><br><small class="text-muted">Grow your student community</small></span>
                                </a>
                                <a class="dropdown-item" href="#">
                                    <i class="fas fa-chart-line me-2" style="color: #fd7e14;"></i>
                                    <span><strong>Track Your Success</strong><br><small class="text-muted">Monitor earnings and engagement</small></span>
                                </a>
                            </div>
                            <div class="col-md-4">
                                <h6 class="dropdown-header">Get Started</h6>
                                <a class="dropdown-item" href="auth/register.php">
                                    <i class="fas fa-user-plus me-2" style="color: #28a745;"></i>
                                    Become an Instructor
                                </a>
                                <a class="dropdown-item" href="teaching-requirements.php">
                                    <i class="fas fa-list-check me-2" style="color: #6c757d;"></i>
                                    Teaching Requirements
                                </a>
                                <a class="dropdown-item" href="course-guidelines.php">
                                    <i class="fas fa-book me-2" style="color: #6c757d;"></i>
                                    Course Guidelines
                                </a>
                                <a class="dropdown-item" href="instructor-support.php">
                                    <i class="fas fa-headset me-2" style="color: #6c757d;"></i>
                                    Instructor Support
                                </a>

                            </div>
                            <div class="col-md-4">
                                <div class="instructor-cta">
                                    <div style="background: linear-gradient(135deg, #276BE4, #1e5bcc); padding: 24px; border-radius: 12px; color: white; text-align: center; box-shadow: 0 4px 20px rgba(39, 107, 228, 0.3);">
                                        <i class="fas fa-graduation-cap" style="font-size: 2.5rem; margin-bottom: 16px; opacity: 0.9;"></i>
                                        <h6 style="color: white; margin-bottom: 8px; font-weight: 700;">Become a Teacher</h6>
                                        <p style="font-size: 13px; margin-bottom: 16px; color: rgba(255,255,255,0.9); line-height: 1.4;">Share your knowledge and earn money teaching thousands of students worldwide</p>
                                        <div style="margin-bottom: 16px;">
                                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                                <span style="font-size: 12px;">📚 Upload unlimited courses</span>
                                            </div>
                                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                                <span style="font-size: 12px;">🌟 Build your brand</span>
                                            </div>
                                            <div style="display: flex; justify-content: space-between;">
                                                <span style="font-size: 12px;">📈 Grow your audience</span>
                                            </div>
                                        </div>
                                        <a href="auth/register.php" style="background: white; color: #276BE4; padding: 12px 24px; border-radius: 25px; text-decoration: none; font-weight: 700; font-size: 13px; display: inline-block; transition: all 0.3s ease;">Start Teaching Today</a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- About Us - Direct Link -->
            <a href="/about.php" style="color: #1c1d1f; font-size: 14px; padding: 10px 16px; text-decoration: none; line-height: 1.5;">About Us</a>
        </div>

        <!-- Dynamic Search Bar -->
        <div class="d-none d-lg-flex flex-grow-1 mx-4" style="max-width: 420px; position: relative;">
            <div class="main-search-container" id="mainSearchContainer">
                <div class="main-search-input-wrapper">
                    <i class="fas fa-search main-search-icon"></i>
                    <input type="text" 
                           placeholder="Search courses, categories, instructors..." 
                           id="main-search-input"
                           class="main-search-input"
                           autocomplete="off" />
                    <button type="button" class="main-search-clear-btn" id="mainSearchClear" style="display: none;">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <!-- Dynamic Search Dropdown -->
                <div class="main-search-dropdown" id="mainSearchDropdown">
                    <div class="main-search-results" id="mainSearchResults">
                        <!-- Dynamic results will be inserted here -->
                    </div>
                </div>
            </div>
        </div>



        <!-- Cart & Auth (Right Side) - Desktop -->
        <div class="d-none d-lg-flex align-items-center" style="gap: 16px;">
            <a href="/cart.php" style="color: #1c1d1f; font-size: 20px; text-decoration: none; position: relative;">
                <i class="fas fa-shopping-cart"></i>
                <span class="cart-counter" style="position: absolute; top: -8px; right: -8px; background: #ec5252; 
                                                  color: white; border-radius: 10px; padding: 2px 5px; 
                                                  font-size: 10px; display: none;">0</span>
            </a>

            <?php 
            // Start session if not already started and headers haven't been sent
            if (session_status() == PHP_SESSION_NONE && !headers_sent()) {
                // Force proper session configuration to match CMS
                ini_set('session.save_handler', 'files');
                ini_set('session.save_path', '/tmp');
                ini_set('session.use_cookies', 1);
                ini_set('session.cookie_lifetime', 0);
                ini_set('session.cookie_path', '/');
                ini_set('session.cookie_domain', '');
                ini_set('session.cookie_secure', 0);
                ini_set('session.cookie_httponly', 1);
                
                if (!session_id()) {
                    session_start();
                }
            }
            
            // Check if user is logged in and determine user type
            $is_logged_in = isset($_SESSION['user_id']) || isset($_SESSION['student_id']) || isset($_SESSION['instructor_id']) || isset($_SESSION['cms_admin_id']);
            
            // Enhanced admin detection - check multiple possible admin session variables
            $is_admin = isset($_SESSION['cms_admin_id']) || 
                       isset($_SESSION['admin_id']) || 
                       isset($_SESSION['cms_admin_username']) ||
                       isset($_SESSION['cms_admin_role']) ||
                       (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'admin') ||
                       (isset($_SESSION['role']) && $_SESSION['role'] === 'admin') ||
                       (isset($_SESSION['cms_admin_name']) && !empty($_SESSION['cms_admin_name']));
            
            $is_instructor = isset($_SESSION['instructor_id']) || (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'instructor');
            $user_name = '';
            
            if ($is_logged_in) {
                // Get user name for display - prioritize CMS admin name
                if (isset($_SESSION['cms_admin_name'])) {
                    $user_name = $_SESSION['cms_admin_name'];
                } elseif (isset($_SESSION['cms_admin_username'])) {
                    $user_name = $_SESSION['cms_admin_username'];
                } elseif (isset($_SESSION['user_name'])) {
                    $user_name = $_SESSION['user_name'];
                } elseif (isset($_SESSION['first_name'])) {
                    $user_name = $_SESSION['first_name'];
                } elseif (isset($_SESSION['full_name'])) {
                    $user_name = $_SESSION['full_name'];
                } elseif (isset($_SESSION['admin_name'])) {
                    $user_name = $_SESSION['admin_name'];
                } else {
                    $user_name = 'User';
                }
            } else {
                // If no session but CMS admin indicator is visible, force admin detection
                if (class_exists('DOMDocument')) {
                    // Check if running in CMS admin context
                    $current_url = $_SERVER['REQUEST_URI'] ?? '';
                    if (strpos($current_url, '/cms/') !== false || 
                        (isset($_COOKIE['cms_admin_session']) && !empty($_COOKIE['cms_admin_session']))) {
                        $is_admin = true;
                        $is_logged_in = true;
                        $user_name = 'Admin';
                    }
                }
            }
            ?>
            
            <?php if ($is_logged_in): ?>
                <!-- Logged In User Menu -->
                <div class="dropdown">
                    <a href="#" class="dropdown-toggle d-flex align-items-center" style="color: #1c1d1f; text-decoration: none; font-size: 14px; font-weight: 600; padding: 8px 12px; border: 1px solid #e8eaed; border-radius: 20px; background: #f8f9fa;" data-bs-toggle="dropdown">
                        <i class="fas fa-user-circle me-2" style="font-size: 16px;"></i>
                        <?= htmlspecialchars($user_name) ?>
                        <i class="fas fa-chevron-down ms-2" style="font-size: 10px;"></i>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <?php if ($is_admin): ?>
                            <!-- Admin Dashboard -->
                            <li><a class="dropdown-item" href="/admin/dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                        <?php elseif ($is_instructor): ?>
                            <!-- Teacher Dashboard -->
                            <li><a class="dropdown-item" href="/teacher/dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                        <?php else: ?>
                            <!-- Student Dashboard -->
                            <li><a class="dropdown-item" href="/student/dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                        <?php endif; ?>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="/auth/logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                    </ul>
                </div>
            <?php else: ?>
                <!-- Not Logged In - Show Login/Signup -->
                <div style="display: flex; align-items: center; gap: 8px;">
                    <a href="/auth/login.php" style="color: #1c1d1f; text-decoration: none; font-size: 14px; 
                                                     font-weight: 600; padding: 8px 16px; border: 1px solid #1c1d1f; 
                                                     border-radius: 4px;">Log in</a>
                    <a href="/auth/register.php" style="background-color: #fef9e7; color: #8b5a00; text-decoration: none; 
                                                        font-size: 14px; font-weight: 600; padding: 8px 16px; 
                                                        border: 1px solid #f59e0b; border-radius: 4px;">Sign up</a>
                </div>
            <?php endif; ?>
        </div>

        <!-- 📱 MOBILE ICONS - Only show on mobile -->
        <div class="d-lg-none d-flex align-items-center mobile-icons-container" style="gap: 20px;">
            <!-- Search Icon -->
            <button class="mobile-icon-btn mobile-search-toggle" id="mobileSearchToggle" type="button">
                <i class="fas fa-search"></i>
            </button>

            <!-- Cart Icon - Same functionality as desktop -->
            <a href="/cart.php" class="mobile-icon-btn mobile-cart-icon" style="position: relative;">
                <i class="fas fa-shopping-cart"></i>
                <span class="cart-counter" style="position: absolute; top: -8px; right: -8px; background: #ec5252; 
                                                  color: white; border-radius: 10px; padding: 2px 5px; 
                                                  font-size: 10px; display: none;">0</span>
            </a>

            <!-- Menu List Icon -->
            <button class="mobile-icon-btn mobile-menu-toggle" id="mobileMenuToggle" type="button">
                <i class="fas fa-bars"></i>
            </button>
        </div>

        <!-- 📱 MOBILE MENU DROPDOWN - Toggleable -->
        <div class="mobile-menu-dropdown d-lg-none" id="mobileMenuDropdown" style="display: none;">
            <div class="container-fluid px-4 py-3">
                <div class="mobile-menu-content">
                    <!-- Navigation Links -->
                    <div class="mobile-nav-section">
                        <h6 class="mobile-nav-title">Navigate</h6>
                        <a href="/index.php" class="mobile-nav-link">
                            <i class="fas fa-home me-3"></i>Home
                        </a>
                        <a href="/courses.php" class="mobile-nav-link">
                            <i class="fas fa-book me-3"></i>Courses
                        </a>
                        
                        <!-- Categories Expandable Section -->
                        <div class="mobile-expandable-section">
                            <a href="#" class="mobile-nav-link mobile-categories-toggle" onclick="toggleMobileSection('categories', event)">
                                <i class="fas fa-th-large me-3"></i>Categories
                                <i class="fas fa-chevron-right ms-auto mobile-chevron"></i>
                            </a>
                            <div class="mobile-expandable-content" id="mobile-categories-content">
                                <?php if (!empty($categories)): ?>
                                    <?php
                                    // Use the same category themes logic as desktop
                                    $mobile_category_themes = [
                                        'Technology' => [
                                            'icon' => 'fas fa-code',
                                            'keywords' => ['programming', 'cyber', 'security', 'software', 'coding', 'development', 'web', 'mobile', 'app', 'tech', 'computer', 'data', 'ai', 'machine learning', 'blockchain', 'cloud'],
                                            'categories' => []
                                        ],
                                        'Business & Entrepreneurship' => [
                                            'icon' => 'fas fa-briefcase',
                                            'keywords' => ['business', 'entrepreneur', 'venture', 'startup', 'management', 'marketing', 'finance', 'sales', 'leadership', 'strategy', 'commerce'],
                                            'categories' => []
                                        ],
                                        'Professional Development' => [
                                            'icon' => 'fas fa-graduation-cap',
                                            'keywords' => ['learning', 'development', 'facilitation', 'assessment', 'skills', 'training', 'coaching', 'certification', 'professional'],
                                            'categories' => []
                                        ],
                                        'Healthcare & Medical' => [
                                            'icon' => 'fas fa-heartbeat',
                                            'keywords' => ['health', 'medical', 'nursing', 'pharmacy', 'medicine', 'healthcare', 'clinical', 'therapy', 'rehabilitation'],
                                            'categories' => []
                                        ],
                                        'Creative & Arts' => [
                                            'icon' => 'fas fa-palette',
                                            'keywords' => ['design', 'art', 'creative', 'photography', 'video', 'music', 'writing', 'media', 'graphic', 'animation'],
                                            'categories' => []
                                        ]
                                    ];
                                    
                                    // Auto-categorize for mobile
                                    foreach ($categories as $category) {
                                        $cat_name = strtolower($category['category_name']);
                                        $categorized = false;
                                        
                                        foreach ($mobile_category_themes as $theme_name => $theme_data) {
                                            foreach ($theme_data['keywords'] as $keyword) {
                                                if (strpos($cat_name, $keyword) !== false) {
                                                    $mobile_category_themes[$theme_name]['categories'][] = $category;
                                                    $categorized = true;
                                                    break 2;
                                                }
                                            }
                                        }
                                        
                                        if (!$categorized) {
                                            if (!isset($mobile_category_themes['Other Categories'])) {
                                                $mobile_category_themes['Other Categories'] = [
                                                    'icon' => 'fas fa-th-large',
                                                    'keywords' => [],
                                                    'categories' => []
                                                ];
                                            }
                                            $mobile_category_themes['Other Categories']['categories'][] = $category;
                                        }
                                    }
                                    
                                    // Remove empty themes
                                    $mobile_category_themes = array_filter($mobile_category_themes, function($theme) {
                                        return !empty($theme['categories']);
                                    });
                                    ?>
                                    
                                    <?php foreach ($mobile_category_themes as $theme_name => $theme_data): ?>
                                        <div class="mobile-category-theme">
                                            <div class="mobile-theme-header">
                                                <i class="<?= $theme_data['icon'] ?> me-2"></i>
                                                <span><?= htmlspecialchars($theme_name) ?></span>
                                            </div>
                                            <?php foreach ($theme_data['categories'] as $category): 
                                                $encrypted_cat_id = IdEncryption::encrypt($category['category_id']);
                                            ?>
                                                <a href="/category.php?category_id=<?= urlencode($encrypted_cat_id) ?>" class="mobile-category-item">
                                                    <?= htmlspecialchars($category['category_name']) ?>
                                                    <span class="mobile-course-count"><?= $category['course_count'] ?></span>
                                                </a>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="mobile-no-categories">
                                        <p>No categories available</p>
                                        <a href="/courses.php" class="mobile-nav-link">Browse All Courses</a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Teach Expandable Section -->
                        <div class="mobile-expandable-section">
                            <a href="#" class="mobile-nav-link mobile-teach-toggle" onclick="toggleMobileSection('teach', event)">
                                <i class="fas fa-graduation-cap me-3"></i>Teach
                                <i class="fas fa-chevron-right ms-auto mobile-chevron"></i>
                            </a>
                            <div class="mobile-expandable-content" id="mobile-teach-content">
                                <div class="mobile-teach-section">
                                    <div class="mobile-section-title">What You Can Do</div>
                                    <a href="#" class="mobile-teach-item">
                                        <i class="fas fa-money-bill-wave me-3" style="color: #28a745;"></i>
                                        <div>
                                            <strong>Earn Money</strong>
                                            <small>Generate income from your expertise</small>
                                        </div>
                                    </a>
                                    <a href="#" class="mobile-teach-item">
                                        <i class="fas fa-upload me-3" style="color: #007bff;"></i>
                                        <div>
                                            <strong>Upload Courses</strong>
                                            <small>Create and share your knowledge</small>
                                        </div>
                                    </a>
                                    <a href="#" class="mobile-teach-item">
                                        <i class="fas fa-users me-3" style="color: #6f42c1;"></i>
                                        <div>
                                            <strong>Build Your Audience</strong>
                                            <small>Grow your student community</small>
                                        </div>
                                    </a>
                                </div>
                                
                                <div class="mobile-teach-section">
                                    <div class="mobile-section-title">Get Started</div>
                                    <a href="auth/register.php" class="mobile-teach-item">
                                        <i class="fas fa-user-plus me-3" style="color: #f59e0b;"></i>
                                        Become an Instructor
                                    </a>
                                    <a href="teaching-requirements.php" class="mobile-teach-item">
                                        <i class="fas fa-list-check me-3" style="color: #6c757d;"></i>
                                        Teaching Requirements
                                    </a>
                                    <a href="course-guidelines.php" class="mobile-teach-item">
                                        <i class="fas fa-book me-3" style="color: #6c757d;"></i>
                                        Course Guidelines
                                    </a>
                                </div>
                                
                                <div class="mobile-teach-cta">
                                    <div class="mobile-cta-card">
                                        <i class="fas fa-graduation-cap"></i>
                                        <h6>Become a Teacher</h6>
                                        <p>Share your knowledge and earn money teaching students worldwide</p>
                                        <a href="auth/register.php" class="mobile-cta-btn">Start Teaching Today</a>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <a href="/about.php" class="mobile-nav-link">
                            <i class="fas fa-info-circle me-3"></i>About Us
                        </a>
                    </div>

                    <div class="mobile-nav-divider"></div>

                    <!-- Auth Section -->
                    <div class="mobile-auth-section">
                        <?php if ($is_logged_in): ?>
                            <!-- Logged In User -->
                            <div class="mobile-user-info">
                                <div class="mobile-user-avatar">
                                    <i class="fas fa-user-circle"></i>
                                </div>
                                <div class="mobile-user-details">
                                    <span class="mobile-user-name"><?= htmlspecialchars($user_name) ?></span>
                                    <span class="mobile-user-role">
                                        <?php if ($is_admin): ?>Admin<?php elseif ($is_instructor): ?>Instructor<?php else: ?>Student<?php endif; ?>
                                    </span>
                                </div>
                            </div>
                            
                            <div class="mobile-user-actions">
                                <?php if ($is_admin): ?>
                                    <a href="/admin/dashboard.php" class="mobile-nav-link">
                                        <i class="fas fa-tachometer-alt me-3"></i>Dashboard
                                    </a>
                                <?php elseif ($is_instructor): ?>
                                    <a href="/teacher/dashboard.php" class="mobile-nav-link">
                                        <i class="fas fa-tachometer-alt me-3"></i>Dashboard
                                    </a>
                                <?php else: ?>
                                    <a href="/student/dashboard.php" class="mobile-nav-link">
                                        <i class="fas fa-tachometer-alt me-3"></i>Dashboard
                                    </a>
                                <?php endif; ?>
                                
                                <a href="/auth/logout.php" class="mobile-nav-link mobile-logout">
                                    <i class="fas fa-sign-out-alt me-3"></i>Logout
                                </a>
                            </div>
                        <?php else: ?>
                            <!-- Not Logged In -->
                            <h6 class="mobile-nav-title">Get Started</h6>
                            <div class="mobile-auth-buttons">
                                <a href="/auth/login.php" class="mobile-auth-btn mobile-login-btn">
                                    <i class="fas fa-sign-in-alt me-2"></i>Log in
                                </a>
                                <a href="/auth/register.php" class="mobile-auth-btn mobile-signup-btn">
                                    <i class="fas fa-user-plus me-2"></i>Sign up
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 🔍 MOBILE SEARCH BAR - Full Width Row Underneath Navbar -->
    <div class="mobile-search-bar d-lg-none" id="mobileSearchBar" style="display: none;">
        <div class="container-fluid px-4">
            <div class="main-search-container" style="width: 100%; position: relative;">
                <div class="main-search-input-wrapper">
                    <i class="fas fa-search main-search-icon"></i>
                    <input type="text" 
                           placeholder="Search courses, categories, instructors..." 
                           id="mobile-search-input"
                           class="main-search-input"
                           autocomplete="off" />
                    <button type="button" class="main-search-clear-btn" id="mobileSearchClear" style="display: none;">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <!-- Dynamic Search Dropdown -->
                <div class="main-search-dropdown" id="mobileSearchDropdown">
                    <div class="main-search-results" id="mobileSearchResults">
                        <!-- Dynamic results will be inserted here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</nav>
<!-- Clean Navbar Styling -->
<style>
    /* Clean Navbar Styling */
    .navbar {
        background: white !important;
        border-bottom: 1px solid #e8eaed !important;
        box-shadow: 0 1px 2px 0 rgba(60,64,67,0.1) !important;
        padding: 10px 0 !important;
        transition: all 0.3s ease !important;
    }
    
    /* Bigger Logo, No Text */
    .navbar-logo {
        height: 45px !important;
        width: auto !important;
        transition: transform 0.2s ease !important;
    }
    
    .navbar-logo:hover {
        transform: scale(1.05) !important;
    }
    
    /* Navigation Links - Clean Style */
    .navbar-nav-desktop {
        gap: 0 !important;
    }
    
    .navbar-nav-desktop .nav-link {
        color: #1c1d1f !important;
        font-weight: 400 !important;
        font-size: 14px !important;
        padding: 10px 16px !important;
        text-decoration: none !important;
        transition: color 0.2s ease !important;
        white-space: nowrap !important;
    }
    
    .navbar-nav-desktop .nav-link:hover,
    .navbar-nav-desktop .nav-link.active {
        color: #a435f0 !important;
    }
    
    /* Search Container */
    .search-container {
        max-width: 380px !important;
    }
    
    .search-wrapper {
        position: relative !important;
        width: 100% !important;
    }
    
    .search-input {
        width: 100% !important;
        padding: 12px 45px 12px 16px !important;
        border: 1px solid #d1d7dc !important;
        border-radius: 50px !important;
        font-size: 14px !important;
        background-color: #fff !important;
        outline: none !important;
        transition: border-color 0.2s ease !important;
    }
    
    .search-input:focus {
        border-color: #1c1d1f !important;
    }
    
    .search-btn {
        position: absolute !important;
        right: 8px !important;
        top: 50% !important;
        transform: translateY(-50%) !important;
        background: none !important;
        border: none !important;
        padding: 8px !important;
        color: #6c757d !important;
        cursor: pointer !important;
    }
    
    /* Small Clean Cart Icon */
    .cart-icon {
        position: relative !important;
        color: #1c1d1f !important;
        text-decoration: none !important;
        font-size: 20px !important;
        transition: color 0.2s ease !important;
    }
    
    .cart-icon:hover {
        color: #a435f0 !important;
    }
    
    .cart-badge {
        position: absolute !important;
        top: -8px !important;
        right: -8px !important;
        background-color: #ec5252 !important;
        color: white !important;
        border-radius: 10px !important;
        padding: 2px 5px !important;
        font-size: 10px !important;
        font-weight: 600 !important;
        min-width: 16px !important;
        text-align: center !important;
        display: none !important;
    }
    
    /* Small Clean Auth Buttons */
    .auth-btns {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
    }
    
    .btn-login {
        color: #1c1d1f !important;
        text-decoration: none !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        padding: 8px 16px !important;
        border: 1px solid #1c1d1f !important;
        border-radius: 4px !important;
        transition: all 0.2s ease !important;
    }
    
    .btn-login:hover {
        background-color: #1c1d1f !important;
        color: white !important;
    }
    
    .btn-signup {
        background-color: #a435f0 !important;
        color: white !important;
        text-decoration: none !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        padding: 8px 16px !important;
        border: 1px solid #a435f0 !important;
        border-radius: 4px !important;
        transition: all 0.2s ease !important;
    }
    
    .btn-signup:hover {
        background-color: #8710d8 !important;
        border-color: #8710d8 !important;
        color: white !important;
    }
    
    /* User Button */
    .user-btn {
        color: #1c1d1f !important;
        text-decoration: none !important;
        font-size: 24px !important;
        padding: 8px !important;
        transition: color 0.2s ease !important;
    }
    
    .user-btn:hover {
        color: #a435f0 !important;
    }
    
    /* Dropdown Styling */
    .dropdown-menu {
        border: 1px solid #d1d7dc !important;
        border-radius: 8px !important;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1) !important;
        padding: 8px 0 !important;
        min-width: 180px !important;
    }
    
    .dropdown-item {
        font-size: 14px !important;
        padding: 8px 16px !important;
        color: #1c1d1f !important;
        transition: background-color 0.2s ease !important;
    }
    
    .dropdown-item:hover {
        background-color: #f7f9fa !important;
        color: #1c1d1f !important;
    }
    
    /* Mobile Navbar */
    .navbar-toggler {
        border: none !important;
        padding: 4px 8px !important;
    }
    
    .navbar-toggler:focus {
        box-shadow: none !important;
    }
    
    /* Mobile Menu */
    #mobileNav .navbar-nav .nav-link {
        color: #1c1d1f !important;
        font-size: 16px !important;
        padding: 12px 16px !important;
        border-bottom: 1px solid #f1f1f1 !important;
    }
    
    /* Mega Dropdown Styling */
    .mega-dropdown-wrapper {
        position: static !important;
    }
    
    .mega-dropdown {
        position: absolute !important;
        top: 100% !important;
        left: 10% !important;
        width: 80% !important;
        max-width: 1200px !important;
        margin-top: 0 !important;
        border: 1px solid #e8eaed !important;
        border-radius: 8px !important;
        box-shadow: 0 10px 25px rgba(0,0,0,0.1) !important;
        background: white !important;
        padding: 30px !important;
        z-index: 1050 !important;
        transform: translateX(-10%) !important;
    }
    
    .mega-dropdown .dropdown-header {
        font-size: 16px !important;
        font-weight: 600 !important;
        color: #1c1d1f !important;
        margin-bottom: 12px !important;
        padding-bottom: 8px !important;
        border-bottom: 2px solid #a435f0 !important;
    }
    
    .mega-dropdown .dropdown-item {
        padding: 8px 0 !important;
        font-size: 14px !important;
        color: #5c6670 !important;
        border: none !important;
        background: none !important;
        transition: color 0.2s ease !important;
    }
    
    .mega-dropdown .dropdown-item:hover {
        color: #a435f0 !important;
        background: none !important;
        padding-left: 8px !important;
        transition: all 0.2s ease !important;
    }
    
    .featured-course {
        padding: 12px !important;
        border: 1px solid #f1f1f1 !important;
        border-radius: 8px !important;
        background: #fafafa !important;
    }
    
    .featured-course img {
        margin-bottom: 8px !important;
    }
    
    .instructor-cta {
        margin-top: 8px !important;
    }
    
    /* Normal Dropdown */
    .normal-dropdown {
        border: 1px solid #d1d7dc !important;
        border-radius: 8px !important;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1) !important;
        padding: 8px 0 !important;
        min-width: 200px !important;
    }
    
    .normal-dropdown .dropdown-item {
        font-size: 14px !important;
        padding: 8px 16px !important;
        color: #1c1d1f !important;
        transition: background-color 0.2s ease !important;
    }
    
    .normal-dropdown .dropdown-item:hover {
        background-color: #f7f9fa !important;
        color: #a435f0 !important;
    }
    
    /* Navigation hover effects */
    .navbar-nav-desktop a:hover {
        color: #a435f0 !important;
    }
    
    /* Responsive */
    @media (max-width: 991.98px) {
        .navbar-logo {
            height: 35px !important;
        }
        
        .search-container {
            display: none !important;
        }
        
        .mega-dropdown {
            display: none !important;
        }
    }
    
    /* Mobile Responsive */
    @media (max-width: 991px) {
        .search-container {
            display: none !important;
        }
        
        .navbar-nav {
            margin-top: 1rem !important;
            padding-top: 1rem !important;
            border-top: 1px solid #e5e5e5 !important;
        }
        
        .navbar-nav .nav-link {
            padding: 12px 0 !important;
            margin: 0 !important;
        }
    }
    
    /* Logo Hover Effect */
    .navbar-brand img:hover {
        transform: scale(1.05) !important;
        transition: transform 0.2s ease !important;
    }
    
    /* Main Search Bar Styles */
    .main-search-container {
        position: relative;
        width: 100%;
    }

    .main-search-input-wrapper {
        position: relative;
        display: flex;
        align-items: center;
        background: #f8f9fa;
        border: 2px solid #e9ecef;
        border-radius: 50px;
        padding: 0 16px;
        transition: all 0.3s ease;
    }

    .main-search-input-wrapper:focus-within {
        border-color: #276be4;
        box-shadow: 0 0 0 3px rgba(39, 107, 228, 0.1);
        background: white;
    }

    .main-search-icon {
        color: #6c757d;
        font-size: 16px;
        margin-right: 12px;
    }

    .main-search-input {
        flex: 1;
        border: none;
        background: transparent;
        padding: 14px 0;
        font-size: 14px;
        color: #212529;
        outline: none;
        min-width: 0;
    }

    .main-search-input::placeholder {
        color: #9ca3af;
    }

    .main-search-clear-btn {
        background: none;
        border: none;
        color: #6c757d;
        cursor: pointer;
        padding: 4px;
        border-radius: 50%;
        transition: all 0.2s ease;
        margin-left: 8px;
    }

    .main-search-clear-btn:hover {
        background: #e9ecef;
        color: #495057;
    }

    /* Search Dropdown Styles */
    .main-search-dropdown {
        position: absolute;
        top: calc(100% + 8px);
        left: 0;
        right: 0;
        background: white;
        border-radius: 12px;
        box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
        border: 1px solid #e9ecef;
        max-height: 500px;
        overflow-y: auto;
        display: none;
        z-index: 9999;
    }

    .main-search-dropdown.show {
        display: block;
    }

    .main-search-results {
        padding: 12px 0;
    }

    /* Search Result Item Styles */
    .main-search-item {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 12px 16px;
        text-decoration: none;
        color: inherit;
        transition: all 0.2s ease;
        border-left: 3px solid transparent;
    }

    .main-search-item:hover {
        background-color: #f8f9fa;
        text-decoration: none;
        color: inherit;
        border-left-color: #276be4;
    }

    .main-search-thumb {
        width: 50px;
        height: 35px;
        border-radius: 6px;
        overflow: hidden;
        flex-shrink: 0;
        border: 1px solid #e9ecef;
        background: #f8f9fa;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .main-search-thumb img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }

    .main-search-info {
        flex: 1;
        min-width: 0;
        overflow: hidden;
    }

    .main-search-title {
        font-size: 14px;
        font-weight: 600;
        color: #212529;
        margin: 0 0 4px 0;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        line-height: 1.3;
    }

    .main-search-meta {
        font-size: 12px;
        color: #6c757d;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .main-search-price {
        font-weight: 600;
        color: #28a745;
    }

    .main-search-rating {
        display: flex;
        align-items: center;
        gap: 2px;
        color: #ffc107;
    }

    .main-search-category {
        background: rgba(39, 107, 228, 0.1);
        color: #276be4;
        padding: 2px 6px;
        border-radius: 12px;
        font-size: 10px;
        font-weight: 500;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    /* Search Section Headers */
    .main-search-section {
        padding: 8px 16px 4px;
        font-size: 11px;
        font-weight: 600;
        color: #6c757d;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        background: #f8f9fa;
        margin: 8px 0 4px;
    }

    .main-search-section:first-child {
        margin-top: 0;
    }

    /* Category Suggestions */
    .main-search-category-item {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 10px 16px;
        text-decoration: none;
        color: inherit;
        transition: all 0.2s ease;
        cursor: pointer;
    }

    .main-search-category-item:hover {
        background-color: #f8f9fa;
        text-decoration: none;
        color: inherit;
    }

    .main-search-category-name {
        font-size: 13px;
        font-weight: 500;
        color: #212529;
    }

    .main-search-category-count {
        font-size: 11px;
        color: #6c757d;
        background: #e9ecef;
        padding: 2px 6px;
        border-radius: 10px;
    }

    /* No Results State */
    .main-search-no-results {
        text-align: center;
        padding: 32px 16px;
        color: #6c757d;
    }

    .main-search-no-results i {
        font-size: 48px;
        color: #dee2e6;
        margin-bottom: 12px;
        display: block;
    }

    .main-search-no-results h6 {
        margin-bottom: 8px;
        color: #495057;
    }

    .main-search-no-results p {
        margin: 0;
        font-size: 13px;
    }

    /* View All Results Button */
    .main-search-footer {
        padding: 12px 16px;
        border-top: 1px solid #e9ecef;
        margin-top: 8px;
        background: #f8f9fa;
    }

    .main-search-view-all {
        width: 100%;
        padding: 10px 16px;
        background: #276be4;
        color: white;
        border: none;
        border-radius: 6px;
        font-weight: 600;
        font-size: 13px;
        cursor: pointer;
        transition: all 0.2s ease;
    }

    .main-search-view-all:hover {
        background: #1e5bcc;
        transform: translateY(-1px);
    }

    /* Loading State */
    .main-search-loading {
        text-align: center;
        padding: 24px 16px;
        color: #6c757d;
    }

    .main-search-loading i {
        font-size: 24px;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }

    /* 📱 MOBILE RESPONSIVE STYLES - Clean & Beautiful */
    
    /* Mobile Icons Container */
    .mobile-icons-container {
        padding-right: 8px;
        overflow: visible !important; /* Allow cart counter to show outside */
        position: relative;
    }
    
    /* Mobile Icon Buttons - Glowing & Clean */
    .mobile-icon-btn {
        width: 44px;
        height: 44px;
        border-radius: 12px;
        border: none;
        background: linear-gradient(135deg, #f59e0b, #f59e0b);
        color: white;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 18px;
        text-decoration: none;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        cursor: pointer;
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.25);
        position: relative;
        overflow: visible; /* Allow cart counter to show outside */
    }
    
    /* Mobile Cart Counter - Special positioning */
    .mobile-icon-btn .cart-counter {
        position: absolute !important;
        top: -6px !important;
        right: -6px !important;
        background: #ec5252 !important;
        color: white !important;
        border-radius: 12px !important;
        padding: 3px 6px !important;
        font-size: 10px !important;
        font-weight: 700 !important;
        min-width: 18px !important;
        height: 18px !important;
        /* Removed display: none !important so JavaScript can control it */
        z-index: 10 !important;
        border: 2px solid white !important;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2) !important;
        line-height: 1 !important;
        text-align: center !important;
    }
    
    /* Breathing/Pulsing Animation for Active States */
    .mobile-icon-btn::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(135deg, rgba(255, 255, 255, 0.2), rgba(255, 255, 255, 0.1));
        opacity: 0;
        transition: opacity 0.3s ease;
        border-radius: 12px;
    }
    
    .mobile-icon-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 20px rgba(245, 158, 11, 0.4);
        background: linear-gradient(135deg, #f59e0b, #d97706);
    }
    
    .mobile-icon-btn:hover::before {
        opacity: 1;
    }
    
    .mobile-icon-btn:active {
        transform: translateY(0);
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.25);
    }
    
    .mobile-icon-btn i {
        z-index: 1;
        position: relative;
    }
    
    /* Active state for search toggle */
    .mobile-search-toggle.active {
        background: linear-gradient(135deg, #3b82f6, #2563eb);
        box-shadow: 0 4px 12px rgba(59, 130, 246, 0.25);
        animation: pulse 2s ease-in-out infinite alternate;
    }
    
    @keyframes pulse {
        0% { box-shadow: 0 4px 12px rgba(59, 130, 246, 0.25); }
        100% { box-shadow: 0 6px 20px rgba(59, 130, 246, 0.4); }
    }
    
    /* Mobile Search Bar */
    .mobile-search-bar {
        position: absolute;
        top: 100%;
        left: 0;
        right: 0;
        background: white;
        border-bottom: 1px solid #e8eaed;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        z-index: 1000;
        transform: translateY(-10px);
        opacity: 0;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    }
    
    .mobile-search-bar.show {
        display: block !important;
        transform: translateY(0);
        opacity: 1;
    }
    
    .mobile-search-bar .main-search-container {
        margin: 16px 0;
    }
    
    /* Mobile Menu Dropdown */
    .mobile-menu-dropdown {
        position: absolute;
        top: 100%;
        left: 0;
        right: 0;
        background: white;
        border-bottom: 1px solid #e8eaed;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        z-index: 999;
        max-height: calc(100vh - 70px);
        overflow-y: auto;
        transform: translateY(-10px);
        opacity: 0;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    }
    
    .mobile-menu-dropdown.show {
        display: block !important;
        transform: translateY(0);
        opacity: 1;
    }
    
    /* Mobile Menu Content */
    .mobile-menu-content {
        background: linear-gradient(135deg, #fefdf8, #fef9e7);
        border-radius: 12px;
        margin: 8px;
        overflow: hidden;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
    }
    
    .mobile-nav-section {
        padding: 20px 0 16px;
    }
    
    .mobile-nav-title {
        font-size: 13px;
        font-weight: 700;
        color: #8b5a00;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin: 0 0 12px 20px;
    }
    
    .mobile-nav-link {
        display: flex;
        align-items: center;
        padding: 14px 20px;
        text-decoration: none;
        color: #1c1d1f;
        font-size: 15px;
        font-weight: 500;
        border-radius: 0;
        transition: all 0.3s ease;
        position: relative;
    }
    
    .mobile-nav-link::before {
        content: '';
        position: absolute;
        left: 0;
        top: 0;
        bottom: 0;
        width: 4px;
        background: linear-gradient(135deg, #f59e0b, #d97706);
        opacity: 0;
        transform: scaleY(0);
        transition: all 0.3s ease;
    }
    
    .mobile-nav-link:hover {
        background: rgba(245, 158, 11, 0.1);
        color: #f59e0b;
        padding-left: 24px;
    }
    
    .mobile-nav-link:hover::before {
        opacity: 1;
        transform: scaleY(1);
    }
    
    .mobile-nav-link i:first-child {
        width: 20px;
        color: #f59e0b;
    }
    
    .mobile-nav-divider {
        height: 1px;
        background: linear-gradient(90deg, transparent, #e8eaed, transparent);
        margin: 16px 20px;
    }
    
    /* Mobile Auth Section */
    .mobile-auth-section {
        padding: 16px 0 20px;
    }
    
    .mobile-user-info {
        display: flex;
        align-items: center;
        padding: 16px 20px;
        background: linear-gradient(135deg, rgba(245, 158, 11, 0.1), rgba(245, 158, 11, 0.05));
        margin: 0 12px 16px;
        border-radius: 12px;
    }
    
    .mobile-user-avatar {
        width: 48px;
        height: 48px;
        border-radius: 50%;
        background: linear-gradient(135deg, #f59e0b, #d97706);
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 24px;
        margin-right: 12px;
    }
    
    .mobile-user-details {
        flex: 1;
    }
    
    .mobile-user-name {
        display: block;
        font-weight: 600;
        color: #1c1d1f;
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .mobile-user-role {
        font-size: 12px;
        color: #8b5a00;
        text-transform: uppercase;
        font-weight: 600;
        letter-spacing: 0.5px;
    }
    
    .mobile-user-actions {
        margin-top: 8px;
    }
    
    .mobile-logout {
        color: #dc3545 !important;
    }
    
    .mobile-logout:hover {
        background: rgba(220, 53, 69, 0.1) !important;
        color: #dc3545 !important;
    }
    
    .mobile-logout i {
        color: #dc3545 !important;
    }
    
    /* Mobile Auth Buttons */
    .mobile-auth-buttons {
        display: flex;
        flex-direction: column;
        gap: 12px;
        padding: 0 20px;
    }
    
    .mobile-auth-btn {
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 14px 20px;
        border-radius: 12px;
        text-decoration: none;
        font-weight: 600;
        font-size: 15px;
        transition: all 0.3s ease;
        border: 2px solid;
    }
    
    .mobile-login-btn {
        background: transparent;
        color: #f59e0b;
        border-color: #f59e0b;
    }
    
    .mobile-login-btn:hover {
        background: #f59e0b;
        color: white;
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.25);
    }
    
    .mobile-signup-btn {
        background: linear-gradient(135deg, #f59e0b, #d97706);
        color: white;
        border-color: #f59e0b;
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.25);
    }
    
    .mobile-signup-btn:hover {
        background: linear-gradient(135deg, #d97706, #b45309);
        transform: translateY(-2px);
        box-shadow: 0 6px 16px rgba(245, 158, 11, 0.35);
        color: white;
    }
    
    /* Smooth slide-in animations */
    .mobile-nav-link,
    .mobile-auth-btn {
        animation: slideInRight 0.3s ease forwards;
        opacity: 0;
        transform: translateX(20px);
    }
    
    .mobile-menu-dropdown.show .mobile-nav-link:nth-child(1) { animation-delay: 0.1s; }
    .mobile-menu-dropdown.show .mobile-nav-link:nth-child(2) { animation-delay: 0.15s; }
    .mobile-menu-dropdown.show .mobile-nav-link:nth-child(3) { animation-delay: 0.2s; }
    .mobile-menu-dropdown.show .mobile-nav-link:nth-child(4) { animation-delay: 0.25s; }
    .mobile-menu-dropdown.show .mobile-nav-link:nth-child(5) { animation-delay: 0.3s; }
    .mobile-menu-dropdown.show .mobile-auth-btn:nth-child(1) { animation-delay: 0.35s; }
    .mobile-menu-dropdown.show .mobile-auth-btn:nth-child(2) { animation-delay: 0.4s; }
    
    @keyframes slideInRight {
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }
    
    /* 📋 MOBILE EXPANDABLE SECTIONS */
    
    .mobile-expandable-section {
        margin: 4px 0;
    }
    
    .mobile-chevron {
        transition: transform 0.3s ease;
        font-size: 12px !important;
    }
    
    .mobile-expandable-section.expanded .mobile-chevron {
        transform: rotate(90deg);
    }
    
    .mobile-expandable-content {
        max-height: 0;
        overflow: hidden;
        transition: max-height 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        background: linear-gradient(135deg, rgba(245, 158, 11, 0.05), rgba(245, 158, 11, 0.02));
        margin-left: 20px;
        border-left: 3px solid rgba(245, 158, 11, 0.3);
        border-radius: 0 8px 8px 0;
    }
    
    .mobile-expandable-section.expanded .mobile-expandable-content {
        max-height: 1000px; /* Large enough for content */
    }
    
    /* 🎨 MOBILE CATEGORIES STYLING */
    
    .mobile-category-theme {
        padding: 12px 16px 8px;
        border-bottom: 1px solid rgba(245, 158, 11, 0.1);
    }
    
    .mobile-category-theme:last-child {
        border-bottom: none;
    }
    
    .mobile-theme-header {
        display: flex;
        align-items: center;
        font-weight: 600;
        color: #8b5a00;
        font-size: 13px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 8px;
        padding: 8px 0;
        border-bottom: 1px solid rgba(245, 158, 11, 0.2);
    }
    
    .mobile-theme-header i {
        color: #f59e0b;
        width: 16px;
    }
    
    .mobile-category-item {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 10px 8px;
        text-decoration: none;
        color: #1c1d1f;
        font-size: 14px;
        border-radius: 6px;
        transition: all 0.3s ease;
        margin: 2px 0;
    }
    
    .mobile-category-item:hover {
        background: rgba(245, 158, 11, 0.1);
        color: #f59e0b;
        text-decoration: none;
        padding-left: 12px;
    }
    
    .mobile-course-count {
        background: linear-gradient(135deg, #f59e0b, #d97706);
        color: white;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
        min-width: 20px;
        text-align: center;
    }
    
    .mobile-no-categories {
        padding: 16px;
        text-align: center;
        color: #6c757d;
    }
    
    /* 🎓 MOBILE TEACH STYLING */
    
    .mobile-teach-section {
        padding: 16px 16px 12px;
        border-bottom: 1px solid rgba(245, 158, 11, 0.1);
    }
    
    .mobile-teach-section:last-child {
        border-bottom: none;
    }
    
    .mobile-section-title {
        font-weight: 600;
        color: #8b5a00;
        font-size: 13px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 12px;
        padding-bottom: 8px;
        border-bottom: 1px solid rgba(245, 158, 11, 0.2);
    }
    
    .mobile-teach-item {
        display: flex;
        align-items: flex-start;
        padding: 12px 8px;
        text-decoration: none;
        color: #1c1d1f;
        border-radius: 8px;
        transition: all 0.3s ease;
        margin: 4px 0;
    }
    
    .mobile-teach-item:hover {
        background: rgba(245, 158, 11, 0.1);
        color: #1c1d1f;
        text-decoration: none;
        transform: translateX(4px);
    }
    
    .mobile-teach-item i {
        width: 20px;
        flex-shrink: 0;
        margin-top: 2px;
    }
    
    .mobile-teach-item div {
        flex: 1;
    }
    
    .mobile-teach-item strong {
        display: block;
        font-size: 14px;
        font-weight: 600;
        color: #1c1d1f;
        margin-bottom: 2px;
    }
    
    .mobile-teach-item small {
        font-size: 12px;
        color: #6c757d;
        line-height: 1.3;
    }
    
    /* 🎯 MOBILE TEACH CTA CARD */
    
    .mobile-teach-cta {
        padding: 16px;
    }
    
    .mobile-cta-card {
        background: linear-gradient(135deg, #f59e0b, #d97706);
        padding: 20px;
        border-radius: 12px;
        color: white;
        text-align: center;
        box-shadow: 0 4px 16px rgba(245, 158, 11, 0.3);
    }
    
    .mobile-cta-card i {
        font-size: 2rem;
        margin-bottom: 12px;
        opacity: 0.9;
    }
    
    .mobile-cta-card h6 {
        color: white;
        font-weight: 700;
        font-size: 16px;
        margin-bottom: 8px;
    }
    
    .mobile-cta-card p {
        font-size: 13px;
        margin-bottom: 16px;
        color: rgba(255, 255, 255, 0.9);
        line-height: 1.4;
    }
    
    .mobile-cta-btn {
        background: white;
        color: #f59e0b;
        padding: 10px 20px;
        border-radius: 20px;
        text-decoration: none;
        font-weight: 700;
        font-size: 13px;
        display: inline-block;
        transition: all 0.3s ease;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    }
    
    .mobile-cta-btn:hover {
        background: #fef9e7;
        color: #d97706;
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        text-decoration: none;
    }
    
    /* Responsive adjustments */
    @media (max-width: 576px) {
        .mobile-icons-container {
            gap: 16px;
        }
        
        .mobile-icon-btn {
            width: 40px;
            height: 40px;
            font-size: 16px;
        }
        
        .mobile-nav-link {
            padding: 12px 16px;
            font-size: 14px;
        }
        
        .mobile-auth-btn {
            padding: 12px 16px;
            font-size: 14px;
        }
        
        .mobile-expandable-content {
            margin-left: 16px;
        }
        
        .mobile-category-theme,
        .mobile-teach-section {
            padding-left: 12px;
            padding-right: 12px;
        }
    }
</style>

<!-- Navbar JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Update cart count from localStorage - Both Desktop & Mobile
    function updateCartCount() {
        const cartItems = JSON.parse(localStorage.getItem('cart') || '[]');
        const cartCount = cartItems.length;
        
        // Update all cart counters (desktop and mobile)
        const cartBadges = document.querySelectorAll('.cart-counter');
        
        console.log(`🛒 Updating cart count: ${cartCount} items found`);
        console.log(`🛒 Found ${cartBadges.length} cart counter elements`);
        
        cartBadges.forEach((cartBadge, index) => {
            console.log(`🛒 Updating cart badge ${index + 1}:`, cartBadge);
            console.log(`🛒 Badge ${index + 1} classes:`, cartBadge.className);
            console.log(`🛒 Badge ${index + 1} parent:`, cartBadge.parentElement);
            console.log(`🛒 Badge ${index + 1} current display:`, cartBadge.style.display);
            
            cartBadge.textContent = cartCount;
            cartBadge.style.display = cartCount > 0 ? 'inline-block' : 'none';
            
            console.log(`🛒 Badge ${index + 1} set to display:`, cartBadge.style.display);
            console.log(`🛒 Badge ${index + 1} text content:`, cartBadge.textContent);
        });
    }
    
    // Initialize cart count on page load
    updateCartCount();
    
    // Initialize dynamic search functionality
    initializeMainSearch();
    
    // Update cart count when localStorage changes
    window.addEventListener('storage', updateCartCount);
    
    // Search form enhancement
    const searchForm = document.querySelector('.search-container form');
    const searchInput = document.querySelector('.search-input');
    
    if (searchForm && searchInput) {
        // Add search suggestions (placeholder for future enhancement)
        searchInput.addEventListener('focus', function() {
            this.style.borderColor = '#1c1d1f';
        });
        
        searchInput.addEventListener('blur', function() {
            this.style.borderColor = '#d1d7dc';
        });
        
        // Prevent empty searches
        searchForm.addEventListener('submit', function(e) {
            if (!searchInput.value.trim()) {
                e.preventDefault();
                searchInput.focus();
            }
        });
    }
    
    // Navbar scroll effect
    let lastScrollY = window.scrollY;
    
    window.addEventListener('scroll', function() {
        const navbar = document.querySelector('.navbar');
        const currentScrollY = window.scrollY;
        
        if (currentScrollY > lastScrollY && currentScrollY > 100) {
            // Scrolling down
            navbar.style.transform = 'translateY(-100%)';
        } else {
            // Scrolling up
            navbar.style.transform = 'translateY(0)';
        }
        
        lastScrollY = currentScrollY;
    });
    
    // Mobile menu enhancement
    const navbarToggler = document.querySelector('.navbar-toggler');
    const navbarCollapse = document.querySelector('.navbar-collapse');
    
    if (navbarToggler && navbarCollapse) {
        navbarToggler.addEventListener('click', function() {
            setTimeout(() => {
                if (navbarCollapse.classList.contains('show')) {
                    document.body.style.overflow = 'hidden';
                } else {
                    document.body.style.overflow = 'auto';
                }
            }, 100);
        });
    }
    
    // Dropdown functionality
    const dropdowns = document.querySelectorAll('.navbar-dropdown');
    
    dropdowns.forEach(dropdown => {
        const dropdownMenu = dropdown.querySelector('.dropdown-menu');
        let hoverTimeout;
        
        // Show dropdown on hover
        dropdown.addEventListener('mouseenter', function() {
            clearTimeout(hoverTimeout);
            
            // Hide all other dropdowns
            dropdowns.forEach(otherDropdown => {
                if (otherDropdown !== dropdown) {
                    const otherMenu = otherDropdown.querySelector('.dropdown-menu');
                    if (otherMenu) {
                        otherMenu.style.display = 'none';
                    }
                }
            });
            
            // Show current dropdown
            if (dropdownMenu) {
                dropdownMenu.style.display = 'block';
                dropdownMenu.style.opacity = '0';
                dropdownMenu.style.transform = 'translateY(-10px)';
                
                // Animate in
                setTimeout(() => {
                    dropdownMenu.style.transition = 'all 0.3s ease';
                    dropdownMenu.style.opacity = '1';
                    dropdownMenu.style.transform = 'translateY(0)';
                }, 10);
            }
        });
        
        // Hide dropdown on mouse leave with delay
        dropdown.addEventListener('mouseleave', function() {
            hoverTimeout = setTimeout(() => {
                if (dropdownMenu) {
                    dropdownMenu.style.transition = 'all 0.2s ease';
                    dropdownMenu.style.opacity = '0';
                    dropdownMenu.style.transform = 'translateY(-10px)';
                    
                    setTimeout(() => {
                        dropdownMenu.style.display = 'none';
                    }, 200);
                }
            }, 100);
        });
        
        // Keep dropdown open when hovering over the menu itself
        if (dropdownMenu) {
            dropdownMenu.addEventListener('mouseenter', function() {
                clearTimeout(hoverTimeout);
            });
            
            dropdownMenu.addEventListener('mouseleave', function() {
                hoverTimeout = setTimeout(() => {
                    dropdownMenu.style.transition = 'all 0.2s ease';
                    dropdownMenu.style.opacity = '0';
                    dropdownMenu.style.transform = 'translateY(-10px)';
                    
                    setTimeout(() => {
                        dropdownMenu.style.display = 'none';
                    }, 200);
                }, 100);
            });
        }
    });
    
    // Close dropdowns when clicking outside
    document.addEventListener('click', function(event) {
        if (!event.target.closest('.navbar-dropdown')) {
            dropdowns.forEach(dropdown => {
                const menu = dropdown.querySelector('.dropdown-menu');
                if (menu) {
                    menu.style.display = 'none';
                }
            });
        }
    });

    // Dynamic Search Implementation
    function initializeMainSearch() {
        const searchInput = document.getElementById('main-search-input');
        const searchContainer = document.getElementById('mainSearchContainer');
        const dropdown = document.getElementById('mainSearchDropdown');
        const clearBtn = document.getElementById('mainSearchClear');
        
        let searchTimeout;
        let currentSearchRequest;
        
        if (!searchInput || !searchContainer || !dropdown) return;
        
        // Real-time search as user types
        searchInput.addEventListener('input', performMainSearch);
        
        // Show/hide clear button
        searchInput.addEventListener('input', function() {
            clearBtn.style.display = this.value.trim() ? 'block' : 'none';
        });
        
        // Clear search
        clearBtn.addEventListener('click', function() {
            searchInput.value = '';
            clearBtn.style.display = 'none';
            dropdown.classList.remove('show');
            searchInput.focus();
        });
        
        // Show dropdown when input is focused and has content
        searchInput.addEventListener('focus', function() {
            if (this.value.trim()) {
                performMainSearch();
            }
        });
        
        // Enter key search - redirect to full search page
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                const query = this.value.trim();
                if (query) {
                    window.location.href = `/search.php?q=${encodeURIComponent(query)}`;
                }
            }
        });
        
        // Hide dropdown when clicking outside
        document.addEventListener('click', function(event) {
            if (!searchContainer.contains(event.target)) {
                dropdown.classList.remove('show');
            }
        });
        
        function performMainSearch() {
            const query = searchInput.value.trim();
            const results = document.getElementById('mainSearchResults');
            
            // Clear previous timeout
            clearTimeout(searchTimeout);
            
            // Cancel previous request
            if (currentSearchRequest) {
                currentSearchRequest.abort();
            }
            
            // If no search query, hide dropdown
            if (!query) {
                dropdown.classList.remove('show');
                return;
            }
            
            // Show loading state
            showMainSearchLoading();
            dropdown.classList.add('show');
            
            // Debounced search
            searchTimeout = setTimeout(() => {
                const params = new URLSearchParams();
                params.append('q', query);
                params.append('limit', '8');
                
                currentSearchRequest = new AbortController();
                
                fetch(`/api/main_navbar_search.php?${params.toString()}`, {
                    signal: currentSearchRequest.signal
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        displayMainSearchResults(data.data, query);
                    } else {
                        showMainSearchError();
                    }
                })
                .catch(error => {
                    if (error.name !== 'AbortError') {
                        console.error('Search error:', error);
                        showMainSearchError();
                    }
                });
            }, 300);
        }
        
        function displayMainSearchResults(data, query) {
            const results = document.getElementById('mainSearchResults');
            const { courses } = data;
            
            let html = '';
            
            // Show courses if available
            if (courses && courses.length > 0) {
                courses.forEach(course => {
                    const courseInitials = course.title ? course.title.substring(0, 2).toUpperCase() : 'CO';
                    
                    html += `
                        <a href="${course.url}" class="main-search-item">
                            <div class="main-search-thumb">
                                <img src="${course.thumbnail}" 
                                     alt="${course.title}" 
                                     onerror="this.style.display='none'; this.parentElement.innerHTML='${courseInitials}'; this.parentElement.style.background='linear-gradient(135deg, #276be4 0%, #1e5bcc 100%)'; this.parentElement.style.color='white'; this.parentElement.style.fontWeight='600'; this.parentElement.style.fontSize='11px';">
                            </div>
                            <div class="main-search-info">
                                <div class="main-search-title">${course.title}</div>
                                <div class="main-search-meta">
                                    <span class="main-search-price">${course.price}</span>
                                    <span class="main-search-category">${course.category}</span>
                                    <span>${course.teacher_name}</span>
                                </div>
                            </div>
                        </a>
                    `;
                });
            }
            
            // Add "View All Results" button if there are results
            if (courses.length > 0) {
                html += `
                    <div class="main-search-footer">
                        <button class="main-search-view-all" onclick="viewAllMainResults('${query}')">
                            <i class="fas fa-search me-2"></i>
                            View All Results for "${query}"
                        </button>
                    </div>
                `;
            }
            
            // Show no results if nothing found
            if (!courses.length) {
                html = `
                    <div class="main-search-no-results">
                        <i class="fas fa-search"></i>
                        <h6>No results found</h6>
                        <p>Try searching for courses, categories, or instructors</p>
                    </div>
                `;
            }
            
            results.innerHTML = html;
        }
        
        function showMainSearchLoading() {
            const results = document.getElementById('mainSearchResults');
            results.innerHTML = `
                <div class="main-search-loading">
                    <i class="fas fa-spinner"></i>
                    <p>Searching...</p>
                </div>
            `;
        }
        
        function showMainSearchError() {
            const results = document.getElementById('mainSearchResults');
            results.innerHTML = `
                <div class="main-search-no-results">
                    <i class="fas fa-exclamation-triangle"></i>
                    <h6>Search Error</h6>
                    <p>Please try again in a moment</p>
                </div>
            `;
        }
    }
    
    // Global functions for search interactions
    window.searchByCategory = function(categoryName) {
        window.location.href = `/search.php?category=${encodeURIComponent(categoryName)}`;
    };
    
    window.viewAllMainResults = function(query) {
        window.location.href = `/search.php?q=${encodeURIComponent(query)}`;
    };

    // 📱 MOBILE NAVBAR FUNCTIONALITY
    initializeMobileNavbar();
    
    function initializeMobileNavbar() {
        const mobileSearchToggle = document.getElementById('mobileSearchToggle');
        const mobileMenuToggle = document.getElementById('mobileMenuToggle');
        const mobileSearchBar = document.getElementById('mobileSearchBar');
        const mobileMenuDropdown = document.getElementById('mobileMenuDropdown');
        const mobileSearchInput = document.getElementById('mobile-search-input');
        const mobileSearchClear = document.getElementById('mobileSearchClear');
        
        if (!mobileSearchToggle || !mobileMenuToggle || !mobileSearchBar || !mobileMenuDropdown) {
            return; // Mobile elements not found
        }
        
        // 🔍 MOBILE SEARCH TOGGLE FUNCTIONALITY
        mobileSearchToggle.addEventListener('click', function(e) {
            console.log('🔍 Mobile search icon clicked!');
            e.preventDefault();
            e.stopPropagation();
            
            const isSearchVisible = mobileSearchBar.classList.contains('show');
            console.log('📱 Search currently visible:', isSearchVisible);
            
            if (isSearchVisible) {
                // Hide search
                console.log('🙈 Hiding mobile search');
                hideMobileSearch();
            } else {
                // Show search and hide menu if visible
                console.log('👁️ Showing mobile search');
                hideMobileMenu();
                showMobileSearch();
            }
        });
        
        // 📋 MOBILE MENU TOGGLE FUNCTIONALITY
        mobileMenuToggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const isMenuVisible = mobileMenuDropdown.classList.contains('show');
            
            if (isMenuVisible) {
                // Hide menu
                hideMobileMenu();
            } else {
                // Show menu and hide search if visible
                hideMobileSearch();
                showMobileMenu();
            }
        });
        
        // 🎯 MOBILE SEARCH FUNCTIONALITY - Same as desktop
        if (mobileSearchInput) {
            let mobileSearchTimeout;
            let mobileCurrentSearchRequest;
            
            // Input event for real-time search
            mobileSearchInput.addEventListener('input', function() {
                const query = this.value.trim();
                
                // Show/hide clear button
                if (mobileSearchClear) {
                    mobileSearchClear.style.display = query ? 'block' : 'none';
                }
                
                // Perform search
                performMobileSearch();
            });
            
            // Clear search
            if (mobileSearchClear) {
                mobileSearchClear.addEventListener('click', function() {
                    mobileSearchInput.value = '';
                    mobileSearchClear.style.display = 'none';
                    const mobileDropdown = document.getElementById('mobileSearchDropdown');
                    if (mobileDropdown) {
                        mobileDropdown.classList.remove('show');
                    }
                    mobileSearchInput.focus();
                });
            }
            
            // Focus and search behavior
            mobileSearchInput.addEventListener('focus', function() {
                if (this.value.trim()) {
                    performMobileSearch();
                }
            });
            
            // Enter key search
            mobileSearchInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    const query = this.value.trim();
                    if (query) {
                        window.location.href = `/search.php?q=${encodeURIComponent(query)}`;
                    }
                }
            });
            
            function performMobileSearch() {
                const query = mobileSearchInput.value.trim();
                const mobileResults = document.getElementById('mobileSearchResults');
                const mobileDropdown = document.getElementById('mobileSearchDropdown');
                
                if (!mobileResults || !mobileDropdown) return;
                
                // Clear previous timeout
                clearTimeout(mobileSearchTimeout);
                
                // Cancel previous request
                if (mobileCurrentSearchRequest) {
                    mobileCurrentSearchRequest.abort();
                }
                
                // If no search query, hide dropdown
                if (!query) {
                    mobileDropdown.classList.remove('show');
                    return;
                }
                
                // Show loading state
                showMobileSearchLoading();
                mobileDropdown.classList.add('show');
                
                // Debounced search
                mobileSearchTimeout = setTimeout(() => {
                    const params = new URLSearchParams();
                    params.append('q', query);
                    params.append('limit', '6'); // Fewer results for mobile
                    
                    mobileCurrentSearchRequest = new AbortController();
                    
                    fetch(`/api/main_navbar_search.php?${params.toString()}`, {
                        signal: mobileCurrentSearchRequest.signal
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            displayMobileSearchResults(data.data, query);
                        } else {
                            showMobileSearchError();
                        }
                    })
                    .catch(error => {
                        if (error.name !== 'AbortError') {
                            console.error('Mobile search error:', error);
                            showMobileSearchError();
                        }
                    });
                }, 300);
            }
            
            function displayMobileSearchResults(data, query) {
                const results = document.getElementById('mobileSearchResults');
                if (!results) return;
                
                const { courses } = data;
                let html = '';
                
                // Show courses if available
                if (courses && courses.length > 0) {
                    courses.forEach(course => {
                        const courseInitials = course.title ? course.title.substring(0, 2).toUpperCase() : 'CO';
                        
                        html += `
                            <a href="${course.url}" class="main-search-item">
                                <div class="main-search-thumb">
                                    <img src="${course.thumbnail}" 
                                         alt="${course.title}" 
                                         onerror="this.style.display='none'; this.parentElement.innerHTML='${courseInitials}'; this.parentElement.style.background='linear-gradient(135deg, #f59e0b 0%, #d97706 100%)'; this.parentElement.style.color='white'; this.parentElement.style.fontWeight='600'; this.parentElement.style.fontSize='11px';">
                                </div>
                                <div class="main-search-info">
                                    <div class="main-search-title">${course.title}</div>
                                    <div class="main-search-meta">
                                        <span class="main-search-price">${course.price}</span>
                                        <span class="main-search-category">${course.category}</span>
                                        <span>${course.teacher_name}</span>
                                    </div>
                                </div>
                            </a>
                        `;
                    });
                }
                
                // Add "View All Results" button if there are results
                if (courses.length > 0) {
                    html += `
                        <div class="main-search-footer">
                            <button class="main-search-view-all" onclick="viewAllMainResults('${query}')">
                                <i class="fas fa-search me-2"></i>
                                View All Results for "${query}"
                            </button>
                        </div>
                    `;
                }
                
                // Show no results if nothing found
                if (!courses.length) {
                    html = `
                        <div class="main-search-no-results">
                            <i class="fas fa-search"></i>
                            <h6>No results found</h6>
                            <p>Try searching for courses, categories, or instructors</p>
                        </div>
                    `;
                }
                
                results.innerHTML = html;
            }
            
            function showMobileSearchLoading() {
                const results = document.getElementById('mobileSearchResults');
                if (!results) return;
                
                results.innerHTML = `
                    <div class="main-search-loading">
                        <i class="fas fa-spinner"></i>
                        <p>Searching...</p>
                    </div>
                `;
            }
            
            function showMobileSearchError() {
                const results = document.getElementById('mobileSearchResults');
                if (!results) return;
                
                results.innerHTML = `
                    <div class="main-search-no-results">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h6>Search Error</h6>
                        <p>Please try again in a moment</p>
                    </div>
                `;
            }
        }
        
        // 🎯 HELPER FUNCTIONS
        function showMobileSearch() {
            console.log('📱 showMobileSearch() called');
            console.log('📱 mobileSearchBar element:', mobileSearchBar);
            
            mobileSearchBar.style.display = 'block'; // Force display
            mobileSearchBar.classList.add('show');
            mobileSearchToggle.classList.add('active');
            
            console.log('📱 Added show class, current classes:', mobileSearchBar.className);
            console.log('📱 Current style.display:', mobileSearchBar.style.display);
            
            setTimeout(() => {
                if (mobileSearchInput) {
                    mobileSearchInput.focus();
                    console.log('📱 Search input focused');
                }
            }, 300); // Wait for animation
        }
        
        function hideMobileSearch() {
            console.log('📱 hideMobileSearch() called');
            mobileSearchBar.classList.remove('show');
            mobileSearchToggle.classList.remove('active');
            const mobileDropdown = document.getElementById('mobileSearchDropdown');
            if (mobileDropdown) {
                mobileDropdown.classList.remove('show');
            }
            console.log('📱 Removed show class');
        }
        
        function showMobileMenu() {
            mobileMenuDropdown.classList.add('show');
            mobileMenuToggle.classList.add('active');
        }
        
        function hideMobileMenu() {
            mobileMenuDropdown.classList.remove('show');
            mobileMenuToggle.classList.remove('active');
        }
        
        // 👆 CLICK OUTSIDE TO CLOSE
        document.addEventListener('click', function(event) {
            const navbar = document.getElementById('main-navbar');
            
            // If click is outside navbar, close all mobile elements
            if (navbar && !navbar.contains(event.target)) {
                hideMobileSearch();
                hideMobileMenu();
            }
        });
        
        // 🔄 WINDOW RESIZE HANDLER - Hide mobile elements on desktop
        window.addEventListener('resize', function() {
            if (window.innerWidth >= 992) { // Bootstrap lg breakpoint
                hideMobileSearch();
                hideMobileMenu();
            }
        });
        
        console.log('📱 Mobile navbar initialized successfully!');
    }

    // 🔄 GLOBAL FUNCTION FOR EXPANDABLE SECTIONS
    window.toggleMobileSection = function(sectionName, event) {
        event.preventDefault();
        event.stopPropagation();
        
        console.log('📱 Toggling mobile section:', sectionName);
        
        const section = document.querySelector(`.mobile-${sectionName}-toggle`).parentElement;
        const content = document.getElementById(`mobile-${sectionName}-content`);
        const chevron = section.querySelector('.mobile-chevron');
        
        if (!section || !content) {
            console.error('📱 Section elements not found:', sectionName);
            return;
        }
        
        const isExpanded = section.classList.contains('expanded');
        
        // Close all other sections first
        document.querySelectorAll('.mobile-expandable-section.expanded').forEach(otherSection => {
            if (otherSection !== section) {
                otherSection.classList.remove('expanded');
                const otherChevron = otherSection.querySelector('.mobile-chevron');
                if (otherChevron) {
                    otherChevron.style.transform = 'rotate(0deg)';
                }
            }
        });
        
        // Toggle current section
        if (isExpanded) {
            // Collapse
            console.log('📱 Collapsing section:', sectionName);
            section.classList.remove('expanded');
            if (chevron) {
                chevron.style.transform = 'rotate(0deg)';
            }
        } else {
            // Expand
            console.log('📱 Expanding section:', sectionName);
            section.classList.add('expanded');
            if (chevron) {
                chevron.style.transform = 'rotate(90deg)';
            }
        }
    };
});
</script>

<!-- SITE-WIDE CART NOTIFICATION POSITIONING FIX -->
<style>
/* Force ALL cart notifications below navbar - works with any notification system */
.cart-notification,
.notification,
.toast,
.alert-notification,
[class*="notification"],
[class*="toast"],
div[style*="position: fixed"][style*="top: 20px"],
div[style*="position: fixed"][style*="top:20px"] {
    top: 70px !important;
}

/* Bootstrap toasts positioning fix */
.toast-container {
    top: 70px !important;
}

/* Additional fallback for inline-styled notifications */
body > div[style*="position: fixed"][style*="top"] {
    top: 70px !important;
}

/* Ensure notifications don't overlap with navbar */
#main-navbar + * .notification,
#main-navbar ~ * .notification {
    top: 70px !important;
}
</style>
