<?php
/**
 * Revenue Sharing Purchase Processor
 * Automatically calculates and distributes revenue between admin and teachers
 */

class PurchaseProcessor {
    private $conn;
    private $admin_percentage;
    private $teacher_percentage;
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
        $this->loadRevenueSettings();
    }
    
    /**
     * Load current revenue sharing settings
     */
    private function loadRevenueSettings() {
        try {
            $stmt = $this->conn->prepare("
                SELECT setting_name, setting_value 
                FROM revenue_settings 
                WHERE setting_name IN ('admin_percentage', 'teacher_percentage')
            ");
            $stmt->execute();
            $result = $stmt->get_result();
            
            // Default values
            $this->admin_percentage = 20.0;
            $this->teacher_percentage = 80.0;
            
            while ($row = $result->fetch_assoc()) {
                if ($row['setting_name'] === 'admin_percentage') {
                    $this->admin_percentage = (float)$row['setting_value'];
                } elseif ($row['setting_name'] === 'teacher_percentage') {
                    $this->teacher_percentage = (float)$row['setting_value'];
                }
            }
            
        } catch (Exception $e) {
            error_log("Error loading revenue settings: " . $e->getMessage());
            // Use defaults
            $this->admin_percentage = 20.0;
            $this->teacher_percentage = 80.0;
        }
    }
    
    /**
     * Process a course purchase and distribute revenue
     * 
     * @param int $user_id Student making the purchase
     * @param int $course_id Course being purchased
     * @param float $amount Purchase amount
     * @param string $payment_method Payment method used
     * @param string $transaction_id Transaction ID from payment gateway
     * @return array Result with purchase_id and revenue breakdown
     */
    public function processPurchase($user_id, $course_id, $amount, $payment_method = null, $transaction_id = null) {
        try {
            // Start database transaction
            $this->conn->begin_transaction();
            
            // Get course and teacher information
            $course_info = $this->getCourseInfo($course_id);
            if (!$course_info) {
                throw new Exception("Course not found");
            }
            
            // Calculate revenue split
            $revenue_split = $this->calculateRevenueSplit($amount);
            
            // Create course purchase record
            $purchase_id = $this->createPurchaseRecord(
                $user_id, 
                $course_id, 
                $course_info['teacher_id'], 
                $amount, 
                $revenue_split, 
                $payment_method, 
                $transaction_id
            );
            
            // Record admin earnings
            $this->recordAdminEarnings($purchase_id, $course_id, $course_info['teacher_id'], $revenue_split['admin_amount']);
            
            // Record teacher earnings
            $this->recordTeacherEarnings($course_info['teacher_id'], $purchase_id, $course_id, $revenue_split['teacher_amount']);
            
            // Create enrollment record
            $this->createEnrollment($user_id, $course_id, $amount, $purchase_id);
            
            // Commit transaction
            $this->conn->commit();
            
            return [
                'success' => true,
                'purchase_id' => $purchase_id,
                'revenue_split' => $revenue_split,
                'course_title' => $course_info['title'],
                'teacher_name' => $course_info['teacher_name']
            ];
            
        } catch (Exception $e) {
            // Rollback transaction on error
            $this->conn->rollback();
            
            error_log("Purchase processing error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get course information including teacher details
     */
    private function getCourseInfo($course_id) {
        $stmt = $this->conn->prepare("
            SELECT c.*, 
                   CONCAT(u.first_name, ' ', u.last_name) as teacher_name
            FROM courses c
            LEFT JOIN users u ON c.teacher_id = u.user_id
            WHERE c.course_id = ?
        ");
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    /**
     * Calculate revenue split based on current settings
     */
    private function calculateRevenueSplit($amount) {
        $admin_amount = ($amount * $this->admin_percentage) / 100;
        $teacher_amount = ($amount * $this->teacher_percentage) / 100;
        
        return [
            'total_amount' => $amount,
            'admin_amount' => round($admin_amount, 2),
            'teacher_amount' => round($teacher_amount, 2),
            'admin_percentage' => $this->admin_percentage,
            'teacher_percentage' => $this->teacher_percentage
        ];
    }
    
    /**
     * Create purchase record in database
     */
    private function createPurchaseRecord($user_id, $course_id, $teacher_id, $amount, $revenue_split, $payment_method, $transaction_id) {
        // Use basic course_purchases table structure that exists
        $stmt = $this->conn->prepare("
            INSERT INTO course_purchases (
                user_id, course_id, amount, payment_id, payment_status, purchase_date
            ) VALUES (?, ?, ?, ?, 'completed', NOW())
        ");
        
        if (!$stmt) {
            throw new Exception("Failed to prepare course_purchases insert: " . $this->conn->error);
        }
        
        $stmt->bind_param("iids", 
            $user_id, 
            $course_id, 
            $revenue_split['total_amount'],
            $transaction_id
        );
        
        $stmt->execute();
        return $this->conn->insert_id;
    }
    
    /**
     * Record admin earnings
     */
    private function recordAdminEarnings($purchase_id, $course_id, $teacher_id, $amount) {
        $stmt = $this->conn->prepare("
            INSERT INTO admin_earnings (purchase_id, course_id, teacher_id, amount)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->bind_param("iiid", $purchase_id, $course_id, $teacher_id, $amount);
        $stmt->execute();
    }
    
    /**
     * Record teacher earnings
     */
    private function recordTeacherEarnings($teacher_id, $purchase_id, $course_id, $amount) {
        $stmt = $this->conn->prepare("
            INSERT INTO teacher_earnings (teacher_id, purchase_id, course_id, amount, status)
            VALUES (?, ?, ?, ?, 'available')
        ");
        $stmt->bind_param("iiid", $teacher_id, $purchase_id, $course_id, $amount);
        $stmt->execute();
    }
    
    /**
     * Create enrollment record
     */
    private function createEnrollment($user_id, $course_id, $amount, $purchase_id) {
        // Check if enrollment already exists
        $stmt = $this->conn->prepare("
            SELECT enrollment_id FROM enrollments 
            WHERE user_id = ? AND course_id = ?
        ");
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            // Create new enrollment (use correct column name: enrolled_date not enrollment_date)
            $stmt = $this->conn->prepare("
                INSERT INTO enrollments (user_id, course_id, course_price, purchase_id, enrolled_date)
                VALUES (?, ?, ?, ?, NOW())
            ");
            
            if (!$stmt) {
                throw new Exception("Failed to prepare enrollment insert: " . $this->conn->error);
            }
            
            $stmt->bind_param("iidi", $user_id, $course_id, $amount, $purchase_id);
            $stmt->execute();
        } else {
            // Update existing enrollment with purchase info (use correct column name: enrolled_date)
            $stmt = $this->conn->prepare("
                UPDATE enrollments 
                SET course_price = ?, purchase_id = ?
                WHERE user_id = ? AND course_id = ?
            ");
            
            if (!$stmt) {
                throw new Exception("Failed to prepare enrollment update: " . $this->conn->error);
            }
            
            $stmt->bind_param("diii", $amount, $purchase_id, $user_id, $course_id);
            $stmt->execute();
        }
    }
    
    /**
     * Get teacher earnings summary
     */
    public function getTeacherEarnings($teacher_id) {
        $stmt = $this->conn->prepare("
            SELECT 
                SUM(CASE WHEN status = 'available' THEN amount ELSE 0 END) as available_balance,
                SUM(CASE WHEN status = 'withdrawn' THEN amount ELSE 0 END) as total_withdrawn,
                SUM(amount) as total_earned,
                COUNT(*) as total_sales
            FROM teacher_earnings 
            WHERE teacher_id = ?
        ");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    /**
     * Get admin earnings summary
     */
    public function getAdminEarnings() {
        $stmt = $this->conn->prepare("
            SELECT 
                SUM(amount) as total_earned,
                COUNT(*) as total_transactions
            FROM admin_earnings
        ");
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    /**
     * Get revenue statistics
     */
    public function getRevenueStats($start_date = null, $end_date = null) {
        $where_clause = "";
        $params = [];
        $types = "";
        
        if ($start_date && $end_date) {
            $where_clause = "WHERE created_at BETWEEN ? AND ?";
            $params = [$start_date, $end_date];
            $types = "ss";
        }
        
        // Check if the table has revenue sharing columns
        $check_columns = $this->conn->query("SHOW COLUMNS FROM course_purchases");
        $has_revenue_columns = false;
        if ($check_columns) {
            $columns = [];
            while ($col = $check_columns->fetch_assoc()) {
                $columns[] = $col['Field'];
            }
            $has_revenue_columns = in_array('admin_amount', $columns) && in_array('teacher_amount', $columns);
        }
        
        if ($has_revenue_columns) {
            // Use full revenue sharing query
            $sql = "
                SELECT 
                    COUNT(*) as total_purchases,
                    SUM(amount) as total_revenue,
                    SUM(admin_amount) as admin_earned,
                    SUM(teacher_amount) as teacher_earned,
                    AVG(admin_percentage) as avg_admin_percentage,
                    AVG(teacher_percentage) as avg_teacher_percentage
                FROM course_purchases 
                WHERE status = 'completed' $where_clause
            ";
        } else {
            // Use basic query for existing table structure
            $sql = "
                SELECT 
                    COUNT(*) as total_purchases,
                    SUM(amount) as total_revenue,
                    SUM(amount * {$this->admin_percentage} / 100) as admin_earned,
                    SUM(amount * {$this->teacher_percentage} / 100) as teacher_earned,
                    {$this->admin_percentage} as avg_admin_percentage,
                    {$this->teacher_percentage} as avg_teacher_percentage
                FROM course_purchases 
                WHERE payment_status = 'completed' $where_clause
            ";
        }
        
        $stmt = $this->conn->prepare($sql);
        
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    /**
     * Update revenue settings
     */
    public function updateRevenueSettings($admin_percentage, $teacher_percentage) {
        if (abs(($admin_percentage + $teacher_percentage) - 100) > 0.01) {
            throw new Exception("Percentages must add up to 100%");
        }
        
        $this->conn->begin_transaction();
        
        try {
            // Update admin percentage
            $stmt = $this->conn->prepare("
                INSERT INTO revenue_settings (setting_name, setting_value, updated_at) 
                VALUES ('admin_percentage', ?, NOW())
                ON DUPLICATE KEY UPDATE 
                setting_value = VALUES(setting_value), 
                updated_at = NOW()
            ");
            $stmt->bind_param("d", $admin_percentage);
            $stmt->execute();
            
            // Update teacher percentage
            $stmt = $this->conn->prepare("
                INSERT INTO revenue_settings (setting_name, setting_value, updated_at) 
                VALUES ('teacher_percentage', ?, NOW())
                ON DUPLICATE KEY UPDATE 
                setting_value = VALUES(setting_value), 
                updated_at = NOW()
            ");
            $stmt->bind_param("d", $teacher_percentage);
            $stmt->execute();
            
            $this->conn->commit();
            
            // Reload settings
            $this->loadRevenueSettings();
            
            return true;
            
        } catch (Exception $e) {
            $this->conn->rollback();
            throw $e;
        }
    }
}

// Usage example function
function processCoursePurchase($user_id, $course_id, $amount, $payment_method = null, $transaction_id = null) {
    global $conn;
    
    $processor = new PurchaseProcessor($conn);
    return $processor->processPurchase($user_id, $course_id, $amount, $payment_method, $transaction_id);
}

// Function to get teacher earnings (for use in teacher dashboard)
function getTeacherEarnings($teacher_id) {
    global $conn;
    
    $processor = new PurchaseProcessor($conn);
    return $processor->getTeacherEarnings($teacher_id);
}

// Function to get admin earnings (for use in admin dashboard)
function getAdminEarnings() {
    global $conn;
    
    $processor = new PurchaseProcessor($conn);
    return $processor->getAdminEarnings();
}
?>
