<?php
/**
 * Permanent Quiz Auto-Linking System
 * Automatically links unlinked quizzes to appropriate videos
 */

class QuizAutoLinker {
    private $conn;
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
    }
    
    /**
     * Auto-link all unlinked quizzes to their appropriate videos
     */
    public function autoLinkUnlinkedQuizzes() {
        $results = [
            'success' => false,
            'linked_count' => 0,
            'errors' => [],
            'details' => []
        ];
        
        try {
            // Find all unlinked quizzes (video_id is NULL but course_id exists)
            $unlinked_query = "
                SELECT q.quiz_id, q.title as quiz_title, q.course_id, c.title as course_title
                FROM quizzes q
                JOIN courses c ON q.course_id = c.course_id
                WHERE q.video_id IS NULL OR q.video_id = 0
                ORDER BY q.course_id, q.created_at
            ";
            
            $unlinked_result = $this->conn->query($unlinked_query);
            
            if ($unlinked_result && $unlinked_result->num_rows > 0) {
                while ($quiz = $unlinked_result->fetch_assoc()) {
                    $linked = $this->linkQuizToVideo($quiz);
                    if ($linked['success']) {
                        $results['linked_count']++;
                        $results['details'][] = $linked['message'];
                    } else {
                        $results['errors'][] = $linked['message'];
                    }
                }
            }
            
            $results['success'] = true;
            
        } catch (Exception $e) {
            $results['errors'][] = "Auto-linking failed: " . $e->getMessage();
        }
        
        return $results;
    }
    
    /**
     * Link a specific quiz to an appropriate video
     */
    private function linkQuizToVideo($quiz) {
        $result = ['success' => false, 'message' => ''];
        
        try {
            // Get videos for this course
            $videos_query = "
                SELECT video_id, title, created_at 
                FROM videos 
                WHERE course_id = ? 
                ORDER BY created_at ASC
            ";
            
            $videos_stmt = $this->conn->prepare($videos_query);
            $videos_stmt->bind_param("i", $quiz['course_id']);
            $videos_stmt->execute();
            $videos_result = $videos_stmt->get_result();
            
            if ($videos_result && $videos_result->num_rows > 0) {
                // Strategy 1: Try to match by similar title
                $best_video = $this->findBestVideoMatch($quiz, $videos_result);
                
                if ($best_video) {
                    // Link the quiz to the best matching video
                    $update_query = "UPDATE quizzes SET video_id = ? WHERE quiz_id = ?";
                    $update_stmt = $this->conn->prepare($update_query);
                    $update_stmt->bind_param("ii", $best_video['video_id'], $quiz['quiz_id']);
                    
                    if ($update_stmt->execute()) {
                        $result['success'] = true;
                        $result['message'] = "✅ Linked Quiz '{$quiz['quiz_title']}' to Video '{$best_video['title']}'";
                    } else {
                        $result['message'] = "❌ Failed to update Quiz ID {$quiz['quiz_id']}";
                    }
                    $update_stmt->close();
                } else {
                    $result['message'] = "⚠️ No suitable video found for Quiz '{$quiz['quiz_title']}'";
                }
            } else {
                $result['message'] = "❌ No videos found in course '{$quiz['course_title']}'";
            }
            $videos_stmt->close();
            
        } catch (Exception $e) {
            $result['message'] = "❌ Error linking Quiz ID {$quiz['quiz_id']}: " . $e->getMessage();
        }
        
        return $result;
    }
    
    /**
     * Find the best video match for a quiz based on title similarity and order
     */
    private function findBestVideoMatch($quiz, $videos_result) {
        $videos = [];
        while ($video = $videos_result->fetch_assoc()) {
            $videos[] = $video;
        }
        
        if (empty($videos)) {
            return null;
        }
        
        $quiz_title_lower = strtolower($quiz['quiz_title']);
        $best_match = null;
        $highest_score = 0;
        
        foreach ($videos as $video) {
            $video_title_lower = strtolower($video['title']);
            $score = 0;
            
            // Strategy 1: Exact title match (highest priority)
            if (strpos($quiz_title_lower, $video_title_lower) !== false || 
                strpos($video_title_lower, $quiz_title_lower) !== false) {
                $score = 100;
            }
            // Strategy 2: Common keywords
            else {
                $quiz_words = explode(' ', $quiz_title_lower);
                $video_words = explode(' ', $video_title_lower);
                $common_words = array_intersect($quiz_words, $video_words);
                $score = count($common_words) * 10;
            }
            
            if ($score > $highest_score) {
                $highest_score = $score;
                $best_match = $video;
            }
        }
        
        // If no good match found, use first video (chronological order)
        if ($highest_score === 0 && !empty($videos)) {
            $best_match = $videos[0]; // First video in the course
        }
        
        return $best_match;
    }
    
    /**
     * Auto-link a quiz when it's created (to be called from quiz creation process)
     */
    public static function autoLinkNewQuiz($conn, $quiz_id, $course_id, $video_id = null) {
        // If video_id is already provided, no need to auto-link
        if (!empty($video_id)) {
            return ['success' => true, 'message' => 'Quiz already linked to video'];
        }
        
        $linker = new QuizAutoLinker($conn);
        
        // Get quiz details
        $quiz_query = "SELECT quiz_id, title, course_id FROM quizzes WHERE quiz_id = ?";
        $quiz_stmt = $conn->prepare($quiz_query);
        $quiz_stmt->bind_param("i", $quiz_id);
        $quiz_stmt->execute();
        $quiz_result = $quiz_stmt->get_result();
        
        if ($quiz_result && $quiz_result->num_rows > 0) {
            $quiz = $quiz_result->fetch_assoc();
            return $linker->linkQuizToVideo($quiz);
        }
        
        return ['success' => false, 'message' => 'Quiz not found'];
    }
    
    /**
     * Periodic maintenance function to link any missed quizzes
     */
    public function performMaintenanceAutoLink() {
        return $this->autoLinkUnlinkedQuizzes();
    }
}
?>
