<?php
/**
 * Referral Reward Processing System
 * Panadite Academy - Handles all referral rewards and commissions
 */

class ReferralProcessor {
    private $conn;
    private $settings;
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
        $this->loadSettings();
    }
    
    /**
     * Load referral settings from database
     */
    private function loadSettings() {
        $query = "SELECT setting_name, setting_value FROM referral_settings WHERE is_active = 1";
        $result = $this->conn->query($query);
        
        $this->settings = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $this->settings[$row['setting_name']] = $row['setting_value'];
            }
        }
    }
    
    /**
     * Process referral when a user registers with a referral code
     */
    public function processReferralRegistration($referral_code, $new_user_id, $new_user_email) {
        // Load secure referral generator if not already loaded
        if (!class_exists('SecureReferralGenerator')) {
            require_once __DIR__ . '/secure_referral_generator.php';
        }
        
        // First try secure referral validation (new system)
        $referrer = null;
        if (preg_match('/^(TCH|STU)[A-F0-9]{12}$/', $referral_code)) {
            $referrer = SecureReferralGenerator::validateSecureCode($referral_code, $this->conn);
        }
        
        // If secure validation failed, try legacy system for backward compatibility
        if (!$referrer && class_exists('LegacyReferralValidator')) {
            $referrer = LegacyReferralValidator::validateLegacyCode($referral_code, $this->conn);
        }
        
        // Final fallback - check old referrals table directly
        if (!$referrer) {
            $query = "SELECT u.user_id, u.username, u.user_role, u.first_name, u.last_name, u.email 
                     FROM referrals r 
                     JOIN users u ON r.referrer_id = u.user_id 
                     WHERE r.referral_code = ?";
            $stmt = $this->conn->prepare($query);
            $stmt->bind_param("s", $referral_code);
            $stmt->execute();
            $referrer = $stmt->get_result()->fetch_assoc();
        }
        
        if (!$referrer) {
            return ['success' => false, 'message' => 'Invalid or expired referral code'];
        }
        
        // Create or update referral connection record in the referrals table
        // This ensures compatibility with existing commission processing
        $referrer_id = $referrer['user_id'];
        
        // Check if referral record already exists for this referrer
        $check_query = "SELECT id FROM referrals WHERE referrer_id = ? AND referral_code = ?";
        $check_stmt = $this->conn->prepare($check_query);
        $check_stmt->bind_param("is", $referrer_id, $referral_code);
        $check_stmt->execute();
        $existing_referral = $check_stmt->get_result()->fetch_assoc();
        
        if ($existing_referral) {
            // Update existing referral record with correct referrer_type
            $update_query = "UPDATE referrals SET referred_user_id = ?, referrer_type = ?, status = 'registered' WHERE id = ?";
            $update_stmt = $this->conn->prepare($update_query);
            $referrer_type = $referrer['user_role']; // Use actual user role from database
            $update_stmt->bind_param("isi", $new_user_id, $referrer_type, $existing_referral['id']);
            $update_result = $update_stmt->execute();
            
            // CHECK if update actually worked
            if (!$update_result || $update_stmt->affected_rows === 0) {
                error_log("REFERRAL UPDATE FAILED: Code={$referral_code}, ReferralID={$existing_referral['id']}, SQL Error: " . $this->conn->error);
                return ['success' => false, 'message' => 'Failed to update referral connection'];
            }
            
            $referral_id = $existing_referral['id'];
        } else {
            // Create new referral record for commission tracking with correct referrer_type
            // Use 'registered' status so commission processing can find it
            $insert_query = "INSERT INTO referrals (referrer_id, referred_user_id, referral_code, referrer_type, status, created_at) 
                           VALUES (?, ?, ?, ?, 'registered', NOW())";
            $insert_stmt = $this->conn->prepare($insert_query);
            $referrer_type = $referrer['user_role']; // Use actual user role from database
            $insert_stmt->bind_param("iiss", $referrer_id, $new_user_id, $referral_code, $referrer_type);
            $insert_result = $insert_stmt->execute();
            
            // CHECK if insert actually worked
            if (!$insert_result || $this->conn->insert_id === 0) {
                error_log("REFERRAL INSERT FAILED: Code={$referral_code}, Referrer={$referrer_id}, Referred={$new_user_id}, SQL Error: " . $this->conn->error);
                return ['success' => false, 'message' => 'Failed to create referral connection'];
            }
            
            $referral_id = $this->conn->insert_id;
        }
        
        // Log successful referral processing
        error_log("REFERRAL CONNECTION SUCCESS: Code={$referral_code}, Referrer={$referrer_id}, Referred={$new_user_id}, ReferralID={$referral_id}");
        
        return [
            'success' => true, 
            'message' => 'Referral processed successfully', 
            'referral_id' => $referral_id,
            'referrer_id' => $referrer_id,
            'referrer_name' => ($referrer['first_name'] ?? '') . ' ' . ($referrer['last_name'] ?? ''),
            'referrer_role' => $referrer['user_role'] ?? 'unknown'
        ];
    }
    
    /**
     * Process referral rewards when a referred user makes a purchase
     */
    public function processPurchaseRewards($user_id, $purchase_id, $course_id, $purchase_amount) {
        // Check if referral rewards are active
        if (!isset($this->settings['referral_reward_active']) || $this->settings['referral_reward_active'] != '1') {
            return ['success' => false, 'message' => 'Referral rewards are not active'];
        }
        
        // Check minimum purchase amount
        $min_amount = floatval($this->settings['min_purchase_amount_for_reward'] ?? 0);
        if ($purchase_amount < $min_amount) {
            return ['success' => false, 'message' => 'Purchase amount below minimum for referral rewards'];
        }
        
        // Find referrals where this user was referred
        // Look for both 'registered' and 'completed' status to handle referrals created during registration
        $query = "SELECT * FROM referrals WHERE referred_user_id = ? AND status IN ('registered', 'completed')";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $referrals = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        
        $rewards_processed = 0;
        
        foreach ($referrals as $referral) {
            // Process student discount reward
            if ($referral['referrer_type'] === 'student') {
                $this->processStudentReward($referral, $purchase_id, $purchase_amount);
                $rewards_processed++;
            }
            
            // Process teacher commission reward
            if ($referral['referrer_type'] === 'teacher') {
                $this->processTeacherCommission($referral, $purchase_id, $course_id, $purchase_amount);
                $rewards_processed++;
            }
            
            // Update referral status to completed
            $update_query = "UPDATE referrals SET status = 'completed', completed_at = CURRENT_TIMESTAMP, purchase_that_triggered_reward = ? WHERE id = ?";
            $stmt = $this->conn->prepare($update_query);
            $stmt->bind_param("ii", $purchase_id, $referral['id']);
            $stmt->execute();
        }
        
        return ['success' => true, 'rewards_processed' => $rewards_processed];
    }
    
    /**
     * Process student referral discount reward
     */
    private function processStudentReward($referral, $purchase_id, $purchase_amount) {
        $discount_percentage = floatval($this->settings['student_discount_percentage'] ?? 10);
        $max_discount = floatval($this->settings['student_max_discount_per_referral'] ?? 500);
        $duration_days = intval($this->settings['student_discount_duration_days'] ?? 30);
        
        // Calculate discount amount
        $discount_amount = min(($purchase_amount * $discount_percentage / 100), $max_discount);
        
        // Set expiry date
        $expires_at = date('Y-m-d H:i:s', strtotime("+{$duration_days} days"));
        
        // Insert student discount
        $insert_query = "INSERT INTO student_referral_discounts (student_id, referral_id, discount_percentage, discount_amount, expires_at) VALUES (?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($insert_query);
        $stmt->bind_param("iidds", $referral['referrer_id'], $referral['id'], $discount_percentage, $discount_amount, $expires_at);
        $stmt->execute();
        
        // Update referral record
        $update_query = "UPDATE referrals SET reward_granted = 1, reward_amount = ? WHERE id = ?";
        $stmt = $this->conn->prepare($update_query);
        $stmt->bind_param("di", $discount_amount, $referral['id']);
        $stmt->execute();
        
        return $discount_amount;
    }
    
    /**
     * Process teacher referral commission
     */
    private function processTeacherCommission($referral, $purchase_id, $course_id, $purchase_amount) {
        $commission_percentage = floatval($this->settings['teacher_commission_percentage'] ?? 5);
        $commission_amount = $purchase_amount * $commission_percentage / 100;
        
        // Insert teacher commission
        $insert_query = "INSERT INTO teacher_referral_commissions (teacher_id, referral_id, referred_student_id, purchase_id, commission_percentage, commission_amount, course_id, purchase_amount) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($insert_query);
        $stmt->bind_param("iiiiddid", $referral['referrer_id'], $referral['id'], $referral['referred_user_id'], $purchase_id, $commission_percentage, $commission_amount, $course_id, $purchase_amount);
        $insert_result = $stmt->execute();
        
        // CHECK if insert actually worked
        if (!$insert_result || $stmt->affected_rows === 0) {
            error_log("TEACHER COMMISSION INSERT FAILED: ReferralID={$referral['id']}, TeacherID={$referral['referrer_id']}, Amount={$commission_amount}, SQL Error: " . $this->conn->error);
            throw new Exception("Failed to insert teacher commission: " . $this->conn->error);
        }
        
        error_log("TEACHER COMMISSION SUCCESS: Inserted R{$commission_amount} commission for teacher {$referral['referrer_id']} from referral {$referral['id']}");
        
        // Update referral record
        $update_query = "UPDATE referrals SET commission_earned = commission_earned + ? WHERE id = ?";
        $stmt = $this->conn->prepare($update_query);
        $stmt->bind_param("di", $commission_amount, $referral['id']);
        $stmt->execute();
        
        return $commission_amount;
    }
    
    /**
     * Create a new referral code for a user
     */
    public function createReferral($user_id, $user_type, $referred_email) {
        // Generate unique referral code
        $user_query = "SELECT first_name, last_name FROM users WHERE user_id = ?";
        $stmt = $this->conn->prepare($user_query);
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $user_data = $stmt->get_result()->fetch_assoc();
        
        if (!$user_data) {
            return ['success' => false, 'message' => 'User not found'];
        }
        
        // Create referral code
        $referral_code = strtoupper(substr($user_data['first_name'], 0, 2) . substr($user_data['last_name'], 0, 2) . $user_id . rand(10, 99));
        
        // Set expiry (30 days from now)
        $expires_at = date('Y-m-d H:i:s', strtotime('+30 days'));
        
        // Insert referral
        $insert_query = "INSERT INTO referrals (referrer_id, referrer_type, referred_email, referral_code, expires_at) VALUES (?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($insert_query);
        $stmt->bind_param("issss", $user_id, $user_type, $referred_email, $referral_code, $expires_at);
        
        if ($stmt->execute()) {
            return ['success' => true, 'referral_code' => $referral_code, 'referral_id' => $this->conn->insert_id];
        } else {
            return ['success' => false, 'message' => 'Failed to create referral'];
        }
    }
    
    /**
     * Get available discounts for a student
     */
    public function getStudentDiscounts($student_id) {
        $query = "SELECT * FROM student_referral_discounts WHERE student_id = ? AND is_used = 0 AND expires_at > NOW()";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
    
    /**
     * Apply student discount to a purchase
     */
    public function applyStudentDiscount($discount_id, $purchase_id) {
        $update_query = "UPDATE student_referral_discounts SET is_used = 1, used_on_purchase = ?, used_at = CURRENT_TIMESTAMP WHERE id = ?";
        $stmt = $this->conn->prepare($update_query);
        $stmt->bind_param("ii", $purchase_id, $discount_id);
        return $stmt->execute();
    }
    
    /**
     * Get teacher commission statistics
     */
    public function getTeacherCommissions($teacher_id) {
        $query = "SELECT * FROM teacher_referral_commissions WHERE teacher_id = ? ORDER BY created_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
    
    /**
     * Get referral analytics data
     */
    public function getAnalytics($start_date = null, $end_date = null) {
        $where_clause = "";
        $params = [];
        
        if ($start_date && $end_date) {
            $where_clause = "WHERE DATE(created_at) BETWEEN ? AND ?";
            $params = [$start_date, $end_date];
        }
        
        $query = "
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as daily_referrals,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as daily_successful,
                SUM(reward_amount) as daily_rewards,
                SUM(commission_earned) as daily_commissions
            FROM referrals 
            $where_clause
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        ";
        
        $stmt = $this->conn->prepare($query);
        if (!empty($params)) {
            $stmt->bind_param("ss", ...$params);
        }
        $stmt->execute();
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

}
?>
