/**
 * Panadite Academy - Complete Guest-to-Authenticated Shopping Cart System
 * Allows users to add courses to cart as guests, then complete purchase after login
 */

class PanaditeCartSystem {
    constructor() {
        this.cartKey = 'panadite_guest_cart';
        this.init();
    }

    init() {
        this.updateCartCounter();
        this.bindCartButtons();
        this.setupCartNotifications();
    }

    // Add course to cart (works for both guests and authenticated users)
    async addToCart(courseId, courseData = null) {
        try {
            // If user is authenticated, use API
            if (this.isAuthenticated()) {
                return await this.addToAuthenticatedCart(courseId);
            }

            // For guests, use localStorage
            return this.addToGuestCart(courseId, courseData);
        } catch (error) {
            console.error('Error adding to cart:', error);
            this.showNotification('Error adding item to cart', 'error');
            return false;
        }
    }

    // Add to guest cart using localStorage
    addToGuestCart(courseId, courseData) {
        let cart = this.getGuestCart();
        
        // Check if course already in cart
        if (cart.some(item => item.id == courseId)) {
            this.showNotification('Course already in cart!', 'info');
            return false;
        }

        // If courseData not provided, fetch from DOM
        if (!courseData) {
            courseData = this.extractCourseDataFromDOM(courseId);
        }

        // Add to cart
        cart.push({
            id: courseId,
            title: courseData.title,
            price: courseData.price,
            thumbnail: courseData.thumbnail,
            instructor: courseData.instructor,
            level: courseData.level,
            description: courseData.description,
            added_date: new Date().toISOString(),
            type: 'guest'
        });

        // Save to localStorage
        localStorage.setItem(this.cartKey, JSON.stringify(cart));
        
        this.updateCartCounter();
        this.showNotification(`"${courseData.title}" added to cart!`, 'success');
        
        return true;
    }

    // Add to authenticated user's cart via API
    async addToAuthenticatedCart(courseId) {
        try {
            const response = await fetch('/student/api/add-to-cart.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    course_id: courseId
                })
            });

            const result = await response.json();
            
            if (result.success) {
                this.updateCartCounter();
                this.showNotification(result.message, 'success');
                return true;
            } else {
                this.showNotification(result.message || 'Failed to add to cart', 'error');
                return false;
            }
        } catch (error) {
            console.error('API Error:', error);
            this.showNotification('Network error. Please try again.', 'error');
            return false;
        }
    }

    // Extract course data from DOM element
    extractCourseDataFromDOM(courseId) {
        // Escape special characters in courseId for querySelector
        const escapedCourseId = CSS.escape(courseId);
        const courseCard = document.querySelector(`[data-course-id="${escapedCourseId}"]`);
        
        if (!courseCard) {
            console.error('Course card not found for ID:', courseId);
            console.error('Tried selector:', `[data-course-id="${escapedCourseId}"]`);
            
            // Try finding all cards and log them for debugging
            const allCards = document.querySelectorAll('[data-course-id]');
            console.log('All course cards on page:', allCards.length);
            allCards.forEach((card, index) => {
                console.log(`Card ${index}:`, card.getAttribute('data-course-id'));
            });
            
            return {
                title: 'Course',
                price: 0,
                thumbnail: '/uploads/thumbnails/default-course.jpg',
                instructor: 'Panadite Academy',
                level: 'Beginner'
            };
        }

        console.log('Found course card:', courseCard);

        // Extract price from data-price attribute (support multiple class prefixes)
        let price = 0;
        const priceElement = courseCard.querySelector('.course-price[data-price], .pa-course-price[data-price], .fc-course-price[data-price], .course-price, .pa-course-price, .fc-course-price');
        if (priceElement) {
            // Try data-price attribute first
            const dataPrice = priceElement.getAttribute('data-price');
            if (dataPrice) {
                price = parseFloat(dataPrice) || 0;
            } else {
                // Parse from text content
                const priceText = priceElement.textContent;
                if (priceText.includes('Free')) {
                    price = 0;
                } else {
                    const priceMatch = priceText.match(/R?([\d,]+\.?\d*)/);
                    price = priceMatch ? parseFloat(priceMatch[1].replace(',', '')) : 0;
                }
            }
        }
        console.log('Extracted price:', price);

        // Extract level from data-level attribute (support multiple class prefixes)
        let level = 'Beginner';
        const levelElement = courseCard.querySelector('.course-level[data-level], .pa-course-level-badge[data-level], .fc-course-level-badge[data-level], .course-level, .pa-course-level-badge, .fc-course-level-badge');
        if (levelElement) {
            level = levelElement.getAttribute('data-level') || levelElement.textContent?.trim() || 'Beginner';
        }
        console.log('Extracted level:', level);

        // Extract title (support multiple class prefixes)
        const title = courseCard.querySelector('.course-title, .pa-course-title, .fc-course-title')?.textContent?.trim() || 'Course';
        console.log('Extracted title:', title);

        // Extract instructor (support multiple class prefixes)
        const instructor = courseCard.querySelector('.course-instructor, .pa-course-instructor, .fc-course-instructor')?.textContent?.trim() || 'Panadite Academy';
        console.log('Extracted instructor:', instructor);

        // Extract thumbnail
        const thumbnail = courseCard.querySelector('img')?.src || '/uploads/thumbnails/default-course.jpg';
        console.log('Extracted thumbnail:', thumbnail);

        // Extract description (support multiple class prefixes)
        const description = courseCard.querySelector('.course-description, .pa-course-description, .fc-course-description')?.textContent?.trim() || '';
        console.log('Extracted description:', description);

        return {
            title: title,
            price: price,
            thumbnail: thumbnail,
            instructor: instructor,
            level: level,
            description: description
        };
    }

    // Get guest cart from localStorage
    getGuestCart() {
        try {
            const cart = localStorage.getItem(this.cartKey);
            return cart ? JSON.parse(cart) : [];
        } catch (error) {
            console.error('Error reading cart:', error);
            return [];
        }
    }

    // Get total cart count
    getCartCount() {
        if (this.isAuthenticated()) {
            // For authenticated users, get count from server
            return this.getAuthenticatedCartCount();
        }
        
        // For guests, count localStorage items
        return this.getGuestCart().length;
    }

    // Update cart counter in navbar
    updateCartCounter() {
        const counter = document.querySelector('.cart-counter, .cart-count');
        if (counter) {
            const count = this.getCartCount();
            counter.textContent = count;
            counter.style.display = count > 0 ? 'inline' : 'none';
        }
    }

    // Check if user is authenticated
    isAuthenticated() {
        // Check for authentication indicators
        return document.body.classList.contains('user-authenticated') ||
               document.querySelector('[data-user-role]') !== null ||
               window.userRole !== undefined;
    }

    // Bind cart buttons to add to cart functionality
    bindCartButtons() {
        // Handle cart buttons
        document.addEventListener('click', async (e) => {
            if (e.target.closest('.cart-btn, .pa-course-cart-btn, .add-to-cart-btn')) {
                e.preventDefault();
                
                const button = e.target.closest('.cart-btn, .pa-course-cart-btn, .add-to-cart-btn');
                
                // Try to get course ID from button first, then from parent card
                let courseId = button.getAttribute('data-course-id');
                
                if (!courseId) {
                    const courseCard = button.closest('[data-course-id]');
                    if (courseCard) {
                        courseId = courseCard.getAttribute('data-course-id');
                    }
                }
                
                if (courseId) {
                    console.log('Cart button clicked for course ID:', courseId);
                    
                    // Add loading state
                    const originalText = button.innerHTML;
                    button.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
                    button.disabled = true;
                    
                    // Add to cart
                    const success = await this.addToCart(courseId);
                    
                    // Reset button
                    setTimeout(() => {
                        button.innerHTML = originalText;
                        button.disabled = false;
                    }, 1000);
                } else {
                    console.error('No course ID found on button or parent card');
                }
            }
        });

        // Handle "Go to Cart" buttons
        document.addEventListener('click', (e) => {
            if (e.target.closest('.go-to-cart-btn')) {
                e.preventDefault();
                this.goToCart();
            }
        });
    }

    // Navigate to cart page
    goToCart() {
        if (this.isAuthenticated()) {
            // Authenticated users go directly to cart
            window.location.href = '/student/cart.php';
        } else {
            // Guests go to cart page (which will show login prompt)
            window.location.href = '/cart.php';
        }
    }

    // Show notification
    showNotification(message, type = 'info') {
        // Remove existing notifications
        const existingNotifications = document.querySelectorAll('.cart-notification');
        existingNotifications.forEach(n => n.remove());

        // Create notification
        const notification = document.createElement('div');
        notification.className = `cart-notification cart-notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="bx ${this.getNotificationIcon(type)}"></i>
                <span>${message}</span>
                <button class="notification-close">&times;</button>
            </div>
        `;

        // Add styles
        notification.style.cssText = `
            position: fixed;
            top: 140px;
            right: 20px;
            z-index: 10000;
            background: ${this.getNotificationColor(type)};
            color: white;
            padding: 15px 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateX(300px);
            transition: transform 0.3s ease;
            max-width: 350px;
        `;

        // Add to page
        document.body.appendChild(notification);

        // Animate in
        setTimeout(() => {
            notification.style.transform = 'translateX(0)';
        }, 100);

        // Handle close button
        notification.querySelector('.notification-close').addEventListener('click', () => {
            this.hideNotification(notification);
        });

        // Auto-hide after 5 seconds
        setTimeout(() => {
            this.hideNotification(notification);
        }, 5000);
    }

    hideNotification(notification) {
        notification.style.transform = 'translateX(300px)';
        setTimeout(() => {
            notification.remove();
        }, 300);
    }

    getNotificationIcon(type) {
        const icons = {
            success: 'bx-check-circle',
            error: 'bx-error',
            info: 'bx-info-circle',
            warning: 'bx-error-circle'
        };
        return icons[type] || icons.info;
    }

    getNotificationColor(type) {
        const colors = {
            success: '#10b981',
            error: '#ef4444',
            info: '#3b82f6',
            warning: '#f59e0b'
        };
        return colors[type] || colors.info;
    }

    // Sync guest cart to authenticated cart after login
    async syncGuestCartAfterLogin() {
        const guestCart = this.getGuestCart();
        
        if (guestCart.length === 0) {
            return;
        }

        try {
            // Sync each item to authenticated cart
            for (const item of guestCart) {
                await this.addToAuthenticatedCart(item.id);
            }

            // Clear guest cart after successful sync
            localStorage.removeItem(this.cartKey);
            
            this.showNotification(`${guestCart.length} items synced to your account!`, 'success');
            
        } catch (error) {
            console.error('Error syncing cart:', error);
            this.showNotification('Some cart items could not be synced', 'warning');
        }
    }

    // Setup cart notifications styling
    setupCartNotifications() {
        // Add notification styles to head
        if (!document.querySelector('#cart-notification-styles')) {
            const style = document.createElement('style');
            style.id = 'cart-notification-styles';
            style.textContent = `
                .cart-notification .notification-content {
                    display: flex;
                    align-items: center;
                    gap: 10px;
                }
                
                .cart-notification .notification-close {
                    background: none;
                    border: none;
                    color: white;
                    font-size: 18px;
                    cursor: pointer;
                    padding: 0;
                    margin-left: auto;
                }
                
                .cart-notification .notification-close:hover {
                    opacity: 0.8;
                }
            `;
            document.head.appendChild(style);
        }
    }
}

// Initialize cart system when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.panaditeCart = new PanaditeCartSystem();
    
    // If user just logged in and there's a cart redirect, sync cart
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('cart_redirect') === '1') {
        window.panaditeCart.syncGuestCartAfterLogin();
    }
});

// Export for use in other scripts
if (typeof module !== 'undefined' && module.exports) {
    module.exports = PanaditeCartSystem;
}
