<?php
/**
 * PayFast ITN (Instant Transaction Notification) Handler
 * This file processes payment notifications from PayFast
 */

// Don't start session for ITN - this is a server-to-server call
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../config/db_connect.php';
require_once __DIR__ . '/payfast.php';

// Log the ITN for debugging
error_log("PayFast ITN received: " . json_encode($_POST));

// Verify that this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(400);
    exit('Invalid request method');
}

// Get the POST data
$postData = $_POST;

// Verify the ITN
if (!verifyPayfastItn($postData)) {
    error_log("PayFast ITN verification failed");
    http_response_code(400);
    exit('ITN verification failed');
}

// Extract payment information
$payment_status = $postData['payment_status'] ?? '';
$payment_id = $postData['m_payment_id'] ?? '';
$pf_payment_id = $postData['pf_payment_id'] ?? '';
$amount_gross = floatval($postData['amount_gross'] ?? 0);
$amount_fee = floatval($postData['amount_fee'] ?? 0);
$amount_net = floatval($postData['amount_net'] ?? 0);

// Parse our custom payment ID to get user and item info
// Format: PFA-timestamp-userid-random
$payment_parts = explode('-', $payment_id);
if (count($payment_parts) >= 3) {
    $user_id = intval($payment_parts[2]);
} else {
    error_log("Invalid payment ID format: " . $payment_id);
    http_response_code(400);
    exit('Invalid payment ID');
}

// Determine item type and ID from the item_name or custom fields
$item_name = $postData['item_name'] ?? '';
$item_type = 'course'; // Default to course
$item_id = 0;

// Try to extract course/video ID from item_name or custom field
if (isset($postData['custom_str1'])) {
    $custom_data = json_decode($postData['custom_str1'], true);
    if ($custom_data) {
        $item_type = $custom_data['item_type'] ?? 'course';
        $item_id = intval($custom_data['item_id'] ?? 0);
    }
}

// If we still don't have item_id, try to parse from item_name
if (!$item_id && preg_match('/Course ID: (\d+)/', $item_name, $matches)) {
    $item_id = intval($matches[1]);
}

if (!$item_id) {
    error_log("Could not determine item ID from payment: " . json_encode($postData));
    http_response_code(400);
    exit('Could not determine item ID');
}

try {
    // Check if we already have this payment recorded
    $existingPaymentQuery = "
        SELECT payment_id, payment_status 
        FROM payments 
        WHERE payment_id = ? OR pf_payment_id = ?
    ";
    
    $stmt = $conn->prepare($existingPaymentQuery);
    $stmt->bind_param("ss", $payment_id, $pf_payment_id);
    $stmt->execute();
    $existingResult = $stmt->get_result();
    
    if ($existingResult->num_rows > 0) {
        $existing = $existingResult->fetch_assoc();
        
        // Update the payment status if it has changed
        if ($existing['payment_status'] !== $payment_status) {
            $updateQuery = "
                UPDATE payments 
                SET payment_status = ?, updated_at = NOW() 
                WHERE payment_id = ? OR pf_payment_id = ?
            ";
            
            $stmt = $conn->prepare($updateQuery);
            $stmt->bind_param("sss", $payment_status, $payment_id, $pf_payment_id);
            $stmt->execute();
        }
    } else {
        // Record the new payment
        $insertQuery = "
            INSERT INTO payments (
                user_id, payment_id, pf_payment_id, item_type, item_id,
                amount, payment_status, payment_method, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, 'payfast', NOW())
        ";
        
        $stmt = $conn->prepare($insertQuery);
        $stmt->bind_param("isssids", $user_id, $payment_id, $pf_payment_id, $item_type, $item_id, $amount_gross, $payment_status);
        $stmt->execute();
    }
    
    // If payment is complete, process the enrollment/access
    if ($payment_status === 'COMPLETE') {
        if ($item_type === 'course') {
            // Check if user is already enrolled
            $enrollmentQuery = "
                SELECT enrollment_id 
                FROM enrollments 
                WHERE student_id = ? AND course_id = ?
            ";
            
            $stmt = $conn->prepare($enrollmentQuery);
            $stmt->bind_param("ii", $user_id, $item_id);
            $stmt->execute();
            $enrollmentResult = $stmt->get_result();
            
            if ($enrollmentResult->num_rows == 0) {
                // Enroll the user
                $enrollQuery = "
                    INSERT INTO enrollments (student_id, course_id, enrolled_date, progress) 
                    VALUES (?, ?, NOW(), 0)
                ";
                
                $stmt = $conn->prepare($enrollQuery);
                $stmt->bind_param("ii", $user_id, $item_id);
                $stmt->execute();
                
                // Get course title for notification
                $courseQuery = "SELECT title FROM courses WHERE course_id = ?";
                $stmt = $conn->prepare($courseQuery);
                $stmt->bind_param("i", $item_id);
                $stmt->execute();
                $courseResult = $stmt->get_result();
                
                if ($courseResult->num_rows > 0) {
                    $course = $courseResult->fetch_assoc();
                    
                    // Create enrollment notification
                    createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation.");
                    
                    // Create payment confirmation notification
                    createNotification($user_id, 'payment', "Payment of R" . number_format($amount_gross, 2) . " confirmed for course: " . $course['title']);
                }
            }
        } elseif ($item_type === 'video') {
            // For video purchases, we just need to ensure the payment is recorded
            // Access is controlled by the hasUserPurchasedVideo function
            
            // Get video title for notification
            $videoQuery = "SELECT title FROM videos WHERE video_id = ?";
            $stmt = $conn->prepare($videoQuery);
            $stmt->bind_param("i", $item_id);
            $stmt->execute();
            $videoResult = $stmt->get_result();
            
            if ($videoResult->num_rows > 0) {
                $video = $videoResult->fetch_assoc();
                
                // Create payment confirmation notification
                createNotification($user_id, 'payment', "Payment of R" . number_format($amount_gross, 2) . " confirmed for video: " . $video['title']);
            }
        }
    }
    
    // Log successful processing
    error_log("PayFast ITN processed successfully for payment: " . $payment_id);
    
    // Send OK response to PayFast
    echo "OK";
    
} catch (Exception $e) {
    error_log("PayFast ITN processing error: " . $e->getMessage());
    http_response_code(500);
    echo "Error processing payment";
}

/**
 * Create a notification for the user
 */
function createNotification($user_id, $type, $message) {
    global $conn;
    
    $insertNotification = "
        INSERT INTO notifications (user_id, type, message, created_at) 
        VALUES (?, ?, ?, NOW())
    ";
    
    $stmt = $conn->prepare($insertNotification);
    $stmt->bind_param("iss", $user_id, $type, $message);
    $stmt->execute();
}
?>
