<?php
/**
 * Panadite Academy - Password Reset Page
 * Secure password reset with token verification
 */

// Start session and include necessary files
session_start();
require_once __DIR__ . '/includes/site_config.php';
require_once __DIR__ . '/includes/db_connection.php';

// Initialize variables
$token = $_GET['token'] ?? '';
$message = '';
$error = '';
$success = false;
$token_valid = false;
$user_data = null;

// Validate token
if (!empty($token)) {
    // Check if token exists and is valid
    $stmt = $conn->prepare("
        SELECT prt.*, u.email, u.first_name 
        FROM password_reset_tokens prt 
        JOIN users u ON prt.user_id = u.user_id 
        WHERE prt.token = ? AND prt.expires_at > NOW() AND prt.used = 0
    ");
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $token_valid = true;
        $user_data = $result->fetch_assoc();
    } else {
        $error = 'Invalid or expired reset token. Please request a new password reset.';
    }
}

// Process password reset
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $token_valid) {
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Validation
    if (empty($new_password)) {
        $error = 'Please enter a new password.';
    } elseif (strlen($new_password) < 8) {
        $error = 'Password must be at least 8 characters long.';
    } elseif ($new_password !== $confirm_password) {
        $error = 'Passwords do not match.';
    } else {
        // Hash the new password
        $password_hash = password_hash($new_password, PASSWORD_DEFAULT);
        
        // Update user password
        $update_stmt = $conn->prepare("UPDATE users SET password = ? WHERE user_id = ?");
        $update_stmt->bind_param("si", $password_hash, $user_data['user_id']);
        
        if ($update_stmt->execute()) {
            // Mark token as used
            $mark_used_stmt = $conn->prepare("UPDATE password_reset_tokens SET used = 1 WHERE token = ?");
            $mark_used_stmt->bind_param("s", $token);
            $mark_used_stmt->execute();
            
            $success = true;
            $message = 'Your password has been successfully updated! You can now login with your new password.';
        } else {
            $error = 'Failed to update password. Please try again.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password - <?php echo $site_name; ?></title>
    
    <!-- Bootstrap 5.2.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Animate.css -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    
    <style>
        :root {
            --primary-color: #276BE4;
            --secondary-color: #FFA915;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --warning-color: #ffc107;
        }

        body {
            background: whitesmoke;
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 0;
            position: relative;
            overflow-x: hidden;
        }

        /* Subtle animated background elements */
        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: 
                radial-gradient(circle at 20% 20%, rgba(39, 107, 228, 0.02) 0%, transparent 50%),
                radial-gradient(circle at 80% 80%, rgba(255, 169, 21, 0.02) 0%, transparent 50%),
                radial-gradient(circle at 40% 60%, rgba(39, 107, 228, 0.015) 0%, transparent 50%);
            animation: gentleFloat 20s ease-in-out infinite;
            z-index: -1;
        }

        @keyframes gentleFloat {
            0%, 100% { transform: translateY(0px) rotate(0deg); }
            50% { transform: translateY(-10px) rotate(1deg); }
        }

        .main-content {
            min-height: calc(100vh - 200px);
            display: flex;
            align-items: center;
            padding: 40px 0;
        }

        .reset-container {
            max-width: 550px;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.15);
            overflow: hidden;
            backdrop-filter: blur(10px);
            position: relative;
        }

        .reset-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
        }

        .header {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            color: white;
            padding: 40px 30px;
            text-align: center;
            position: relative;
        }

        .header i {
            font-size: 48px;
            margin-bottom: 15px;
            opacity: 0.9;
        }

        .header h1 {
            margin: 0;
            font-size: 28px;
            font-weight: 700;
        }

        .header p {
            margin: 10px 0 0 0;
            opacity: 0.9;
            font-size: 16px;
        }

        .form-section {
            padding: 40px 30px;
        }

        .form-floating {
            margin-bottom: 20px;
            position: relative;
        }

        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 12px 50px 12px 16px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.9);
        }

        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(39, 107, 228, 0.25);
            background: white;
        }

        .form-floating label {
            color: #6c757d;
            font-weight: 500;
        }

        .password-toggle {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #6c757d;
            cursor: pointer;
            z-index: 10;
            padding: 0;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .password-toggle:hover {
            color: var(--primary-color);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            border-radius: 12px;
            padding: 14px 30px;
            font-size: 16px;
            font-weight: 600;
            width: 100%;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }

        .btn-success {
            background: linear-gradient(135deg, var(--success-color), #218838);
            border: none;
            border-radius: 12px;
            padding: 14px 30px;
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-success:hover {
            background: linear-gradient(135deg, #218838, #1e7e34);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(40, 167, 69, 0.3);
        }

        .alert {
            border-radius: 12px;
            border: none;
            padding: 16px 20px;
            margin-bottom: 25px;
            font-weight: 500;
            position: relative;
            overflow: hidden;
        }

        .alert::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
            background: currentColor;
        }

        .alert-success {
            background: linear-gradient(135deg, #d4edda, #c3e6cb);
            color: #155724;
        }

        .alert-danger {
            background: linear-gradient(135deg, #f8d7da, #f5c6cb);
            color: #721c24;
        }

        .password-strength {
            margin-top: 10px;
        }

        .strength-bar {
            height: 6px;
            background: #e9ecef;
            border-radius: 3px;
            overflow: hidden;
            margin-bottom: 8px;
        }

        .strength-fill {
            height: 100%;
            transition: all 0.3s ease;
            border-radius: 3px;
        }

        .strength-weak { background: var(--danger-color); width: 25%; }
        .strength-fair { background: var(--warning-color); width: 50%; }
        .strength-good { background: #17a2b8; width: 75%; }
        .strength-strong { background: var(--success-color); width: 100%; }

        .strength-text {
            font-size: 12px;
            font-weight: 500;
        }

        .success-animation {
            text-align: center;
            padding: 20px;
        }

        .success-animation i {
            font-size: 64px;
            color: var(--success-color);
            margin-bottom: 20px;
            animation: pulse 2s infinite;
        }

        .back-links {
            text-align: center;
            padding: 20px 30px 30px;
            background: #f8f9fa;
            border-top: 1px solid #e9ecef;
        }

        .back-links a {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
            margin: 0 15px;
            transition: all 0.3s ease;
        }

        .back-links a:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }

        .user-info {
            background: linear-gradient(135deg, #e3f2fd, #bbdefb);
            border: 1px solid #2196f3;
            border-radius: 12px;
            padding: 15px 20px;
            margin-bottom: 25px;
            text-align: center;
        }

        .user-info h5 {
            color: #1565c0;
            margin: 0 0 5px 0;
            font-weight: 600;
        }

        .user-info p {
            color: #1976d2;
            margin: 0;
            font-size: 14px;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }

        @media (max-width: 768px) {
            .main-content {
                padding: 20px 0;
            }
            
            .reset-container {
                margin: 0 15px;
            }
            
            .header {
                padding: 30px 20px;
            }
            
            .form-section {
                padding: 30px 20px;
            }
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/includes/navbar.php'; ?>
    
    <div class="main-content">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-lg-6 col-md-8">
                    <div class="reset-container animate__animated animate__fadeInUp">
                        
                        <?php if ($success): ?>
                        <!-- Success State -->
                        <div class="header">
                            <div class="success-animation">
                                <i class="fas fa-check-circle"></i>
                            </div>
                            <h1>Password Updated!</h1>
                            <p>Your password has been successfully changed</p>
                        </div>
                        
                        <div class="form-section">
                            <div class="alert alert-success animate__animated animate__fadeIn">
                                <i class="fas fa-shield-alt me-2"></i>
                                <?php echo htmlspecialchars($message); ?>
                            </div>
                            
                            <div class="text-center">
                                <a href="auth/login.php" class="btn btn-success">
                                    <i class="fas fa-sign-in-alt me-2"></i>
                                    Login Now
                                </a>
                            </div>
                        </div>
                        
                        <?php elseif (!empty($token) && !$token_valid): ?>
                        <!-- Invalid Token State -->
                        <div class="header">
                            <i class="fas fa-exclamation-triangle" style="color: var(--warning-color);"></i>
                            <h1>Invalid Reset Link</h1>
                            <p>This password reset link is invalid or has expired</p>
                        </div>
                        
                        <div class="form-section">
                            <div class="alert alert-danger animate__animated animate__shakeX">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                <?php echo htmlspecialchars($error); ?>
                            </div>
                            
                            <div class="text-center">
                                <a href="forgot-password.php" class="btn btn-primary">
                                    <i class="fas fa-redo me-2"></i>
                                    Request New Reset Link
                                </a>
                            </div>
                        </div>
                        
                        <?php elseif (empty($token)): ?>
                        <!-- No Token State -->
                        <div class="header">
                            <i class="fas fa-question-circle" style="color: var(--warning-color);"></i>
                            <h1>Missing Reset Token</h1>
                            <p>No password reset token provided</p>
                        </div>
                        
                        <div class="form-section">
                            <div class="alert alert-danger">
                                <i class="fas fa-link me-2"></i>
                                Please use the complete reset link from your email, or request a new password reset.
                            </div>
                            
                            <div class="text-center">
                                <a href="forgot-password.php" class="btn btn-primary">
                                    <i class="fas fa-key me-2"></i>
                                    Reset Password
                                </a>
                            </div>
                        </div>
                        
                        <?php else: ?>
                        <!-- Password Reset Form -->
                        <div class="header">
                            <i class="fas fa-lock"></i>
                            <h1>Set New Password</h1>
                            <p>Choose a strong password for your account</p>
                        </div>
                        
                        <div class="form-section">
                            <?php if ($user_data): ?>
                            <div class="user-info animate__animated animate__fadeIn">
                                <h5><i class="fas fa-user me-2"></i><?php echo htmlspecialchars($user_data['first_name']); ?></h5>
                                <p><?php echo htmlspecialchars($user_data['email']); ?></p>
                            </div>
                            <?php endif; ?>
                            
                            <?php if ($error): ?>
                            <div class="alert alert-danger animate__animated animate__shakeX">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                <?php echo htmlspecialchars($error); ?>
                            </div>
                            <?php endif; ?>
                            
                            <form method="POST" class="animate__animated animate__fadeIn" id="resetForm">
                                <div class="form-floating">
                                    <input type="password" 
                                           class="form-control" 
                                           id="new_password" 
                                           name="new_password" 
                                           placeholder="Enter new password"
                                           required
                                           minlength="8">
                                    <label for="new_password">
                                        <i class="fas fa-lock me-2"></i>New Password
                                    </label>
                                    <button type="button" class="password-toggle" onclick="togglePassword('new_password')">
                                        <i class="fas fa-eye" id="toggleIcon1"></i>
                                    </button>
                                </div>
                                
                                <div class="password-strength" id="passwordStrength" style="display: none;">
                                    <div class="strength-bar">
                                        <div class="strength-fill" id="strengthFill"></div>
                                    </div>
                                    <div class="strength-text" id="strengthText">Password strength</div>
                                </div>
                                
                                <div class="form-floating">
                                    <input type="password" 
                                           class="form-control" 
                                           id="confirm_password" 
                                           name="confirm_password" 
                                           placeholder="Confirm new password"
                                           required>
                                    <label for="confirm_password">
                                        <i class="fas fa-lock me-2"></i>Confirm Password
                                    </label>
                                    <button type="button" class="password-toggle" onclick="togglePassword('confirm_password')">
                                        <i class="fas fa-eye" id="toggleIcon2"></i>
                                    </button>
                                </div>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-shield-alt me-2"></i>
                                    Update Password
                                </button>
                            </form>
                        </div>
                        <?php endif; ?>
                        
                        <div class="back-links">
                            <a href="auth/login.php">
                                <i class="fas fa-arrow-left me-1"></i>Back to Login
                            </a>
                            <a href="index.php">
                                <i class="fas fa-home me-1"></i>Home
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include __DIR__ . '/includes/footer.php'; ?>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Password visibility toggle
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = document.getElementById(fieldId === 'new_password' ? 'toggleIcon1' : 'toggleIcon2');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        
        // Password strength checker
        function checkPasswordStrength(password) {
            let strength = 0;
            let feedback = [];
            
            if (password.length >= 8) strength++;
            else feedback.push('At least 8 characters');
            
            if (/[a-z]/.test(password)) strength++;
            else feedback.push('Lowercase letter');
            
            if (/[A-Z]/.test(password)) strength++;
            else feedback.push('Uppercase letter');
            
            if (/[0-9]/.test(password)) strength++;
            else feedback.push('Number');
            
            if (/[^A-Za-z0-9]/.test(password)) strength++;
            else feedback.push('Special character');
            
            return { strength, feedback };
        }
        
        // Real-time password strength
        document.addEventListener('DOMContentLoaded', function() {
            const passwordField = document.getElementById('new_password');
            const strengthDiv = document.getElementById('passwordStrength');
            const strengthFill = document.getElementById('strengthFill');
            const strengthText = document.getElementById('strengthText');
            
            if (passwordField) {
                passwordField.addEventListener('input', function() {
                    const password = this.value;
                    
                    if (password.length > 0) {
                        strengthDiv.style.display = 'block';
                        const result = checkPasswordStrength(password);
                        
                        // Update strength bar
                        strengthFill.className = 'strength-fill';
                        if (result.strength <= 2) {
                            strengthFill.classList.add('strength-weak');
                            strengthText.textContent = 'Weak password';
                            strengthText.style.color = 'var(--danger-color)';
                        } else if (result.strength <= 3) {
                            strengthFill.classList.add('strength-fair');
                            strengthText.textContent = 'Fair password';
                            strengthText.style.color = 'var(--warning-color)';
                        } else if (result.strength <= 4) {
                            strengthFill.classList.add('strength-good');
                            strengthText.textContent = 'Good password';
                            strengthText.style.color = '#17a2b8';
                        } else {
                            strengthFill.classList.add('strength-strong');
                            strengthText.textContent = 'Strong password';
                            strengthText.style.color = 'var(--success-color)';
                        }
                        
                        if (result.feedback.length > 0) {
                            strengthText.textContent += ' (Missing: ' + result.feedback.join(', ') + ')';
                        }
                    } else {
                        strengthDiv.style.display = 'none';
                    }
                });
            }
            
            // Form submission
            const form = document.getElementById('resetForm');
            const submitBtn = document.querySelector('.btn-primary');
            
            if (form && submitBtn) {
                form.addEventListener('submit', function() {
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Updating...';
                    submitBtn.disabled = true;
                });
            }
        });
    </script>
</body>
</html>
