<?php
/**
 * Database Setup Script for Study Reminders
 * Creates the study_reminders table and sets up the automated reminder system
 */

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type for proper display
header('Content-Type: text/html; charset=utf-8');

echo "<!DOCTYPE html>
<html>
<head>
    <title>Study Reminders Database Setup</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        .info { color: #17a2b8; }
        pre { background: #f8f9fa; padding: 10px; border-radius: 5px; overflow-x: auto; }
    </style>
</head>
<body>
<div class='container'>
<h1>🚀 Study Reminders Database Setup</h1>
";

echo "<p>Starting database setup process...</p>";

// Try to include database connection with error handling
try {
    $config_path = '../config/database.php';
    
    echo "<p class='info'>📂 Looking for database config at: " . realpath($config_path) . "</p>";
    
    if (!file_exists($config_path)) {
        // Try alternative paths
        $alt_paths = [
            dirname(__DIR__) . '/config/database.php',
            dirname(__DIR__) . '/config/db_connection.php',
            dirname(__DIR__) . '/includes/db_connection.php',
            dirname(__DIR__) . '/config/connection.php'
        ];
        
        foreach ($alt_paths as $path) {
            echo "<p class='info'>🔍 Trying alternative path: " . $path . "</p>";
            if (file_exists($path)) {
                $config_path = $path;
                echo "<p class='success'>✅ Found database config at: " . $path . "</p>";
                break;
            }
        }
    }
    
    if (!file_exists($config_path)) {
        throw new Exception("Database configuration file not found. Please check the path.");
    }
    
    require_once $config_path;
    
    if (!isset($conn) || !$conn) {
        throw new Exception("Database connection not established. Please check your database configuration.");
    }
    
    echo "<p class='success'>✅ Database connection established successfully!</p>";
    
} catch (Exception $e) {
    echo "<p class='error'>❌ Database connection error: " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p class='warning'>⚠️ Creating manual database connection...</p>";
    
    // Create manual connection with common default settings
    $servername = "localhost";
    $username = "root";  // Default XAMPP username
    $password = "";      // Default XAMPP password
    $database = "panadite_academy"; // Common database name
    
    try {
        $conn = new mysqli($servername, $username, $password, $database);
        
        if ($conn->connect_error) {
            throw new Exception("Connection failed: " . $conn->connect_error);
        }
        
        echo "<p class='success'>✅ Manual database connection successful!</p>";
        echo "<p class='info'>📊 Using database: $database on $servername</p>";
        
    } catch (Exception $e2) {
        echo "<p class='error'>❌ Manual connection also failed: " . htmlspecialchars($e2->getMessage()) . "</p>";
        echo "<div class='error'>
                <h3>Database Setup Instructions:</h3>
                <ol>
                    <li>Make sure your web server (XAMPP/WAMP) is running</li>
                    <li>Ensure MySQL/MariaDB service is started</li>
                    <li>Check your database configuration file</li>
                    <li>Verify database credentials</li>
                </ol>
                <p><strong>Common database config locations:</strong></p>
                <ul>
                    <li>config/database.php</li>
                    <li>config/db_connection.php</li>
                    <li>includes/db_connection.php</li>
                </ul>
              </div>";
        exit;
    }
}

function createStudyRemindersTable($conn) {
    echo "<p class='info'>🏗️ Creating study_reminders table...</p>";
    
    $sql = "
    CREATE TABLE IF NOT EXISTS `study_reminders` (
        `reminder_id` int(11) NOT NULL AUTO_INCREMENT,
        `student_id` int(11) NOT NULL,
        `course_id` int(11) NOT NULL,
        `reminder_date` date NOT NULL,
        `reminder_time` time NOT NULL,
        `reminder_type` enum('next_lesson','review_session','assignment','quiz_prep','study_session','custom') NOT NULL DEFAULT 'custom',
        `description` text NOT NULL,
        `email_sent` tinyint(1) NOT NULL DEFAULT 0,
        `reminder_sent` tinyint(1) NOT NULL DEFAULT 0,
        `is_completed` tinyint(1) NOT NULL DEFAULT 0,
        `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`reminder_id`),
        KEY `idx_student_id` (`student_id`),
        KEY `idx_course_id` (`course_id`),
        KEY `idx_reminder_date` (`reminder_date`),
        KEY `idx_student_course` (`student_id`, `course_id`),
        KEY `idx_upcoming_reminders` (`student_id`, `reminder_date`, `reminder_time`),
        KEY `idx_email_notifications` (`email_sent`, `reminder_date`, `reminder_time`),
        KEY `idx_pending_reminders` (`reminder_sent`, `reminder_date`, `reminder_time`),
        CONSTRAINT `fk_study_reminders_student` FOREIGN KEY (`student_id`) REFERENCES `users` (`user_id`) ON DELETE CASCADE,
        CONSTRAINT `fk_study_reminders_course` FOREIGN KEY (`course_id`) REFERENCES `courses` (`course_id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql) === TRUE) {
        echo "<p class='success'>✅ Table 'study_reminders' created successfully.</p>";
        return true;
    } else {
        echo "<p class='error'>❌ Error creating table: " . htmlspecialchars($conn->error) . "</p>";
        return false;
    }
}

function createReminderLogTable($conn) {
    echo "<p class='info'>🏗️ Creating reminder_log table for tracking sent reminders...</p>";
    
    $sql = "
    CREATE TABLE IF NOT EXISTS `reminder_log` (
        `log_id` int(11) NOT NULL AUTO_INCREMENT,
        `reminder_id` int(11) NOT NULL,
        `student_id` int(11) NOT NULL,
        `email_address` varchar(255) NOT NULL,
        `sent_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
        `email_status` enum('sent','failed','pending') NOT NULL DEFAULT 'pending',
        `error_message` text,
        PRIMARY KEY (`log_id`),
        KEY `idx_reminder_id` (`reminder_id`),
        KEY `idx_student_id` (`student_id`),
        KEY `idx_sent_at` (`sent_at`),
        CONSTRAINT `fk_reminder_log_reminder` FOREIGN KEY (`reminder_id`) REFERENCES `study_reminders` (`reminder_id`) ON DELETE CASCADE,
        CONSTRAINT `fk_reminder_log_student` FOREIGN KEY (`student_id`) REFERENCES `users` (`user_id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql) === TRUE) {
        echo "<p class='success'>✅ Table 'reminder_log' created successfully.</p>";
        return true;
    } else {
        echo "<p class='error'>❌ Error creating reminder_log table: " . htmlspecialchars($conn->error) . "</p>";
        return false;
    }
}

function setupAutomaticReminderSystem($conn) {
    echo "<p class='info'>⚙️ Setting up automatic reminder system...</p>";
    
    // Create a configuration table for the reminder system
    $sql = "
    CREATE TABLE IF NOT EXISTS `reminder_system_config` (
        `config_id` int(11) NOT NULL AUTO_INCREMENT,
        `config_key` varchar(100) NOT NULL UNIQUE,
        `config_value` text NOT NULL,
        `description` text,
        `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`config_id`),
        UNIQUE KEY `idx_config_key` (`config_key`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql) === TRUE) {
        echo "<p class='success'>✅ Table 'reminder_system_config' created successfully.</p>";
        
        // Insert default configuration
        $configs = [
            ['reminder_check_interval', '5', 'Interval in minutes to check for due reminders'],
            ['email_from_address', 'noreply@panaditeacademy.com', 'From email address for reminders'],
            ['email_from_name', 'Panadite Academy', 'From name for reminder emails'],
            ['reminder_advance_minutes', '0', 'Send reminders X minutes before scheduled time'],
            ['max_reminder_attempts', '3', 'Maximum attempts to send a reminder email'],
            ['system_enabled', '1', 'Enable/disable the automatic reminder system']
        ];
        
        foreach ($configs as $config) {
            $stmt = $conn->prepare("INSERT IGNORE INTO reminder_system_config (config_key, config_value, description) VALUES (?, ?, ?)");
            $stmt->bind_param("sss", $config[0], $config[1], $config[2]);
            $stmt->execute();
        }
        
        echo "<p class='success'>✅ Default reminder system configuration inserted.</p>";
        return true;
    } else {
        echo "<p class='error'>❌ Error creating reminder_system_config table: " . htmlspecialchars($conn->error) . "</p>";
        return false;
    }
}

function createSampleReminders($conn) {
    echo "<p class='info'>📝 Creating sample reminders for testing...</p>";
    
    // Try different approaches to find users and courses
    $user = null;
    $course = null;
    
    // Try to find students with different role values
    $student_queries = [
        "SELECT user_id FROM users WHERE role = 'student' LIMIT 1",
        "SELECT user_id FROM users WHERE user_type = 'student' LIMIT 1", 
        "SELECT user_id FROM users WHERE role_id = 2 LIMIT 1",  // Common student role ID
        "SELECT DISTINCT user_id FROM enrollments LIMIT 1",      // From enrollments table
        "SELECT user_id FROM users WHERE user_id > 0 LIMIT 1"   // Any user
    ];
    
    foreach ($student_queries as $query) {
        try {
            $result = $conn->query($query);
            if ($result && $result->num_rows > 0) {
                $user = $result->fetch_assoc();
                echo "<p class='success'>✅ Found user with ID: {$user['user_id']}</p>";
                break;
            }
        } catch (Exception $e) {
            // Continue to next query if this one fails
            continue;
        }
    }
    
    // Try to find courses
    $course_queries = [
        "SELECT course_id FROM courses LIMIT 1",
        "SELECT id as course_id FROM courses LIMIT 1",  // Some tables use 'id' instead
        "SELECT DISTINCT course_id FROM enrollments LIMIT 1"  // From enrollments table
    ];
    
    foreach ($course_queries as $query) {
        try {
            $result = $conn->query($query);
            if ($result && $result->num_rows > 0) {
                $course = $result->fetch_assoc();
                echo "<p class='success'>✅ Found course with ID: {$course['course_id']}</p>";
                break;
            }
        } catch (Exception $e) {
            // Continue to next query if this one fails
            continue;
        }
    }
    
    if (!$user || !$course) {
        echo "<p class='warning'>⚠️ Could not find suitable user and course data. Skipping sample data creation.</p>";
        echo "<p class='info'>💡 This is normal if your database uses different table structures. You can still create reminders through the interface!</p>";
        return true;
    }
    
    // Create sample reminders
    $tomorrow = date('Y-m-d', strtotime('+1 day'));
    $next_week = date('Y-m-d', strtotime('+7 days'));
    
    $sample_reminders = [
        [
            'student_id' => $user['user_id'],
            'course_id' => $course['course_id'],
            'reminder_date' => $tomorrow,
            'reminder_time' => '09:00:00',
            'reminder_type' => 'next_lesson',
            'description' => 'Time for your next lesson - Introduction to Programming Basics',
            'email_sent' => 1
        ],
        [
            'student_id' => $user['user_id'],
            'course_id' => $course['course_id'],
            'reminder_date' => $next_week,
            'reminder_time' => '14:30:00',
            'reminder_type' => 'review_session',
            'description' => 'Review session: Variables and Data Types',
            'email_sent' => 1
        ]
    ];
    
    $stmt = $conn->prepare("INSERT INTO study_reminders (student_id, course_id, reminder_date, reminder_time, reminder_type, description, email_sent) VALUES (?, ?, ?, ?, ?, ?, ?)");
    
    foreach ($sample_reminders as $reminder) {
        $stmt->bind_param("iissssi", 
            $reminder['student_id'],
            $reminder['course_id'],
            $reminder['reminder_date'],
            $reminder['reminder_time'],
            $reminder['reminder_type'],
            $reminder['description'],
            $reminder['email_sent']
        );
        $stmt->execute();
    }
    
    echo "<p class='success'>✅ Sample reminders created successfully.</p>";
    return true;
}

// Main execution
try {
    echo "<h2 class='info'>🚀 Starting Study Reminders Database Setup...</h2>";
    
    // Create main table
    if (!createStudyRemindersTable($conn)) {
        throw new Exception("Failed to create study_reminders table");
    }
    
    // Create log table
    if (!createReminderLogTable($conn)) {
        throw new Exception("Failed to create reminder_log table");
    }
    
    // Setup automatic reminder system
    if (!setupAutomaticReminderSystem($conn)) {
        throw new Exception("Failed to setup automatic reminder system");
    }
    
    // Create sample data (optional)
    createSampleReminders($conn);
    
    echo "<div style='background: linear-gradient(135deg, #28a745, #20c997); color: white; padding: 20px; border-radius: 10px; margin: 20px 0;'>
            <h2>🎉 Study Reminders Database Setup Complete!</h2>
            <p>✅ All tables created successfully</p>
            <p>✅ Automatic reminder system configured</p>
            <p>✅ Sample data added</p>
          </div>";
    
    echo "<div class='info' style='background: #e3f2fd; padding: 15px; border-radius: 8px; margin: 15px 0;'>
            <h3>📋 Next Steps:</h3>
            <ol>
                <li>Set up the automatic reminder cron job (check_reminders.php)</li>
                <li>Configure email settings in the reminder_system_config table</li>
                <li>Test the reminder creation and email sending functionality</li>
            </ol>
          </div>";
    
    // Show table information
    echo "<h3>📊 Table Information:</h3>";
    echo "<div style='background: #f8f9fa; padding: 15px; border-radius: 8px;'>";
    
    $tables = ['study_reminders', 'reminder_log', 'reminder_system_config'];
    foreach ($tables as $table) {
        $result = $conn->query("SHOW TABLE STATUS LIKE '$table'");
        if ($result && $row = $result->fetch_assoc()) {
            echo "<p><strong>$table:</strong> {$row['Rows']} rows, " . number_format($row['Data_length']) . " bytes</p>";
        }
    }
    echo "</div>";
    
    // Cron setup instructions
    echo "<div class='warning' style='background: #fff3cd; padding: 20px; border-radius: 10px; margin: 20px 0; border-left: 5px solid #ffc107;'>
            <h3>⏰ Set Up Automatic Reminders</h3>
            <p><strong>For Windows (Task Scheduler):</strong></p>
            <ol>
                <li>Open Task Scheduler</li>
                <li>Create Basic Task → Run every 5 minutes</li>
                <li>Action: Start a program</li>
                <li>Program: <code>php.exe</code></li>
                <li>Arguments: <code>" . dirname(__DIR__) . "\\cron\\check_reminders.php</code></li>
            </ol>
            
            <p><strong>For Linux/Mac (Crontab):</strong></p>
            <pre>*/5 * * * * cd " . dirname(__DIR__) . "/cron && php check_reminders.php</pre>
            
            <p><strong>Manual Test:</strong></p>
            <pre>php " . dirname(__DIR__) . "/cron/check_reminders.php</pre>
          </div>";
    
    echo "<div style='text-align: center; margin: 30px 0;'>
            <a href='../student/course-detail.php' style='background: #007bff; color: white; padding: 15px 30px; text-decoration: none; border-radius: 8px; display: inline-block; margin: 10px;'>
                🚀 Go to Course Page
            </a>
            <a href='test_reminders.php' style='background: #28a745; color: white; padding: 15px 30px; text-decoration: none; border-radius: 8px; display: inline-block; margin: 10px;'>
                🧪 Test System
            </a>
            <a href='index.php' style='background: #6c757d; color: white; padding: 15px 30px; text-decoration: none; border-radius: 8px; display: inline-block; margin: 10px;'>
                🏠 Setup Home
            </a>
          </div>";
    
} catch (Exception $e) {
    echo "<div class='error' style='background: #f8d7da; padding: 20px; border-radius: 10px; border-left: 5px solid #dc3545;'>
            <h3>❌ Setup Failed</h3>
            <p>" . htmlspecialchars($e->getMessage()) . "</p>
            <p><strong>Common solutions:</strong></p>
            <ul>
                <li>Ensure MySQL/MariaDB service is running</li>
                <li>Check database credentials</li>
                <li>Verify database name exists</li>
                <li>Make sure you have proper permissions</li>
            </ul>
            <button onclick='location.reload()' style='background: #dc3545; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer;'>
                🔄 Try Again
            </button>
          </div>";
}

if (isset($conn) && $conn) {
    $conn->close();
}

echo "</div></body></html>";
?>
