<?php
/**
 * Test Script for Study Reminders System
 * Use this to test the automatic reminder functionality
 */

require_once '../config/database.php';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Study Reminders System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <style>
        body { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; }
        .test-container { background: white; border-radius: 20px; box-shadow: 0 20px 60px rgba(0,0,0,0.1); }
        .test-header { background: linear-gradient(135deg, #4ECDC4 0%, #44A08D 100%); color: white; padding: 2rem; border-radius: 20px 20px 0 0; }
        .test-output { background: #000; color: #00ff00; font-family: 'Courier New', monospace; padding: 1rem; border-radius: 10px; max-height: 300px; overflow-y: auto; }
        .status-indicator { padding: 0.5rem 1rem; border-radius: 20px; font-weight: bold; }
        .status-success { background: #d4edda; color: #155724; }
        .status-warning { background: #fff3cd; color: #856404; }
        .status-error { background: #f8d7da; color: #721c24; }
    </style>
</head>
<body>
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-md-10">
                <div class="test-container">
                    <div class="test-header text-center">
                        <h1><i class="bx bx-test-tube me-2"></i>Study Reminders Test Suite</h1>
                        <p class="mb-0">Test and verify your automatic reminder system</p>
                    </div>
                    
                    <div class="p-4">
                        <!-- System Status Check -->
                        <div class="card mb-4">
                            <div class="card-header bg-primary text-white">
                                <h5><i class="bx bx-check-circle me-2"></i>System Status Check</h5>
                            </div>
                            <div class="card-body">
                                <?php
                                $status_checks = [];
                                
                                // Check database tables
                                try {
                                    $tables = ['study_reminders', 'reminder_log', 'reminder_system_config'];
                                    foreach ($tables as $table) {
                                        $result = $conn->query("SHOW TABLES LIKE '$table'");
                                        if ($result->num_rows > 0) {
                                            $status_checks[] = ['type' => 'success', 'message' => "✅ Table '$table' exists"];
                                        } else {
                                            $status_checks[] = ['type' => 'error', 'message' => "❌ Table '$table' missing"];
                                        }
                                    }
                                    
                                    // Check for sample data
                                    $reminder_count = $conn->query("SELECT COUNT(*) as count FROM study_reminders")->fetch_assoc()['count'];
                                    $status_checks[] = ['type' => 'info', 'message' => "📊 Found $reminder_count reminders in database"];
                                    
                                    // Check configuration
                                    $config_count = $conn->query("SELECT COUNT(*) as count FROM reminder_system_config")->fetch_assoc()['count'];
                                    $status_checks[] = ['type' => 'info', 'message' => "⚙️ Found $config_count configuration settings"];
                                    
                                    // Check if cron file exists
                                    $cron_file = dirname(__DIR__) . '/cron/check_reminders.php';
                                    if (file_exists($cron_file)) {
                                        $status_checks[] = ['type' => 'success', 'message' => "✅ Cron script file exists"];
                                    } else {
                                        $status_checks[] = ['type' => 'error', 'message' => "❌ Cron script file missing"];
                                    }
                                    
                                } catch (Exception $e) {
                                    $status_checks[] = ['type' => 'error', 'message' => "❌ Database error: " . $e->getMessage()];
                                }
                                
                                foreach ($status_checks as $check) {
                                    $class = $check['type'] === 'success' ? 'status-success' : 
                                            ($check['type'] === 'error' ? 'status-error' : 'status-warning');
                                    echo "<div class='status-indicator $class mb-2'>{$check['message']}</div>";
                                }
                                ?>
                            </div>
                        </div>
                        
                        <!-- Quick Test Actions -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="card mb-4">
                                    <div class="card-header bg-success text-white">
                                        <h6><i class="bx bx-play-circle me-2"></i>Quick Tests</h6>
                                    </div>
                                    <div class="card-body">
                                        <button class="btn btn-outline-primary btn-sm w-100 mb-2" onclick="createTestReminder()">
                                            <i class="bx bx-plus me-2"></i>Create Test Reminder (2 min from now)
                                        </button>
                                        <button class="btn btn-outline-info btn-sm w-100 mb-2" onclick="runCronCheck()">
                                            <i class="bx bx-refresh me-2"></i>Run Cron Check Manually
                                        </button>
                                        <button class="btn btn-outline-warning btn-sm w-100 mb-2" onclick="viewReminders()">
                                            <i class="bx bx-list-ul me-2"></i>View All Reminders
                                        </button>
                                        <button class="btn btn-outline-danger btn-sm w-100" onclick="clearTestData()">
                                            <i class="bx bx-trash me-2"></i>Clear Test Data
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="card mb-4">
                                    <div class="card-header bg-info text-white">
                                        <h6><i class="bx bx-cog me-2"></i>System Configuration</h6>
                                    </div>
                                    <div class="card-body">
                                        <?php
                                        try {
                                            $configs = $conn->query("SELECT config_key, config_value FROM reminder_system_config ORDER BY config_key");
                                            while ($config = $configs->fetch_assoc()) {
                                                echo "<small><strong>{$config['config_key']}:</strong> {$config['config_value']}</small><br>";
                                            }
                                        } catch (Exception $e) {
                                            echo "<small class='text-danger'>Error loading configuration</small>";
                                        }
                                        ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Test Output -->
                        <div class="card">
                            <div class="card-header bg-dark text-white">
                                <h6><i class="bx bx-terminal me-2"></i>Test Output</h6>
                            </div>
                            <div class="card-body p-0">
                                <div class="test-output" id="testOutput">
                                    Ready to run tests... Click a button above to start testing!
                                </div>
                            </div>
                        </div>
                        
                        <!-- Cron Setup Instructions -->
                        <div class="card mt-4">
                            <div class="card-header bg-warning text-dark">
                                <h6><i class="bx bx-time me-2"></i>Automatic Reminder Setup</h6>
                            </div>
                            <div class="card-body">
                                <p><strong>To enable automatic reminders, set up a cron job:</strong></p>
                                
                                <div class="alert alert-dark">
                                    <h6>For Linux/Mac (crontab):</h6>
                                    <code>*/5 * * * * cd <?= dirname(__DIR__) ?>/cron && php check_reminders.php</code>
                                </div>
                                
                                <div class="alert alert-info">
                                    <h6>For Windows (Task Scheduler):</h6>
                                    <ol>
                                        <li>Open Task Scheduler</li>
                                        <li>Create Basic Task</li>
                                        <li>Set trigger: Daily, repeat every 5 minutes</li>
                                        <li>Action: Start a program</li>
                                        <li>Program: <code>php.exe</code></li>
                                        <li>Arguments: <code><?= dirname(__DIR__) ?>\cron\check_reminders.php</code></li>
                                    </ol>
                                </div>
                                
                                <div class="alert alert-success">
                                    <h6>Manual Test:</h6>
                                    <p>Run this command to test manually:</p>
                                    <code>php <?= dirname(__DIR__) ?>/cron/check_reminders.php</code>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        function addLog(message, type = 'info') {
            const output = document.getElementById('testOutput');
            const timestamp = new Date().toLocaleTimeString();
            const icon = type === 'success' ? '✅' : type === 'error' ? '❌' : type === 'warning' ? '⚠️' : 'ℹ️';
            output.innerHTML += `<div>[${timestamp}] ${icon} ${message}</div>`;
            output.scrollTop = output.scrollHeight;
        }
        
        function createTestReminder() {
            addLog('Creating test reminder for 2 minutes from now...');
            
            const testDate = new Date();
            testDate.setMinutes(testDate.getMinutes() + 2);
            
            const reminderData = {
                action: 'send_study_reminder',
                reminder: {
                    date: testDate.toISOString().split('T')[0],
                    time: testDate.toTimeString().split(' ')[0].substring(0, 5),
                    type: 'custom',
                    description: 'TEST REMINDER - This is an automatic test reminder created at ' + new Date().toLocaleString(),
                    emailNotification: true
                },
                course_id: 1, // Assuming course ID 1 exists
                student_id: 1  // Assuming student ID 1 exists
            };
            
            fetch('../student/api/send_study_reminder_email.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(reminderData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    addLog('✅ Test reminder created successfully! Reminder ID: ' + data.reminder_id, 'success');
                    addLog('📅 Scheduled for: ' + data.scheduled_datetime, 'info');
                    addLog('⏰ The automatic reminder should be sent in about 2 minutes', 'warning');
                } else {
                    addLog('❌ Failed to create test reminder: ' + data.message, 'error');
                }
            })
            .catch(error => {
                addLog('❌ Error creating test reminder: ' + error.message, 'error');
            });
        }
        
        function runCronCheck() {
            addLog('Running manual cron check...');
            
            fetch('../cron/check_reminders.php?debug=1', {
                method: 'GET'
            })
            .then(response => response.text())
            .then(data => {
                addLog('📋 Cron check completed:', 'success');
                const lines = data.split('\n');
                lines.forEach(line => {
                    if (line.trim()) {
                        addLog(line.trim(), 'info');
                    }
                });
            })
            .catch(error => {
                addLog('❌ Error running cron check: ' + error.message, 'error');
            });
        }
        
        function viewReminders() {
            addLog('Loading all reminders...');
            
            fetch('get_reminders.php', {
                method: 'GET'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    addLog('📊 Found ' + data.reminders.length + ' reminders:', 'success');
                    data.reminders.forEach(reminder => {
                        const status = reminder.reminder_sent ? '✅ Sent' : '⏳ Pending';
                        addLog(`ID: ${reminder.reminder_id} | ${reminder.reminder_date} ${reminder.reminder_time} | ${reminder.description} | ${status}`, 'info');
                    });
                } else {
                    addLog('❌ Failed to load reminders: ' + data.message, 'error');
                }
            })
            .catch(error => {
                addLog('❌ Error loading reminders: ' + error.message, 'error');
            });
        }
        
        function clearTestData() {
            if (confirm('Are you sure you want to clear all test reminder data?')) {
                addLog('Clearing test data...');
                
                fetch('clear_test_data.php', {
                    method: 'POST'
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        addLog('✅ Test data cleared successfully!', 'success');
                    } else {
                        addLog('❌ Failed to clear test data: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    addLog('❌ Error clearing test data: ' + error.message, 'error');
                });
            }
        }
        
        // Auto-refresh reminder status every 30 seconds
        setInterval(() => {
            addLog('🔄 Auto-checking for new reminder activity...', 'info');
            viewReminders();
        }, 30000);
    </script>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
