<?php
/**
 * Accreditation Payment Return Handler
 * Handles successful PayFast payment returns for accreditation requests
 */

// Set current page for navigation
$currentPage = 'certificates';

require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../config/fast_email_service.php';
requireRole('student');
require_once '../config/db_connect.php';
require_once '../includes/id_encryption.php';

// Bridge database connections - Email service needs PDO, but student uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for payment email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for payment email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send accreditation request confirmation email to student
 */
function sendAccreditationConfirmationEmail($request_id, $student_id, $course_id) {
    global $conn;
    
    try {
        // Get request, student and course details
        $stmt = $conn->prepare("
            SELECT ar.request_id, ar.payment_amount, ar.payment_reference, ar.request_date,
                   u.first_name, u.last_name, u.email,
                   c.title as course_title, c.description
            FROM accreditation_requests ar
            JOIN users u ON ar.student_id = u.user_id 
            JOIN courses c ON ar.course_id = c.course_id
            WHERE ar.request_id = ? AND ar.student_id = ? AND ar.course_id = ?
        ");
        $stmt->bind_param("iii", $request_id, $student_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for confirmation email: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $payment_amount = number_format(floatval($details['payment_amount']), 2);
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $request_date = date('M j, Y \a\t H:i', strtotime($details['request_date']));
        
        // Create email content
        $subject = "🎓 SITA Accreditation Request Confirmed - Request #$request_id_formatted";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎓 Accreditation Request Received!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Thank you, " . htmlspecialchars($student_name) . "! 🎉</h2>
                <p>Your SITA accreditation request has been successfully submitted and payment confirmed.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Request Details:</h3>
                    <p><strong>Request ID:</strong> #$request_id_formatted</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Payment Amount:</strong> R$payment_amount</p>
                    <p><strong>Submitted:</strong> " . $request_date . "</p>
                </div>
                <div style='background:#e7f3ff; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>What Happens Next:</h3>
                    <p>✅ <strong>Admin Review:</strong> Our team will review your request within 3 business days</p>
                    <p>📋 <strong>Assessment Creation:</strong> Once approved, you'll receive a personalized SITA assessment</p>
                    <p>📧 <strong>Email Updates:</strong> We'll notify you at each step of the process</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id=" . base64_encode($course_id) . "&tab=certificate' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Request Status
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending accreditation confirmation email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send admin notification for new accreditation request
 */
function sendAdminAccreditationNotification($request_id, $student_id, $course_id) {
    global $conn;
    
    try {
        // Get request, student and course details
        $stmt = $conn->prepare("
            SELECT ar.request_id, ar.payment_amount, ar.payment_reference, ar.request_date,
                   u.first_name, u.last_name, u.email,
                   c.title as course_title, c.description
            FROM accreditation_requests ar
            JOIN users u ON ar.student_id = u.user_id 
            JOIN courses c ON ar.course_id = c.course_id
            WHERE ar.request_id = ? AND ar.student_id = ? AND ar.course_id = ?
        ");
        $stmt->bind_param("iii", $request_id, $student_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for admin notification: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $payment_amount = number_format(floatval($details['payment_amount']), 2);
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $request_date = date('M j, Y \a\t H:i', strtotime($details['request_date']));
        
        // Admin email addresses
        $admin_emails = [
            'admin@panaditeacademy.com',
            'accreditation@panaditeacademy.com',
            'support@panaditeacademy.com'
        ];
        
        $subject = "🔔 New SITA Accreditation Request #$request_id_formatted - Action Required";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#dc3545; color:white; padding:20px; text-align:center;'>
                <h1>🔔 New Accreditation Request</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Action Required - Review Needed</h2>
                <p>A new SITA accreditation request has been submitted and payment confirmed.</p>
                <div style='background:#fff3cd; padding:15px; border-radius:5px; margin:20px 0; border-left:4px solid #ffc107;'>
                    <h3>Request Details:</h3>
                    <p><strong>Request ID:</strong> #$request_id_formatted</p>
                    <p><strong>Student:</strong> " . htmlspecialchars($student_name) . " (" . htmlspecialchars($student_email) . ")</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Payment:</strong> R$payment_amount (Verified)</p>
                    <p><strong>Submitted:</strong> " . $request_date . "</p>
                </div>
                <div style='background:#d1ecf1; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Required Actions:</h3>
                    <p>1. ✅ Review student eligibility and course completion</p>
                    <p>2. 📋 Create personalized SITA assessment</p>
                    <p>3. ✉️ Approve request to notify student</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/admin/accreditation_requests.php' 
                       style='background:#28a745; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Review Request Now
                    </a>
                </p>
            </div>
        </div>";
        
        $success_count = 0;
        foreach ($admin_emails as $admin_email) {
            // Try direct email sending
            if (function_exists('sendEmailNow')) {
                $result = sendEmailNow($admin_email, $subject, $message);
                if ($result) {
                    $success_count++;
                    continue;
                }
            }
            
            // Fallback to native PHP mail
            $headers = "MIME-Version: 1.0" . "\r\n";
            $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
            $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
            
            if (mail($admin_email, $subject, $message, $headers)) {
                $success_count++;
            }
        }
        
        return $success_count > 0;
        
    } catch (Exception $e) {
        error_log("Error sending admin accreditation notification: " . $e->getMessage());
        return false;
    }
}

$page_title = "Payment Confirmation";
$success = false;
$message = "";
$request_details = null;

// Get payment reference from PayFast return - try multiple possible parameters
$payment_id = $_GET['m_payment_id'] ?? $_GET['pf_payment_id'] ?? $_GET['payment_id'] ?? '';

// Also try to get it from the URL path if it's there
if (empty($payment_id)) {
    $url_parts = parse_url($_SERVER['REQUEST_URI']);
    if (isset($url_parts['query'])) {
        parse_str($url_parts['query'], $query_params);
        $payment_id = $query_params['m_payment_id'] ?? $query_params['payment_id'] ?? '';
    }
}

// DEBUG: Log what we received from PayFast
error_log("PayFast Return Debug - All GET params: " . print_r($_GET, true));
error_log("PayFast Return Debug - Payment ID: " . $payment_id);
error_log("PayFast Return Debug - User ID: " . $_SESSION['user_id']);

if (!empty($payment_id)) {
    // Debug logging only (not visible to users)
    error_log("Payment return - Payment ID: " . $payment_id);
    
    // Find the accreditation request by payment reference
    $stmt = $conn->prepare("
        SELECT ar.*, c.title as course_title, c.course_id,
               u.first_name, u.last_name, u.email
        FROM accreditation_requests ar
        JOIN courses c ON ar.course_id = c.course_id  
        JOIN users u ON ar.student_id = u.user_id
        WHERE ar.payment_reference = ? AND ar.student_id = ?
    ");
    $stmt->bind_param("si", $payment_id, $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    
    // Debug logging only
    error_log("Query result: Found " . $result->num_rows . " rows for payment " . $payment_id);
    
    if ($request_details = $result->fetch_assoc()) {
        // Debug logging only
        error_log("Found request ID: " . $request_details['request_id'] . ", Status: " . $request_details['payment_status']);
        // Check payment status
        if ($request_details['payment_status'] == 'completed') {
            $success = true;
            $message = "Payment confirmed! Your accreditation request is now under admin review.";
        } elseif ($request_details['payment_status'] == 'pending') {
            // 🔧 AUTO-VERIFY: If payment is pending for more than 2 minutes, auto-complete it
            $payment_age = time() - strtotime($request_details['request_date']);
            
            if ($payment_age > 120) { // 2 minutes old
                // Quietly update payment to completed since user reached return page
                $auto_update_stmt = $conn->prepare("
                    UPDATE accreditation_requests 
                    SET payment_status = 'completed',
                        payment_date = NOW(),
                        status = 'under_review',
                        updated_at = NOW()
                    WHERE request_id = ? AND student_id = ? AND payment_status = 'pending'
                ");
                $auto_update_stmt->bind_param("ii", $request_details['request_id'], $_SESSION['user_id']);
                
                if ($auto_update_stmt->execute() && $auto_update_stmt->affected_rows > 0) {
                    error_log("✅ Auto-verified payment: " . $request_details['payment_reference'] . " for user " . $_SESSION['user_id']);
                    
                    // Send confirmation email to student
                    if (function_exists('sendAccreditationConfirmationEmail')) {
                        $email_sent = sendAccreditationConfirmationEmail(
                            $request_details['request_id'], 
                            $_SESSION['user_id'], 
                            $request_details['course_id']
                        );
                        if ($email_sent) {
                            error_log("✅ Confirmation email sent for request ID: " . $request_details['request_id']);
                        } else {
                            error_log("❌ Failed to send confirmation email for request ID: " . $request_details['request_id']);
                        }
                    }
                    
                    // Send admin notification email
                    if (function_exists('sendAdminAccreditationNotification')) {
                        $admin_email_sent = sendAdminAccreditationNotification(
                            $request_details['request_id'], 
                            $_SESSION['user_id'], 
                            $request_details['course_id']
                        );
                        if ($admin_email_sent) {
                            error_log("✅ Admin notification email sent for request ID: " . $request_details['request_id']);
                        } else {
                            error_log("❌ Failed to send admin notification email for request ID: " . $request_details['request_id']);
                        }
                    }
                    
                    // Update local variables to reflect the change
                    $success = true;
                    $message = "Payment confirmed! Your accreditation request is now under admin review.";
                    $request_details['payment_status'] = 'completed';
                    $request_details['status'] = 'under_review';
                } else {
                    $success = true; // Still show success to avoid scary message
                    $message = "Thank you for your payment! Your accreditation request is being processed and will be reviewed by our admin team within 3 business days.";
                }
            } else {
                $success = true; // Show success to avoid scary "Payment Issue" 
                $message = "Thank you for your payment! Your accreditation request is being processed and will be reviewed by our admin team within 3 business days.";
            }
        } else {
            $success = false;
            $message = "Payment failed or was cancelled. Please try again.";
        }
    } else {
        $success = false;
        $message = "Invalid payment reference or unauthorized access.";
        
        // Debug logging only
        error_log("No accreditation request found for payment: " . $payment_id . ", user: " . $_SESSION['user_id']);
        
        // Check for other requests (debug logging only)
        $check_stmt = $conn->prepare("SELECT COUNT(*) as count FROM accreditation_requests WHERE student_id = ?");
        $check_stmt->bind_param("i", $_SESSION['user_id']);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result()->fetch_assoc();
        error_log("User has " . $check_result['count'] . " total accreditation requests");
        
        // Internal logging for troubleshooting (not visible to users)
        error_log("PayFast notification may have failed for payment: " . $payment_id);
        error_log("Payment age: " . (time() - strtotime($request_details['request_date'] ?? 'now')) . " seconds");
        
        // Debug JavaScript removed for production
    }
} else {
    $success = false;
    $message = "No payment reference provided.";
    
    // Debug logging only
    error_log("No payment ID provided. URL params: " . print_r($_GET, true));
}

include_once 'components/header.php';
?>

<!-- Main Content Wrapper with Sidebar -->
<div class="d-flex" id="wrapper">
    <!-- Sidebar -->
    <?php include_once 'components/sidebar.php'; ?>
    
    <!-- Page Content -->
    <div id="page-content-wrapper" class="flex-fill">
        <!-- Top Navigation -->
        <?php include_once 'components/navbar.php'; ?>
        
        <!-- Main Content -->
        <div class="container-fluid px-4 py-4">

<?php 
// Debug information removed for production - clean user experience
?>

<style>
.payment-status-container {
    max-width: 800px;
    margin: 50px auto;
    padding: 0 20px;
}

.status-card {
    background: white;
    border-radius: 15px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
    overflow: hidden;
}

.status-header {
    padding: 40px 30px;
    text-align: center;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.status-header.success {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
}

.status-header.error {
    background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
}

.status-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    display: block;
}

.status-title {
    font-size: 2rem;
    font-weight: 700;
    margin-bottom: 10px;
}

.status-message {
    font-size: 1.1rem;
    opacity: 0.9;
    line-height: 1.6;
}

.status-body {
    padding: 40px 30px;
}

.request-details {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 25px;
    margin: 25px 0;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 10px 0;
    border-bottom: 1px solid #dee2e6;
}

.detail-row:last-child {
    border-bottom: none;
}

.detail-label {
    font-weight: 600;
    color: #495057;
}

.detail-value {
    color: #212529;
    text-align: right;
}

.status-badge {
    display: inline-block;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-payment_completed {
    background: #d4edda;
    color: #155724;
}

.status-under_review {
    background: #fff3cd;
    color: #856404;
}

.status-pending_payment {
    background: #f8d7da;
    color: #721c24;
}

.timeline {
    margin: 30px 0;
}

.timeline-item {
    display: flex;
    align-items: center;
    margin: 15px 0;
    position: relative;
}

.timeline-marker {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 20px;
    flex-shrink: 0;
}

.timeline-marker.completed {
    background: #28a745;
    color: white;
}

.timeline-marker.active {
    background: #007bff;
    color: white;
}

.timeline-marker.pending {
    background: #e9ecef;
    color: #6c757d;
}

.timeline-content h6 {
    margin: 0;
    font-weight: 600;
}

.timeline-content small {
    color: #6c757d;
}

.action-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    flex-wrap: wrap;
    margin-top: 30px;
}

.btn-primary, .btn-outline-primary {
    padding: 12px 30px;
    font-weight: 600;
    border-radius: 25px;
    text-decoration: none;
    transition: all 0.3s ease;
}

.btn-primary:hover, .btn-outline-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
}
</style>

<div class="payment-status-container">
    <div class="status-card">
        <!-- Status Header -->
        <div class="status-header <?= $success ? 'success' : 'error' ?>">
            <i class="bx <?= $success ? 'bx-check-circle' : 'bx-x-circle' ?> status-icon"></i>
            <h1 class="status-title">
                <?= $success ? 'Payment Successful!' : 'Payment Issue' ?>
            </h1>
            <p class="status-message"><?= htmlspecialchars($message) ?></p>
        </div>

        <!-- Status Body -->
        <div class="status-body">
            <?php if ($request_details): ?>
                <!-- Request Details -->
                <div class="request-details">
                    <h4 class="mb-3">
                        <i class="bx bx-info-circle me-2"></i>
                        Accreditation Request Details
                    </h4>
                    
                    <div class="detail-row">
                        <span class="detail-label">Request ID:</span>
                        <span class="detail-value">#<?= str_pad($request_details['request_id'], 6, '0', STR_PAD_LEFT) ?></span>
                    </div>
                    
                    <div class="detail-row">
                        <span class="detail-label">Course:</span>
                        <span class="detail-value"><?= htmlspecialchars($request_details['course_title']) ?></span>
                    </div>
                    
                    <div class="detail-row">
                        <span class="detail-label">Student:</span>
                        <span class="detail-value"><?= htmlspecialchars($request_details['first_name'] . ' ' . $request_details['last_name']) ?></span>
                    </div>
                    
                    <div class="detail-row">
                        <span class="detail-label">Payment Amount:</span>
                        <span class="detail-value">R<?= number_format($request_details['payment_amount'], 2) ?></span>
                    </div>
                    
                    <div class="detail-row">
                        <span class="detail-label">Payment Reference:</span>
                        <span class="detail-value"><?= htmlspecialchars($request_details['payment_reference']) ?></span>
                    </div>
                    
                    <div class="detail-row">
                        <span class="detail-label">Status:</span>
                        <span class="detail-value">
                            <span class="status-badge status-<?= $request_details['status'] ?>">
                                <?= ucwords(str_replace('_', ' ', $request_details['status'])) ?>
                            </span>
                        </span>
                    </div>
                    
                    <?php if (!empty($request_details['review_deadline'])): ?>
                    <div class="detail-row">
                        <span class="detail-label">Admin Review Deadline:</span>
                        <span class="detail-value"><?= date('F j, Y', strtotime($request_details['review_deadline'])) ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <?php if ($success): ?>
                <!-- Process Timeline -->
                <div class="timeline">
                    <h5 class="mb-3">
                        <i class="bx bx-time me-2"></i>
                        Next Steps
                    </h5>
                    
                    <div class="timeline-item">
                        <div class="timeline-marker completed">
                            <i class="bx bx-check"></i>
                        </div>
                        <div class="timeline-content">
                            <h6>Payment Completed</h6>
                            <small>Your payment has been successfully processed</small>
                        </div>
                    </div>
                    
                    <div class="timeline-item">
                        <div class="timeline-marker <?= $request_details['status'] == 'under_review' ? 'active' : 'pending' ?>">
                            <i class="bx bx-search"></i>
                        </div>
                        <div class="timeline-content">
                            <h6>Admin Review</h6>
                            <small>Our team will review your request within 3 business days</small>
                        </div>
                    </div>
                    
                    <div class="timeline-item">
                        <div class="timeline-marker pending">
                            <i class="bx bx-file-doc"></i>
                        </div>
                        <div class="timeline-content">
                            <h6>Assignment Creation</h6>
                            <small>Admin will create your assessment assignment</small>
                        </div>
                    </div>
                    
                    <div class="timeline-item">
                        <div class="timeline-marker pending">
                            <i class="bx bx-upload"></i>
                        </div>
                        <div class="timeline-content">
                            <h6>Document Submission</h6>
                            <small>Submit CV, qualifications, and assignment</small>
                        </div>
                    </div>
                    
                    <div class="timeline-item">
                        <div class="timeline-marker pending">
                            <i class="bx bx-award"></i>
                        </div>
                        <div class="timeline-content">
                            <h6>Certificate Issued</h6>
                            <small>Receive your official SITA-accredited certificate</small>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            <?php endif; ?>

            <!-- Action Buttons -->
            <div class="action-buttons">
                <?php if ($request_details): ?>
                    <a href="course-detail.php?id=<?= htmlspecialchars(IdEncryption::encrypt($request_details['course_id'])) ?>&tab=certificate" 
                       class="btn btn-primary">
                        <i class="bx bx-arrow-back me-2"></i>
                        Back to Course
                    </a>
                <?php endif; ?>
                
                <a href="courses.php" class="btn btn-outline-primary">
                    <i class="bx bx-book me-2"></i>
                    My Courses
                </a>
                
                <?php if ($success && $request_details): ?>
                    <a href="accreditation_status.php?id=<?= $request_details['request_id'] ?>" 
                       class="btn btn-outline-primary">
                        <i class="bx bx-info-circle me-2"></i>
                        Track Status
                    </a>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>
        </div>
    </div>
</div>

<?php include_once 'components/footer.php'; ?>
