<?php
session_start();
require_once '../../includes/config.php';
require_once '../../includes/functions.php';
require_once '../../includes/course_progress_service_quiz_only.php';

// Ensure user is logged in as student
requireRole('student');

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get POST data
$course_id = intval($_POST['course_id'] ?? 0);
$student_id = $_SESSION['user_id'];

// Validate required fields
if (!$course_id) {
    echo json_encode(['error' => 'Missing course ID']);
    exit;
}

// Verify student is enrolled in the course
$enrollment_check = $conn->prepare("
    SELECT enrollment_id 
    FROM enrollments 
    WHERE student_id = ? AND course_id = ? AND status = 'active'
");
$enrollment_check->bind_param("ii", $student_id, $course_id);
$enrollment_check->execute();

if ($enrollment_check->get_result()->num_rows === 0) {
    echo json_encode(['error' => 'Not enrolled in this course']);
    exit;
}

try {
    // Initialize progress service with database connection
    $progressService = new QuizOnlyProgressService($conn);
    
    // Calculate video progress
    $video_progress_query = $conn->prepare("
        SELECT 
            COUNT(l.lesson_id) as total_lessons,
            COUNT(CASE WHEN vp.completed = 1 THEN 1 END) as completed_lessons,
            AVG(COALESCE(vp.progress_percentage, 0)) as avg_progress
        FROM lessons l
        LEFT JOIN video_progress vp ON l.lesson_id = vp.lesson_id AND vp.student_id = ?
        WHERE l.course_id = ?
    ");
    $video_progress_query->bind_param("ii", $student_id, $course_id);
    $video_progress_query->execute();
    $video_data = $video_progress_query->get_result()->fetch_assoc();
    
    $video_completion_percentage = round($video_data['avg_progress'], 1);
    $lessons_completed = intval($video_data['completed_lessons']);
    $total_lessons = intval($video_data['total_lessons']);
    $all_videos_completed = $video_completion_percentage >= 100;
    
    // Get quiz eligibility
    $quiz_eligibility = $progressService->isCertificateEligible($student_id, $course_id);
    $quiz_passed = $quiz_eligibility['eligible'];
    $quiz_percentage = $quiz_eligibility['current_percentage'];
    $required_grade = $quiz_eligibility['minimum_required'];
    
    // Overall eligibility requires both video completion AND quiz passing
    $overall_eligible = $all_videos_completed && $quiz_passed;
    
    echo json_encode([
        'success' => true,
        'eligible' => $overall_eligible,
        'requirements' => [
            'videos' => [
                'completed' => $all_videos_completed,
                'completion_percentage' => $video_completion_percentage,
                'lessons_completed' => $lessons_completed,
                'total_lessons' => $total_lessons
            ],
            'quizzes' => [
                'passed' => $quiz_passed,
                'current_grade' => $quiz_percentage,
                'required_grade' => $required_grade
            ]
        ],
        'message' => $overall_eligible 
            ? 'Congratulations! You are eligible for a certificate.' 
            : 'Complete all requirements to become eligible for a certificate.',
        'next_steps' => $overall_eligible 
            ? ['Visit the Certificate tab to request your certificate'] 
            : array_filter([
                !$all_videos_completed ? 'Complete all video lessons (100% watched)' : null,
                !$quiz_passed ? "Achieve a quiz average of {$required_grade}% or higher" : null
            ])
    ]);

} catch (Exception $e) {
    error_log("Certificate eligibility check error: " . $e->getMessage());
    echo json_encode([
        'error' => 'Failed to check certificate eligibility',
        'details' => $e->getMessage()
    ]);
}
?>
