<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');
require_once '../config/db_connect.php';
require_once '../includes/course_progress_service_quiz_only.php';

$page_title = 'Course Detail';
$user_id = $_SESSION['user_id'];

// Initialize progress service for comprehensive course completion checking
$progressService = new QuizOnlyProgressService($conn);

// Decrypt the course ID from URL
$course_id = getSecureId('id');

// Redirect if invalid or missing encrypted ID
if (!$course_id) {
    header('Location: courses.php?error=invalid_course');
    exit();
}

// Get course information
$course = null;
if (isset($conn) && $course_id > 0) {
    $stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        $stmt->close();
    }
}

if (!$course) {
    header("Location: courses.php");
    exit;
}

// Check enrollment
$is_enrolled = false;
$enrollment_data = null;
if (isset($conn)) {
    $stmt = $conn->prepare("SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?");
    if ($stmt) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $enrollment_data = $result->fetch_assoc();
        $is_enrolled = (bool)$enrollment_data;
        $stmt->close();
    }
}

// Get course lessons/videos from database
$lessons = [];

// First try to get real videos from the videos table
if (isset($conn)) {
    $video_query = "
        SELECT v.*, 
               COALESCE(vp.completion_percentage, 0) as watch_progress,
               CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END as completed,
               CASE WHEN vp.last_position > 0 THEN vp.last_position ELSE 0 END as last_position
        FROM videos v
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
        WHERE v.course_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC, v.created_at ASC
    ";
    
    if ($stmt = $conn->prepare($video_query)) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            // Fix video URL path for web access
            if (!empty($row['video_url'])) {
                // If it's already a full URL (starts with http), keep it
                if (strpos($row['video_url'], 'http') === 0) {
                    $row['video_url'] = $row['video_url'];
                } else {
                    // Convert relative path to full URL
                    $row['video_url'] = '/' . ltrim($row['video_url'], './');
                }
            }
            
            // Fix thumbnail path for web access
            if (!empty($row['thumbnail'])) {
                // If it's already a full URL (starts with http), keep it
                if (strpos($row['thumbnail'], 'http') === 0) {
                    $row['thumbnail'] = $row['thumbnail'];
                } else {
                    // Convert relative path to full URL
                    $row['thumbnail'] = '/' . ltrim($row['thumbnail'], './');
                }
            } else {
                // Set default placeholder thumbnail if none exists
                $row['thumbnail'] = '/assets/images/default-video-thumbnail.jpg';
            }
            
            // Check for related content
            $row['has_assignment'] = false;
            $row['has_quiz'] = false;
            $row['has_materials'] = false;
            
            // Check assignments for this course
            $assignment_check = $conn->prepare("SELECT COUNT(*) as count FROM assignments WHERE course_id = ?");
            if ($assignment_check) {
                $assignment_check->bind_param("i", $course_id);
                $assignment_check->execute();
                $assignment_result = $assignment_check->get_result();
                $assignment_data = $assignment_result->fetch_assoc();
                $row['has_assignment'] = $assignment_data['count'] > 0;
                $assignment_check->close();
            }
            
            // Check quizzes for this course
            $quiz_check = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE course_id = ?");
            if ($quiz_check) {
                $quiz_check->bind_param("i", $course_id);
                $quiz_check->execute();
                $quiz_result = $quiz_check->get_result();
                $quiz_data = $quiz_result->fetch_assoc();
                $row['has_quiz'] = $quiz_data['count'] > 0;
                $quiz_check->close();
            }
            
            // Check materials for this course
            $material_check = $conn->prepare("SELECT COUNT(*) as count FROM course_materials WHERE course_id = ?");
            if ($material_check) {
                $material_check->bind_param("i", $course_id);
                $material_check->execute();
                $material_result = $material_check->get_result();
                $material_data = $material_result->fetch_assoc();
                $row['has_materials'] = $material_data['count'] > 0;
                $material_check->close();
            }
            
            $row['lesson_id'] = $row['video_id']; // For compatibility
            $lessons[] = $row;
        }
        $stmt->close();
    }
}

// If no videos found, create structured lessons based on available course content
if (empty($lessons)) {
    // Get real assignments for this course
    $assignments = [];
    if (isset($conn)) {
        $assignment_query = "SELECT title, assignment_id, due_date, points FROM assignments WHERE course_id = ? ORDER BY created_at ASC";
        if ($stmt = $conn->prepare($assignment_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $assignments[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Get real materials for this course
    $materials = [];
    if (isset($conn)) {
        $materials_query = "SELECT title, material_id, category FROM course_materials WHERE course_id = ? ORDER BY date_added ASC";
        if ($stmt = $conn->prepare($materials_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $materials[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Get real quizzes for this course
    $quizzes = [];
    if (isset($conn)) {
        $quiz_query = "SELECT title, quiz_id FROM quizzes WHERE course_id = ? ORDER BY created_at ASC";
        if ($stmt = $conn->prepare($quiz_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $quizzes[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Create lessons based on actual course assignments and materials
    $course_title = $course['title'] ?? 'Course';
    
    // Create lessons based on real assignments first
    $lesson_count = 0;
    foreach ($assignments as $assignment) {
        $lesson_count++;
        $lessons[] = [
            'lesson_id' => $lesson_count,
            'video_id' => $lesson_count,
            'title' => $assignment['title'] . " - Tutorial",
            'duration' => rand(10, 30) . ":" . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
            'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
            'video_path' => '',
            'thumbnail' => '',
            'description' => "Learn about " . $assignment['title'],
            'completed' => $lesson_count <= 2 ? true : false,
            'has_assignment' => true,
            'has_quiz' => count($quizzes) > ($lesson_count - 1),
            'has_materials' => count($materials) > ($lesson_count - 1),
            'watch_progress' => $lesson_count <= 2 ? 100 : ($lesson_count == 3 ? 65 : 0),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'video_order' => $lesson_count,
            'is_published' => 1,
            'last_position' => 0
        ];
    }
    
    // If no assignments, create lessons based on materials
    if (empty($assignments) && !empty($materials)) {
        foreach ($materials as $index => $material) {
            $lesson_count++;
            $lessons[] = [
                'lesson_id' => $lesson_count,
                'video_id' => $lesson_count,
                'title' => $material['title'],
                'duration' => rand(15, 35) . ":" . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
                'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
                'video_path' => '',
                'thumbnail' => '',
                'description' => $material['description'] ?? "Learn about " . $material['title'],
                'completed' => $index < 2 ? true : false,
                'has_assignment' => false,
                'has_quiz' => count($quizzes) > $index,
                'has_materials' => true,
                'watch_progress' => $index < 2 ? 100 : ($index == 2 ? 65 : 0),
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
                'video_order' => $lesson_count,
                'is_published' => 1,
                'last_position' => 0
            ];
        }
    }
    
    // No demo data - keep lessons empty if no real content exists
}

// Fix lesson ID detection - use first available lesson if none specified
$current_lesson_id = isset($_GET['lesson']) ? (int)$_GET['lesson'] : null;
$current_lesson = null;

// Only process lessons if they exist
if (!empty($lessons)) {
    if ($current_lesson_id) {
        // Look for specific lesson
        foreach ($lessons as $lesson) {
            if ($lesson['lesson_id'] == $current_lesson_id) {
                $current_lesson = $lesson;
                break;
            }
        }
    }
    
    // If no specific lesson found, use first lesson
    if (!$current_lesson) {
        $current_lesson = $lessons[0];
        $current_lesson_id = $current_lesson['lesson_id']; // Set to actual lesson ID
    }
} else {
    // No lessons available, set to null to avoid JavaScript errors
    $current_lesson_id = null;
}

// Debug: Log the lesson ID being used
error_log("COURSE DETAIL DEBUG: current_lesson_id = " . ($current_lesson_id ?? 'NULL'));
error_log("COURSE DETAIL DEBUG: Available lessons: " . json_encode(array_column($lessons, 'lesson_id')));

include_once 'components/header.php';

// ===== COMPREHENSIVE REVIEW SYSTEM QUERIES =====

// Get review statistics for this course
$review_stats = [
    'average_rating' => 0,
    'total_reviews' => 0,
    'rating_breakdown' => [5 => 0, 4 => 0, 3 => 0, 2 => 0, 1 => 0]
];

if (isset($conn)) {
    $stats_query = "SELECT 
        COUNT(*) as total_reviews,
        ROUND(AVG(rating), 1) as average_rating,
        SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
        SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
        SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
        SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
        SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star
    FROM course_reviews 
    WHERE course_id = ? AND is_approved = TRUE";
    
    if ($stmt = $conn->prepare($stats_query)) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($stats = $result->fetch_assoc()) {
            $review_stats['total_reviews'] = $stats['total_reviews'];
            $review_stats['average_rating'] = $stats['average_rating'] ?: 0;
            $review_stats['rating_breakdown'] = [
                5 => $stats['five_star'],
                4 => $stats['four_star'],
                3 => $stats['three_star'],
                2 => $stats['two_star'],
                1 => $stats['one_star']
            ];
        }
        $stmt->close();
    }
}

// Get all reviews for this course with user details
$reviews = [];
if (isset($conn)) {
    $reviews_query = "SELECT 
        cr.review_id,
        cr.rating,
        cr.review_title,
        cr.review_text,
        cr.helpful_count,
        cr.created_at,
        u.first_name,
        u.last_name,
        u.profile_picture,
        (SELECT COUNT(*) FROM review_likes WHERE review_id = cr.review_id AND is_helpful = TRUE) as helpful_votes,
        (SELECT COUNT(*) FROM review_replies WHERE review_id = cr.review_id) as reply_count
    FROM course_reviews cr
    JOIN users u ON cr.student_id = u.user_id
    WHERE cr.course_id = ? AND cr.is_approved = TRUE
    ORDER BY cr.created_at DESC
    LIMIT 20";
    
    if ($stmt = $conn->prepare($reviews_query)) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $reviews[] = $row;
        }
        $stmt->close();
    }
}

// Check if current user has already reviewed this course
$user_has_reviewed = false;
$user_review = null;
if (isset($_SESSION['user_id']) && isset($conn)) {
    $user_review_query = "SELECT * FROM course_reviews WHERE course_id = ? AND student_id = ?";
    if ($stmt = $conn->prepare($user_review_query)) {
        $stmt->bind_param("ii", $course_id, $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($user_review = $result->fetch_assoc()) {
            $user_has_reviewed = true;
        }
        $stmt->close();
    }
}

// Check if user is enrolled (for review eligibility)
$user_enrolled = false;
if (isset($_SESSION['user_id']) && isset($conn)) {
    $enrollment_query = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
    if ($stmt = $conn->prepare($enrollment_query)) {
        $stmt->bind_param("ii", $_SESSION['user_id'], $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user_enrolled = $result->num_rows > 0;
        $stmt->close();
    }
}

// For backward compatibility with existing code
$average_rating = $review_stats['average_rating'];
$rating_count = $review_stats['total_reviews'];

?>

<style>
:root {
    --panadite-primary: #2563eb;
    --panadite-secondary: #1d4ed8;
    --panadite-light: #dbeafe;
    --panadite-dark: #1e40af;
    --text-primary: #1f2937;
    --text-secondary: #6b7280;
    --border-color: #e5e7eb;
}

/* Bootstrap Accordion Sidebar Styles */
.course-content-sidebar {
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    position: sticky;
    top: 20px;
    max-height: calc(100vh - 40px);
    overflow-y: auto;
}

.course-content-sidebar .accordion {
    border: none;
}

.course-content-sidebar .accordion-item {
    border: none;
    border-bottom: 1px solid #e9ecef;
}

.course-content-sidebar .accordion-item:last-child {
    border-bottom: none;
}

.course-content-sidebar .accordion-button {
    background: #f8f9fa;
    border: none;
    color: var(--text-primary);
    font-weight: 600;
    padding: 1rem 1.25rem;
    font-size: 0.95rem;
}

.course-content-sidebar .accordion-button:not(.collapsed) {
    background: var(--panadite-primary);
    color: white;
    box-shadow: none;
}

.course-content-sidebar .accordion-button:focus {
    box-shadow: none;
    border: none;
}

.course-content-sidebar .accordion-button::after {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16' fill='%23212529'%3e%3cpath fill-rule='evenodd' d='M1.646 4.646a.5.5 0 0 1 .708 0L8 10.293l5.646-5.647a.5.5 0 0 1 .708.708l-6 6a.5.5 0 0 1-.708 0l-6-6a.5.5 0 0 1 0-.708z'/%3e%3c/svg%3e");
}

.course-content-sidebar .accordion-button:not(.collapsed)::after {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16' fill='%23ffffff'%3e%3cpath fill-rule='evenodd' d='M1.646 4.646a.5.5 0 0 1 .708 0L8 10.293l5.646-5.647a.5.5 0 0 1 .708.708l-6 6a.5.5 0 0 1-.708 0l-6-6a.5.5 0 0 1 0-.708z'/%3e%3c/svg%3e");
}

/* Lesson Items in Accordion */
.lesson-item-accordion {
    cursor: pointer;
    transition: all 0.2s ease;
}

.lesson-item-accordion:hover {
    background: #f8f9fa;
}

.lesson-item-accordion.active {
    background: var(--panadite-light);
    border-left: 4px solid var(--panadite-primary);
}

.lesson-item-accordion.completed {
    opacity: 0.8;
}

.lesson-thumbnail-small {
    width: 50px;
    height: 35px;
    position: relative;
    border-radius: 6px;
    overflow: hidden;
}

.lesson-thumbnail-small img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.thumbnail-placeholder-small {
    width: 100%;
    height: 100%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
}

.completion-check {
    position: absolute;
    top: -5px;
    right: -5px;
    background: white;
    border-radius: 50%;
    width: 20px;
    height: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.8rem;
}

.lesson-title {
    font-weight: 600;
    color: var(--text-primary);
    font-size: 0.9rem;
    line-height: 1.3;
    margin-bottom: 0.25rem;
}

.lesson-meta {
    margin-bottom: 0.5rem;
}

/* Content Badges for Video-Specific Materials */
.lesson-content-badges {
    display: flex;
    flex-wrap: wrap;
    gap: 0.375rem;
    margin-top: 0.5rem;
}

.content-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.125rem 0.5rem;
    font-size: 0.7rem;
    font-weight: 500;
    border-radius: 12px;
    border: 1px solid;
    transition: all 0.2s ease;
    cursor: pointer;
    text-decoration: none;
}

.content-badge i {
    font-size: 0.75rem;
}

.content-badge span {
    font-weight: 600;
    letter-spacing: 0.025em;
}

/* Quiz Badge - No Background, Square Style */
.badge-quiz {
    background: transparent;
    color: #92400e;
    border: 1px solid #92400e;
    border-radius: 2px;
    box-shadow: none;
}

.badge-quiz:hover {
    background: transparent;
    color: #78350f;
    border-color: #78350f;
    transform: translateY(-1px);
    box-shadow: none;
}

/* Assignment Badge - Blue Colors */
.badge-assignment {
    background: #dbeafe;
    color: #1e40af;
    border-color: #3b82f6;
}

.badge-assignment:hover {
    background: #93c5fd;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(59, 130, 246, 0.3);
}

/* Materials Badge - Green Colors */
.badge-materials {
    background: #d1fae5;
    color: #065f46;
    border-color: #10b981;
}

.badge-materials:hover {
    background: #6ee7b7;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(16, 185, 129, 0.3);
}

/* Notes Badge - Purple Colors */
.badge-notes {
    background: #e0e7ff;
    color: #4338ca;
    border-color: #6366f1;
}

.badge-notes:hover {
    background: #a5b4fc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(99, 102, 241, 0.3);
}

/* Assignment, Quiz, and Material Items */
.assignment-item, .quiz-item, .material-item {
    cursor: pointer;
    transition: all 0.2s ease;
}

.assignment-item:hover, .quiz-item:hover, .material-item:hover {
    background: #f8f9fa;
}

.assignment-title, .quiz-title, .material-title {
    font-weight: 600;
    color: var(--text-primary);
    font-size: 0.9rem;
    margin-bottom: 0.25rem;
}

/* Course Content Sidebar Height Fix */
.course-content-sidebar {
    height: 100%;
    min-height: 600px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
    overflow-y: auto;
}

.course-content-sidebar .accordion {
    border: none;
    height: 100%;
}

.course-content-sidebar .accordion-item {
    border: none;
    border-bottom: 1px solid #e5e7eb;
}

.course-content-sidebar .accordion-item:last-child {
    border-bottom: none;
}

/* Make accordion content more compact */
.course-content-sidebar .accordion-body {
    padding: 0.5rem;
}

/* Compact quiz info card */
.quiz-info-card {
    margin-bottom: 0.5rem !important;
    padding: 0.5rem !important;
}

.quiz-info-card .quiz-details {
    margin-bottom: 0.5rem !important;
}

.quiz-info-card .quiz-results {
    margin-bottom: 0.5rem !important;
    padding: 0.5rem !important;
}

/* Ensure both columns have equal height */
.row > .col-md-3,
.row > .col-md-9 {
    height: auto;
    min-height: 600px;
}

/* Custom scrollbar for sidebar */
.course-content-sidebar::-webkit-scrollbar {
    width: 6px;
}

.course-content-sidebar::-webkit-scrollbar-track {
    background: #f1f1f1;
}

.course-content-sidebar::-webkit-scrollbar-thumb {
    background: #c1c1c1;
    border-radius: 3px;
}

.course-content-sidebar::-webkit-scrollbar-thumb:hover {
    background: #a8a8a8;
}

/* Enhanced Course Description Tab Styles */
.enhanced-course-description {
    padding: 2rem;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
}

/* Course Header Section */
.course-header-section {
    margin-bottom: 2.5rem;
    padding-bottom: 2rem;
    border-bottom: 1px solid #e5e7eb;
}

.course-title-section {
    margin-bottom: 2rem;
}

.course-main-title {
    font-size: 2.25rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.75rem;
    line-height: 1.2;
}

.course-subtitle {
    font-size: 1.125rem;
    color: #6b7280;
    line-height: 1.6;
    margin-bottom: 0;
}

/* Course Meta Grid */
.course-meta-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
    gap: 1.5rem;
}

.meta-card {
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 1.5rem;
    text-align: center;
    transition: all 0.2s ease;
}

.meta-card:hover {
    background: #f1f5f9;
    border-color: #2563eb;
    transform: translateY(-2px);
}

.meta-icon {
    background: #2563eb;
    color: white;
    width: 48px;
    height: 48px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 1rem;
    font-size: 1.25rem;
}

.meta-value {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.meta-label {
    font-size: 0.875rem;
    color: #6b7280;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Instructor Card */
.instructor-card {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 16px;
    padding: 1.5rem;
    border: 1px solid #e2e8f0;
}

.instructor-header {
    margin-bottom: 1.5rem;
}

.instructor-header h5 {
    color: #1f2937;
    font-weight: 600;
    margin-bottom: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.instructor-content {
    display: flex;
    gap: 1rem;
    align-items: flex-start;
}

.instructor-avatar {
    width: 64px;
    height: 64px;
    border-radius: 50%;
    overflow: hidden;
    flex-shrink: 0;
}

.instructor-avatar img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.avatar-placeholder {
    width: 100%;
    height: 100%;
    background: #2563eb;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
}

.instructor-details {
    flex: 1;
}

.instructor-name {
    font-size: 1.125rem;
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.instructor-title {
    color: #6b7280;
    font-size: 0.875rem;
    margin-bottom: 1rem;
}

.instructor-stats {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.instructor-stats .stat-item {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    font-size: 0.8rem;
    color: #6b7280;
}

.instructor-stats .stat-item i {
    color: #2563eb;
}

/* Learning Objectives Section */
.learning-objectives-section {
    margin-bottom: 2.5rem;
    padding: 2rem;
    background: #f0fdf4;
    border-radius: 12px;
    border: 1px solid #bbf7d0;
}

.learning-objectives-section h4 {
    color: #15803d;
    font-weight: 600;
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.objectives-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1rem;
}

.objective-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: white;
    border-radius: 8px;
    border: 1px solid #dcfce7;
    transition: transform 0.2s ease;
}

.objective-item:hover {
    transform: translateX(4px);
}

.objective-item i {
    color: #16a34a;
    font-size: 1.125rem;
    flex-shrink: 0;
}

.objective-item span {
    color: #374151;
    font-size: 0.9rem;
    line-height: 1.4;
}

/* Course Requirements Section */
.course-requirements-section {
    margin-bottom: 2.5rem;
}

.requirements-card,
.course-info-card {
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    padding: 1.5rem;
    height: 100%;
}

.requirements-card h5,
.course-info-card h5 {
    color: #1f2937;
    font-weight: 600;
    margin-bottom: 1.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.requirements-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.requirements-list li {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.5rem 0;
    color: #374151;
    font-size: 0.9rem;
}

.requirements-list li i {
    color: #16a34a;
    font-size: 1rem;
}

.info-list {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #f3f4f6;
}

.info-item:last-child {
    border-bottom: none;
}

.info-label {
    font-weight: 500;
    color: #6b7280;
    font-size: 0.875rem;
}

.info-value {
    font-weight: 600;
    color: #1f2937;
    font-size: 0.875rem;
}

/* Current Lesson Section */
.current-lesson-section {
    margin-bottom: 2rem;
}

.current-lesson-section h4 {
    color: #1f2937;
    font-weight: 600;
    margin-bottom: 1.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.lesson-info-card {
    background: #fef3c7;
    border: 1px solid #fbbf24;
    border-radius: 12px;
    padding: 1.5rem;
}

.lesson-content .lesson-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #92400e;
    margin-bottom: 0.75rem;
}

.lesson-content .lesson-description {
    color: #78350f;
    line-height: 1.6;
    margin-bottom: 1rem;
}

.lesson-meta-info {
    display: flex;
    gap: 1.5rem;
    flex-wrap: wrap;
}

.lesson-meta-info .meta-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #78350f;
    font-size: 0.875rem;
    font-weight: 500;
}

.lesson-meta-info .meta-item i {
    color: #d97706;
}

/* Enhanced Course Description Responsive Design */
@media (max-width: 768px) {
    .enhanced-course-description {
        padding: 1.25rem;
    }
    
    .course-main-title {
        font-size: 1.75rem;
    }
    
    .course-meta-grid {
        grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
        gap: 1rem;
    }
    
    .instructor-content {
        flex-direction: column;
        text-align: center;
        align-items: center;
    }
    
    .objectives-grid {
        grid-template-columns: 1fr;
    }
    
    .instructor-stats {
        justify-content: center;
    }
    
    .info-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.25rem;
    }
    
    .course-header-section {
        margin-bottom: 2rem;
        padding-bottom: 1.5rem;
    }
    
    .learning-objectives-section {
        padding: 1.5rem;
        margin-bottom: 2rem;
    }
    
    .meta-card {
        padding: 1rem;
    }
    
    .instructor-card {
        padding: 1.25rem;
    }
}

/* Enhanced Reviews Tab Styles */
.enhanced-reviews-content {
    padding: 2rem;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
}

/* Reviews Summary Section */
.reviews-summary-section {
    margin-bottom: 2.5rem;
    padding-bottom: 2rem;
    border-bottom: 1px solid #e5e7eb;
}

.overall-rating-card {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border: 1px solid #e2e8f0;
    border-radius: 16px;
    padding: 2rem;
    text-align: center;
    height: 100%;
}

.rating-display .rating-number {
    font-size: 3rem;
    font-weight: 700;
    color: #1f2937;
    display: block;
    margin-bottom: 0.5rem;
}

.rating-stars-large {
    display: flex;
    justify-content: center;
    gap: 0.25rem;
    margin-bottom: 1rem;
}

.rating-stars-large i {
    font-size: 1.5rem;
    color: #fbbf24;
}

.rating-text {
    color: #6b7280;
    font-weight: 500;
    font-size: 1rem;
}

.rating-breakdown-card {
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    padding: 1.5rem;
    height: 100%;
}

.breakdown-title {
    color: #1f2937;
    font-weight: 600;
    margin-bottom: 1.5rem;
}

.rating-breakdown-row {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 0.75rem;
}

.star-label {
    min-width: 60px;
    font-size: 0.875rem;
    color: #6b7280;
}

.rating-bar {
    flex: 1;
    height: 8px;
    background: #f3f4f6;
    border-radius: 4px;
    overflow: hidden;
}

.rating-bar-fill {
    height: 100%;
    background: #fbbf24;
    transition: width 0.3s ease;
}

.rating-count {
    min-width: 30px;
    text-align: right;
    font-size: 0.875rem;
    color: #6b7280;
    font-weight: 500;
}

/* Write Review Section */
.write-review-section {
    margin-bottom: 2.5rem;
}

.write-review-card {
    background: #f0fdf4;
    border: 1px solid #bbf7d0;
    border-radius: 12px;
    padding: 1.5rem;
}

.write-review-header {
    margin-bottom: 1.5rem;
}

.write-review-header h5 {
    color: #15803d;
    font-weight: 600;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.write-review-header p {
    color: #16a34a;
    margin-bottom: 0;
    font-size: 0.9rem;
}

.review-form .form-group {
    margin-bottom: 1.5rem;
}

.review-form .form-label {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.5rem;
}

.star-rating-input {
    display: flex;
    gap: 0.25rem;
    margin-bottom: 0.5rem;
}

.rating-star {
    font-size: 1.75rem;
    color: #d1d5db;
    cursor: pointer;
    transition: color 0.2s ease;
}

.rating-star:hover,
.rating-star.active {
    color: #fbbf24;
}

.form-actions {
    display: flex;
    gap: 1rem;
    margin-top: 1.5rem;
}

/* User Review Section */
.user-review-section {
    margin-bottom: 2.5rem;
}

.user-review-card {
    background: #fffbeb;
    border: 1px solid #fbbf24;
    border-radius: 12px;
    padding: 1.5rem;
}

.user-review-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.user-review-header h6 {
    color: #92400e;
    font-weight: 600;
    margin-bottom: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.user-review-content .review-rating {
    display: flex;
    gap: 0.25rem;
    margin-bottom: 0.75rem;
}

.user-review-content .review-rating i {
    color: #fbbf24;
    font-size: 1rem;
}

.user-review-content .review-title {
    color: #92400e;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.user-review-content .review-text {
    color: #78350f;
    line-height: 1.6;
    margin-bottom: 0.75rem;
}

.user-review-content .review-date {
    color: #a16207;
    font-size: 0.8rem;
}

/* Prompt Sections */
.login-prompt-section,
.enrollment-prompt-section {
    margin-bottom: 2.5rem;
}

.login-prompt-card,
.enrollment-prompt-card {
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 2rem;
    text-align: center;
}

.login-prompt-card h6,
.enrollment-prompt-card h6 {
    color: #1f2937;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.login-prompt-card p,
.enrollment-prompt-card p {
    color: #6b7280;
    margin-bottom: 1.5rem;
}

/* Reviews List Section */
.reviews-list-section {
    margin-bottom: 2rem;
}

.reviews-list-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
}

.reviews-list-header h5 {
    color: #1f2937;
    font-weight: 600;
    margin-bottom: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.reviews-sort .form-select {
    min-width: 150px;
}

.no-reviews-placeholder {
    text-align: center;
    padding: 3rem 2rem;
    background: #f9fafb;
    border-radius: 12px;
    border: 2px dashed #d1d5db;
}

.no-reviews-icon i {
    font-size: 3rem;
    color: #9ca3af;
    margin-bottom: 1rem;
}

.no-reviews-placeholder h6 {
    color: #4b5563;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.no-reviews-placeholder p {
    color: #6b7280;
    margin-bottom: 1.5rem;
}

/* Review Items */
.review-item-enhanced {
    display: flex;
    gap: 1rem;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    transition: all 0.2s ease;
}

.review-item-enhanced:hover {
    border-color: #2563eb;
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.1);
}

.review-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    overflow: hidden;
    flex-shrink: 0;
}

.review-avatar img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.review-avatar .avatar-placeholder {
    width: 100%;
    height: 100%;
    background: #2563eb;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
}

.review-content {
    flex: 1;
}

.review-header {
    margin-bottom: 1rem;
}

.reviewer-name {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.review-meta {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.review-rating {
    display: flex;
    gap: 0.125rem;
}

.review-rating i {
    color: #fbbf24;
    font-size: 0.875rem;
}

.review-date {
    color: #6b7280;
    font-size: 0.8rem;
}

.review-body .review-title {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.5rem;
    font-size: 1rem;
}

.review-body .review-text {
    color: #374151;
    line-height: 1.6;
    margin-bottom: 1rem;
}

.review-actions {
    display: flex;
    gap: 1rem;
}

.review-action-btn {
    background: none;
    border: 1px solid #e5e7eb;
    border-radius: 6px;
    padding: 0.375rem 0.75rem;
    font-size: 0.8rem;
    color: #6b7280;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.review-action-btn:hover {
    background: #f3f4f6;
    border-color: #2563eb;
    color: #2563eb;
}

.load-more-section {
    text-align: center;
    margin-top: 2rem;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .enhanced-reviews-content {
        padding: 1.25rem;
    }
    
    .reviews-list-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .review-item-enhanced {
        flex-direction: column;
        gap: 1rem;
    }
    
    .review-avatar {
        align-self: flex-start;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .review-meta {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
    
    .review-actions {
        flex-wrap: wrap;
        gap: 0.5rem;
    }
    
    .rating-breakdown-row {
        gap: 0.5rem;
    }
    
    .star-label {
        min-width: 50px;
        font-size: 0.8rem;
    }
}

/* Global Mobile Responsive Base */
* {
    box-sizing: border-box;
}

html, body {
    overflow-x: hidden;
    width: 100%;
    max-width: 100vw;
}

.container-fluid, .container {
    width: 100%;
    max-width: 100%;
    padding-left: 15px;
    padding-right: 15px;
}

/* Prevent horizontal overflow */
.main-content, .course-content, .course-sidebar {
    max-width: 100%;
    width: 100%;
}

body {
    background: whitesmoke;
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    margin: 0;
    padding: 0;
}

.course-wrapper {
    min-height: 100vh;
    background: whitesmoke;
}

/* Clean Header */
.course-header {
    background: white;
    padding: 1rem 0;
    margin-bottom: 0;
    border-bottom: 3px solid var(--panadite-primary);
    box-shadow: 0 2px 10px rgba(37, 99, 235, 0.1);
}

.course-header h1 {
    font-size: 1.5rem;
    font-weight: 600;
    margin: 0;
    color: var(--panadite-primary);
    text-align: center;
}

/* Main Layout */
.course-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0;
    display: flex;
    min-height: calc(100vh - 120px);
}

/* Mobile responsive grid layout */
@media (max-width: 768px) {
    .course-container {
        display: flex !important;
        flex-direction: column !important;
        grid-template-columns: none !important;
        max-width: 100% !important;
        width: 100% !important;
        padding: 0.5rem !important;
        gap: 1rem !important;
    }
}

/* Course Profile Card */
.course-profile-card {
    background: linear-gradient(135deg, var(--panadite-light) 0%, white 100%);
    border: 2px solid var(--panadite-primary);
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    display: flex;
    gap: 1.5rem;
    align-items: center;
}

.course-image {
    width: 80px;
    height: 80px;
    border-radius: 12px;
    background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
    background-size: cover;
    background-position: center;
    flex-shrink: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 2rem;
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.2);
}

.course-profile-info h3 {
    color: var(--panadite-primary);
    font-size: 1.3rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
}

.course-profile-meta {
    display: flex;
    gap: 2rem;
    font-size: 0.9rem;
    color: #64748b;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

/* Video Section */
.video-section {
    background: white;
    padding: 2rem;
    border-right: 1px solid #e5e7eb;
    min-height: 100vh;
    overflow-y: auto;
    flex: 1;
    box-sizing: border-box;
}

/* Sidebar */
.course-sidebar {
    background: white;
    padding: 1.5rem;
    border-left: 1px solid #e5e7eb;
    width: 350px;
    flex-shrink: 0;
    overflow-y: auto;
    max-height: 100vh;
}

.course-info {
    margin-bottom: 2rem;
    padding-bottom: 1.5rem;
    border-bottom: 2px solid #f3f4f6;
}

.course-title {
    font-size: 1.8rem;
    font-weight: 700;
    color: var(--panadite-primary);
    margin-bottom: 0.5rem;
}

.course-instructor {
    color: #64748b;
    font-size: 1rem;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.progress-section {
    background: var(--panadite-light);
    padding: 1rem;
    border-radius: 12px;
    margin-bottom: 1.5rem;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: #e5e7eb;
    border-radius: 4px;
    overflow: hidden;
    margin: 0.5rem 0;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
    transition: width 0.3s ease;
}

.progress-text {
    font-size: 0.9rem;
    color: var(--panadite-dark);
    font-weight: 600;
}

/* Video Player */
.video-container {
    position: relative;
    background: #000;
    border-radius: 12px;
    overflow: hidden;
    box-shadow: 0 8px 25px rgba(0,0,0,0.2);
    margin-bottom: 1.5rem;
    cursor: pointer; /* Show it's interactive */
}

.video-player {
    width: 100%;
    height: 400px;
    background: linear-gradient(45deg, #1a1a1a 25%, #2a2a2a 50%, #1a1a1a 75%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    position: relative;
}

/* Video element styling for proper display */
#lessonVideo {
    width: 100%;
    height: 100%;
    object-fit: cover; /* Fills container while maintaining aspect ratio */
    background: #000;
    border-radius: 12px 12px 0 0; /* Match container border radius */
}

/* Alternative: use object-fit: contain to show full video without cropping */
.video-fit-contain #lessonVideo {
    object-fit: contain; /* Shows full video with possible letterboxing */
}

.video-controls {
    position: absolute;
    bottom: 0;
    left: 0;
    right: 0;
    background: linear-gradient(135deg, rgba(30, 41, 59, 0.95) 0%, rgba(51, 65, 85, 0.95) 100%);
    padding: 1rem;
    color: white;
    border-top: 1px solid #475569;
    opacity: 0;
    transition: opacity 0.3s ease;
    backdrop-filter: blur(10px);
}

.video-container:hover .video-controls {
    opacity: 1;
}

.time-bar {
    width: 100%;
    height: 4px;
    background: rgba(255,255,255,0.3);
    border-radius: 2px;
    margin-bottom: 0.5rem;
    cursor: pointer;
}

.time-progress {
    height: 100%;
    background: var(--panadite-primary);
    border-radius: 2px;
    width: 30%;
    transition: width 0.1s ease;
}

.control-buttons {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 0.5rem;
    gap: 1rem;
}

.control-left {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    flex: 1;
}

.control-center {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    justify-content: center;
    flex: 1;
}

.control-right {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    justify-content: flex-end;
    flex: 1;
}

/* Enhanced Control Buttons */
.control-btn {
    background: none;
    border: none;
    color: white;
    cursor: pointer;
    font-size: 1.2rem;
    padding: 0.5rem;
    border-radius: 4px;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
}

.control-btn:hover {
    background: rgba(255, 255, 255, 0.1);
    transform: scale(1.05);
}

.control-btn:active {
    transform: scale(0.95);
}

/* Dropdown Menus */
.dropdown {
    position: relative;
}

.dropdown-menu {
    position: absolute;
    bottom: 50px;
    left: 50%;
    transform: translateX(-50%);
    background: rgba(15, 23, 42, 0.95);
    backdrop-filter: blur(10px);
    border-radius: 8px;
    padding: 0.5rem 0;
    min-width: 120px;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
    z-index: 1000;
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.dropdown-menu.show {
    opacity: 1;
    visibility: visible;
}

.dropdown-item {
    padding: 0.5rem 1rem;
    cursor: pointer;
    color: white;
    transition: all 0.2s ease;
    font-size: 0.9rem;
}

.dropdown-item:hover {
    background: rgba(37, 99, 235, 0.5);
}

.dropdown-item.active {
    background: var(--panadite-primary);
    font-weight: 600;
}

/* Volume Control */
.volume-control {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.volume-slider {
    width: 80px;
    height: 4px;
    background: rgba(255, 255, 255, 0.3);
    border-radius: 2px;
    outline: none;
    -webkit-appearance: none;
    cursor: pointer;
}

.volume-slider::-webkit-slider-thumb {
    -webkit-appearance: none;
    width: 16px;
    height: 16px;
    background: var(--panadite-primary);
    border-radius: 50%;
    cursor: pointer;
}

.volume-slider::-moz-range-thumb {
    width: 16px;
    height: 16px;
    background: var(--panadite-primary);
    border-radius: 50%;
    cursor: pointer;
    border: none;
}

/* Speed Label Styling */
#speedLabel {
    font-size: 0.9rem;
    font-weight: 600;
    min-width: 30px;
}

/* Note Modal Styling */
.modal-content {
    border: none;
    border-radius: 12px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
}

.modal-header {
    background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
    color: white;
    border-radius: 12px 12px 0 0;
    padding: 1.5rem;
    border-bottom: none;
}

.modal-header .btn-close {
    filter: invert(1);
    opacity: 0.8;
}

.modal-header .btn-close:hover {
    opacity: 1;
}

.modal-title {
    font-weight: 600;
    display: flex;
    align-items: center;
}

.modal-body {
    padding: 2rem;
}

.form-label {
    font-weight: 600;
    color: var(--panadite-dark);
    margin-bottom: 0.5rem;
}

#noteTextarea {
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    padding: 1rem;
    font-size: 1rem;
    line-height: 1.5;
    transition: all 0.3s ease;
    resize: vertical;
    min-height: 120px;
}

#noteTextarea:focus {
    border-color: var(--panadite-primary);
    box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
    outline: none;
}

.note-tips {
    background: #f8fafc;
    padding: 1rem;
    border-radius: 8px;
    border-left: 4px solid var(--panadite-primary);
    margin-top: 1rem;
}

.modal-footer {
    padding: 1.5rem 2rem;
    border-top: 1px solid #e5e7eb;
    border-radius: 0 0 12px 12px;
}

.modal-footer .btn {
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    border-radius: 8px;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

/* Custom Modal Styles */
.custom-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
}

.custom-modal-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.4);
    backdrop-filter: blur(2px);
}

.custom-modal-container {
    position: relative;
    background: white;
    border-radius: 12px;
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: modalSlideIn 0.3s ease-out;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateY(-30px) scale(0.95);
    }
    to {
        opacity: 1;
        transform: translateY(0) scale(1);
    }
}

.custom-modal-header {
    background: linear-gradient(135deg, #4285f4 0%, #34a853 100%);
    color: white;
    padding: 20px 24px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.custom-modal-title {
    display: flex;
    align-items: center;
    font-weight: 600;
    font-size: 18px;
}

.custom-modal-title i {
    font-size: 20px;
    margin-right: 10px;
}

.custom-modal-title #modalTimestamp {
    color: #fff3cd;
    font-weight: 700;
    margin-left: 8px;
}

.custom-modal-close {
    background: none;
    border: none;
    color: white;
    font-size: 24px;
    cursor: pointer;
    padding: 8px;
    border-radius: 4px;
    transition: background-color 0.2s ease;
    line-height: 1;
}

.custom-modal-close:hover {
    background: rgba(255, 255, 255, 0.2);
}

.custom-modal-body {
    padding: 24px;
}

.note-input-section {
    margin-bottom: 16px;
}

.note-input-section label {
    display: block;
    font-weight: 600;
    color: #333;
    margin-bottom: 8px;
    font-size: 14px;
}

.note-input-section textarea {
    width: 100%;
    min-height: 120px;
    padding: 12px;
    border: 2px solid #e1e5e9;
    border-radius: 8px;
    font-size: 14px;
    line-height: 1.5;
    resize: vertical;
    transition: border-color 0.2s ease, box-shadow 0.2s ease;
    box-sizing: border-box;
    font-family: inherit;
}

.note-input-section textarea:focus {
    outline: none;
    border-color: #4285f4;
    box-shadow: 0 0 0 3px rgba(66, 133, 244, 0.1);
}

.note-input-section textarea.invalid {
    border-color: #dc3545;
    box-shadow: 0 0 0 3px rgba(220, 53, 69, 0.1);
}

.note-tips-section {
    padding: 12px;
    background: #f8f9fa;
    border-radius: 6px;
    display: flex;
    align-items: flex-start;
    font-size: 13px;
    color: #6c757d;
    border-left: 3px solid #ffc107;
}

.note-tips-section i {
    font-size: 16px;
    margin-right: 8px;
    margin-top: 1px;
    color: #ffc107;
}

.custom-modal-footer {
    padding: 20px 24px;
    background: #f8f9fa;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
    border-top: 1px solid #e9ecef;
}

.custom-btn {
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 14px;
}

.custom-btn-secondary {
    background: #6c757d;
    color: white;
}

.custom-btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.custom-btn-primary {
    background: linear-gradient(135deg, #4285f4 0%, #34a853 100%);
    color: white;
}

.custom-btn-primary:hover {
    background: linear-gradient(135deg, #3367d6 0%, #2d8f3f 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(66, 133, 244, 0.3);
}

/* Video Notifications Animations */
@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes slideOut {
    from {
        transform: translateX(0);
        opacity: 1;
    }
    to {
        transform: translateX(100%);
        opacity: 0;
    }
}

/* Video Notes Panel */
.video-notes-panel {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    margin-top: 1.5rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    border-left: 4px solid var(--panadite-primary);
}

.notes-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 1rem;
}

.notes-title {
    font-size: 1.2rem;
    font-weight: 600;
    color: var(--panadite-dark);
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.notes-list {
    max-height: 300px;
    overflow-y: auto;
}

.note-item {
    display: flex;
    gap: 1rem;
    padding: 1rem;
    background: #f8fafc;
    border-radius: 8px;
    margin-bottom: 0.5rem;
    transition: all 0.2s ease;
}

.note-item:hover {
    background: #e2e8f0;
    transform: translateY(-1px);
}

.note-timestamp {
    color: var(--panadite-primary);
    font-weight: 600;
    font-size: 0.9rem;
    cursor: pointer;
    min-width: 60px;
}

.note-timestamp:hover {
    text-decoration: underline;
}

.note-content {
    flex: 1;
    color: var(--panadite-dark);
    line-height: 1.5;
}

.note-actions {
    display: flex;
    gap: 0.25rem;
}

.note-action {
    background: none;
    border: none;
    color: #64748b;
    cursor: pointer;
    padding: 0.25rem;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.note-action:hover {
    color: var(--panadite-primary);
    background: rgba(37, 99, 235, 0.1);
}

.no-notes {
    text-align: center;
    color: #64748b;
    font-style: italic;
    padding: 2rem;
}

/* Keyboard Shortcuts Panel */
.shortcuts-info {
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    padding: 1rem;
    border-radius: 8px;
    margin-top: 1rem;
    font-size: 0.85rem;
}

.shortcuts-title {
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: var(--panadite-primary);
}

.shortcuts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 0.5rem;
}

.shortcut-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.25rem 0;
}

.shortcut-key {
    background: rgba(255, 255, 255, 0.1);
    padding: 0.2rem 0.5rem;
    border-radius: 4px;
    font-family: monospace;
    font-size: 0.8rem;
}

.time-display {
    font-size: 0.9rem;
    font-weight: 500;
    color: #e2e8f0;
}

.video-controls button {
    transition: all 0.2s ease;
    border-radius: 4px;
    padding: 0.25rem;
}

.video-controls button:hover {
    background: rgba(255, 255, 255, 0.1);
    transform: scale(1.05);
}

/* Current Lesson Info */
.current-lesson {
    background: var(--panadite-light);
    padding: 1.5rem;
    border-radius: 12px;
    border-left: 4px solid var(--panadite-primary);
}

.lesson-title {
    font-size: 1.3rem;
    font-weight: 600;
    color: var(--panadite-primary);
    margin-bottom: 0.5rem;
}

.lesson-description {
    color: #64748b;
    line-height: 1.5;
    margin-bottom: 1rem;
}

.lesson-meta {
    display: flex;
    gap: 1rem;
    font-size: 0.9rem;
    color: var(--panadite-dark);
}

/* Sidebar */
.course-sidebar {
    background: #f8fafc;
    padding: 1.5rem;
    overflow-y: auto;
    border-left: 1px solid #e5e7eb;
}

.sidebar-section {
    margin-bottom: 2rem;
}

.section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--panadite-primary);
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

/* Lessons List */
.lessons-list {
    gap: 0.5rem;
}

.lesson-item {
    background: white;
    border-radius: 12px;
    padding: 1rem;
    margin-bottom: 1rem;
    cursor: pointer;
    transition: all 0.3s ease;
    border: 2px solid transparent;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    display: flex;
    gap: 1rem;
    align-items: flex-start;
}

/* Lesson Thumbnail Column */
.lesson-thumbnail-column {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    flex-shrink: 0;
}

/* Lesson Thumbnail Styles */
.lesson-thumbnail {
    position: relative;
    width: 80px;
    height: 45px;
    border-radius: 6px;
    overflow: hidden;
    background: #f3f4f6;
}

/* Progress under thumbnail */
.thumbnail-progress {
    width: 80px;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.25rem;
}

.thumbnail-progress .lesson-progress-bar {
    width: 100%;
    height: 3px;
    background-color: #e5e7eb;
    border-radius: 2px;
    overflow: hidden;
}

.thumbnail-progress .lesson-progress-fill {
    height: 100%;
    background: linear-gradient(90deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
    border-radius: 2px;
    transition: width 0.3s ease;
}

.thumbnail-progress .progress-percentage {
    font-size: 0.7rem;
    font-weight: 600;
    color: var(--panadite-primary);
    text-align: center;
    white-space: nowrap;
}

/* Thumbnail Indicators */
.thumbnail-indicators {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
    margin-top: 0.25rem;
}

.thumbnail-badge {
    font-size: 0.6rem;
    font-weight: 600;
    padding: 0.2rem 0.4rem;
    border-radius: 4px;
}

/* Empty State Styling */
.empty-state {
    text-align: center;
    padding: 3rem 2rem;
    color: #6b7280;
}

.empty-icon {
    font-size: 4rem;
    color: #d1d5db;
    margin-bottom: 1rem;
}

.empty-state h4 {
    font-size: 1.25rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
}

.empty-state p {
    font-size: 0.95rem;
    line-height: 1.5;
    margin: 0;
    max-width: 300px;
    margin: 0 auto;
}
    display: flex;
    align-items: center;
    gap: 0.2rem;
    white-space: nowrap;
    text-align: center;
    width: fit-content;
    align-self: center;
}

.thumbnail-badge i {
    font-size: 0.7rem;
}

.thumbnail-badge.badge-assignment {
    background: #fef3c7;
    color: #d97706;
    border: 1px solid #fbbf24;
}

.thumbnail-badge.badge-quiz {
    background: #dbeafe;
    color: #2563eb;
    border: 1px solid #60a5fa;
}

.thumbnail-badge.badge-materials {
    background: #d1fae5;
    color: #059669;
    border: 1px solid #34d399;
}

.thumbnail-img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.thumbnail-placeholder {
    width: 100%;
    height: 100%;
    background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 2rem;
}

.play-overlay {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: rgba(37, 99, 235, 0.9);
    color: white;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 12px;
    animation: pulse 2s infinite;
}

.completion-badge {
    position: absolute;
    top: 2px;
    right: 2px;
    background: #10b981;
    color: white;
    width: 16px;
    height: 16px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 10px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.2);
}
/* Lesson Content */
.lesson-content {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

@media (max-width: 768px) {
    body {
        font-size: 14px;
    }
    
    .main-content {
        flex-direction: column !important;
        gap: 1rem;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 !important;
        padding: 0.5rem !important;
    }
    
    .course-content {
        width: 100% !important;
        max-width: 100% !important;
        order: 1;
        margin: 0 !important;
        padding: 0 !important;
        flex: none !important;
    }
    
    .course-sidebar {
        width: 100% !important;
        max-width: 100% !important;
        order: 2;
        margin-left: 0 !important;
        margin: 0 !important;
        padding: 0 !important;
        flex: none !important;
    }
    
    .course-header {
        padding: 1rem 0.5rem !important;
        width: 100%;
        max-width: 100%;
    }
    
    .course-title {
        font-size: 1.5rem !important;
        line-height: 1.3;
    }
    
    .course-description {
        font-size: 0.9rem !important;
    }
    
    .course-meta {
        flex-wrap: wrap;
        gap: 0.5rem;
    }
    
    .meta-item {
        font-size: 0.8rem;
    }
    
    .video-container {
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 0 1rem 0 !important;
        padding: 0 !important;
    }
    
    .video-player {
        width: 100% !important;
        height: 220px !important;
    }
    
    .video-controls {
        padding: 0.5rem !important;
        width: 100%;
    }
    
    .control-buttons {
        gap: 0.5rem;
        flex-wrap: wrap;
    }
    
    .control-btn {
        width: 35px;
        height: 35px;
        font-size: 1rem;
    }
    
    .course-stats {
        display: grid !important;
        grid-template-columns: 1fr 1fr !important;
        gap: 0.5rem !important;
        margin-bottom: 1rem !important;
        width: 100% !important;
    }
    
    .stat-card {
        flex: none !important;
        margin: 0 !important;
        padding: 1rem !important;
        min-width: 0 !important;
    }
    
    .sidebar-section {
        width: 100% !important;
        max-width: 100% !important;
    }
    
    .lessons-list {
        width: 100% !important;
        max-width: 100% !important;
    }
    
    .lesson-item {
        padding: 0.75rem;
        gap: 0.75rem;
    }
    
    .lesson-thumbnail-column {
        gap: 0.4rem;
    }
    
    .lesson-thumbnail {
        width: 70px;
        height: 39px;
    }
    
    .thumbnail-progress {
        width: 70px;
    }
    
    .thumbnail-badge {
        font-size: 0.55rem;
        padding: 0.15rem 0.3rem;
    }
    
    .lesson-content .lesson-name {
        font-size: 0.9rem;
        line-height: 1.3;
    }
    
    .lesson-content .lesson-duration {
        font-size: 0.8rem;
    }
    
    .video-container {
        margin-bottom: 1rem;
    }
    
    .video-controls {
        padding: 0.5rem;
    }
    
    .control-buttons {
        gap: 0.5rem;
    }
    
    .control-btn {
        width: 35px;
        height: 35px;
        font-size: 1rem;
    }
}

@media (max-width: 480px) {
    .course-header {
        padding: 1rem;
    }
    
    .course-title {
        font-size: 1.5rem;
    }
    
    .course-description {
        font-size: 0.9rem;
    }
    
    .course-meta {
        flex-wrap: wrap;
        gap: 0.5rem;
    }
    
    .meta-item {
        font-size: 0.8rem;
    }
    
    .video-player {
        height: 200px;
    }
    
    .lesson-item {
        padding: 0.5rem;
        gap: 0.5rem;
    }
    
    .lesson-thumbnail {
        width: 60px;
        height: 34px;
    }
    
    .thumbnail-progress {
        width: 60px;
    }
    
    .thumbnail-badge {
        font-size: 0.5rem;
        padding: 0.1rem 0.25rem;
    }
    
    .lesson-content .lesson-name {
        font-size: 0.85rem;
    }
    
    .lesson-content .lesson-number,
    .lesson-content .lesson-duration {
        font-size: 0.75rem;
    }
    
    .progress-percentage {
        font-size: 0.65rem;
    }
    
    .completion-badge {
        width: 14px;
        height: 14px;
        font-size: 8px;
    }
    
    .play-overlay {
        width: 20px;
        height: 20px;
        font-size: 10px;
    }
}

@keyframes pulse {
    0%, 100% { 
        transform: translate(-50%, -50%) scale(1); 
        opacity: 1; 
    }
    50% { 
        transform: translate(-50%, -50%) scale(1.1); 
        opacity: 0.8; 
    }
}

.lesson-item:hover {
    border-color: var(--panadite-primary);
    box-shadow: 0 2px 8px rgba(37, 99, 235, 0.1);
}

.lesson-item.active {
    border-color: var(--panadite-primary);
    background: var(--panadite-light);
}

.lesson-item.completed::before {
    content: '✓';
    position: absolute;
    top: 0.5rem;
    right: 0.5rem;
    width: 20px;
    height: 20px;
    background: #10b981;
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.8rem;
    font-weight: bold;
}

.lesson-number {
    font-size: 0.8rem;
    color: var(--panadite-primary);
    font-weight: 600;
    margin-bottom: 0.25rem;
}

.lesson-name {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.lesson-duration {
    font-size: 0.8rem;
    color: #64748b;
    margin-bottom: 0.5rem;
}

.lesson-indicators {
    display: flex;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
    flex-wrap: wrap;
}

.indicator-badge {
    padding: 0.15rem 0.4rem;
    border-radius: 2px;
    font-size: 0.6rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.15rem;
}

.badge-assignment {
    background: #fef3c7;
    color: #92400e;
}

.badge-quiz {
    background: transparent;
    color: #6b21a8;
    border: 1px solid #6b21a8;
}

.badge-materials {
    background: #d1fae5;
    color: #065f46;
}

.lesson-progress-bar {
    width: 100%;
    height: 4px;
    background: #e5e7eb;
    border-radius: 2px;
    margin-top: 0.5rem;
    overflow: hidden;
}

.lesson-progress-fill {
    height: 100%;
    background: linear-gradient(90deg, var(--panadite-primary) 0%, #10b981 100%);
    transition: width 0.3s ease;
    border-radius: 2px;
}

.progress-percentage {
    font-size: 0.7rem;
    color: var(--panadite-primary);
    font-weight: 600;
    margin-top: 0.25rem;
}

/* Dynamic Question Input */
.question-section {
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 2px solid var(--panadite-primary);
    border-radius: 12px;
    padding: 2rem;
    margin-top: 2rem;
    display: none;
    animation: slideIn 0.5s ease-out;
}

.question-section.show {
    display: block;
}

@keyframes slideIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.question-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.question-icon {
    width: 50px;
    height: 50px;
    background: var(--panadite-primary);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.question-text {
    flex: 1;
}

.question-text h4 {
    color: var(--panadite-primary);
    font-size: 1.3rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
}

.question-text p {
    color: #64748b;
    margin: 0;
    font-size: 1rem;
}

.question-form {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.question-input {
    width: 100%;
    padding: 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-family: inherit;
    font-size: 1rem;
    resize: vertical;
    min-height: 100px;
    transition: border-color 0.3s ease;
}

.question-input:focus {
    outline: none;
    border-color: var(--panadite-primary);
    box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
}

.question-submit {
    background: var(--panadite-primary);
    color: white;
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    align-self: flex-start;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.question-submit:hover {
    background: var(--panadite-secondary);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
}

/* Lesson Actions */
.lesson-actions {
    display: flex;
    gap: 0.5rem;
    margin-top: 0.5rem;
    flex-wrap: wrap;
}

.action-btn {
    padding: 0.25rem 0.75rem;
    border: 1px solid #d1d5db;
    background: white;
    color: #6b7280;
    border-radius: 6px;
    font-size: 0.75rem;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.action-btn:hover {
    border-color: var(--panadite-primary);
    color: var(--panadite-primary);
    background: var(--panadite-light);
}

/* Course Stats */
.course-stats {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.stat-card {
    background: white;
    padding: 1rem;
    border-radius: 8px;
    text-align: center;
    border: 2px solid var(--panadite-light);
}

.stat-number {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--panadite-primary);
    margin-bottom: 0.25rem;
}

.stat-label {
    font-size: 0.8rem;
    color: #64748b;
    font-weight: 500;
}

/* Navigation Buttons */
.lesson-navigation {
    display: flex;
    gap: 1rem;
    margin-top: 1.5rem;
}

.nav-btn {
    flex: 1;
    padding: 0.75rem 1rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.btn-primary {
    background: var(--panadite-primary);
    color: white;
}

.btn-secondary {
    background: #f3f4f6;
    color: #6b7280;
}

.nav-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.btn-primary:hover {
    background: var(--panadite-secondary);
}

/* Responsive Design */
@media (max-width: 1024px) {
    .course-container {
        grid-template-columns: 1fr;
        grid-template-rows: auto 1fr;
    }
    
    .course-sidebar {
        order: -1;
        max-height: 300px;
        padding: 1rem;
    }
    
    .lessons-list {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 0.5rem;
    }
}

@media (max-width: 768px) {
    .video-section {
        padding: 1rem;
    }
    
    .course-container {
        padding: 0;
    }
    
    .video-player {
        height: 250px;
    }
    
    .course-stats {
        grid-template-columns: 1fr;
    }
    
    .lesson-navigation {
        flex-direction: column;
    }
}

/* ===== SITA ACCREDITED CERTIFICATE SECTION STYLES ===== */

/* Main Accreditation Content */
.accreditation-content {
    background: #f8f9fa;
    border-radius: 20px;
    padding: 2rem;
    min-height: 600px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.05);
}

/* Gradient Header Enhancement */
.bg-gradient-primary {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 50%, #1d4ed8 100%) !important;
    box-shadow: 0 10px 40px rgba(37, 99, 235, 0.3);
    border: none;
    position: relative;
    overflow: hidden;
}

.bg-gradient-primary::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, rgba(255,255,255,0.1) 0%, transparent 50%);
    pointer-events: none;
}

.accreditation-header h3 {
    font-weight: 700;
    letter-spacing: -0.5px;
    position: relative;
    z-index: 2;
}

.accreditation-header p {
    position: relative;
    z-index: 2;
}

.accreditation-badge {
    background: rgba(255, 255, 255, 0.95) !important;
    backdrop-filter: blur(10px);
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
    border: 2px solid rgba(255, 255, 255, 0.3);
    transition: all 0.3s ease;
}

.accreditation-badge:hover {
    transform: scale(1.02);
    box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
}

.sita-logo {
    max-height: 40px;
    filter: drop-shadow(0 2px 8px rgba(0, 0, 0, 0.1));
    transition: filter 0.3s ease;
}

.sita-logo:hover {
    filter: drop-shadow(0 4px 12px rgba(0, 0, 0, 0.2));
}

/* Process Steps Enhancement */
.process-steps {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-top: 1rem;
}

.step-item {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    padding: 1.5rem;
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border-radius: 16px;
    border: 2px solid #e5e7eb;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.06);
    position: relative;
    overflow: hidden;
}

.step-item::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
    transform: scaleY(0);
    transition: transform 0.3s ease;
}

.step-item:hover {
    border-color: #3b82f6;
    box-shadow: 0 8px 25px rgba(59, 130, 246, 0.15);
    transform: translateX(8px);
}

.step-item:hover::before {
    transform: scaleY(1);
}

.step-number {
    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
    color: white;
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    font-size: 1rem;
    flex-shrink: 0;
    box-shadow: 0 6px 20px rgba(59, 130, 246, 0.4);
    position: relative;
    z-index: 2;
}

.step-number::before {
    content: '';
    position: absolute;
    top: -2px;
    left: -2px;
    right: -2px;
    bottom: -2px;
    background: linear-gradient(135deg, #60a5fa, #3b82f6);
    border-radius: 50%;
    z-index: -1;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.step-item:hover .step-number::before {
    opacity: 1;
}

.step-content {
    flex: 1;
    line-height: 1.6;
    color: #374151;
}

.step-content strong {
    color: #1f2937;
    display: block;
    margin-bottom: 0.5rem;
    font-size: 1.05rem;
}

/* Requirements Grid Enhancement */
.requirements-grid {
    display: grid;
    gap: 1rem;
    margin-top: 1rem;
}

.requirement-card {
    display: flex;
    align-items: center;
    padding: 1.25rem;
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border-radius: 12px;
    border: 2px solid #e5e7eb;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    font-size: 0.95rem;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    position: relative;
    overflow: hidden;
}

.requirement-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 2px;
    background: linear-gradient(90deg, #3b82f6, #1d4ed8);
    transform: scaleX(0);
    transition: transform 0.3s ease;
}

.requirement-card:hover {
    border-color: #d1d5db;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    transform: translateY(-2px);
}

.requirement-card:hover::before {
    transform: scaleX(1);
}

.requirement-card i {
    font-size: 1.3rem;
    margin-right: 0.75rem;
    transition: transform 0.2s ease;
}

.requirement-card:hover i {
    transform: scale(1.1);
}

/* Enhanced Alert Styles */
.alert {
    border-radius: 16px;
    padding: 2rem;
    border: none;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    position: relative;
    overflow: hidden;
}

.alert::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, rgba(255,255,255,0.1) 0%, transparent 50%);
    pointer-events: none;
}

.alert-info {
    background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%);
    border-left: 6px solid #3b82f6;
}

.alert-warning {
    background: linear-gradient(135deg, #fef3c7 0%, #fed7aa 100%);
    border-left: 6px solid #f59e0b;
}

.alert-heading {
    font-weight: 700;
    color: #1f2937;
    position: relative;
    z-index: 2;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.alert p, .alert ul, .alert small {
    position: relative;
    z-index: 2;
}

/* Card Enhancements */
.card {
    border: none;
    border-radius: 20px;
    overflow: hidden;
    box-shadow: 0 6px 30px rgba(0, 0, 0, 0.08);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    background: #ffffff;
}

.card:hover {
    transform: translateY(-8px);
    box-shadow: 0 12px 50px rgba(0, 0, 0, 0.15);
}

.card-header {
    border-bottom: none;
    font-weight: 600;
    padding: 1.5rem 2rem;
    position: relative;
}

.card-header::after {
    content: '';
    position: absolute;
    bottom: 0;
    left: 2rem;
    right: 2rem;
    height: 1px;
    background: linear-gradient(90deg, transparent, rgba(0,0,0,0.1), transparent);
}

.card-header.bg-light {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%) !important;
}

.card-header.bg-primary {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%) !important;
    color: white !important;
}

.card-header.bg-primary::after {
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
}

.card-body {
    padding: 2rem;
}

.card-title {
    margin-bottom: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

/* Form Styling Enhancement */
.form-control {
    border-radius: 12px;
    border: 2px solid #e5e7eb;
    padding: 0.875rem 1.25rem;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    font-size: 0.95rem;
    background: rgba(255, 255, 255, 0.8);
    backdrop-filter: blur(10px);
}

.form-control:focus {
    border-color: #3b82f6;
    box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.1);
    background: #ffffff;
    transform: translateY(-1px);
}

.form-label {
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.75rem;
    font-size: 0.95rem;
}

.form-check {
    padding: 1rem;
    background: rgba(248, 250, 252, 0.8);
    border-radius: 12px;
    border: 1px solid #e5e7eb;
    transition: all 0.3s ease;
}

.form-check:hover {
    background: rgba(59, 130, 246, 0.05);
    border-color: #3b82f6;
}

.form-check-input {
    border-radius: 6px;
    border: 2px solid #d1d5db;
    margin-top: 0.25rem;
    width: 1.2rem;
    height: 1.2rem;
}

.form-check-input:checked {
    background-color: #3b82f6;
    border-color: #3b82f6;
    box-shadow: 0 0 0 2px rgba(59, 130, 246, 0.2);
}

.form-check-label {
    color: #374151;
    line-height: 1.6;
    margin-left: 0.75rem;
    font-size: 0.95rem;
}

/* Button Styling Enhancement */
.btn {
    border-radius: 12px;
    font-weight: 600;
    padding: 0.875rem 2.5rem;
    border: none;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    text-transform: none;
    letter-spacing: 0.25px;
    position: relative;
    overflow: hidden;
}

.btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.5s ease;
}

.btn:hover::before {
    left: 100%;
}

.btn-primary {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
    transform: translateY(-3px);
    box-shadow: 0 10px 30px rgba(59, 130, 246, 0.4);
    color: white;
}

.btn-outline-primary {
    border: 2px solid #3b82f6;
    color: #3b82f6;
    background: transparent;
}

.btn-outline-primary:hover {
    background: #3b82f6;
    color: white;
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(59, 130, 246, 0.3);
}

.btn-lg {
    padding: 1.125rem 3rem;
    font-size: 1.1rem;
}

/* Progress Bar Enhancement */
.progress {
    border-radius: 12px;
    overflow: hidden;
    background: #e5e7eb;
    box-shadow: inset 0 2px 6px rgba(0, 0, 0, 0.1);
    position: relative;
}

.progress::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, rgba(255,255,255,0.1) 25%, transparent 25%, transparent 50%, rgba(255,255,255,0.1) 50%, rgba(255,255,255,0.1) 75%, transparent 75%, transparent);
    background-size: 1rem 1rem;
    animation: progress-shimmer 2s linear infinite;
    pointer-events: none;
}

@keyframes progress-shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

.progress-bar {
    border-radius: 12px;
    background: linear-gradient(90deg, #3b82f6 0%, #1d4ed8 100%);
    transition: width 0.8s cubic-bezier(0.4, 0, 0.2, 1);
    font-weight: 600;
    font-size: 0.875rem;
    position: relative;
    overflow: hidden;
}

.progress-bar-striped {
    background-image: linear-gradient(45deg, rgba(255, 255, 255, 0.15) 25%, transparent 25%, transparent 50%, rgba(255, 255, 255, 0.15) 50%, rgba(255, 255, 255, 0.15) 75%, transparent 75%, transparent);
    background-size: 1.5rem 1.5rem;
    animation: progress-bar-stripes 1s linear infinite;
}

@keyframes progress-bar-stripes {
    0% { background-position: 1.5rem 0; }
    100% { background-position: 0 0; }
}

/* Status Timeline Enhancement */
.status-timeline {
    position: relative;
    padding: 2rem 0;
    background: rgba(255, 255, 255, 0.5);
    border-radius: 16px;
    margin: 1rem 0;
}

.timeline-item {
    position: relative;
    display: flex;
    align-items: center;
    gap: 1.5rem;
    padding: 1.5rem 0;
    border-left: 4px solid #e5e7eb;
    margin-left: 2rem;
    padding-left: 3rem;
    transition: all 0.3s ease;
}

.timeline-item:hover {
    background: rgba(59, 130, 246, 0.02);
    border-radius: 12px;
    margin-right: 1rem;
}

.timeline-item.completed {
    border-left-color: #10b981;
}

.timeline-item.active {
    border-left-color: #3b82f6;
}

.timeline-marker {
    position: absolute;
    left: -2rem;
    width: 3.5rem;
    height: 3.5rem;
    border-radius: 50%;
    background: white;
    border: 4px solid #e5e7eb;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    color: #6b7280;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
}

.timeline-item.completed .timeline-marker {
    background: #10b981;
    border-color: #10b981;
    color: white;
    box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
}

.timeline-item.active .timeline-marker {
    background: #3b82f6;
    border-color: #3b82f6;
    color: white;
    box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
}

.timeline-content h6 {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.5rem;
    font-size: 1.1rem;
}

.timeline-content small {
    color: #6b7280;
    font-size: 0.875rem;
}

/* Status Badges Enhancement */
.badge {
    font-size: 0.85rem;
    padding: 0.6rem 1.2rem;
    border-radius: 25px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
}

.badge:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
}

.badge.bg-warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%) !important;
}

.badge.bg-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%) !important;
}

.badge.bg-primary {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%) !important;
}

.badge.bg-danger {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%) !important;
}

/* Text Color Enhancements */
.text-primary {
    color: #3b82f6 !important;
}

.text-muted {
    color: #6b7280 !important;
}

.text-warning {
    color: #f59e0b !important;
}

.text-success {
    color: #10b981 !important;
}

/* Icon Enhancements */
.bx {
    transition: all 0.3s ease;
}

.card-header .bx,
.alert .bx,
.step-content .bx {
    margin-right: 0.5rem;
}

.bx:hover {
    transform: scale(1.1);
}

/* Shadow and Depth Enhancement */
.shadow-sm {
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.08) !important;
}

.shadow {
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1) !important;
}

/* Responsive Design for SITA Section */
@media (max-width: 768px) {
    .accreditation-content {
        padding: 1.5rem;
    }
    
    .accreditation-header {
        text-align: center;
    }
    
    .accreditation-header .row > div {
        margin-bottom: 1.5rem;
    }
    
    .process-steps {
        gap: 0.75rem;
    }
    
    .step-item {
        padding: 1rem;
        margin: 0;
    }
    
    .step-number {
        width: 32px;
        height: 32px;
        font-size: 0.875rem;
    }
    
    .requirements-grid {
        gap: 0.75rem;
    }
    
    .requirement-card {
        padding: 1rem;
        font-size: 0.9rem;
    }
    
    .card-body {
        padding: 1.5rem;
    }
    
    .btn-lg {
        padding: 1rem 2rem;
        font-size: 1rem;
    }
    
    .timeline-item {
        margin-left: 1rem;
        padding-left: 2rem;
    }
    
    .timeline-marker {
        left: -1.5rem;
        width: 3rem;
        height: 3rem;
        font-size: 1rem;
    }
}

@media (max-width: 576px) {
    .accreditation-content {
        padding: 1rem;
        margin: 0 -0.5rem;
    }
    
    .step-item {
        flex-direction: column;
        text-align: center;
        gap: 0.75rem;
    }
    
    .step-number {
        margin: 0 auto;
    }
}

/* Hover Effects and Animations */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.accreditation-content .card {
    animation: fadeInUp 0.6s ease-out;
}

.accreditation-content .card:nth-child(2) {
    animation-delay: 0.1s;
}

.accreditation-content .card:nth-child(3) {
    animation-delay: 0.2s;
}

/* Loading States */
.btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none !important;
}

.btn:disabled:hover {
    transform: none !important;
    box-shadow: none !important;
}

/* Dynamic Progress Bar Animations */
@keyframes shine {
    0% {
        left: -100%;
    }
    50% {
        left: 100%;
    }
    100% {
        left: 100%;
    }
}

.dynamic-progress-bar:hover {
    transform: scale(1.02);
    box-shadow: 0 4px 15px rgba(33, 150, 243, 0.6) !important;
}

.dynamic-progress-container {
    position: relative;
    overflow: hidden;
}

.dynamic-progress-bar {
    background: linear-gradient(90deg, #2196f3 0%, #1976d2 50%, #0d47a1 100%) !important;
    position: relative;
    overflow: hidden;
}

.dynamic-progress-bar::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, 
        rgba(255,255,255,0.1) 25%, 
        transparent 25%, 
        transparent 50%, 
        rgba(255,255,255,0.1) 50%, 
        rgba(255,255,255,0.1) 75%, 
        transparent 75%);
    background-size: 20px 20px;
    animation: move 1s linear infinite;
}

@keyframes move {
    0% {
        background-position: 0 0;
    }
    100% {
        background-position: 20px 0;
    }
}

</style>

<div class="course-wrapper">
    <!-- Header -->
    <div class="course-header">
        <h1><?= htmlspecialchars($course['title']) ?></h1>
    </div>

    <!-- Bootstrap Container for Grid Layout -->
    <div class="container-fluid mt-4">
        <div class="row">
            <!-- Course Content Accordion Sidebar (25% width) -->
            <div class="col-md-3">
                <div class="course-content-sidebar">
                    <div class="accordion" id="courseContentAccordion">
                        
                        <?php if (empty($lessons)): ?>
                            <!-- No Lessons Available -->
                            <div class="accordion-item">
                                <div class="empty-state p-4">
                                    <div class="text-center text-muted">
                                        <i class="bx bx-video-off fs-1"></i>
                                        <p class="mt-2 mb-0">No lessons available</p>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <!-- Individual Lesson Accordions -->
                            <?php foreach ($lessons as $index => $lesson): ?>
                                <?php 
                                // Get video-specific content for badges and links
                                $hasVideoQuiz = false;
                                $hasVideoAssignment = false; 
                                $hasVideoMaterials = false;
                                $hasVideoNotes = false;
                                $videoQuiz = null;
                                $videoAssignments = [];
                                $videoMaterials = [];
                                
                                if (isset($conn) && !empty($lesson['lesson_id'])) {
                                    // First, check if video has any quiz (simple check)
                                    $quiz_check = $conn->prepare("SELECT quiz_id, title, time_limit, passing_score FROM quizzes WHERE video_id = ? AND is_published = 1 LIMIT 1");
                                    if ($quiz_check) {
                                        $quiz_check->bind_param("i", $lesson['lesson_id']);
                                        $quiz_check->execute();
                                        $quiz_result = $quiz_check->get_result();
                                        if ($quiz_data = $quiz_result->fetch_assoc()) {
                                            $hasVideoQuiz = true;
                                            $videoQuiz = $quiz_data;
                                            
                                            // Now get student's attempt data separately if quiz exists
                                            $attempt_check = $conn->prepare("
                                                SELECT attempt_id, score, passed, completed_at 
                                                FROM quiz_attempts 
                                                WHERE quiz_id = ? AND student_id = ? 
                                                ORDER BY completed_at DESC 
                                                LIMIT 1
                                            ");
                                            if ($attempt_check) {
                                                $attempt_check->bind_param("ii", $quiz_data['quiz_id'], $user_id);
                                                $attempt_check->execute();
                                                $attempt_result = $attempt_check->get_result();
                                                if ($attempt_data = $attempt_result->fetch_assoc()) {
                                                    // Merge attempt data with quiz data
                                                    $videoQuiz = array_merge($videoQuiz, $attempt_data);
                                                }
                                                $attempt_check->close();
                                            }
                                        }
                                        $quiz_check->close();
                                    }
                                    
                                    // Get video-specific assignments data
                                    $assignment_check = $conn->prepare("SELECT assignment_id, title, due_date FROM assignments WHERE video_id = ?");
                                    if ($assignment_check) {
                                        $assignment_check->bind_param("i", $lesson['lesson_id']);
                                        $assignment_check->execute();
                                        $assignment_result = $assignment_check->get_result();
                                        while ($assignment_data = $assignment_result->fetch_assoc()) {
                                            $videoAssignments[] = $assignment_data;
                                        }
                                        $hasVideoAssignment = count($videoAssignments) > 0;
                                        $assignment_check->close();
                                    }
                                    
                                    // Get video-specific materials data
                                    $materials_check = $conn->prepare("SELECT material_id, title, file_path, file_type FROM course_materials WHERE video_id = ?");
                                    if ($materials_check) {
                                        $materials_check->bind_param("i", $lesson['lesson_id']);
                                        $materials_check->execute();
                                        $materials_result = $materials_check->get_result();
                                        while ($materials_data = $materials_result->fetch_assoc()) {
                                            $videoMaterials[] = $materials_data;
                                        }
                                        $hasVideoMaterials = count($videoMaterials) > 0;
                                        $materials_check->close();
                                    }
                                    
                                    // Check for video-specific notes
                                    $notes_check = $conn->prepare("SELECT COUNT(*) as count FROM lesson_notes WHERE video_id = ?");
                                    if ($notes_check) {
                                        $notes_check->bind_param("i", $lesson['lesson_id']);
                                        $notes_check->execute();
                                        $notes_result = $notes_check->get_result();
                                        if ($notes_data = $notes_result->fetch_assoc()) {
                                            $hasVideoNotes = $notes_data['count'] > 0;
                                        }
                                        $notes_check->close();
                                    }
                                }
                                ?>
                                
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="headingLesson<?= $lesson['lesson_id'] ?>">
                                        <button class="accordion-button <?= $lesson['lesson_id'] == $current_lesson_id ? '' : 'collapsed' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#collapseLesson<?= $lesson['lesson_id'] ?>" aria-expanded="<?= $lesson['lesson_id'] == $current_lesson_id ? 'true' : 'false' ?>" aria-controls="collapseLesson<?= $lesson['lesson_id'] ?>">
                                            <div class="d-flex align-items-center w-100">
                                                <div class="lesson-thumbnail-small me-3">
                                                    <?php if (!empty($lesson['thumbnail'])): ?>
                                                        <img src="<?= htmlspecialchars($lesson['thumbnail']) ?>" alt="Lesson <?= $index + 1 ?>" class="rounded" style="width: 40px; height: 30px; object-fit: cover;">
                                                    <?php else: ?>
                                                        <div class="thumbnail-placeholder-small d-flex align-items-center justify-content-center" style="width: 40px; height: 30px; background: #f8f9fa; border-radius: 4px;">
                                                            <i class="bx bx-play text-muted"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                    <?php if ($lesson['completed']): ?>
                                                        <div class="completion-check position-absolute" style="top: -5px; right: -5px;">
                                                            <i class="bx bx-check-circle text-success fs-6"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="flex-grow-1">
                                                    <div class="fw-semibold"><?= htmlspecialchars($lesson['title']) ?></div>
                                                    <div class="d-flex align-items-center justify-content-between mb-1">
                                                        <small class="text-muted">
                                                            <i class="bx bx-time"></i> <?= $lesson['duration'] ?>
                                                        </small>
                                                        <small class="text-muted" id="progress-display-<?= $lesson['lesson_id'] ?>" style="font-size: 0.7rem; display: none;">
                                                            <span id="progress-percentage-<?= $lesson['lesson_id'] ?>">0</span>% watched
                                                        </small>
                                                    </div>
                                                    <!-- Content Badges - Separate Row -->
                                                    <div class="d-flex align-items-center flex-wrap" id="lesson-badges-<?= $lesson['lesson_id'] ?>">
                                                        <span class="badge badge-watching-accordion me-1" style="font-size: 0.6rem; display: none;" title="Currently Watching" id="watching-badge-<?= $lesson['lesson_id'] ?>">
                                                            <i class="bx bx-play"></i> Watching
                                                        </span>
                                                        <?php if ($lesson['completed']): ?>
                                                            <span class="badge badge-completed-accordion me-1" style="font-size: 0.6rem;" title="Video Completed" id="completed-badge-<?= $lesson['lesson_id'] ?>">
                                                                <i class="bx bx-check"></i> Completed
                                                            </span>
                                                        <?php else: ?>
                                                            <span class="badge badge-completed-accordion me-1" style="font-size: 0.6rem; display: none;" title="Video Completed" id="completed-badge-<?= $lesson['lesson_id'] ?>">
                                                                <i class="bx bx-check"></i> Completed
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoQuiz): ?>
                                                            <span class="badge badge-quiz-accordion me-1" style="font-size: 0.6rem;" title="Interactive Quiz Available">
                                                                <i class="bx bx-help-circle"></i> Quiz
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoAssignment): ?>
                                                            <span class="badge bg-warning me-1" style="font-size: 0.7rem;" title="Assignment Available">
                                                                <i class="bx bx-task"></i> Task
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoMaterials): ?>
                                                            <span class="badge bg-info me-1" style="font-size: 0.7rem;" title="Materials Available">
                                                                <i class="bx bx-folder-open"></i> Files
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoNotes): ?>
                                                            <span class="badge bg-secondary me-1" style="font-size: 0.7rem;" title="Notes Available">
                                                                <i class="bx bx-note"></i> Notes
                                                            </span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </button>
                                    </h2>
                                    <div id="collapseLesson<?= $lesson['lesson_id'] ?>" class="accordion-collapse collapse <?= $lesson['lesson_id'] == $current_lesson_id ? 'show' : '' ?>" aria-labelledby="headingLesson<?= $lesson['lesson_id'] ?>" data-bs-parent="#courseContentAccordion">
                                        <div class="accordion-body">
                                            <!-- Watch Button -->
                                            <div class="d-grid gap-2 mb-3">
                                                <button class="btn btn-watch-lesson" onclick="loadLesson(<?= $lesson['lesson_id'] ?>)">
                                                    <i class="bx bx-play-circle me-2"></i>
                                                    <?= $lesson['lesson_id'] == $current_lesson_id ? 'Currently Watching' : 'Watch Lesson' ?>
                                                </button>
                                            </div>
                                            
                                            <!-- Progress Bar -->
                                            <div class="progress mb-3" style="height: 6px; background: #e3f2fd; border-radius: 6px;">
                                                <div class="progress-bar" style="width: <?= $lesson['watch_progress'] ?>%; background: #2196f3; border-radius: 6px; transition: width 0.3s ease;" id="accordion-progress-bar-<?= $lesson['lesson_id'] ?>"></div>
                                            </div>
                                            <div class="text-center text-muted small mb-3" id="accordion-progress-text-<?= $lesson['lesson_id'] ?>">
                                                Progress: <?= round($lesson['watch_progress']) ?>% complete
                                            </div>
                                            
                                            <!-- Video-Specific Content Links -->
                                            <?php if ($hasVideoQuiz || $hasVideoAssignment || $hasVideoMaterials || $hasVideoNotes): ?>
                                                <div class="lesson-content-links">
                                                    <h6 class="mb-2"><i class="bx bx-link me-1"></i> Related Content</h6>
                                                    
                                                    <?php if ($hasVideoQuiz && $videoQuiz): ?>
                                                        <?php 
                                                        // Encrypt quiz ID
                                                        $encrypted_quiz_id = '';
                                                        try {
                                                            if (class_exists('IdEncryption') && method_exists('IdEncryption', 'encrypt')) {
                                                                $encrypted_quiz_id = IdEncryption::encrypt($videoQuiz['quiz_id']);
                                                                if (empty($encrypted_quiz_id) || $encrypted_quiz_id == $videoQuiz['quiz_id']) {
                                                                    $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                }
                                                            } else {
                                                                $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                            }
                                                        } catch (Exception $e) {
                                                            $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                        }
                                                        
                                                        // Check if student has completed the quiz
                                                        $hasCompleted = !empty($videoQuiz['attempt_id']);
                                                        $studentScore = $videoQuiz['score'] ?? 0;
                                                        $hasPassed = $videoQuiz['passed'] ?? false;
                                                        $completedAt = $videoQuiz['completed_at'] ?? null;
                                                        ?>
                                                        
                                                        <!-- Quiz Information Card -->
                                                        <div class="quiz-info-card mb-2 p-2 border rounded">
                                                            <div class="quiz-header d-flex justify-content-between align-items-start mb-2">
                                                                <h6 class="mb-0 fw-bold text-primary small">
                                                                    <i class="bx bx-help-circle me-1"></i>
                                                                    Interactive Quiz: <?= htmlspecialchars($videoQuiz['title']) ?>
                                                                </h6>
                                                                <?php if ($hasCompleted): ?>
                                                                    <span class="badge bg-<?= $hasPassed ? 'success' : 'warning' ?> small">
                                                                        <?= $hasPassed ? 'Passed' : 'Failed' ?>
                                                                    </span>
                                                                <?php endif; ?>
                                                            </div>
                                                            
                                                            <!-- Quiz Details -->
                                                            <div class="quiz-details mb-2">
                                                                <div class="row g-1">
                                                                    <div class="col-6">
                                                                        <small class="text-muted d-block">⏱️ Time Limit</small>
                                                                        <small class="fw-semibold"><?= $videoQuiz['time_limit'] ?> minutes</small>
                                                                    </div>
                                                                    <div class="col-6">
                                                                        <small class="text-muted d-block">🎯 Passing Score</small>
                                                                        <small class="fw-semibold"><?= $videoQuiz['passing_score'] ?>%</small>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                            
                                                            <?php if ($hasCompleted): ?>
                                                                <!-- Quiz Results -->
                                                                <div class="quiz-results mb-2 p-2 border rounded">
                                                                    <div class="d-flex justify-content-between align-items-center">
                                                                        <div>
                                                                            <small class="fw-bold text-<?= $hasPassed ? 'success' : 'warning' ?>">
                                                                                Your Score: <?= round($studentScore) ?>%
                                                                            </small>
                                                                            <br>
                                                                            <small class="text-muted">
                                                                                Completed: <?= date('M j, Y', strtotime($completedAt)) ?>
                                                                            </small>
                                                                        </div>
                                                                        <div class="text-end">
                                                                            <?php if ($hasPassed): ?>
                                                                                <i class="bx bx-check-circle text-success"></i>
                                                                            <?php else: ?>
                                                                                <i class="bx bx-x-circle text-warning"></i>
                                                                            <?php endif; ?>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                                
                                                                <!-- Take Quiz Again Button -->
                                                <div class="d-grid">
                                                    <a href="take_quiz.php?id=<?= htmlspecialchars($encrypted_quiz_id) ?>" 
                                                       class="btn btn-start-quiz btn-sm">
                                                        <i class="bx bx-refresh me-1"></i>Take Quiz Again
                                                    </a>
                                                </div>
                                                            <?php else: ?>
                                                                <!-- First Time Quiz Button -->
                                                <div class="d-grid">
                                                    <a href="take_quiz.php?id=<?= htmlspecialchars($encrypted_quiz_id) ?>" 
                                                       class="btn btn-start-quiz btn-sm">
                                                        <i class="bx bx-play-circle me-1"></i>Start Quiz
                                                    </a>
                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($hasVideoAssignment): ?>
                                                        <div class="content-link-item mb-2">
                                                            <a href="#" class="btn btn-outline-warning btn-sm w-100">
                                                                <i class="bx bx-task me-2"></i>Complete Assignment
                                                            </a>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($hasVideoMaterials): ?>
                                                        <div class="content-link-item mb-2">
                                                            <a href="#" class="btn btn-outline-info btn-sm w-100">
                                                                <i class="bx bx-folder-open me-2"></i>Download Materials
                                                            </a>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($hasVideoNotes): ?>
                                                        <div class="content-link-item mb-2">
                                                            <a href="#" class="btn btn-outline-secondary btn-sm w-100">
                                                                <i class="bx bx-note me-2"></i>View Lecture Notes
                                                            </a>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center text-muted">
                                                    <small>No additional content for this lesson</small>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>

                    </div>
                </div>
            </div>

            <!-- Main Content Area (75% width) -->
            <div class="col-md-9">

            <!-- Course Profile Card -->
            <div class="course-profile-card">
                <div class="course-image" style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '') ?>')">
                    <?php if (empty($course['thumbnail'])): ?>
                        <i class="bx bx-book-open"></i>
                    <?php endif; ?>
                </div>
                <div class="course-profile-info">
                    <h3><?= htmlspecialchars($course['title']) ?></h3>
                    <div class="course-profile-meta">
                        <div class="meta-item">
                            <i class="bx bx-user"></i>
                            <?php 
                            // Get instructor info from database
                            $instructor_name = 'Panadite Academy';
                            if (isset($conn) && $course['teacher_id']) {
                                $instructor_query = "SELECT first_name, last_name FROM teachers WHERE teacher_id = ?";
                                if ($stmt = $conn->prepare($instructor_query)) {
                                    $stmt->bind_param("i", $course['teacher_id']);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    if ($instructor = $result->fetch_assoc()) {
                                        $instructor_name = trim($instructor['first_name'] . ' ' . $instructor['last_name']);
                                    }
                                    $stmt->close();
                                }
                            }
                            echo htmlspecialchars($instructor_name);
                            ?>
                        </div>
                        <div class="meta-item">
                            <i class="bx bx-time"></i>
                            <?= count($lessons) ?> Lessons
                        </div>
                        <div class="meta-item">
                            <i class="bx bx-group"></i>
                            <?php 
                            // Get real enrollment count
                            $enrollment_count = 0;
                            if (isset($conn)) {
                                $enrollment_query = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = ?";
                                if ($stmt = $conn->prepare($enrollment_query)) {
                                    $stmt->bind_param("i", $course_id);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    if ($enrollment_data = $result->fetch_assoc()) {
                                        $enrollment_count = $enrollment_data['count'];
                                    }
                                    $stmt->close();
                                }
                            }
                            echo $enrollment_count;
                            ?> Students
                        </div>
                        <div class="meta-item">
                            <i class="bx bx-star"></i>
                            <?php 
                            // Get real course rating
                            $average_rating = 0;
                            $rating_count = 0;
                            if (isset($conn)) {
                                $rating_query = "SELECT AVG(rating) as avg_rating, COUNT(*) as count FROM ratings WHERE course_id = ?";
                                if ($stmt = $conn->prepare($rating_query)) {
                                    $stmt->bind_param("i", $course_id);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    if ($rating_data = $result->fetch_assoc()) {
                                        $average_rating = round($rating_data['avg_rating'], 1);
                                        $rating_count = $rating_data['count'];
                                    }
                                    $stmt->close();
                                }
                            }
                            
                            // Show rating or default
                            if ($rating_count > 0) {
                                echo $average_rating . " (" . $rating_count . " reviews)";
                            } else {
                                echo "New Course";
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Course Info -->
            <div class="course-info">
                <h1 class="course-title"><?= htmlspecialchars($course['title']) ?></h1>
                <p class="course-description"><?= htmlspecialchars($course['description']) ?></p>
                
                <!-- Course Progress -->
                <div class="course-progress">
                    <div class="progress-header">
                        <span>Course Progress</span>
                        <span class="progress-percentage" id="main-course-progress-percentage">
                            <?php 
                            // Calculate real course progress
                            $total_lessons = count($lessons);
                            $completed_lessons = 0;
                            foreach ($lessons as $lesson) {
                                if ($lesson['completed']) {
                                    $completed_lessons++;
                                }
                            }
                            $course_progress_percentage = $total_lessons > 0 ? round(($completed_lessons / $total_lessons) * 100) : 0;
                            
                            // Calculate comprehensive course completion and overall grade
                            $comprehensive_progress = $progressService->calculateCourseProgress($user_id, $course_id);
                            $certificate_eligibility = $progressService->isCertificateEligible($user_id, $course_id);
                            $overall_course_mark = $comprehensive_progress['overall_percentage'];
                            $videos_completed = $course_progress_percentage >= 100; // All videos watched
                            $quiz_requirements_met = $certificate_eligibility['eligible']; // Quiz passing grade met
                            $full_course_completed = $videos_completed && $quiz_requirements_met;
                            
                            echo $course_progress_percentage;
                            ?>%
                        </span>
                    </div>
                    <div class="progress-bar" style="background: #e3f2fd; border-radius: 8px; height: 10px;">
                        <div class="progress-fill" id="main-course-progress-bar" style="width: <?= $course_progress_percentage ?>%; background: #2196f3; border-radius: 8px; transition: width 0.3s ease;"></div>
                    </div>
                </div>
                
                <!-- Overall Course Grade Display -->
                <?php if ($overall_course_mark > 0): ?>
                <div class="course-grade-display mt-3">
                    <div class="grade-info">
                        <span class="grade-label">Overall Course Grade:</span>
                        <span class="grade-value <?= $quiz_requirements_met ? 'text-success' : 'text-warning' ?>">
                            <?= number_format($overall_course_mark, 1) ?>%
                            <?php if ($quiz_requirements_met): ?>
                                <i class="bx bx-check-circle text-success ms-1"></i>
                            <?php endif; ?>
                        </span>
                        <span class="grade-status">
                            (Passing Grade: <?= $certificate_eligibility['minimum_required'] ?>%)
                        </span>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Video Player -->
            <div class="video-container">
                <div class="video-player" id="videoPlayer">
                    <?php if (empty($lessons)): ?>
                        <!-- No lessons available -->
                        <div class="video-placeholder" style="width: 100%; height: 100%; background: linear-gradient(135deg, #1f2937 0%, #374151 100%); display: flex; align-items: center; justify-content: center; flex-direction: column; color: #9ca3af;">
                            <div style="text-align: center;">
                                <i class="bx bx-video-off" style="font-size: 4rem; margin-bottom: 1rem; color: #6b7280;"></i>
                                <h3 style="margin: 0 0 0.5rem 0; color: white;">No Content Available</h3>
                                <p style="margin: 0 0 0.5rem 0;">This course doesn't have any lessons or videos yet.</p>
                                <small>Please check back later or contact your instructor</small>
                            </div>
                        </div>
                    <?php elseif (!empty($current_lesson['video_url'])): ?>
                        <video id="lessonVideo" preload="metadata" 
                               style="width: 100%; height: 100%; background: #000;"
                               <?php if (!empty($current_lesson['thumbnail'])): ?>
                               poster="<?= htmlspecialchars($current_lesson['thumbnail']) ?>"
                               <?php endif; ?>>
                            <source src="<?= htmlspecialchars($current_lesson['video_url']) ?>" type="video/mp4">
                            Your browser does not support the video tag.
                        </video>
                    <?php else: ?>
                        <div class="video-placeholder" style="width: 100%; height: 100%; background: linear-gradient(135deg, #1f2937 0%, #374151 100%); display: flex; align-items: center; justify-content: center; flex-direction: column; color: #9ca3af; position: relative;">
                            <!-- Show thumbnail if available even when video is not available -->
                            <?php if (!empty($current_lesson['thumbnail'])): ?>
                                <img src="<?= htmlspecialchars($current_lesson['thumbnail']) ?>" 
                                     alt="<?= htmlspecialchars($current_lesson['title']) ?>"
                                     style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; opacity: 0.3;">
                            <?php endif; ?>
                            
                            <div style="position: relative; z-index: 1; text-align: center;">
                                <i class="bx bx-video" style="font-size: 4rem; margin-bottom: 1rem;"></i>
                                <h3 style="margin: 0 0 0.5rem 0; color: white;"><?= htmlspecialchars($current_lesson['title']) ?></h3>
                                <p style="margin: 0 0 0.5rem 0;">Video not available</p>
                                <small>Please contact your instructor</small>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="video-controls">
                    <div class="time-bar" onclick="seekVideo(event)">
                        <div class="time-progress" id="timeProgress"></div>
                    </div>
                    <div class="control-buttons">
                        <div class="control-left">
                            <button id="playPauseBtn" onclick="togglePlayPause()" class="control-btn">
                                <i class="bx bx-play"></i>
                            </button>
                            <button onclick="skipBackward()" class="control-btn" title="Skip back 10s">
                                <i class="bx bx-rewind"></i>
                            </button>
                            <button onclick="skipForward()" class="control-btn" title="Skip forward 10s">
                                <i class="bx bx-fast-forward"></i>
                            </button>
                            <div class="time-display">
                                <span id="currentTime">0:00</span> / <span id="totalTime"><?= $current_lesson['duration'] ?></span>
                            </div>
                        </div>
                        <div class="control-center">
                            <!-- Playback Speed -->
                            <div class="dropdown">
                                <button class="control-btn dropdown-toggle" onclick="toggleSpeedMenu()" title="Playback Speed">
                                    <span id="speedLabel">1x</span>
                                </button>
                                <div class="dropdown-menu" id="speedMenu">
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(0.5)">0.5x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(0.75)">0.75x</div>
                                    <div class="dropdown-item active" onclick="setPlaybackSpeed(1)">1x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(1.25)">1.25x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(1.5)">1.5x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(2)">2x</div>
                                </div>
                            </div>

                            <!-- Quality Settings -->
                            <div class="dropdown">
                                <button class="control-btn dropdown-toggle" onclick="toggleQualityMenu()" title="Video Quality">
                                    <i class="bx bx-cog"></i>
                                </button>
                                <div class="dropdown-menu" id="qualityMenu">
                                    <div class="dropdown-item active" onclick="setVideoQuality('auto')">Auto</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('1080p')">1080p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('720p')">720p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('480p')">480p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('360p')">360p</div>
                                </div>
                            </div>

                            <!-- Subtitles -->
                            <button class="control-btn" onclick="toggleSubtitles()" title="Subtitles" id="subtitleBtn">
                                <i class="bx bx-captions"></i>
                            </button>

                            <!-- Notes -->
                            <button class="control-btn" onclick="addNote()" title="Add Note at Current Time">
                                <i class="bx bx-note"></i>
                            </button>
                        </div>
                        <div class="control-right">
                            <!-- Volume Control -->
                            <div class="volume-control">
                                <button onclick="toggleMute()" class="control-btn">
                                    <i class="bx bx-volume-full" id="volumeIcon"></i>
                                </button>
                                <input type="range" id="volumeSlider" class="volume-slider" min="0" max="100" value="100" onchange="setVolume(this.value)">
                            </div>
                            
                            <!-- Picture in Picture -->
                            <button onclick="togglePictureInPicture()" class="control-btn" title="Picture in Picture">
                                <i class="bx bx-dock-top"></i>
                            </button>
                            
                            <!-- Fullscreen -->
                            <button onclick="toggleFullscreen()" class="control-btn">
                                <i class="bx bx-fullscreen"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Course Content Tabs -->
            <div class="course-tabs-container">
                <!-- Tab Navigation -->
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="description">
                        <i class="bx bx-book-open"></i>
                        Course Description
                    </button>
                    <button class="tab-btn" data-tab="reviews">
                        <i class="bx bx-star"></i>
                        Reviews
                    </button>
                    <button class="tab-btn" data-tab="notes">
                        <i class="bx bx-note"></i>
                        Notes
                    </button>
                    <button class="tab-btn" data-tab="certificate">
                        <i class="bx bx-award"></i>
                        Certificate
                    </button>
                    <button class="tab-btn" data-tab="tools">
                        <i class="bx bx-wrench"></i>
                        Learning Tools
                    </button>
                </div>

                <!-- Tab Content -->
                <div class="tab-content">
                    <!-- Enhanced Course Description Tab -->
                    <div class="tab-pane active" id="description">
                        <div class="enhanced-course-description">
                            
                            <!-- Course Header Section -->
                            <div class="course-header-section">
                                <div class="row g-4">
                                    <div class="col-lg-8">
                                        <div class="course-title-section">
                                            <h2 class="course-main-title"><?= htmlspecialchars($course['title']) ?></h2>
                                            <p class="course-subtitle"><?= htmlspecialchars($course['description']) ?></p>
                                        </div>
                                        
                                        <!-- Course Meta Information -->
                                        <div class="course-meta-grid">
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-play-circle"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value"><?= count($lessons) ?></div>
                                                    <div class="meta-label">Video Lessons</div>
                                                </div>
                                            </div>
                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-group"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value"><?= $enrollment_count ?></div>
                                                    <div class="meta-label">Students Enrolled</div>
                                                </div>
                                            </div>
                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-time"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value">
                                                        <?php 
                                                        // Calculate total course duration
                                                        $total_minutes = 0;
                                                        foreach ($lessons as $lesson) {
                                                            if (!empty($lesson['duration'])) {
                                                                // Parse duration (assumes format like "5:30" or "10:15")
                                                                $time_parts = explode(':', $lesson['duration']);
                                                                if (count($time_parts) == 2) {
                                                                    $total_minutes += intval($time_parts[0]) * 60 + intval($time_parts[1]);
                                                                }
                                                            }
                                                        }
                                                        $hours = floor($total_minutes / 60);
                                                        $minutes = $total_minutes % 60;
                                                        echo $hours > 0 ? $hours . 'h ' . $minutes . 'm' : $minutes . ' mins';
                                                        ?>
                                                    </div>
                                                    <div class="meta-label">Total Duration</div>
                                                </div>
                                            </div>
                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-medal"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value">
                                                        <?php
                                                        // Get course level/difficulty
                                                        $course_level = !empty($course['level']) ? htmlspecialchars($course['level']) : 'Beginner';
                                                        echo $course_level;
                                                        ?>
                                                    </div>
                                                    <div class="meta-label">Difficulty Level</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Instructor Information -->
                                    <div class="col-lg-4">
                                        <div class="instructor-card">
                                            <div class="instructor-header">
                                                <h5><i class="bx bx-user-circle"></i> Your Instructor</h5>
                                            </div>
                                            <div class="instructor-content">
                                                <div class="instructor-avatar">
                                                    <?php if (!empty($course['instructor_profile_image'])): ?>
                                                        <img src="<?= htmlspecialchars($course['instructor_profile_image']) ?>" alt="Instructor">
                                                    <?php else: ?>
                                                        <div class="avatar-placeholder">
                                                            <i class="bx bx-user"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="instructor-details">
                                                    <h6 class="instructor-name"><?= htmlspecialchars($instructor_name) ?></h6>
                                                    <p class="instructor-title">
                                                        <?php
                                                        // Get instructor title/bio from database
                                                        $instructor_title = "Course Instructor";
                                                        if (isset($conn)) {
                                                            $instructor_query = "SELECT bio, title FROM users WHERE user_id = ?";
                                                            if ($stmt = $conn->prepare($instructor_query)) {
                                                                $stmt->bind_param("i", $course['teacher_id']);
                                                                $stmt->execute();
                                                                $result = $stmt->get_result();
                                                                if ($instructor_info = $result->fetch_assoc()) {
                                                                    $instructor_title = !empty($instructor_info['title']) ? htmlspecialchars($instructor_info['title']) : $instructor_title;
                                                                }
                                                                $stmt->close();
                                                            }
                                                        }
                                                        echo $instructor_title;
                                                        ?>
                                                    </p>
                                                    <div class="instructor-stats">
                                                        <div class="stat-item">
                                                            <i class="bx bx-book"></i>
                                                            <span>
                                                                <?php
                                                                // Get instructor's total courses
                                                                $instructor_courses = 0;
                                                                if (isset($conn)) {
                                                                    $courses_query = "SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?";
                                                                    if ($stmt = $conn->prepare($courses_query)) {
                                                                        $stmt->bind_param("i", $course['teacher_id']);
                                                                        $stmt->execute();
                                                                        $result = $stmt->get_result();
                                                                        if ($courses_data = $result->fetch_assoc()) {
                                                                            $instructor_courses = $courses_data['count'];
                                                                        }
                                                                        $stmt->close();
                                                                    }
                                                                }
                                                                echo $instructor_courses . ' Courses';
                                                                ?>
                                                            </span>
                                                        </div>
                                                        <div class="stat-item">
                                                            <i class="bx bx-group"></i>
                                                            <span>
                                                                <?php
                                                                // Get instructor's total students
                                                                $instructor_students = 0;
                                                                if (isset($conn)) {
                                                                    $students_query = "SELECT COUNT(DISTINCT e.student_id) as count FROM enrollments e JOIN courses c ON e.course_id = c.course_id WHERE c.teacher_id = ?";
                                                                    if ($stmt = $conn->prepare($students_query)) {
                                                                        $stmt->bind_param("i", $course['teacher_id']);
                                                                        $stmt->execute();
                                                                        $result = $stmt->get_result();
                                                                        if ($students_data = $result->fetch_assoc()) {
                                                                            $instructor_students = $students_data['count'];
                                                                        }
                                                                        $stmt->close();
                                                                    }
                                                                }
                                                                echo $instructor_students . ' Students';
                                                                ?>
                                                            </span>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Learning Objectives Section -->
                            <div class="learning-objectives-section">
                                <h4><i class="bx bx-target-lock"></i> What You'll Learn</h4>
                                <div class="objectives-grid">
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Master the fundamental concepts and principles</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Apply practical skills through hands-on exercises</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Build confidence in the subject matter</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Complete interactive quizzes and assessments</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Gain industry-relevant knowledge and expertise</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Develop problem-solving and critical thinking skills</span>
                                    </div>
                                </div>
                            </div>

                            <!-- Course Requirements Section -->
                            <div class="course-requirements-section">
                                <div class="row g-4">
                                    <div class="col-md-6">
                                        <div class="requirements-card">
                                            <h5><i class="bx bx-list-check"></i> Prerequisites</h5>
                                            <ul class="requirements-list">
                                                <li><i class="bx bx-check"></i> Basic computer literacy</li>
                                                <li><i class="bx bx-check"></i> Internet connection for online content</li>
                                                <li><i class="bx bx-check"></i> Motivation to learn and practice</li>
                                                <li><i class="bx bx-check"></i> No prior experience required</li>
                                            </ul>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="course-info-card">
                                            <h5><i class="bx bx-info-circle"></i> Course Information</h5>
                                            <div class="info-list">
                                                <div class="info-item">
                                                    <span class="info-label">Created:</span>
                                                    <span class="info-value">
                                                        <?= date('M d, Y', strtotime($course['created_at'])) ?>
                                                    </span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Last Updated:</span>
                                                    <span class="info-value">
                                                        <?= date('M d, Y', strtotime($course['updated_at'] ?? $course['created_at'])) ?>
                                                    </span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Language:</span>
                                                    <span class="info-value">English</span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Access:</span>
                                                    <span class="info-value">Lifetime Access</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Current Lesson Information (if viewing a specific lesson) -->
                            <?php if (!empty($current_lesson)): ?>
                            <div class="current-lesson-section">
                                <h4><i class="bx bx-play"></i> Current Lesson</h4>
                                <div class="lesson-info-card">
                                    <div class="lesson-content">
                                        <h5 class="lesson-title"><?= htmlspecialchars($current_lesson['title']) ?></h5>
                                        <p class="lesson-description"><?= htmlspecialchars($current_lesson['description']) ?></p>
                                        <div class="lesson-meta-info">
                                            <div class="meta-item">
                                                <i class="bx bx-time"></i>
                                                <span>Duration: <?= $current_lesson['duration'] ?></span>
                                            </div>
                                            <div class="meta-item">
                                                <i class="bx bx-check-circle"></i>
                                                <span><?= $current_lesson['completed'] ? 'Completed' : 'In Progress' ?></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Enhanced Reviews Tab -->
                    <div class="tab-pane" id="reviews">
                        <div class="enhanced-reviews-content">
                            
                            <!-- Reviews Summary Header -->
                            <div class="reviews-summary-section">
                                <div class="row g-4">
                                    <div class="col-lg-4">
                                        <div class="overall-rating-card">
                                            <div class="rating-display">
                                                <div class="rating-number"><?= $review_stats['average_rating'] > 0 ? number_format($review_stats['average_rating'], 1) : 'New' ?></div>
                                                <div class="rating-stars-large">
                                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                                        <i class='bx <?= $i <= $review_stats['average_rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                    <?php endfor; ?>
                                                </div>
                                                <div class="rating-text"><?= $review_stats['total_reviews'] ?> student review<?= $review_stats['total_reviews'] != 1 ? 's' : '' ?></div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-lg-8">
                                        <div class="rating-breakdown-card">
                                            <h6 class="breakdown-title">Rating Breakdown</h6>
                                            <?php for ($star = 5; $star >= 1; $star--): ?>
                                                <?php 
                                                $count = $review_stats['rating_breakdown'][$star];
                                                $percentage = $review_stats['total_reviews'] > 0 ? ($count / $review_stats['total_reviews']) * 100 : 0;
                                                ?>
                                                <div class="rating-breakdown-row">
                                                    <span class="star-label"><?= $star ?> stars</span>
                                                    <div class="rating-bar">
                                                        <div class="rating-bar-fill" style="width: <?= $percentage ?>%"></div>
                                                    </div>
                                                    <span class="rating-count"><?= $count ?></span>
                                                </div>
                                            <?php endfor; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Write Review Section -->
                            <?php if (isset($_SESSION['user_id']) && $user_enrolled && !$user_has_reviewed): ?>
                            <div class="write-review-section">
                                <div class="write-review-card">
                                    <div class="write-review-header">
                                        <h5><i class="bx bx-edit"></i> Share Your Experience</h5>
                                        <p>Help other students by sharing your experience with this course</p>
                                    </div>
                                    
                                    <form id="reviewForm" class="review-form">
                                        <input type="hidden" name="course_id" value="<?= encryptId($course_id) ?>">
                                        <input type="hidden" name="user_id" value="<?= $_SESSION['user_id'] ?? '' ?>">
                                        <input type="hidden" name="user_token" value="<?= md5(($_SESSION['user_id'] ?? '') . 'panadite_review_token') ?>">
                                        
                                        <div class="form-group">
                                            <label class="form-label">Your Rating</label>
                                            <div class="star-rating-input">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i class="bx bx-star rating-star" data-rating="<?= $i ?>"></i>
                                                <?php endfor; ?>
                                            </div>
                                            <input type="hidden" name="rating" id="selectedRating" required>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="reviewTitle" class="form-label">Review Title <small class="text-muted">(minimum 3 characters)</small></label>
                                            <input type="text" class="form-control" id="reviewTitle" name="review_title" 
                                                   placeholder="Summarize your experience... (minimum 3 characters required)" 
                                                   minlength="3" maxlength="255" required>
                                            <div class="form-text">
                                                <span id="titleCharCount">0</span>/3 characters minimum
                                            </div>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="reviewText" class="form-label">Your Review <small class="text-muted">(minimum 10 characters)</small></label>
                                            <textarea class="form-control" id="reviewText" name="review_text" rows="4" 
                                                      placeholder="Share details about your experience with this course... (minimum 10 characters required)" 
                                                      minlength="10" required></textarea>
                                            <div class="form-text">
                                                <span id="charCount">0</span>/10 characters minimum
                                            </div>
                                        </div>
                                        
                                        <div class="form-actions">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bx bx-send"></i> Submit Review
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary" onclick="cancelReview()">
                                                Cancel
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            <?php elseif (isset($_SESSION['user_id']) && $user_has_reviewed): ?>
                            <div class="user-review-section">
                                <div class="user-review-card">
                                    <div class="user-review-header">
                                        <h6><i class="bx bx-check-circle"></i> Your Review</h6>
                                        <div class="review-actions">
                                            <button class="btn btn-sm btn-outline-primary" onclick="editReview(<?= $user_review['review_id'] ?>, '<?= htmlspecialchars($user_review['review_title'], ENT_QUOTES) ?>', '<?= htmlspecialchars($user_review['review_text'], ENT_QUOTES) ?>', <?= $user_review['rating'] ?>)">
                                                <i class="bx bx-edit"></i> Edit
                                            </button>
                                        </div>
                                    </div>
                                    <div class="user-review-content">
                                        <div class="review-rating">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <i class='bx <?= $i <= $user_review['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                            <?php endfor; ?>
                                        </div>
                                        <h6 class="review-title"><?= htmlspecialchars($user_review['review_title']) ?></h6>
                                        <p class="review-text"><?= htmlspecialchars($user_review['review_text']) ?></p>
                                        <small class="review-date">Reviewed on <?= date('M d, Y', strtotime($user_review['created_at'])) ?></small>
                                    </div>
                                </div>
                            </div>
                            <?php elseif (!isset($_SESSION['user_id'])): ?>
                            <div class="login-prompt-section">
                                <div class="login-prompt-card">
                                    <i class="bx bx-user-circle" style="font-size: 2.5rem; color: #6b7280; margin-bottom: 1rem;"></i>
                                    <h6>Sign in to leave a review</h6>
                                    <p>Share your experience and help other students</p>
                                    <a href="../login.php" class="btn btn-primary btn-sm">
                                        <i class="bx bx-log-in"></i> Sign In
                                    </a>
                                </div>
                            </div>
                            <?php elseif (!$user_enrolled): ?>
                            <div class="enrollment-prompt-section">
                                <div class="enrollment-prompt-card">
                                    <i class="bx bx-lock-alt" style="font-size: 2.5rem; color: #6b7280; margin-bottom: 1rem;"></i>
                                    <h6>Enroll to leave a review</h6>
                                    <p>Only enrolled students can leave reviews</p>
                                </div>
                            </div>
                            <?php endif; ?>

                            <!-- Reviews List -->
                            <div class="reviews-list-section">
                                <div class="reviews-list-header">
                                    <h5><i class="bx bx-message-square-detail"></i> Student Reviews</h5>
                                    <?php if ($review_stats['total_reviews'] > 0): ?>
                                        <div class="dropdown">
                                            <button class="btn btn-outline-secondary dropdown-toggle" type="button" id="reviewSortDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                                                Newest First
                                            </button>
                                            <ul class="dropdown-menu" aria-labelledby="reviewSortDropdown">
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('newest')">Newest First</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('oldest')">Oldest First</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('highest')">Highest Rated</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('lowest')">Lowest Rated</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('helpful')">Most Helpful</a></li>
                                            </ul>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <?php if (empty($reviews)): ?>
                                <div class="no-reviews-placeholder">
                                    <div class="no-reviews-icon">
                                        <i class="bx bx-message-square-dots"></i>
                                    </div>
                                    <h6>No Reviews Yet</h6>
                                    <p>Be the first to share your experience with this course!</p>
                                    <?php if (isset($_SESSION['user_id']) && $user_enrolled && !$user_has_reviewed): ?>
                                        <button class="btn btn-primary btn-sm" onclick="showReviewForm()">
                                            <i class="bx bx-plus"></i> Write First Review
                                        </button>
                                    <?php endif; ?>
                                </div>
                                <?php else: ?>
                                <div class="reviews-list" id="reviewsList">
                                    <?php foreach ($reviews as $review): ?>
                                    <div class="review-item-enhanced">
                                        <div class="review-avatar">
                                            <?php if (!empty($review['profile_picture'])): ?>
                                                <img src="<?= htmlspecialchars($review['profile_picture']) ?>" alt="Reviewer">
                                            <?php else: ?>
                                                <div class="avatar-placeholder">
                                                    <i class="bx bx-user"></i>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="review-content">
                                            <div class="review-header">
                                                <div class="reviewer-info">
                                                    <h6 class="reviewer-name"><?= htmlspecialchars($review['first_name'] . ' ' . substr($review['last_name'], 0, 1) . '.') ?></h6>
                                                    <div class="review-meta">
                                                        <div class="review-rating">
                                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                                <i class='bx <?= $i <= $review['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                            <?php endfor; ?>
                                                        </div>
                                                        <span class="review-date"><?= date('M d, Y', strtotime($review['created_at'])) ?></span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="review-body">
                                                <h6 class="review-title"><?= htmlspecialchars($review['review_title']) ?></h6>
                                                <p class="review-text"><?= nl2br(htmlspecialchars($review['review_text'])) ?></p>
                                            </div>
                                            
                                            <div class="review-actions">
                                                <button class="review-action-btn helpful-btn" data-review-id="<?= encryptId($review['review_id']) ?>">
                                                    <i class="bx bx-like"></i>
                                                    <span class="helpful-count">Helpful (<?= $review['helpful_votes'] ?>)</span>
                                                </button>
                                                <?php if ($review['reply_count'] > 0): ?>
                                                    <button class="review-action-btn" onclick="showReplies('<?= encryptId($review['review_id']) ?>')">
                                                        <i class="bx bx-message"></i>
                                                        <span>Instructor Reply</span>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <?php if (count($reviews) >= 20): ?>
                                <div class="load-more-section">
                                    <button class="btn btn-outline-primary" onclick="loadMoreReviews()">
                                        <i class="bx bx-download"></i> Load More Reviews
                                    </button>
                                </div>
                                <?php endif; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- SITA-Compliant Accredited Certificate Tab -->
                    <div class="tab-pane" id="certificate">
                        <?php 
                        // Include accreditation logic
                        include_once 'includes/accreditation_logic.php';
                        ?>
                        
                        <div class="accreditation-content">
                                <div class="accreditation-header border rounded mb-4 p-4" style="border-color: #e2e8f0;">
                                    <div class="row align-items-center">
                                        <div class="col-md-8">
                                            <h3 class="mb-2 text-dark"><i class="bx bx-award me-2 text-primary"></i> SITA-Accredited Professional Certificate</h3>
                                            <p class="mb-0 text-muted">Request an official, industry-recognized accredited certificate from Panadite Academy</p>
                                        </div>
                                        <div class="col-md-4 text-md-end">
                                            <div class="accreditation-badge bg-white rounded p-3 text-center" style="border: 1px solid #e2e8f0;">
                                                <img src="../assets/images/sita-logo.png" alt="SITA Accredited" class="sita-logo mb-2" style="height: 40px;" onerror="this.style.display='none'">
                                                <div class="badge bg-primary">SITA Compliant</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                            <?php if (!$accreditation_request): ?>
                            <!-- Initial Request Stage -->
                            <div class="accreditation-stage" id="initialRequest">
                                <!-- SITA Information Accordion -->
                                <div class="accordion" id="sitaInfoAccordion">
                                    <!-- Requirements Overview -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingRequirements">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseRequirements" aria-expanded="false" aria-controls="collapseRequirements">
                                                <i class="bx bx-task text-primary me-2"></i>
                                                <strong>Certificate Requirements</strong>
                                                <small class="text-muted ms-2">(<?= $videos_completed && $quiz_requirements_met ? 'Ready' : 'In Progress' ?>)</small>
                                            </button>
                                        </h2>
                                        <div id="collapseRequirements" class="accordion-collapse collapse" aria-labelledby="headingRequirements" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <div class="requirements-grid">
                                                    <div class="requirement-card <?= $videos_completed ? 'completed' : '' ?>">
                                                        <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Complete all course videos (<?= $course_progress_percentage ?>% done)</span>
                                                    </div>
                                                    <div class="requirement-card <?= $quiz_requirements_met ? 'completed' : '' ?>">
                                                        <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Achieve passing grade in quizzes (<?= number_format($overall_course_mark, 1) ?>% / <?= $certificate_eligibility['minimum_required'] ?>%)</span>
                                                    </div>
                                                    <div class="requirement-card <?= $is_enrolled ? 'completed' : '' ?>">
                                                        <i class="bx <?= $is_enrolled ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Active course enrollment</span>
                                                    </div>
                                                    <div class="requirement-card">
                                                        <i class="bx bx-circle text-muted me-2"></i>
                                                        <span>Pass SITA assessment (minimum 70%)</span>
                                                    </div>
                                                    <div class="requirement-card">
                                                        <i class="bx bx-circle text-muted me-2"></i>
                                                        <span>Submit required documents</span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- About SITA Accreditation -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingAbout">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseAbout" aria-expanded="false" aria-controls="collapseAbout">
                                                <i class="bx bx-shield-check text-primary me-2"></i>
                                                <strong>About SITA Accreditation</strong>
                                            </button>
                                        </h2>
                                        <div id="collapseAbout" class="accordion-collapse collapse" aria-labelledby="headingAbout" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <p class="text-muted">SITA (Skills and Industry Training Authority) accreditation ensures your certificate meets professional industry standards and is recognized by employers worldwide. This certification carries significant weight in the professional market and demonstrates your commitment to excellence.</p>
                                                
                                                <div class="alert alert-info">
                                                    <div class="d-flex align-items-center">
                                                        <i class="bx bx-time text-info me-2" style="font-size: 1.5rem;"></i>
                                                        <div>
                                                            <h6 class="alert-heading mb-1">Processing Timeline</h6>
                                                            <p class="mb-0"><strong>Total Processing Time:</strong> Up to 5 months from initial request submission</p>
                                                            <small class="text-muted">SITA accreditation is a comprehensive process that ensures the highest professional standards.</small>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Process Overview -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingProcess">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseProcess" aria-expanded="false" aria-controls="collapseProcess">
                                                <i class="bx bx-list-check text-primary me-2"></i>
                                                <strong>6-Step Process Overview</strong>
                                            </button>
                                        </h2>
                                        <div id="collapseProcess" class="accordion-collapse collapse" aria-labelledby="headingProcess" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <div class="process-steps">
                                                    <div class="step-item">
                                                        <span class="step-number">1</span>
                                                        <div class="step-content">
                                                            <strong>Request Submission:</strong> Submit your accreditation request
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">2</span>
                                                        <div class="step-content">
                                                            <strong>Admin Review:</strong> Our team reviews your eligibility
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">3</span>
                                                        <div class="step-content">
                                                            <strong>SITA Assessment:</strong> Complete a comprehensive final assessment
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">4</span>
                                                        <div class="step-content">
                                                            <strong>Document Submission:</strong> Upload CV, qualifications, ID, and assignment work
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">5</span>
                                                        <div class="step-content">
                                                            <strong>Final Review:</strong> Admin team validates all requirements
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">6</span>
                                                        <div class="step-content">
                                                            <strong>Certificate Issuance:</strong> Receive your official accredited certificate
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <?php if ($full_course_completed && $is_enrolled): ?>
                                <div class="card shadow-sm mt-4">
                                    <div class="card-header bg-primary text-white">
                                        <h5 class="card-title mb-0"><i class="bx bx-send me-2"></i> Request Accredited Certificate</h5>
                                    </div>
                                    <div class="card-body p-4">
                                        <form id="accreditationRequestForm">
                                            <input type="hidden" name="course_id" value="<?= $course_id ?>">
                                            <input type="hidden" name="student_id" value="<?= $user_id ?>">
                                            
                                            <div class="mb-3">
                                                <label class="form-label">Course</label>
                                                <input type="text" class="form-control" value="<?= htmlspecialchars($course['title']) ?>" readonly>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="motivationText" class="form-label">Motivation Statement <small class="text-muted">(Optional)</small></label>
                                                <textarea class="form-control" id="motivationText" name="motivation" rows="4" placeholder="Briefly explain why you want this accredited certificate and how it will benefit your career..."></textarea>
                                            </div>
                                            
                                            <div class="alert alert-warning mb-3">
                                                <h6 class="alert-heading"><i class="bx bx-info-circle me-2"></i>Important Notice</h6>
                                                <p class="mb-2">By submitting this request, you acknowledge:</p>
                                                <ul class="mb-0">
                                                    <li>This is a comprehensive accreditation process taking up to 5 months</li>
                                                    <li>You will need to complete a SITA assessment and submit professional documents</li>
                                                    <li>The certificate fee and assessment requirements will be communicated after approval</li>
                                                </ul>
                                            </div>
                                            
                                            <div class="form-check mb-4">
                                                <input class="form-check-input" type="checkbox" id="agreeTerms" required>
                                                <label class="form-check-label fw-bold" for="agreeTerms">
                                                    I understand and agree to the accreditation process requirements and timeline.
                                                </label>
                                            </div>
                                            
                                            <div class="d-grid">
                                                <button type="submit" class="btn btn-primary btn-lg">
                                                    <i class="bx bx-send"></i> Submit Accreditation Request
                                                </button>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div class="card shadow-sm mt-4">
                                    <div class="card-body text-center py-5">
                                        <div class="mb-3">
                                            <i class="bx bx-lock-alt text-warning" style="font-size: 4rem;"></i>
                                        </div>
                                        <h5 class="text-warning mb-3">Complete All Course Requirements</h5>
                                        <p class="text-muted mb-4">You must complete all videos AND achieve the passing grade in quizzes to earn your certificate.</p>
                                        
                                        <!-- Comprehensive Requirements Progress -->
                                        <div class="requirements-progress mb-4">
                                            <div class="row text-start">
                                                <div class="col-md-6">
                                                    <div class="requirement-item mb-3">
                                                        <div class="d-flex align-items-center">
                                                            <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                            <div>
                                                                <strong>Video Completion:</strong>
                                                                <div class="progress mt-1" style="height: 8px; background: #e3f2fd; border-radius: 6px;">
                                                                    <div class="progress-bar" role="progressbar" style="width: <?= $course_progress_percentage ?>%; background: #2196f3; border-radius: 6px; transition: width 0.3s ease;" id="certificate-video-progress-bar">
                                                                    </div>
                                                                </div>
                                                                <small class="text-muted" id="certificate-video-progress-text"><?= $course_progress_percentage ?>% videos watched</small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="requirement-item mb-3">
                                                        <div class="d-flex align-items-center">
                                                            <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                            <div>
                                                                <strong>Quiz Performance:</strong>
                                                                <div class="progress mt-1" style="height: 8px;">
                                                                    <div class="progress-bar <?= $quiz_requirements_met ? 'bg-success' : 'bg-warning' ?>" role="progressbar" style="width: <?= min(100, ($overall_course_mark / $certificate_eligibility['minimum_required']) * 100) ?>%">
                                                                    </div>
                                                                </div>
                                                                <small class="text-muted">
                                                                    <?= number_format($overall_course_mark, 1) ?>% / <?= $certificate_eligibility['minimum_required'] ?>% required
                                                                </small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Overall Course Grade Display -->
                                        <div class="overall-grade-display mb-4">
                                            <div class="card bg-light">
                                                <div class="card-body py-3">
                                                    <h6 class="mb-1">Overall Course Grade</h6>
                                                    <div class="grade-display">
                                                        <span class="h4 <?= $quiz_requirements_met ? 'text-success' : 'text-warning' ?>">
                                                            <?= number_format($overall_course_mark, 1) ?>%
                                                        </span>
                                                        <small class="text-muted ms-2">
                                                            (Passing: <?= $certificate_eligibility['minimum_required'] ?>%)
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Next Steps -->
                                        <div class="next-steps">
                                            <?php if (!$videos_completed && !$quiz_requirements_met): ?>
                                                <p class="text-info mb-2"><strong>Next Steps:</strong> Complete all videos and improve your quiz scores</p>
                                                <div class="d-flex gap-2 justify-content-center">
                                                    <a href="#lessons" class="btn btn-outline-primary btn-sm">Continue Videos</a>
                                                    <a href="#quizzes" class="btn btn-outline-success btn-sm">Take Quizzes</a>
                                                </div>
                                            <?php elseif (!$videos_completed): ?>
                                                <p class="text-info mb-2"><strong>Next Step:</strong> Complete all remaining videos</p>
                                                <a href="#lessons" class="btn btn-outline-primary">Continue Learning</a>
                                            <?php elseif (!$quiz_requirements_met): ?>
                                                <p class="text-info mb-2"><strong>Next Step:</strong> Improve your quiz performance</p>
                                                <a href="#quizzes" class="btn btn-outline-success">Take Quizzes</a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php else: ?>
                            <!-- Request Status and Progress -->
                            <div class="accreditation-progress">
                                <div class="status-header mb-4">
                                    <div class="row align-items-center">
                                        <div class="col-md-8">
                                            <h5>Your Accreditation Request</h5>
                                            <p class="mb-0">Request ID: #<?= str_pad($accreditation_request['request_id'], 6, '0', STR_PAD_LEFT) ?></p>
                                        </div>
                                        <div class="col-md-4 text-md-end">
                                            <?= getStatusBadge($accreditation_request['status']) ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="status-timeline mb-4">
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['pending', 'approved', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : 'active' ?>">
                                        <div class="timeline-marker"><i class="bx bx-send"></i></div>
                                        <div class="timeline-content">
                                            <h6>Request Submitted</h6>
                                            <small><?= date('M d, Y \a\t H:i', strtotime($accreditation_request['request_date'])) ?></small>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['approved', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : ($accreditation_request['status'] == 'pending' ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-check-circle"></i></div>
                                        <div class="timeline-content">
                                            <h6>Admin Approval</h6>
                                            <?php if ($accreditation_request['approved_date']): ?>
                                                <small><?= date('M d, Y \a\t H:i', strtotime($accreditation_request['approved_date'])) ?></small>
                                            <?php else: ?>
                                                <small>Pending review</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['assessment_created']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-brain"></i></div>
                                        <div class="timeline-content">
                                            <h6>SITA Assessment</h6>
                                            <?php if ($assessment_attempt && $assessment_attempt['end_time']): ?>
                                                <small>Completed: <?= $assessment_attempt['percentage_score'] ?>%</small>
                                            <?php else: ?>
                                                <small>Awaiting completion</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['documents_pending']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-folder-plus"></i></div>
                                        <div class="timeline-content">
                                            <h6>Document Submission</h6>
                                            <small><?= count($submitted_documents) ?> document(s) uploaded</small>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['under_review', 'approved_final']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-award"></i></div>
                                        <div class="timeline-content">
                                            <h6>Certificate Issued</h6>
                                            <?php if ($issued_certificate): ?>
                                                <small><?= date('M d, Y', strtotime($issued_certificate['issue_date'])) ?></small>
                                            <?php else: ?>
                                                <small>Pending</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="current-status-card mb-4">
                                    <div class="card-body">
                                        <h6><i class="bx bx-info-circle"></i> Current Status</h6>
                                        <p><?= getStatusDescription($accreditation_request['status']) ?></p>
                                        
                                        <?php if ($accreditation_request['admin_notes']): ?>
                                        <div class="admin-notes">
                                            <h6><i class="bx bx-message-detail"></i> Admin Notes</h6>
                                            <p class="text-primary"><?= htmlspecialchars($accreditation_request['admin_notes']) ?></p>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="next-steps">
                                            <h6><i class="bx bx-navigation"></i> Next Steps</h6>
                                            <p><?= getNextSteps($accreditation_request['status'], $assessment_attempt) ?></p>
                                        </div>
                                    </div>
                                </div>

                                <!-- SITA Assessment Interface -->
                                <?php if ($accreditation_request['status'] == 'assessment_created' && $current_assessment): ?>
                                <div class="assessment-interface-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-brain"></i> SITA Assessment - <?= htmlspecialchars($current_assessment['title']) ?></h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="assessment-info mb-4">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="info-item">
                                                        <i class="bx bx-time"></i>
                                                        <span><strong>Time Limit:</strong> <?= $current_assessment['time_limit_minutes'] ?> minutes</span>
                                                    </div>
                                                    <div class="info-item">
                                                        <i class="bx bx-target-lock"></i>
                                                        <span><strong>Pass Mark:</strong> <?= $current_assessment['pass_mark_percentage'] ?>%</span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="info-item">
                                                        <i class="bx bx-refresh"></i>
                                                        <span><strong>Attempts Allowed:</strong> <?= $current_assessment['max_attempts'] ?></span>
                                                    </div>
                                                    <?php if ($current_assessment['deadline']): ?>
                                                    <div class="info-item">
                                                        <i class="bx bx-calendar"></i>
                                                        <span><strong>Deadline:</strong> <?= date('M d, Y', strtotime($current_assessment['deadline'])) ?></span>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <?php if ($current_assessment['instructions']): ?>
                                        <div class="assessment-instructions mb-4">
                                            <h6><i class="bx bx-info-circle"></i> Instructions</h6>
                                            <div class="instructions-content">
                                                <?= nl2br(htmlspecialchars($current_assessment['instructions'])) ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($assessment_attempt && $assessment_attempt['status'] == 'completed'): ?>
                                        <div class="assessment-result mb-4">
                                            <div class="result-card <?= $assessment_attempt['passed'] ? 'success' : 'warning' ?>">
                                                <div class="result-header">
                                                    <i class="bx <?= $assessment_attempt['passed'] ? 'bx-check-circle' : 'bx-x-circle' ?>"></i>
                                                    <h6><?= $assessment_attempt['passed'] ? 'Assessment Passed!' : 'Assessment Not Passed' ?></h6>
                                                </div>
                                                <div class="result-details">
                                                    <div class="score-display">
                                                        <span class="score-number"><?= $assessment_attempt['percentage_score'] ?>%</span>
                                                        <span class="score-total">Score: <?= $assessment_attempt['total_score'] ?> points</span>
                                                    </div>
                                                    <div class="attempt-info">
                                                        <small>Completed: <?= date('M d, Y \a\t H:i', strtotime($assessment_attempt['end_time'])) ?></small>
                                                        <small>Time Taken: <?= $assessment_attempt['time_taken_minutes'] ?> minutes</small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="assessment-actions">
                                            <?php if (!$assessment_attempt): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>" class="btn btn-primary btn-lg">
                                                <i class="bx bx-play-circle"></i> Start SITA Assessment
                                            </a>
                                            <?php elseif ($assessment_attempt['status'] == 'in_progress'): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>&resume=<?= $assessment_attempt['attempt_id'] ?>" class="btn btn-warning btn-lg">
                                                <i class="bx bx-play"></i> Resume Assessment
                                            </a>
                                            <?php elseif (!$assessment_attempt['passed'] && $assessment_attempt['attempt_number'] < $current_assessment['max_attempts']): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>" class="btn btn-secondary btn-lg">
                                                <i class="bx bx-refresh"></i> Retake Assessment (Attempt <?= $assessment_attempt['attempt_number'] + 1 ?>)
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Document Upload Interface -->
                                <?php if ($accreditation_request['status'] == 'documents_pending' || $accreditation_request['status'] == 'documents_submitted'): ?>
                                <div class="document-upload-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-folder-plus"></i> Required Document Submission</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="document-requirements mb-4">
                                            <h6><i class="bx bx-list-check"></i> Required Documents</h6>
                                            <div class="requirements-grid">
                                                <?php 
                                                $required_docs = [
                                                    'cv' => ['name' => 'Curriculum Vitae (CV)', 'icon' => 'bx-user', 'formats' => 'PDF, DOC, DOCX'],
                                                    'qualifications' => ['name' => 'Educational Qualifications', 'icon' => 'bx-certification', 'formats' => 'PDF, JPG, PNG'],
                                                    'id_document' => ['name' => 'Government ID Document', 'icon' => 'bx-id-card', 'formats' => 'PDF, JPG, PNG'],
                                                    'assignment' => ['name' => 'Course Assignment Work', 'icon' => 'bx-file', 'formats' => 'PDF, DOC, DOCX, ZIP']
                                                ];
                                                
                                                foreach ($required_docs as $doc_type => $doc_info):
                                                    $uploaded = array_filter($submitted_documents, function($doc) use ($doc_type) {
                                                        return $doc['document_type'] == $doc_type;
                                                    });
                                                    $has_upload = !empty($uploaded);
                                                ?>
                                                <div class="requirement-card <?= $has_upload ? 'completed' : 'pending' ?>">
                                                    <div class="requirement-header">
                                                        <i class="bx <?= $doc_info['icon'] ?> <?= $has_upload ? 'text-success' : 'text-muted' ?>"></i>
                                                        <h6><?= $doc_info['name'] ?></h6>
                                                        <?php if ($has_upload): ?>
                                                        <span class="badge bg-success">✓ Uploaded</span>
                                                        <?php endif; ?>
                                                    </div>
                                                    <p class="requirement-formats">Accepted: <?= $doc_info['formats'] ?></p>
                                                    <?php if ($has_upload): ?>
                                                    <div class="uploaded-files">
                                                        <?php foreach ($uploaded as $doc): ?>
                                                        <div class="uploaded-file">
                                                            <i class="bx bx-file"></i>
                                                            <span><?= htmlspecialchars($doc['document_name']) ?></span>
                                                            <small class="verification-status <?= $doc['verification_status'] ?>">
                                                                <?= ucfirst($doc['verification_status']) ?>
                                                            </small>
                                                        </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>
                                        
                                        <?php if ($accreditation_request['status'] == 'documents_pending'): ?>
                                        <form id="documentUploadForm" enctype="multipart/form-data">
                                            <input type="hidden" name="request_id" value="<?= $accreditation_request['request_id'] ?>">
                                            
                                            <div class="mb-3">
                                                <label for="documentType" class="form-label">Document Type</label>
                                                <select class="form-select" id="documentType" name="document_type" required>
                                                    <option value="">Select document type...</option>
                                                    <option value="cv">Curriculum Vitae (CV)</option>
                                                    <option value="qualifications">Educational Qualifications</option>
                                                    <option value="id_document">Government ID Document</option>
                                                    <option value="assignment">Course Assignment Work</option>
                                                </select>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="documentFiles" class="form-label">Select Files</label>
                                                <input type="file" class="form-control" id="documentFiles" name="documents[]" multiple accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.zip" required>
                                                <div class="form-text">
                                                    <i class="bx bx-info-circle"></i> 
                                                    You can upload multiple files. Maximum 50MB per file.
                                                </div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="documentNotes" class="form-label">Additional Notes <small class="text-muted">(Optional)</small></label>
                                                <textarea class="form-control" id="documentNotes" name="notes" rows="3" placeholder="Any additional information about these documents..."></textarea>
                                            </div>
                                            
                                            <div class="d-grid">
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="bx bx-upload"></i> Upload Documents
                                                </button>
                                            </div>
                                        </form>
                                        <?php else: ?>
                                        <div class="submission-complete">
                                            <div class="alert alert-info">
                                                <i class="bx bx-check-circle"></i>
                                                <strong>Documents Submitted Successfully!</strong>
                                                Your documents are currently under admin review. You will be notified of the outcome.
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Certificate Download Interface -->
                                <?php if ($accreditation_request['status'] == 'certificate_issued' && $issued_certificate): ?>
                                <div class="certificate-issued-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-award"></i> Your SITA-Accredited Certificate</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="certificate-preview mb-4">
                                            <div class="text-center py-4">
                                                <i class="bx bx-award text-warning" style="font-size: 4rem;"></i>
                                                <h4 class="text-success">🎉 Congratulations!</h4>
                                                <p class="lead">Your SITA-accredited certificate has been successfully issued.</p>
                                                
                                                <div class="certificate-details">
                                                    <div class="row">
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Certificate Number:</strong><br>
                                                                <code><?= $issued_certificate['certificate_number'] ?></code>
                                                            </div>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Issue Date:</strong><br>
                                                                <?= date('F d, Y', strtotime($issued_certificate['issue_date'])) ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <div class="row mt-3">
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Verification Code:</strong><br>
                                                                <code><?= $issued_certificate['verification_code'] ?></code>
                                                            </div>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <div class="detail-item">
                                                                <strong>Certificate Type:</strong><br>
                                                                SITA Accredited Professional
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="certificate-actions">
                                            <div class="row g-3">
                                                <div class="col-md-4">
                                                    <a href="download-certificate.php?id=<?= $issued_certificate['certificate_id'] ?>" class="btn btn-primary w-100">
                                                        <i class="bx bx-download"></i> Download PDF
                                                    </a>
                                                </div>
                                                <div class="col-md-4">
                                                    <button class="btn btn-outline-primary w-100" onclick="shareCertificate('<?= $issued_certificate['certificate_number'] ?>')">
                                                        <i class="bx bx-share-alt"></i> Share Certificate
                                                    </button>
                                                </div>
                                                <div class="col-md-4">
                                                    <a href="verify-certificate.php?code=<?= $issued_certificate['verification_code'] ?>" class="btn btn-outline-secondary w-100" target="_blank">
                                                        <i class="bx bx-check-shield"></i> Verify Online
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="certificate-note mt-4">
                                            <div class="alert alert-info">
                                                <i class="bx bx-info-circle"></i>
                                                <strong>Important:</strong> This certificate is digitally signed and can be verified online using the verification code above. 
                                                Share this achievement on your professional profiles!
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <!-- Rejection Notice -->
                                <?php if ($accreditation_request['status'] == 'rejected'): ?>
                                <div class="rejection-notice-card mb-4">
                                    <div class="card-header bg-danger text-white">
                                        <h5><i class="bx bx-x-circle"></i> Request Rejected</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="alert alert-danger">
                                            <h6>Unfortunately, your accreditation request has been rejected.</h6>
                                            <?php if ($accreditation_request['rejection_reason']): ?>
                                            <p><strong>Reason:</strong> <?= htmlspecialchars($accreditation_request['rejection_reason']) ?></p>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="support-contact">
                                            <h6><i class="bx bx-support"></i> Need Help?</h6>
                                            <p>If you believe this rejection was made in error or would like to reapply, please contact our support team:</p>
                                            <a href="mailto:accreditation@panaditeacademy.com" class="btn btn-outline-primary">
                                                <i class="bx bx-envelope"></i> Contact Support
                                            </a>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Notes Tab -->
                    <div class="tab-pane" id="notes">
                        <div class="notes-content">
                            <!-- Notes Header -->
                            <div class="notes-header">
                                <div class="notes-title">
                                    <i class="bx bx-note"></i>
                                    My Notes
                                </div>
                                <button class="btn btn-primary btn-sm" onclick="focusNoteInput()" title="Add new note at current time">
                                    <i class="bx bx-plus"></i> Add Note
                                </button>
                            </div>
                            
                            <!-- Notes Input Area -->
                            <div class="notes-input-area" style="margin-bottom: 2rem;">
                                <div class="input-group">
                                    <textarea class="form-control note-input" 
                                              placeholder="Add a note at the current video time..." 
                                              rows="3"
                                              style="resize: vertical; min-height: 80px;"></textarea>
                                    <button class="btn btn-primary save-note-btn" type="button">
                                        <i class="bx bx-save"></i> Save Note
                                    </button>
                                </div>
                                <small class="text-muted">
                                    <i class="bx bx-info-circle"></i> 
                                    Notes will be saved with the current video timestamp
                                </small>
                            </div>
                            
                            <!-- Notes List -->
                            <div class="notes-list" id="notesList">
                                <div class="no-notes text-center py-4">
                                    <i class="bx bx-note text-muted" style="font-size: 2rem;"></i>
                                    <p class="text-muted mt-2 mb-0">No notes yet. Add your first note while watching!</p>
                                </div>
                            </div>
                            
                            <!-- Notes Actions -->
                            <div class="notes-actions">
                                <button class="btn btn-outline-primary btn-sm" onclick="exportNotes()" title="Export all notes">
                                    <i class="bx bx-export"></i> Export Notes
                                </button>
                                <button class="btn btn-outline-secondary btn-sm" onclick="printNotes()" title="Print notes">
                                    <i class="bx bx-printer"></i> Print Notes
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Learning Tools Tab -->
                    <div class="tab-pane" id="tools">
                        <div class="learning-tools-content">
                            <!-- Smart Study Productivity Hub -->
                            <div class="productivity-header mb-4">
                                <div class="row align-items-center">
                                    <div class="col-md-8">
                                        <h3 class="mb-2">
                                            <i class="bx bx-brain text-primary me-2"></i>
                                            Smart Study Hub
                                        </h3>
                                        <p class="text-muted mb-0">Powerful tools to enhance your learning experience and productivity</p>
                                    </div>
                                    <div class="col-md-4 text-md-end">
                                        <div class="productivity-stats">
                                            <div class="stat-badge">
                                                <i class="bx bx-time-five text-success"></i>
                                                <span class="ms-1">Study Smart</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Two Main Tools Row -->
                            <div class="row g-4">
                                <!-- Smart Study Calendar -->
                                <div class="col-lg-6">
                                    <div class="productivity-card h-100">
                                        <div class="card-header">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div>
                                                    <h5 class="mb-1">
                                                        <i class="bx bx-calendar-event text-primary me-2"></i>
                                                        Smart Study Calendar
                                                    </h5>
                                                    <p class="text-muted small mb-0">Set lesson reminders with email notifications</p>
                                                </div>
                                                <div class="calendar-actions">
                                                    <button class="btn btn-sm btn-primary" onclick="createQuickReminder()">
                                                        <i class="bx bx-plus"></i> Quick Reminder
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body p-0">
                                            <!-- Calendar Navigation -->
                                            <div class="calendar-nav p-3 border-bottom">
                                                <div class="d-flex align-items-center justify-content-between">
                                                    <button class="btn btn-outline-primary btn-sm" onclick="previousMonth()">
                                                        <i class="bx bx-chevron-left"></i>
                                                    </button>
                                                    <h6 class="mb-0 fw-bold" id="currentMonthYear">November 2025</h6>
                                                    <button class="btn btn-outline-primary btn-sm" onclick="nextMonth()">
                                                        <i class="bx bx-chevron-right"></i>
                                                    </button>
                                                </div>
                                            </div>
                                            
                                            <!-- Calendar Grid -->
                                            <div class="calendar-container p-3">
                                                <div class="calendar-grid" id="calendarGrid">
                                                    <!-- Calendar will be rendered here by JavaScript -->
                                                </div>
                                            </div>
                                            
                                            <!-- Upcoming Reminders -->
                                            <div class="upcoming-reminders p-3 border-top bg-light">
                                                <h6 class="mb-2">
                                                    <i class="bx bx-bell text-warning me-1"></i>
                                                    Upcoming Reminders
                                                </h6>
                                                <div id="upcomingRemindersList">
                                                    <div class="text-muted small text-center py-2">
                                                        <i class="bx bx-calendar-plus me-1"></i>
                                                        No reminders set yet
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Picture-in-Picture Video Player -->
                                <div class="col-lg-6">
                                    <div class="productivity-card h-100">
                                        <div class="card-header">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div>
                                                    <h5 class="mb-1">
                                                        <i class="bx bx-video text-success me-2"></i>
                                                        Picture-in-Picture Player
                                                    </h5>
                                                    <p class="text-muted small mb-0">Continue watching while browsing</p>
                                                </div>
                                                <div class="pip-status">
                                                    <span class="badge bg-secondary" id="pipStatus">Ready</span>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body">
                                            <!-- PiP Controls -->
                                            <div class="pip-controls mb-4">
                                                <div class="row g-3">
                                                    <div class="col-6">
                                                        <button class="btn btn-success w-100" onclick="enablePictureInPicture()" id="pipEnableBtn">
                                                            <i class="bx bx-dock-top me-2"></i>
                                                            Enable PiP Mode
                                                        </button>
                                                    </div>
                                                    <div class="col-6">
                                                        <button class="btn btn-outline-secondary w-100" onclick="disablePictureInPicture()" id="pipDisableBtn" disabled>
                                                            <i class="bx bx-dock-bottom me-2"></i>
                                                            Exit PiP Mode
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Current Video Info -->
                                            <div class="current-video-info">
                                                <div class="video-thumbnail-container mb-3">
                                                    <div class="mini-video-preview" id="miniVideoPreview">
                                                        <?php if ($current_lesson): ?>
                                                            <img src="<?= !empty($current_lesson['thumbnail']) ? htmlspecialchars($current_lesson['thumbnail']) : '/assets/images/video-placeholder.jpg' ?>" 
                                                                 alt="Current Video" class="img-fluid rounded">
                                                            <div class="play-overlay">
                                                                <i class="bx bx-play-circle"></i>
                                                            </div>
                                                        <?php else: ?>
                                                            <div class="no-video-placeholder">
                                                                <i class="bx bx-video-off text-muted"></i>
                                                                <p class="text-muted mb-0">No video selected</p>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                
                                                <div class="video-details">
                                                    <h6 class="mb-2" id="currentVideoTitle">
                                                        <?= $current_lesson ? htmlspecialchars($current_lesson['title']) : 'No video selected' ?>
                                                    </h6>
                                                    <div class="video-meta text-muted small">
                                                        <span><i class="bx bx-time me-1"></i>Duration: 
                                                            <span id="videoDuration"><?= $current_lesson && !empty($current_lesson['duration']) ? htmlspecialchars($current_lesson['duration']) : 'N/A' ?></span>
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- PiP Features -->
                                            <div class="pip-features mt-4">
                                                <h6 class="mb-3">Features:</h6>
                                                <div class="feature-list">
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Watch while browsing other tabs</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Resizable floating window</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Always on top display</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Full playback controls</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Browser Support Info -->
                                            <div class="browser-support mt-4">
                                                <div class="alert alert-info small mb-0">
                                                    <i class="bx bx-info-circle me-1"></i>
                                                    <strong>Browser Support:</strong> Works best in Chrome, Firefox, Safari, and Edge. 
                                                    Make sure your browser allows Picture-in-Picture mode.
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Quick Actions Bar -->
                            <div class="quick-actions-bar mt-4">
                                <div class="row g-3">
                                    <div class="col-md-3">
                                        <div class="quick-action-card text-center">
                                            <div class="action-icon mb-2">
                                                <i class="bx bx-bell-plus text-primary"></i>
                                            </div>
                                            <h6 class="mb-1">Next Lesson Reminder</h6>
                                            <button class="btn btn-sm btn-outline-primary" onclick="setNextLessonReminder()">
                                                Set Reminder
                                            </button>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="quick-action-card text-center">
                                            <div class="action-icon mb-2">
                                                <i class="bx bx-calendar-week text-success"></i>
                                            </div>
                                            <h6 class="mb-1">Weekly Schedule</h6>
                                            <button class="btn btn-sm btn-outline-success" onclick="createWeeklySchedule()">
                                                Plan Week
                                            </button>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="quick-action-card text-center">
                                            <div class="action-icon mb-2">
                                                <i class="bx bx-mobile-alt text-info"></i>
                                            </div>
                                            <h6 class="mb-1">Mobile PiP</h6>
                                            <button class="btn btn-sm btn-outline-info" onclick="showMobilePipInfo()">
                                                Learn More
                                            </button>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="quick-action-card text-center">
                                            <div class="action-icon mb-2">
                                                <i class="bx bx-cog text-warning"></i>
                                            </div>
                                            <h6 class="mb-1">Preferences</h6>
                                            <button class="btn btn-sm btn-outline-warning" onclick="openStudyPreferences()">
                                                Settings
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Keyboard Shortcuts -->
                            <div class="shortcuts-section mt-4">
                                <h5><i class="bx bx-keyboard"></i> Enhanced Keyboard Shortcuts</h5>
                                <div class="shortcuts-grid">
                                    <div class="shortcut-item">
                                        <span>Play/Pause</span>
                                        <span class="shortcut-key">Space</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Skip Forward</span>
                                        <span class="shortcut-key">→</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Skip Backward</span>
                                        <span class="shortcut-key">←</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Add Note</span>
                                        <span class="shortcut-key">N</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Fullscreen</span>
                                        <span class="shortcut-key">F</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Volume Up/Down</span>
                                        <span class="shortcut-key">↑ / ↓</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
                    </div>
                </div>
            </div>

            <!-- Custom Note Modal -->
            <div id="customNoteModal" class="custom-modal" style="display: none;">
                <div class="custom-modal-overlay" onclick="closeNoteModal()"></div>
                <div class="custom-modal-container">
                    <div class="custom-modal-header">
                        <div class="custom-modal-title">
                            <i class="bx bx-note"></i>
                            <span id="modalTitleText">Add Note at</span>
                            <span id="modalTimestamp">0:00</span>
                        </div>
                        <button class="custom-modal-close" onclick="closeNoteModal()">
                            <i class="bx bx-x"></i>
                        </button>
                    </div>
                    <div class="custom-modal-body">
                        <div class="note-input-section">
                            <label for="customNoteTextarea">Your note:</label>
                            <textarea id="customNoteTextarea" 
                                placeholder="Add your thoughts, questions, or important points about this moment in the video..."></textarea>
                        </div>
                        <div class="note-tips-section">
                            <i class="bx bx-lightbulb"></i>
                            <span><strong>Tips:</strong> Your notes will be saved with the timestamp so you can easily jump back to this moment later.</span>
                        </div>
                    </div>
                    <div class="custom-modal-footer">
                        <button class="custom-btn custom-btn-secondary" onclick="closeNoteModal()">
                            <i class="bx bx-x"></i>
                            Cancel
                        </button>
                        <button class="custom-btn custom-btn-primary" id="customSaveNoteBtn">
                            <i class="bx bx-check"></i>
                            <span id="saveButtonText">Save Note</span>
                        </button>
                    </div>
                </div>
            </div>



            <!-- Navigation -->
            <div class="lesson-navigation">
                <button class="nav-btn btn-secondary">
                    <i class="bx bx-chevron-left"></i> Previous Lesson
                </button>
                <button class="nav-btn btn-primary">
                    Next Lesson <i class="bx bx-chevron-right"></i>
                </button>
            </div>

            <!-- Dynamic Question Section -->
            <div class="question-section" id="questionSection">
                <div class="question-header">
                    <div class="question-icon">
                        <i class="bx bx-help-circle"></i>
                    </div>
                    <div class="question-text">
                        <h4>Need Help with This Lesson?</h4>
                        <p>Ask your facilitator any questions about the content you just watched</p>
                    </div>
                </div>
                <form class="question-form" onsubmit="submitQuestion(event)">
                    <textarea class="question-input" 
                              placeholder="Type your question here... What didn't you understand about this lesson?"
                              maxlength="500" required></textarea>
                    <button type="submit" class="question-submit">
                        <i class="bx bx-send"></i>
                        Send Question to Facilitator
                    </button>
                </form>
            </div>
        </div>

                </div>
            </div>
        </div>
    </div>
</div>

<script>
let isPlaying = false;
let currentTime = 0;
let duration = 0;

function playVideo() {
    const player = document.getElementById('videoPlayer');
    const overlay = player.querySelector('.video-overlay');
    
    if (!isPlaying) {
        overlay.style.display = 'none';
        isPlaying = true;
        // Simulate video playing
        simulateVideoProgress();
    }
}

function simulateVideoProgress() {
    const totalSeconds = convertDurationToSeconds('<?= $current_lesson['duration'] ?>');
    duration = totalSeconds;
    
    const interval = setInterval(() => {
        if (isPlaying) {
            currentTime += 1;
            updateTimeDisplay();
            updateProgressBar();
            
            if (currentTime >= duration) {
                clearInterval(interval);
                completeLesson();
            }
        }
    }, 1000);
}

function updateTimeDisplay() {
    document.getElementById('currentTime').textContent = formatTime(currentTime);
    document.getElementById('totalTime').textContent = formatTime(duration);
}

function updateProgressBar() {
    const progress = (currentTime / duration) * 100;
    document.getElementById('timeProgress').style.width = progress + '%';
}

function formatTime(seconds) {
    const minutes = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return minutes + ':' + (secs < 10 ? '0' : '') + secs;
}

function convertDurationToSeconds(duration) {
    const parts = duration.split(':');
    return parseInt(parts[0]) * 60 + parseInt(parts[1]);
}

function seekVideo(event) {
    const bar = event.currentTarget;
    const rect = bar.getBoundingClientRect();
    const pos = (event.clientX - rect.left) / rect.width;
    currentTime = Math.floor(pos * duration);
    updateTimeDisplay();
    updateProgressBar();
}

function loadLesson(lessonId) {
    // Use encrypted course ID to avoid invalid course error
    const encryptedCourseId = '<?= $_GET['id'] ?? '' ?>';
    if (encryptedCourseId) {
        window.location.href = `course-detail.php?id=${encryptedCourseId}&lesson=${lessonId}`;
    } else {
        console.error('No encrypted course ID available');
        showVideoNotification('Error: Unable to load lesson', 'error');
    }
}

function completeLesson() {
    // Mark lesson as completed and move to next
    console.log('Lesson completed!');
    
    // Show question section with animation
    setTimeout(() => {
        const questionSection = document.getElementById('questionSection');
        questionSection.classList.add('show');
        
        // Scroll to question section smoothly
        questionSection.scrollIntoView({ 
            behavior: 'smooth',
            block: 'start'
        });
    }, 1000);
    
    // Update lesson progress to 100%
    updateLessonProgress(100);
}

function updateLessonProgress(percentage) {
    // Update current lesson progress in sidebar
    const currentLessonCard = document.querySelector('.lesson-item.active .lesson-progress-fill');
    const currentProgressText = document.querySelector('.lesson-item.active .progress-percentage');
    
    if (currentLessonCard) {
        currentLessonCard.style.width = percentage + '%';
    }
    if (currentProgressText) {
        currentProgressText.textContent = percentage + '% watched';
    }
    
    // Update video progress bar if it exists
    const videoProgressBar = document.querySelector('.video-progress-fill');
    if (videoProgressBar) {
        videoProgressBar.style.width = percentage + '%';
    }
}

// Video notification system
function showVideoNotification(message, type = 'info', duration = 4000) {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.video-notification');
    existingNotifications.forEach(notif => notif.remove());
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `video-notification alert alert-${type === 'success' ? 'success' : type === 'error' ? 'danger' : 'info'} alert-dismissible fade show`;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
        min-width: 300px;
        max-width: 500px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        border-radius: 8px;
        animation: slideInRight 0.3s ease-out;
    `;
    
    notification.innerHTML = `
        <div class="d-flex align-items-center">
            <div class="me-2">
                ${type === 'success' ? '✅' : type === 'error' ? '❌' : 'ℹ️'}
            </div>
            <div class="flex-grow-1">${message}</div>
            <button type="button" class="btn-close ms-2" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after duration
    if (duration > 0) {
        setTimeout(() => {
            if (notification && notification.parentNode) {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 150);
            }
        }, duration);
    }
    
    return notification;
}

// Add CSS animation for notifications
if (!document.querySelector('#video-notification-styles')) {
    const style = document.createElement('style');
    style.id = 'video-notification-styles';
    style.textContent = `
        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        .video-notification {
            animation: slideInRight 0.3s ease-out;
        }
    `;
    document.head.appendChild(style);
}

function submitQuestion(event) {
    event.preventDefault();
    const form = event.target;
    const question = form.querySelector('.question-input').value;
    
    if (question.trim()) {
        // Show loading state
        const submitBtn = form.querySelector('.question-submit');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Sending...';
        submitBtn.disabled = true;
        
        // Simulate sending question
        setTimeout(() => {
            submitBtn.innerHTML = '<i class="bx bx-check"></i> Question Sent!';
            submitBtn.style.background = '#10b981';
            
            setTimeout(() => {
                // Hide question section
                document.getElementById('questionSection').classList.remove('show');
                
                // Show success message
                showNotification('Question sent to facilitator successfully!', 'success');
                
                // Reset form
                form.reset();
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
                submitBtn.style.background = '';
            }, 2000);
        }, 1500);
    }
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : '#2563eb'};
        color: white;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 1000;
        font-weight: 600;
        animation: slideInRight 0.3s ease-out;
    `;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease-out';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

// Add CSS animations for notifications
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);

function viewMaterials(lessonId) {
    showNotification('Opening lesson materials...', 'info');
    // Here you would typically open a materials viewer or download
    console.log('Viewing materials for lesson:', lessonId);
}

function viewAssignment(lessonId) {
    showNotification('Opening assignment...', 'info');
    // Here you would typically redirect to assignment page
    window.open(`assignment.php?lesson_id=${lessonId}`, '_blank');
}

function takeQuiz(lessonId) {
    showNotification('Loading quiz...', 'info');
    // Here you would typically redirect to quiz page
    window.open(`quiz.php?lesson_id=${lessonId}`, '_blank');
}

// Video player - clean implementation
(function() {
    // Private variables to avoid conflicts
    var videoCurrentTime = 0;
    var videoDuration = 0;
    var videoIsPlaying = false;
    var videoElement = null;
    
    // Make functions globally accessible
    window.togglePlayPause = function() {
        console.log('Play/Pause button clicked');
        if (videoElement) {
            console.log('Video element found, current state:', videoElement.paused ? 'paused' : 'playing');
            if (videoElement.paused) {
                videoElement.play().then(() => {
                    console.log('Video play successful');
                }).catch(error => {
                    console.error('Error playing video:', error);
                });
            } else {
                videoElement.pause();
                console.log('Video paused');
            }
        } else {
            console.error('Video element not found!');
        }
    };
    
    window.toggleMute = function() {
        if (videoElement) {
            videoElement.muted = !videoElement.muted;
            const volumeIcon = document.getElementById('volumeIcon');
            if (volumeIcon) {
                volumeIcon.className = videoElement.muted ? 'bx bx-volume-mute' : 'bx bx-volume-full';
            }
        }
    };
    
    window.toggleFullscreen = function() {
        const videoContainer = document.getElementById('videoPlayer');
        if (videoContainer) {
            if (document.fullscreenElement) {
                document.exitFullscreen();
            } else {
                videoContainer.requestFullscreen();
            }
        }
    };
    
    window.seekVideo = function(event) {
        if (videoElement && videoDuration > 0) {
            const progressBar = event.currentTarget;
            const rect = progressBar.getBoundingClientRect();
            const clickX = event.clientX - rect.left;
            const percentage = clickX / rect.width;
            const seekTime = percentage * videoDuration;
            videoElement.currentTime = seekTime;
        }
    };
    
    // Advanced Video Controls
    window.skipBackward = function() {
        if (videoElement) {
            videoElement.currentTime = Math.max(0, videoElement.currentTime - 10);
            console.log('Skipped backward 10 seconds');
        }
    };
    
    window.skipForward = function() {
        if (videoElement && videoDuration > 0) {
            videoElement.currentTime = Math.min(videoDuration, videoElement.currentTime + 10);
            console.log('Skipped forward 10 seconds');
        }
    };
    
    // Playback Speed Controls
    window.toggleSpeedMenu = function() {
        const menu = document.getElementById('speedMenu');
        menu.classList.toggle('show');
        // Close other menus
        document.getElementById('qualityMenu').classList.remove('show');
    };
    
    window.setPlaybackSpeed = function(speed) {
        if (videoElement) {
            videoElement.playbackRate = speed;
            document.getElementById('speedLabel').textContent = speed + 'x';
            
            // Update active state
            document.querySelectorAll('#speedMenu .dropdown-item').forEach(item => {
                item.classList.remove('active');
            });
            event.target.classList.add('active');
            
            // Close menu
            document.getElementById('speedMenu').classList.remove('show');
            console.log('Playback speed set to:', speed + 'x');
        }
    };
    
    // Quality Controls (placeholder - would need multiple video sources)
    window.toggleQualityMenu = function() {
        const menu = document.getElementById('qualityMenu');
        menu.classList.toggle('show');
        // Close other menus
        document.getElementById('speedMenu').classList.remove('show');
    };
    
    window.setVideoQuality = function(quality) {
        // In a full implementation, you'd switch video sources here
        console.log('Quality set to:', quality);
        
        // Update active state
        document.querySelectorAll('#qualityMenu .dropdown-item').forEach(item => {
            item.classList.remove('active');
        });
        event.target.classList.add('active');
        
        // Close menu
        document.getElementById('qualityMenu').classList.remove('show');
        
        // Show notification
        showNotification('Video quality changed to ' + quality);
    };
    
    // Volume Controls
    window.setVolume = function(volume) {
        if (videoElement) {
            videoElement.volume = volume / 100;
            const volumeIcon = document.getElementById('volumeIcon');
            
            if (volume == 0) {
                volumeIcon.className = 'bx bx-volume-mute';
            } else if (volume < 50) {
                volumeIcon.className = 'bx bx-volume-low';
            } else {
                volumeIcon.className = 'bx bx-volume-full';
            }
        }
    };
    
    // Subtitle Controls
    window.toggleSubtitles = function() {
        const subtitleBtn = document.getElementById('subtitleBtn');
        const textTracks = videoElement.textTracks;
        
        if (textTracks.length > 0) {
            const track = textTracks[0];
            if (track.mode === 'showing') {
                track.mode = 'hidden';
                subtitleBtn.style.background = 'none';
                console.log('Subtitles disabled');
            } else {
                track.mode = 'showing';
                subtitleBtn.style.background = 'rgba(37, 99, 235, 0.5)';
                console.log('Subtitles enabled');
            }
        } else {
            showNotification('No subtitles available for this video');
        }
    };
    
    // Picture in Picture
    window.togglePictureInPicture = function() {
        if (videoElement) {
            if (document.pictureInPictureElement) {
                document.exitPictureInPicture();
                console.log('Exited Picture in Picture');
            } else {
                videoElement.requestPictureInPicture().then(() => {
                    console.log('Entered Picture in Picture');
                }).catch(error => {
                    console.error('Picture in Picture failed:', error);
                    showNotification('Picture in Picture not supported');
                });
            }
        }
    };
    
    // Note Taking Feature
    window.addNote = function() {
        if (videoElement && videoDuration > 0) {
            const currentTime = Math.floor(videoElement.currentTime);
            const minutes = Math.floor(currentTime / 60);
            const seconds = currentTime % 60;
            const timestamp = `${minutes}:${seconds.toString().padStart(2, '0')}`;
            
            // Show custom modal instead of browser prompt
            showNoteModal(currentTime, timestamp);
        }
    };
    
    // Show custom note modal
    function showNoteModal(time, timestamp, existingNote = '', noteId = null) {
        const modal = document.getElementById('customNoteModal');
        const titleText = document.getElementById('modalTitleText');
        const timestampSpan = document.getElementById('modalTimestamp');
        const textarea = document.getElementById('customNoteTextarea');
        const saveBtn = document.getElementById('customSaveNoteBtn');
        const saveButtonText = document.getElementById('saveButtonText');
        
        // Update modal content
        if (timestampSpan) timestampSpan.textContent = timestamp;
        if (textarea) textarea.value = existingNote;
        
        // Update title and button text for edit vs add
        if (noteId) {
            titleText.textContent = 'Edit Note at';
            saveButtonText.textContent = 'Update Note';
        } else {
            titleText.textContent = 'Add Note at';
            saveButtonText.textContent = 'Save Note';
        }
        
        // Clear any existing event listeners
        saveBtn.onclick = null;
        
        // Add click event listener for save button
        saveBtn.onclick = function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const noteText = textarea.value.trim();
            if (noteText) {
                if (noteId) {
                    // Update existing note
                    updateExistingNote(noteId, time, noteText, timestamp);
                } else {
                    // Save new note
                    saveVideoNote(time, noteText, timestamp);
                }
                // Close modal
                closeNoteModal();
            } else {
                textarea.focus();
                textarea.classList.add('invalid');
                setTimeout(() => textarea.classList.remove('invalid'), 3000);
            }
        };
        
        // Handle Ctrl+Enter to save
        textarea.onkeydown = function(e) {
            if (e.ctrlKey && e.key === 'Enter') {
                saveBtn.click();
            }
        };
        
        // Show modal
        modal.style.display = 'flex';
        
        // Focus on textarea after a short delay
        setTimeout(() => {
            if (textarea) textarea.focus();
        }, 100);
    }
    
    // Close custom note modal
    window.closeNoteModal = function() {
        const modal = document.getElementById('customNoteModal');
        const textarea = document.getElementById('customNoteTextarea');
        
        modal.style.display = 'none';
        if (textarea) {
            textarea.value = '';
            textarea.classList.remove('invalid');
        }
    };
    
    // Helper Functions
    function saveVideoNote(time, note, timestamp) {
        const lessonId = <?= $current_lesson_id ?>;
        
        const requestData = {
            lesson_id: lessonId,
            timestamp: time,
            note: note,
            formatted_time: timestamp
        };
        
        console.log('Saving note with data:', requestData);
        
        fetch('save_video_note.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(requestData)
        }).then(response => {
            console.log('Save note response status:', response.status);
            console.error('Network error saving note:', error);
            showNotification('Failed to save note - Network error', 'error');
        });
    }
    
    function showNotification(message, type = 'success') {
        const notification = document.createElement('div');
        notification.className = `toast-notification ${type}`;
        notification.textContent = message;
        
        // Style the notification
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${type === 'error' ? 'linear-gradient(135deg, #ff416c 0%, #ff4b2b 100%)' : 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)'};
            color: white;
            padding: 12px 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            z-index: 10000;
            transform: translateX(400px);
            transition: transform 0.3s ease;
            font-weight: 500;
        `;
        
        document.body.appendChild(notification);
        
        // Add animation class after a short delay
        setTimeout(() => {
            notification.classList.add('show');
        }, 100);
        
        // Remove after 3 seconds
        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }
    
    // Close dropdowns when clicking outside
    document.addEventListener('click', function(event) {
        if (!event.target.closest('.dropdown')) {
            document.querySelectorAll('.dropdown-menu').forEach(menu => {
                menu.classList.remove('show');
            });
        }
    });
    
    // Keyboard Shortcuts
    document.addEventListener('keydown', function(event) {
        // Don't trigger shortcuts if user is typing in an input field
        if (event.target.tagName === 'INPUT' || event.target.tagName === 'TEXTAREA') {
            return;
        }
        
        switch(event.code) {
            case 'Space':
                event.preventDefault();
                togglePlayPause();
                break;
            case 'ArrowRight':
                event.preventDefault();
                skipForward();
                break;
            case 'ArrowLeft':
                event.preventDefault();
                skipBackward();
                break;
            case 'ArrowUp':
                event.preventDefault();
                adjustVolume(10);
                break;
            case 'ArrowDown':
                event.preventDefault();
                adjustVolume(-10);
                break;
            case 'KeyM':
                event.preventDefault();
                toggleMute();
                break;
            case 'KeyF':
                event.preventDefault();
                toggleFullscreen();
                break;
            case 'KeyN':
                event.preventDefault();
                addNote();
                break;
        }
    });
    
    // Volume adjustment helper
    function adjustVolume(change) {
        if (videoElement) {
            const volumeSlider = document.getElementById('volumeSlider');
            const currentVolume = parseInt(volumeSlider.value);
            const newVolume = Math.max(0, Math.min(100, currentVolume + change));
            volumeSlider.value = newVolume;
            setVolume(newVolume);
        }
    }
    
    // Toggle notes panel
    window.toggleNotesPanel = function() {
        addNote();
    };
    
    // Load video notes
    function loadVideoNotes() {
        const lessonId = <?= $current_lesson_id ? $current_lesson_id : 'null' ?>;
        
        if (!lessonId) {
            console.log('No lesson ID available, skipping notes load');
            const notesList = document.getElementById('notesList');
            if (notesList) {
                notesList.innerHTML = `
                    <div class="no-notes">
                        <i class="bx bx-note" style="font-size: 2rem; opacity: 0.3; margin-bottom: 0.5rem;"></i>
                        <p>No lessons available in this course yet.</p>
                    </div>
                `;
            }
            return;
        }
        
        console.log('Loading notes for lesson ID:', lessonId);
        
        fetch(`get_video_notes.php?lesson_id=${lessonId}`)
            .then(response => {
                console.log('Notes response status:', response.status);
                return response.text();
            })
            .then(text => {
                console.log('Notes raw response:', text);
                try {
                    const data = JSON.parse(text);
                    console.log('Notes parsed data:', data);
                    
                    const notesList = document.getElementById('notesList');
                    
                    if (data.success && data.notes && data.notes.length > 0) {
                        console.log(`Found ${data.notes.length} notes`);
                        
                        // Helper function to format time within this context
                        const formatTimeLocal = (seconds) => {
                            if (isNaN(seconds) || seconds < 0) return '0:00';
                            const mins = Math.floor(seconds / 60);
                            const secs = Math.floor(seconds % 60);
                            return `${mins}:${secs.toString().padStart(2, '0')}`;
                        };
                        
                        // Helper function to get proper timestamp display
                        const getTimestamp = (note) => {
                            // If formatted_time is valid (not "Invalid Date" etc), use it
                            if (note.formatted_time && 
                                note.formatted_time !== 'Invalid Date' && 
                                note.formatted_time !== 'NaN:NaN' && 
                                note.formatted_time.includes(':')) {
                                return note.formatted_time;
                            }
                            // Otherwise format from timestamp
                            return formatTimeLocal(note.timestamp || 0);
                        };
                        
                        notesList.innerHTML = `
                            <div class="accordion" id="notesAccordion">
                                ${data.notes.map((note, index) => `
                                    <div class="accordion-item compact-note-item">
                                        <h2 class="accordion-header" id="noteHeading${note.id}">
                                            <button class="accordion-button collapsed" type="button" 
                                                    data-bs-toggle="collapse" data-bs-target="#noteCollapse${note.id}" 
                                                    aria-expanded="false" 
                                                    aria-controls="noteCollapse${note.id}">
                                                <div class="note-header-content">
                                                    <div class="note-thumbnail-mini">
                                                        <img src="${note.thumbnail_url || '/assets/images/video-placeholder.jpg'}" 
                                                             alt="Video" 
                                                             onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzIiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAzMiAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjMyIiBoZWlnaHQ9IjI0IiBmaWxsPSIjZjFmNWY5Ii8+CjxwYXRoIGQ9Ik0xMiA5TDE5IDEyLjVMMTIgMTZWOVoiIGZpbGw9IiM2MzY2ZjEiLz4KPC9zdmc+'">
                                                    </div>
                                                    <div class="note-header-info">
                                                        <div class="note-title-compact">${escapeHtml(note.video_title || 'Video Lesson')}</div>
                                                        <div class="note-meta-compact">
                                                            <span class="note-timestamp-compact" onclick="event.stopPropagation(); seekToTimestamp(${note.timestamp})" 
                                                                  title="Jump to ${getTimestamp(note)}">
                                                                <i class="bx bx-play-circle"></i> ${getTimestamp(note)}
                                                            </span>
                                                            <span class="note-time-compact">
                                                                <i class="bx bx-time"></i> ${note.time_ago || 'Recently'}
                                                            </span>
                                                        </div>
                                                    </div>
                                                    <div class="note-preview">${escapeHtml(note.note).substring(0, 40)}${note.note.length > 40 ? '...' : ''}</div>
                                                </div>
                                            </button>
                                        </h2>
                                        <div id="noteCollapse${note.id}" class="accordion-collapse collapse" 
                                             aria-labelledby="noteHeading${note.id}" data-bs-parent="#notesAccordion">
                                            <div class="accordion-body">
                                                <div class="note-content-expanded">
                                                    <div class="note-text-full">${escapeHtml(note.note)}</div>
                                                    <div class="note-actions-compact">
                                                        <button class="btn btn-sm btn-outline-primary" onclick="editNote(${note.id}, '${escapeHtml(note.note)}', ${note.timestamp})" title="Edit note">
                                                            <i class="bx bx-edit"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-outline-danger" onclick="deleteNote(${note.id})" title="Delete note">
                                                            <i class="bx bx-trash"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-outline-success" onclick="seekToTimestamp(${note.timestamp})" title="Jump to ${getTimestamp(note)}">
                                                            <i class="bx bx-skip-next"></i> Jump to ${getTimestamp(note)}
                                                        </button>
                                                    </div>
                                                    <div class="note-details-compact">
                                                        <small class="text-muted">
                                                            <i class="bx bx-calendar"></i> ${note.created_date || 'Unknown date'}
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        `;
                    } else {
                        console.log('No notes found or failed to load:', data.message || 'Unknown reason');
                        notesList.innerHTML = `
                            <div class="no-notes">
                                <i class="bx bx-note" style="font-size: 2rem; opacity: 0.3; margin-bottom: 0.5rem;"></i>
                                <p>No notes yet. Click the note button on the video player or press 'N' to add your first note!</p>
                            </div>
                        `;
                    }
                } catch (e) {
                    console.error('Failed to parse notes JSON:', e, 'Raw text:', text);
                }
            })
            .catch(error => {
                console.error('Network error loading notes:', error);
            });
    }
    
    // Seek to timestamp from note
    window.seekToTimestamp = function(timestamp) {
        if (videoElement) {
            videoElement.currentTime = timestamp;
            showNotification(`Jumped to ${formatTime(timestamp)}`);
        }
    };
    
    // Edit note - use modal instead of prompt
    window.editNote = function(noteId, currentNote, timestamp) {
        const minutes = Math.floor(timestamp / 60);
        const seconds = timestamp % 60;
        const formattedTime = `${minutes}:${seconds.toString().padStart(2, '0')}`;
        
        // Show modal with existing note content
        showNoteModal(timestamp, formattedTime, currentNote, noteId);
    };
    
    // Helper function for updating existing notes
    function updateExistingNote(noteId, timestamp, noteText, formattedTime) {
        fetch('save_video_note.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                lesson_id: <?= $current_lesson_id ?>,
                timestamp: timestamp,
                note: noteText,
                formatted_time: formattedTime
            })
        }).then(response => {
            console.log('Update note response status:', response.status);
            return response.text();
        }).then(text => {
            console.log('Update note raw response:', text);
            try {
                const data = JSON.parse(text);
                if (data.success) {
                    showNotification('Note updated successfully');
                    setTimeout(() => {
                        loadVideoNotes();
                    }, 500);
                } else {
                    console.error('Update note failed:', data);
                    showNotification(data.message || 'Failed to update note', 'error');
                }
            } catch (e) {
                console.error('Failed to parse update response:', e, 'Raw text:', text);
                showNotification('Failed to update note - Invalid response', 'error');
            }
        }).catch(error => {
            console.error('Network error updating note:', error);
            showNotification('Failed to update note - Network error', 'error');
        });
    }
    
    // Delete note
    window.deleteNote = function(noteId) {
        if (confirm('Are you sure you want to delete this note?')) {
            fetch('delete_video_note.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    note_id: noteId
                })
            }).then(response => response.json())
              .then(data => {
                  if (data.success) {
                      showNotification('Note deleted successfully');
                      loadVideoNotes();
                  } else {
                      showNotification('Failed to delete note', 'error');
                  }
              }).catch(error => {
                  console.error('Error deleting note:', error);
                  showNotification('Failed to delete note', 'error');
              });
        }
    };
    
    // HTML escape helper
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Private helper functions
    function updateTimeDisplay() {
        const currentTimeElement = document.getElementById('currentTime');
        const totalTimeElement = document.getElementById('totalTime');
        
        if (currentTimeElement && videoCurrentTime >= 0) {
            currentTimeElement.textContent = formatTime(videoCurrentTime);
        }
        if (totalTimeElement && videoDuration > 0) {
            totalTimeElement.textContent = formatTime(videoDuration);
        }
    }
    
    function updateProgressBar() {
        const progressBar = document.getElementById('timeProgress');
        if (progressBar) {
            if (videoDuration > 0 && videoCurrentTime >= 0) {
                const percentage = Math.max(0, Math.min(100, (videoCurrentTime / videoDuration) * 100));
                progressBar.style.width = percentage + '%';
                console.log('Progress bar updated:', percentage + '%', 'Current:', videoCurrentTime, 'Duration:', videoDuration);
            } else {
                progressBar.style.width = '0%';
                console.log('Progress bar reset to 0%');
            }
        }
    }
    
    function updatePlayPauseButton() {
        const playPauseBtn = document.getElementById('playPauseBtn');
        if (playPauseBtn) {
            const icon = playPauseBtn.querySelector('i');
            if (icon) {
                if (videoElement && !videoElement.paused) {
                    icon.className = 'bx bx-pause';
                } else {
                    icon.className = 'bx bx-play';
                }
            }
        }
    }
    
    function formatTime(seconds) {
        const minutes = Math.floor(seconds / 60);
        const remainingSeconds = Math.floor(seconds % 60);
        return minutes + ':' + (remainingSeconds < 10 ? '0' : '') + remainingSeconds;
    }
    
    function trackWatchProgress() {
        if (videoDuration > 0) {
            const percentage = Math.round((videoCurrentTime / videoDuration) * 100);
            
            // Update lesson progress in sidebar
            updateLessonProgress(percentage);
            
            // Save to database every 30 seconds
            if (Math.floor(videoCurrentTime) % 30 === 0) {
                fetch('save_progress.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        lesson_id: <?= $current_lesson_id ?>,
                        progress: percentage,
                        current_time: Math.floor(videoCurrentTime)
                    })
                }).then(response => response.json())
                  .then(data => {
                      console.log('Progress saved:', data);
                  }).catch(error => {
                      console.error('Error saving progress:', error);
                  });
            }
        }
    }
    
    function updateLessonProgress(percentage) {
        // Update lesson progress in sidebar if available
        const currentLessonItem = document.querySelector('.lesson-item.active .lesson-progress-fill');
        if (currentLessonItem) {
            currentLessonItem.style.width = percentage + '%';
            console.log('Updated sidebar lesson progress:', percentage + '%');
        }
        
        const progressText = document.querySelector('.lesson-item.active .progress-percentage');
        if (progressText) {
            progressText.textContent = percentage + '% watched';
        }
        
        // Also update any progress text in the main content area
        const mainProgressText = document.querySelector('.lesson-progress-text');
        if (mainProgressText) {
            mainProgressText.textContent = percentage + '% watched';
        }
    }
    
    // Initialize when page loads
    document.addEventListener('DOMContentLoaded', function() {
        console.log('DOM loaded, initializing video player...');
        videoElement = document.getElementById('lessonVideo');
        if (videoElement) {
            console.log('Video element found:', videoElement);
            
            // Set up video event handlers
            videoElement.addEventListener('loadedmetadata', function() {
                videoDuration = videoElement.duration;
                console.log('Video metadata loaded, duration:', videoDuration);
                updateTimeDisplay();
            });
            
            videoElement.addEventListener('timeupdate', function() {
                videoCurrentTime = videoElement.currentTime;
                trackWatchProgress();
                updateTimeDisplay();
                updateProgressBar();
                updatePlayPauseButton();
                
                // Update accordion badge for current lesson with dynamic progress
                const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
                if (currentLessonId && !videoElement.paused && videoElement.duration) {
                    const progressPercent = (videoElement.currentTime / videoElement.duration) * 100;
                    updateAccordionBadges(currentLessonId, 'watching', progressPercent);
                    
                    // DIRECT UPDATE: Update main course progress without AJAX - EVERY TIME
                    const mainProgressPercentage = document.getElementById('main-course-progress-percentage');
                    const mainProgressBar = document.getElementById('main-course-progress-bar');
                    
                    // CERTIFICATE VIDEO PROGRESS: Update certificate tab video completion
                    const certificateVideoProgressText = document.getElementById('certificate-video-progress-text');
                    const certificateVideoProgressBar = document.getElementById('certificate-video-progress-bar');
                    
                    // ACCORDION LESSON PROGRESS: Update individual lesson progress in accordion
                    const accordionProgressText = document.getElementById(`accordion-progress-text-${currentLessonId}`);
                    const accordionProgressBar = document.getElementById(`accordion-progress-bar-${currentLessonId}`);
                    
                    console.log(`🔍 DEBUG: Main elements - percentage: ${!!mainProgressPercentage}, bar: ${!!mainProgressBar}`);
                    console.log(`🎯 DEBUG: Certificate elements - text: ${!!certificateVideoProgressText}, bar: ${!!certificateVideoProgressBar}`);
                    console.log(`📋 DEBUG: Accordion elements - text: ${!!accordionProgressText}, bar: ${!!accordionProgressBar}, lessonId: ${currentLessonId}`);
                    
                    if (mainProgressPercentage && mainProgressBar) {
                        // Calculate simple progress: if current video is X% watched, add that to base progress
                        const baseProgress = 33; // Current course progress from PHP
                        const currentVideoContribution = progressPercent * 0.33; // Each video contributes ~33% to total
                        const newTotalProgress = Math.min(100, Math.round(baseProgress + currentVideoContribution));
                        
                        console.log(`📊 DIRECT UPDATE: Main progress ${baseProgress}% + ${currentVideoContribution.toFixed(1)}% = ${newTotalProgress}%`);
                        
                        mainProgressPercentage.textContent = newTotalProgress + '%';
                        mainProgressBar.style.width = newTotalProgress + '%';
                        
                        // UPDATE CERTIFICATE VIDEO PROGRESS TOO
                        if (certificateVideoProgressText && certificateVideoProgressBar) {
                            certificateVideoProgressText.textContent = newTotalProgress + '% videos watched';
                            certificateVideoProgressBar.style.width = newTotalProgress + '%';
                            console.log(`🏆 CERTIFICATE UPDATED: Video progress set to "${newTotalProgress}% videos watched"`);
                        }
                        
                        // UPDATE ACCORDION LESSON PROGRESS TOO
                        if (accordionProgressText && accordionProgressBar) {
                            const roundedProgress = Math.round(progressPercent);
                            accordionProgressText.textContent = `Progress: ${roundedProgress}% complete`;
                            accordionProgressBar.style.width = progressPercent + '%';
                            console.log(`📋 ACCORDION UPDATED: Lesson ${currentLessonId} progress set to "${roundedProgress}% complete"`);
                        }
                        
                        console.log(`✅ UPDATED: Main progress set to "${newTotalProgress}%"`);
                    } else {
                        console.log(`❌ MAIN ELEMENTS NOT FOUND: percentage=${mainProgressPercentage}, bar=${mainProgressBar}`);
                    }
                }
            });
            
            // Set video to saved position if available
            <?php if (isset($current_lesson['last_position']) && $current_lesson['last_position'] > 0): ?>
                videoElement.currentTime = <?= $current_lesson['last_position'] ?>;
            <?php endif; ?>
            
            // Force initial update after a short delay
            setTimeout(function() {
                if (videoElement.duration) {
                    videoDuration = videoElement.duration;
                    videoCurrentTime = videoElement.currentTime || 0;
                    updateTimeDisplay();
                    updateProgressBar();
                    updatePlayPauseButton();
                    console.log('Force updated time display, duration:', videoDuration, 'currentTime:', videoCurrentTime);
                } else {
                    // Reset everything to initial state if no video loaded
                    videoDuration = 0;
                    videoCurrentTime = 0;
                    updateTimeDisplay();
                    updateProgressBar();
                    updatePlayPauseButton();
                    console.log('Video not loaded, reset to initial state');
                }
                
                // Load existing notes for this lesson
                loadVideoNotes();
            }, 1000);
        } else {
            console.error('Video element not found!');
        }
    });
    
    // Enhanced video progress tracking with AJAX
    let progressUpdateInterval;
    let courseProgressUpdateCounter = 0;
    
    function startProgressTracking() {
        if (progressUpdateInterval) clearInterval(progressUpdateInterval);
        
        progressUpdateInterval = setInterval(function() {
            if (videoIsPlaying && videoElement && videoElement.duration) {
                const currentTime = videoElement.currentTime;
                const duration = videoElement.duration;
                const progressPercentage = Math.round((currentTime / duration) * 100);
                
                // Update lesson progress in sidebar
                updateLessonProgress(progressPercentage);
                
                // Send AJAX update every 5 seconds
                updateVideoProgressAJAX(progressPercentage, currentTime);
                
                // Update main course progress every 5 seconds for testing
                courseProgressUpdateCounter++;
                if (courseProgressUpdateCounter >= 1) {
                    console.log('Triggering main course progress update...');
                    updateMainCourseProgress();
                    courseProgressUpdateCounter = 0;
                }
            }
        }, 1000);
    }
    
    function stopProgressTracking() {
        if (progressUpdateInterval) {
            clearInterval(progressUpdateInterval);
            progressUpdateInterval = null;
            courseProgressUpdateCounter = 0;
        }
    }
    
    // AJAX function to update video progress
    function updateVideoProgressAJAX(progressPercentage, currentTime) {
        const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
        if (!currentLessonId) return;
        
        $.ajax({
            url: 'ajax/update_video_progress.php',
            method: 'POST',
            data: {
                lesson_id: currentLessonId,
                course_id: <?= $course_id ?? 'null' ?>,
                progress_percentage: Math.round(progressPercentage),
                current_time: Math.round(currentTime),
                completed: progressPercentage >= 95 ? 1 : 0
            },
            success: function(response) {
                console.log('Progress updated:', response);
                
                // Update certificate eligibility if video completed
                if (progressPercentage >= 95) {
                    updateCertificateEligibility();
                    // Update main course progress when video is completed
                    setTimeout(() => {
                        updateMainCourseProgress();
                    }, 1000); // Small delay to ensure server updates are processed
                }
            },
            error: function(xhr, status, error) {
                console.log('Progress update failed:', error);
            }
        });
    }
    
    // Auto-advance to next video
    function autoAdvanceToNextVideo() {
        const lessons = <?= json_encode($lessons) ?>;
        const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
        
        // Find current lesson index
        const currentIndex = lessons.findIndex(lesson => lesson.lesson_id == currentLessonId);
        
        if (currentIndex !== -1 && currentIndex < lessons.length - 1) {
            const nextLesson = lessons[currentIndex + 1];
            
            // Show notification
            showVideoNotification(`🎉 Video completed! Auto-advancing to: ${nextLesson.title}`, 'success', 3000);
            
            // Auto-advance after 3 seconds
            setTimeout(() => {
                loadLesson(nextLesson.lesson_id);
            }, 3000);
        } else {
            // Last video completed
            showVideoNotification('🎉 Course completed! All videos watched.', 'success', 5000);
            updateCertificateEligibility();
        }
    }
    
    // Update certificate eligibility check
    function updateCertificateEligibility() {
        $.ajax({
            url: 'ajax/check_certificate_eligibility.php',
            method: 'POST',
            data: {
                course_id: <?= $course_id ?? 'null' ?>
            },
            success: function(response) {
                if (response.eligible) {
                    showVideoNotification('🏆 Congratulations! You\'re now eligible for a certificate!', 'success', 5000);
                    
                    // Update certificate tab if visible
                    const certificateTab = document.querySelector('[data-bs-target="#certificate"]');
                    if (certificateTab) {
                        certificateTab.classList.add('badge-success');
                    }
                }
            },
            error: function(xhr, status, error) {
                console.log('Certificate eligibility check failed:', error);
            }
        });
    }
    
    // Update main course progress display dynamically
    function updateMainCourseProgress() {
        console.log('🔄 Starting main course progress update...');
        
        $.ajax({
            url: 'ajax/get_course_progress.php',
            method: 'POST',
            data: {
                course_id: <?= $course_id ?? 'null' ?>
            },
            success: function(response) {
                console.log('📊 Course progress AJAX response:', response);
                
                if (response.success) {
                    // Update main progress percentage
                    const mainProgressPercentage = document.getElementById('main-course-progress-percentage');
                    const mainProgressBar = document.getElementById('main-course-progress-bar');
                    
                    console.log('🎯 Found elements:', {
                        percentage: mainProgressPercentage,
                        bar: mainProgressBar
                    });
                    
                    if (mainProgressPercentage && mainProgressBar) {
                        const newProgress = Math.round(response.video_progress);
                        const currentText = mainProgressPercentage.textContent;
                        
                        console.log(`📈 Updating progress from "${currentText}" to "${newProgress}%"`);
                        
                        // Update percentage text
                        mainProgressPercentage.textContent = newProgress + '%';
                        
                        // Update progress bar with smooth animation
                        mainProgressBar.style.width = newProgress + '%';
                        
                        console.log(`✅ Main course progress updated to: ${newProgress}%`);
                        
                        // Show notification for milestones
                        if (newProgress === 25 || newProgress === 50 || newProgress === 75 || newProgress === 100) {
                            showVideoNotification(`🎯 Course Progress: ${newProgress}% completed!`, 'success', 3000);
                        }
                    } else {
                        console.error('❌ Could not find progress elements:', {
                            percentageElement: 'main-course-progress-percentage',
                            barElement: 'main-course-progress-bar'
                        });
                    }
                } else {
                    console.error('❌ Course progress update failed:', response);
                }
            },
            error: function(xhr, status, error) {
                console.error('❌ Course progress AJAX failed:', {
                    status: xhr.status,
                    statusText: xhr.statusText,
                    error: error,
                    response: xhr.responseText
                });
            }
        });
    }
    
    // Function to update accordion badges dynamically
    function updateAccordionBadges(lessonId, status, progressPercent = 0) {
        const watchingBadge = document.getElementById(`watching-badge-${lessonId}`);
        const completedBadge = document.getElementById(`completed-badge-${lessonId}`);
        const progressDisplay = document.getElementById(`progress-display-${lessonId}`);
        const progressPercentage = document.getElementById(`progress-percentage-${lessonId}`);
        
        console.log(`Updating badges for lesson ${lessonId}, status: ${status}, progress: ${progressPercent}%`);
        
        if (!watchingBadge || !completedBadge) {
            console.log(`Badge elements not found for lesson ${lessonId}`);
            return;
        }
        
        switch(status) {
            case 'watching':
                watchingBadge.style.display = 'inline-flex';
                completedBadge.style.display = 'none';
                
                // Show and update progress display
                if (progressDisplay && progressPercentage) {
                    progressDisplay.style.display = 'block';
                    progressPercentage.textContent = Math.round(progressPercent);
                }
                
                console.log(`Lesson ${lessonId}: Showing watching badge with ${progressPercent}% progress`);
                break;
                
            case 'completed':
                watchingBadge.style.display = 'none';
                completedBadge.style.display = 'inline-flex';
                
                // Hide progress display and show 100%
                if (progressDisplay && progressPercentage) {
                    progressDisplay.style.display = 'none';
                }
                
                console.log(`Lesson ${lessonId}: Showing completed badge`);
                break;
                
            case 'idle':
                watchingBadge.style.display = 'none';
                
                // Hide progress display when idle
                if (progressDisplay) {
                    progressDisplay.style.display = 'none';
                }
                break;
        }
    }
    
    // Enhanced video event listeners
    if (videoElement) {
        videoElement.addEventListener('play', function() {
            videoIsPlaying = true;
            startProgressTracking();
            
            // Show watching badge for current lesson
            const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
            if (currentLessonId) {
                const progressPercent = videoElement.duration ? (videoElement.currentTime / videoElement.duration) * 100 : 0;
                updateAccordionBadges(currentLessonId, 'watching', progressPercent);
            }
            
            // IMMEDIATE TEST: Update main course progress when video starts
            console.log('🚀 IMMEDIATE TEST: Updating main course progress...');
            updateMainCourseProgress();
            
            console.log('Video started playing - progress tracking enabled');
        });
        
        videoElement.addEventListener('pause', function() {
            videoIsPlaying = false;
            stopProgressTracking();
            
            // Hide watching badge when paused
            const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
            if (currentLessonId) {
                updateAccordionBadges(currentLessonId, 'idle');
            }
            
            console.log('Video paused - progress tracking stopped');
        });
        
        videoElement.addEventListener('ended', function() {
            videoIsPlaying = false;
            stopProgressTracking();
            
            // Mark as completed and update progress
            updateVideoProgressAJAX(100, videoElement.duration);
            updateLessonProgress(100);
            
            // Show completed badge
            const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
            if (currentLessonId) {
                updateAccordionBadges(currentLessonId, 'completed', 100);
            }
            
            console.log('Video ended - marked as completed');
            
            // Auto-advance to next video
            autoAdvanceToNextVideo();
        });
        
        videoElement.addEventListener('loadedmetadata', function() {
            // Reset auto-advance flag when new video loads
            videoElement.autoAdvanceTriggered = false;
            console.log('Video metadata loaded - ready for playback');
        });
        
        // Start tracking when page loads if video is already playing
        if (!videoElement.paused) {
            videoIsPlaying = true;
            startProgressTracking();
            
            // Show watching badge if video is auto-playing
            const currentLessonId = <?= $current_lesson_id ?? 'null' ?>;
            if (currentLessonId) {
                updateAccordionBadges(currentLessonId, 'watching');
            }
        }
    }
    
})();

// Tabbed Interface Functionality
function initializeTabs() {
    // Tab switching functionality
    const tabButtons = document.querySelectorAll('.tab-btn');
    const tabPanes = document.querySelectorAll('.tab-pane');

    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');
            
            // Remove active class from all buttons and panes
            tabButtons.forEach(btn => btn.classList.remove('active'));
            tabPanes.forEach(pane => pane.classList.remove('active'));
            
            // Add active class to clicked button and corresponding pane
            this.classList.add('active');
            document.getElementById(targetTab).classList.add('active');
        });
    });
}

// Initialize tabs when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeTabs();
});

// Initialize review functionality immediately when page loads
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing review system...');
    initializeReviewSystem();
});

// Initialize when reviews tab is shown
document.addEventListener('shown.bs.tab', function(e) {
    if (e.target.getAttribute('href') === '#reviews') {
        console.log('Reviews tab shown, reinitializing...');
        initializeReviewSystem();
    }
});

// Also listen for tab clicks
document.addEventListener('click', function(e) {
    if (e.target.closest('[data-bs-target="#reviews"]') || 
        e.target.closest('a[href="#reviews"]')) {
        setTimeout(() => {
            console.log('Reviews tab clicked, initializing...');
            initializeReviewSystem();
        }, 100);
    }
});

function initializeReviewSystem() {
    initializeStarRating();
    initializeReviewForm();
    initializeReviewSorting();
}

// Add Note function for Notes tab
function addNote() {
    const currentTime = videoElement ? videoElement.currentTime : 0;
    document.getElementById('modalTimestamp').textContent = formatTime(currentTime);
    document.getElementById('customNoteModal').style.display = 'flex';
}

// All JavaScript functionality is now contained in the IIFE above

// SITA Accreditation Request Form Handler
document.addEventListener('DOMContentLoaded', function() {
    const accreditationForm = document.getElementById('accreditationRequestForm');
    if (accreditationForm) {
        accreditationForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            
            // Disable form and show loading
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin me-1"></i> Submitting Request...';
            
            // Get form data
            const formData = new FormData(this);
            
            // Submit via fetch
            fetch('api/submit_accreditation_request.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show success message
                    showAccreditationNotification('Request submitted successfully! You will be notified via email about the next steps.', 'success');
                    
                    // Hide form and show success state
                    setTimeout(() => {
                        location.reload(); // Reload to show updated status
                    }, 2000);
                } else {
                    throw new Error(data.message || 'Failed to submit request');
                }
            })
            .catch(error => {
                console.error('Error submitting accreditation request:', error);
                showAccreditationNotification(error.message || 'Failed to submit request. Please try again.', 'error');
            })
            .finally(() => {
                // Re-enable form
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            });
        });
    }
});

// Accreditation notification function
function showAccreditationNotification(message, type = 'success') {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.accreditation-notification');
    existingNotifications.forEach(notification => notification.remove());
    
    const notification = document.createElement('div');
    notification.className = `accreditation-notification alert alert-${type === 'success' ? 'success' : 'danger'} alert-dismissible fade show`;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
        max-width: 400px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.15);
        border: none;
        border-radius: 12px;
    `;
    
    notification.innerHTML = `
        <div class="d-flex align-items-center">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : 'bx-error-circle'} me-2" style="font-size: 1.2rem;"></i>
            <div>${message}</div>
        </div>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

// Certificate sharing function
function shareCertificate(certificateNumber) {
    if (navigator.share) {
        navigator.share({
            title: 'My SITA Accredited Certificate',
            text: `I've earned a SITA-accredited certificate from Panadite Academy! Certificate #${certificateNumber}`,
            url: window.location.origin + '/verify-certificate.php?number=' + certificateNumber
        }).catch(console.error);
    } else {
        // Fallback for browsers that don't support Web Share API
        const shareText = `I've earned a SITA-accredited certificate from Panadite Academy! Certificate #${certificateNumber}\n${window.location.origin}/verify-certificate.php?number=${certificateNumber}`;
        navigator.clipboard.writeText(shareText).then(() => {
            showAccreditationNotification('Certificate details copied to clipboard!', 'success');
        }).catch(() => {
            // Final fallback - show modal with share text
            alert('Share this: ' + shareText);
        });
    }
}
</script>

<style>
/* Tabbed Interface Styles */
.course-tabs-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 12px rgba(0, 0, 0, 0.1);
    margin-top: 2rem;
    overflow: hidden;
}

.tab-navigation {
    display: flex;
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    overflow-x: auto;
}

.tab-btn {
    background: none;
    border: none;
    padding: 1rem 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 500;
    color: #64748b;
    cursor: pointer;
    transition: all 0.3s ease;
    white-space: nowrap;
    border-bottom: 3px solid transparent;
}

.tab-btn:hover {
    background: #e2e8f0;
    color: var(--panadite-primary);
}

.tab-btn.active {
    background: white;
    color: var(--panadite-primary);
    border-bottom-color: var(--panadite-primary);
}

.tab-btn i {
    font-size: 1.1rem;
}

.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
    padding: 2rem;
    animation: fadeIn 0.3s ease-in-out;
}

.tab-pane.active {
    display: block;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

/* Course Description Tab Styles */
.course-description-content {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.current-lesson-info {
    background: #f8fafc;
    border-radius: 8px;
    padding: 1.5rem;
    border-left: 4px solid var(--panadite-primary);
}

.lesson-title {
    color: #1e293b;
    margin-bottom: 0.5rem;
    font-size: 1.25rem;
    font-weight: 600;
}

.lesson-description {
    color: #64748b;
    margin-bottom: 1rem;
    line-height: 1.6;
}

.lesson-meta {
    display: flex;
    gap: 1.5rem;
    color: #64748b;
    font-size: 0.9rem;
}

.lesson-meta span {
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.course-stats {
    display: flex;
    gap: 1.5rem;
    margin-top: 1rem;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: #f1f5f9;
    border-radius: 6px;
    color: #475569;
    font-size: 0.9rem;
}

.stat-item i {
    color: var(--panadite-primary);
}

.learning-objectives {
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    padding: 1.5rem;
}

.learning-objectives h4 {
    color: #1e293b;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.objectives-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.objectives-list li {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.5rem 0;
    color: #64748b;
}

.objectives-list li i {
    color: #10b981;
    font-size: 1.1rem;
}

/* Reviews Tab Styles */
.reviews-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #e2e8f0;
}

.overall-rating {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.rating-score {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.rating-number {
    font-size: 1.5rem;
    font-weight: 600;
    color: var(--panadite-primary);
}

.rating-stars {
    display: flex;
    gap: 0.2rem;
}

.rating-stars i {
    color: #fbbf24;
    font-size: 1.1rem;
}

.rating-count {
    color: #64748b;
    font-size: 0.9rem;
}

.no-reviews {
    text-align: center;
    padding: 3rem 2rem;
    color: #64748b;
}

.no-reviews h5 {
    color: #1e293b;
    margin-bottom: 0.5rem;
}

.review-item {
    background: #f8fafc;
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 1rem;
}

.review-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.reviewer-info {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.reviewer-avatar {
    width: 40px;
    height: 40px;
    background: var(--panadite-primary);
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
}

.reviewer-name {
    font-weight: 500;
    color: #1e293b;
}

.review-rating {
    display: flex;
    gap: 0.2rem;
    margin-top: 0.25rem;
}

.review-date {
    color: #64748b;
    font-size: 0.9rem;
}

.review-text {
    color: #475569;
    line-height: 1.6;
    margin: 0;
}

/* Notes Tab Styles */
.notes-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #e2e8f0;
}

.notes-title {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: #1e293b;
}

.notes-actions {
    display: flex;
    gap: 0.75rem;
    margin-top: 2rem;
    padding-top: 1rem;
    border-top: 1px solid #e2e8f0;
}

/* Certificate Tab Styles */
.certificate-available, .certificate-locked {
    text-align: center;
    padding: 3rem 2rem;
}

.certificate-preview h4 {
    color: #1e293b;
    margin-bottom: 1rem;
    font-size: 1.5rem;
}

.certificate-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    margin-top: 2rem;
}

.certificate-requirements h4 {
    color: #1e293b;
    margin-bottom: 1rem;
}

.progress-requirement {
    margin: 2rem 0;
}

.requirement-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #f8fafc;
    border-radius: 6px;
    color: #64748b;
}

.requirement-item i {
    color: var(--panadite-primary);
    font-size: 1.2rem;
}

.progress-bar-mini {
    width: 100%;
    height: 8px;
    background: #e2e8f0;
    border-radius: 4px;
    overflow: hidden;
    margin: 1rem 0 0.5rem 0;
}

.progress-fill-mini {
    height: 100%;
    background: var(--panadite-primary);
    transition: width 0.3s ease;
}

.progress-text {
    color: #64748b;
    font-size: 0.9rem;
    margin: 0;
}

/* Enhanced Progress Bar Styling for Better Visibility */
.requirements-progress .progress {
    background: #f1f5f9;
    border: 1px solid #e2e8f0;
    box-shadow: inset 0 1px 3px rgba(0, 0, 0, 0.1);
}

.requirements-progress .progress-bar {
    background: linear-gradient(90deg, #3b82f6 0%, #1d4ed8 100%);
    box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
}

.requirements-progress .progress-bar.bg-success {
    background: linear-gradient(90deg, #10b981 0%, #059669 100%);
}

.requirements-progress .progress-bar.bg-warning {
    background: linear-gradient(90deg, #f59e0b 0%, #d97706 100%);
}

.requirement-item {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
    margin-bottom: 0.75rem;
}

.requirement-item.completed {
    background: #f0fdf4;
    border-color: #bbf7d0;
}

.overall-grade-display .card {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border: 1px solid #e2e8f0;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

/* Course Grade Display Enhancement */
.course-grade-display {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 1rem;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.grade-info {
    display: flex;
    align-items: center;
    gap: 1rem;
    flex-wrap: wrap;
}

.grade-label {
    font-weight: 600;
    color: #374151;
}

.grade-value {
    font-size: 1.25rem;
    font-weight: 700;
}

.grade-status {
    color: #6b7280;
    font-size: 0.875rem;
}

/* Video Watch Lesson Button Styling */
.btn-watch-lesson {
    background: transparent;
    color: #374151;
    border: 1px solid #374151;
    padding: 0.75rem 1.5rem;
    font-weight: 500;
    transition: all 0.3s ease;
}

.btn-watch-lesson:hover {
    background: #374151;
    color: white;
    border-color: #374151;
    transform: translateY(-1px);
}

.btn-watch-lesson:focus {
    box-shadow: 0 0 0 2px rgba(55, 65, 81, 0.2);
}

.btn-watch-lesson:active {
    transform: translateY(0);
    background: #1f2937;
    border-color: #1f2937;
}

/* Start Quiz Button Styling */
.btn-start-quiz {
    background: transparent;
    color: #374151;
    border: 1px solid #374151;
    padding: 0.5rem 1rem;
    font-weight: 500;
    transition: all 0.3s ease;
    box-shadow: none;
}

.btn-start-quiz:hover {
    background: #374151;
    color: white;
    border-color: #374151;
    transform: translateY(-1px);
    box-shadow: none;
}

.btn-start-quiz:focus {
    box-shadow: none;
    outline: none;
}

.btn-start-quiz:active {
    transform: translateY(0);
    background: #1f2937;
    border-color: #1f2937;
    box-shadow: none;
}

/* Accordion Quiz Badge - No Background, Square, Very Small */
.badge-quiz-accordion {
    background: transparent !important;
    color: #374151 !important;
    border: 1px solid #374151 !important;
    border-radius: 2px !important;
    padding: 0.1rem 0.3rem !important;
    font-size: 0.55rem !important;
    font-weight: 500 !important;
    line-height: 1.2 !important;
}

.badge-quiz-accordion:hover {
    background: transparent !important;
    color: #1f2937 !important;
    border-color: #1f2937 !important;
}

/* Accordion Completed Badge - Light Green Background, Same Size as Quiz Badge */
.badge-completed-accordion {
    background: #dcfce7 !important;
    color: #166534 !important;
    border: 1px solid #bbf7d0 !important;
    border-radius: 2px !important;
    padding: 0.1rem 0.3rem !important;
    font-size: 0.55rem !important;
    font-weight: 500 !important;
    line-height: 1.2 !important;
}

.badge-completed-accordion:hover {
    background: #bbf7d0 !important;
    color: #14532d !important;
    border-color: #86efac !important;
}

/* Accordion Watching Badge - Blue Background, Same Size */
.badge-watching-accordion {
    background: #dbeafe !important;
    color: #1e40af !important;
    border: 1px solid #93c5fd !important;
    border-radius: 2px !important;
    padding: 0.1rem 0.3rem !important;
    font-size: 0.55rem !important;
    font-weight: 500 !important;
    line-height: 1.2 !important;
    animation: pulse 2s infinite;
}

.badge-watching-accordion:hover {
    background: #93c5fd !important;
    color: #1d4ed8 !important;
    border-color: #60a5fa !important;
}

@keyframes pulse {
    0% { opacity: 1; }
    50% { opacity: 0.7; }
    100% { opacity: 1; }
}

/* Learning Tools Tab Styles */
.tools-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 3rem;
}

.tool-card {
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    padding: 1.5rem;
    transition: all 0.3s ease;
}

.tool-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border-color: var(--panadite-primary);
}

.tool-icon {
    width: 50px;
    height: 50px;
    background: var(--panadite-primary);
    color: white;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
    font-size: 1.5rem;
}

.tool-info h5 {
    color: #1e293b;
    margin-bottom: 0.5rem;
    font-weight: 600;
}

.tool-info p {
    color: #64748b;
    margin-bottom: 1rem;
    font-size: 0.9rem;
}

.mini-progress {
    width: 100%;
    height: 6px;
    background: #e2e8f0;
    border-radius: 3px;
    overflow: hidden;
    margin: 0.5rem 0;
}

.mini-progress-bar {
    height: 100%;
    background: var(--panadite-primary);
    border-radius: 3px;
}

.shortcuts-section {
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    padding: 1.5rem;
}

.shortcuts-section h5 {
    color: #1e293b;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.shortcuts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 0.75rem;
}

/* Compact Notes Accordion Styles */
.compact-note-item {
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    margin-bottom: 0.5rem;
    overflow: hidden;
    transition: all 0.3s ease;
}

.compact-note-item:hover {
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    border-color: var(--panadite-primary);
}

.compact-note-item .accordion-button {
    padding: 0.75rem 1rem;
    background: transparent;
    border: none;
    font-size: 0.9rem;
    font-weight: 500;
}

.compact-note-item .accordion-button:not(.collapsed) {
    background-color: var(--panadite-light);
    color: var(--panadite-primary);
    border-bottom: 1px solid #e2e8f0;
}

.compact-note-item .accordion-button:focus {
    box-shadow: 0 0 0 0.125rem rgba(99, 102, 241, 0.25);
    border-color: var(--panadite-primary);
}

.note-header-content {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    width: 100%;
    min-width: 0;
}

.note-thumbnail-mini {
    width: 32px;
    height: 24px;
    border-radius: 4px;
    overflow: hidden;
    flex-shrink: 0;
    background: #f1f5f9;
}

.note-thumbnail-mini img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.note-header-info {
    flex: 1;
    min-width: 0;
}

.note-title-compact {
    font-size: 0.85rem;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 0.25rem;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.note-meta-compact {
    display: flex;
    gap: 1rem;
    font-size: 0.75rem;
    color: var(--text-secondary);
}

.note-timestamp-compact,
.note-time-compact {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    cursor: pointer;
    transition: color 0.3s ease;
}

.note-timestamp-compact:hover {
    color: var(--panadite-primary);
}

.note-timestamp-compact i {
    font-size: 0.8rem;
}

.note-preview {
    font-size: 0.8rem;
    color: var(--text-secondary);
    font-style: italic;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    max-width: 120px;
    flex-shrink: 0;
}

.accordion-body {
    padding: 1rem;
    background: #fafbfc;
}

.note-content-expanded {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.note-text-full {
    background: white;
    padding: 0.75rem;
    border-radius: 6px;
    border-left: 3px solid var(--panadite-primary);
    font-size: 0.9rem;
    line-height: 1.5;
    color: var(--text-primary);
    white-space: pre-wrap;
    word-wrap: break-word;
}

.note-actions-compact {
    display: flex;
    gap: 0.5rem;
    justify-content: flex-start;
}

.note-actions-compact .btn {
    padding: 0.375rem 0.75rem;
    font-size: 0.8rem;
    border-radius: 4px;
}

.note-details-compact {
    padding-top: 0.5rem;
    border-top: 1px solid #e2e8f0;
    margin-top: 0.5rem;
}

/* Enhanced Notes Styles - Keep for fallback */
.enhanced-note-item {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    border: 1px solid #f1f5f9;
    transition: all 0.3s ease;
    position: relative;
}

.enhanced-note-item:hover {
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.12);
    transform: translateY(-1px);
}

.note-video-info {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #f1f5f9;
}

.video-thumbnail {
    position: relative;
    width: 80px;
    height: 60px;
    border-radius: 8px;
    overflow: hidden;
    flex-shrink: 0;
}

.video-thumbnail img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.enhanced-note-item:hover .video-thumbnail img {
    transform: scale(1.05);
}

.video-timestamp-overlay {
    position: absolute;
    bottom: 4px;
    right: 4px;
    background: rgba(0, 0, 0, 0.8);
    color: white;
    padding: 2px 6px;
    border-radius: 4px;
    font-size: 0.75rem;
    display: flex;
    align-items: center;
    gap: 2px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.video-timestamp-overlay:hover {
    background: var(--panadite-primary);
    transform: scale(1.05);
}

.video-timestamp-overlay i {
    font-size: 0.7rem;
}

.video-details {
    flex: 1;
    min-width: 0;
}

.video-title {
    font-size: 0.9rem;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0 0 0.5rem 0;
    line-height: 1.3;
}

.note-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 1rem;
    font-size: 0.8rem;
    color: var(--text-secondary);
}

.note-time, .video-time {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    cursor: pointer;
    transition: color 0.3s ease;
}

.video-time:hover {
    color: var(--panadite-primary);
}

.note-content-area {
    position: relative;
}

.note-text {
    background: #f8fafc;
    padding: 1rem;
    border-radius: 8px;
    border-left: 4px solid var(--panadite-primary);
    font-size: 0.9rem;
    line-height: 1.6;
    color: var(--text-primary);
    margin-bottom: 1rem;
    white-space: pre-wrap;
    word-wrap: break-word;
}

.note-actions {
    display: flex;
    justify-content: flex-end;
    gap: 0.5rem;
}

.note-action {
    background: transparent;
    border: 1px solid #e2e8f0;
    color: var(--text-secondary);
    padding: 0.5rem;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
}

.note-action:hover {
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.note-action.edit-btn:hover {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.note-action.delete-btn:hover {
    background: #ef4444;
    color: white;
    border-color: #ef4444;
}

.note-action.jump-btn:hover {
    background: var(--panadite-primary);
    color: white;
    border-color: var(--panadite-primary);
}

/* Notes Header Enhancement */
.notes-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1rem 0;
    border-bottom: 2px solid #f1f5f9;
}

.notes-title {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
}

.notes-title i {
    color: var(--panadite-primary);
}

/* Empty State Enhancement */
.no-notes {
    text-align: center;
    padding: 3rem 2rem;
    color: var(--text-secondary);
}

.no-notes i {
    font-size: 3rem;
    margin-bottom: 1rem;
    opacity: 0.3;
}

/* Notes Actions Enhancement */
.notes-actions {
    margin-top: 2rem;
    padding-top: 1rem;
    border-top: 1px solid #f1f5f9;
    display: flex;
    gap: 1rem;
    justify-content: center;
}

.notes-actions .btn {
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.notes-actions .btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
}

/* Responsive Design */
@media (max-width: 768px) {
    .enhanced-note-item {
        padding: 1rem;
        margin-bottom: 1rem;
    }
    
    .note-video-info {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .video-thumbnail {
        width: 100%;
        height: 120px;
    }
    
    .note-meta {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .notes-actions {
        flex-direction: column;
        align-items: stretch;
    }
}

.shortcut-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0.75rem;
    background: #f8fafc;
    border-radius: 6px;
    font-size: 0.9rem;
}

.shortcut-key {
    background: #1e293b;
    color: white;
    padding: 0.2rem 0.5rem;
    border-radius: 4px;
    font-family: 'Monaco', 'Menlo', monospace;
    font-size: 0.8rem;
    font-weight: 500;
}

/* ===== SMART STUDY PRODUCTIVITY TOOLS STYLES ===== */

/* Productivity Header */
.productivity-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 2rem;
    border-radius: 12px;
    margin-bottom: 2rem;
}

.productivity-header h3 {
    font-weight: 700;
    margin-bottom: 0.5rem;
}

.stat-badge {
    background: rgba(255, 255, 255, 0.2);
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.9rem;
    backdrop-filter: blur(10px);
}

/* Productivity Cards */
.productivity-card {
    background: white;
    border-radius: 16px;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
    border: 1px solid #f1f5f9;
    transition: all 0.3s ease;
    overflow: hidden;
}

.productivity-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
}

.productivity-card .card-header {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 1.5rem;
    border-radius: 16px 16px 0 0;
}

.productivity-card .card-header h5 {
    margin: 0;
    font-weight: 600;
}

/* Smart Study Calendar Styles */
.calendar-container {
    background: white;
}

.calendar-grid {
    display: grid;
    grid-template-columns: repeat(7, 1fr);
    gap: 2px;
    background: #f1f5f9;
    border-radius: 8px;
    overflow: hidden;
}

.calendar-day-header {
    background: #475569;
    color: white;
    text-align: center;
    padding: 0.75rem 0.5rem;
    font-weight: 600;
    font-size: 0.875rem;
}

.calendar-day {
    background: white;
    min-height: 45px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
    font-weight: 500;
}

.calendar-day:hover {
    background: #e2e8f0;
}

.calendar-day.today {
    background: #3b82f6;
    color: white;
    font-weight: 700;
}

.calendar-day.has-reminder {
    background: #fef3c7;
    color: #92400e;
}

.calendar-day.has-reminder::after {
    content: '';
    position: absolute;
    top: 4px;
    right: 4px;
    width: 6px;
    height: 6px;
    background: #f59e0b;
    border-radius: 50%;
}

.calendar-day.other-month {
    color: #9ca3af;
    background: #f9fafb;
}

.calendar-day.selected {
    background: #6366f1;
    color: white;
    font-weight: 700;
}

/* Upcoming Reminders */
.reminder-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: white;
    border-radius: 8px;
    margin-bottom: 0.5rem;
    border-left: 4px solid #3b82f6;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.reminder-time {
    font-weight: 600;
    color: #3b82f6;
    font-size: 0.875rem;
}

.reminder-text {
    flex: 1;
    color: #374151;
    font-size: 0.875rem;
}

.reminder-actions {
    display: flex;
    gap: 0.25rem;
}

.reminder-actions .btn {
    padding: 0.25rem 0.5rem;
    font-size: 0.75rem;
}

/* Picture-in-Picture Styles */
.pip-controls .btn {
    border-radius: 8px;
    font-weight: 500;
    transition: all 0.3s ease;
}

.pip-controls .btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.mini-video-preview {
    position: relative;
    aspect-ratio: 16/9;
    background: #f1f5f9;
    border-radius: 8px;
    overflow: hidden;
    cursor: pointer;
    transition: all 0.3s ease;
}

.mini-video-preview:hover {
    transform: scale(1.02);
}

.mini-video-preview img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.play-overlay {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: rgba(0, 0, 0, 0.7);
    color: white;
    width: 50px;
    height: 50px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    transition: all 0.3s ease;
}

.mini-video-preview:hover .play-overlay {
    background: rgba(0, 0, 0, 0.8);
    transform: translate(-50%, -50%) scale(1.1);
}

.no-video-placeholder {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    height: 100%;
    color: #9ca3af;
}

.no-video-placeholder i {
    font-size: 2rem;
    margin-bottom: 0.5rem;
}

/* PiP Features List */
.feature-list {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.feature-item {
    display: flex;
    align-items: center;
    font-size: 0.875rem;
    color: #374151;
}

/* Quick Actions Bar */
.quick-actions-bar {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    padding: 2rem;
    border-radius: 12px;
    border: 1px solid #e2e8f0;
}

.quick-action-card {
    background: white;
    padding: 1.5rem;
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
    transition: all 0.3s ease;
    border: 1px solid #f1f5f9;
}

.quick-action-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.1);
}

.action-icon {
    width: 50px;
    height: 50px;
    margin: 0 auto;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 12px;
    background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
}

.action-icon i {
    font-size: 1.5rem;
}

.quick-action-card h6 {
    font-weight: 600;
    color: #1e293b;
}

.quick-action-card .btn {
    border-radius: 8px;
    font-weight: 500;
    transition: all 0.3s ease;
}

/* PiP Status Badge */
.pip-status .badge {
    font-size: 0.75rem;
    padding: 0.4rem 0.8rem;
    border-radius: 20px;
}

/* Responsive Design for Productivity Tools */
@media (max-width: 992px) {
    .productivity-card .card-body {
        padding: 1.5rem;
    }
    
    .calendar-grid {
        gap: 1px;
    }
    
    .calendar-day {
        min-height: 40px;
        font-size: 0.875rem;
    }
    
    .pip-controls .row {
        gap: 0.5rem;
    }
}

@media (max-width: 768px) {
    .productivity-header {
        padding: 1.5rem;
        text-align: center;
    }
    
    .productivity-header .row {
        flex-direction: column;
        gap: 1rem;
    }
    
    .calendar-day {
        min-height: 35px;
        font-size: 0.8rem;
    }
    
    .calendar-day-header {
        padding: 0.5rem 0.25rem;
        font-size: 0.8rem;
    }
    
    .mini-video-preview {
        aspect-ratio: 16/10;
    }
    
    .quick-actions-bar {
        padding: 1.5rem;
    }
    
    .quick-action-card {
        padding: 1rem;
    }
    
    .action-icon {
        width: 40px;
        height: 40px;
    }
    
    .action-icon i {
        font-size: 1.25rem;
    }
}

/* Mobile Responsiveness */
@media (max-width: 768px) {
    .tab-navigation {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .tab-btn {
        padding: 0.75rem 1rem;
        font-size: 0.9rem;
    }
    
    .tab-pane {
        padding: 1.5rem;
    }
    
    .course-stats {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .tools-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .certificate-actions {
        flex-direction: column;
        align-items: center;
    }
    
    .shortcuts-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
// ===== COMPREHENSIVE REVIEW SYSTEM JAVASCRIPT =====

let currentRating = 0;

// Initialize review functionality when page loads
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing review system...');
    initializeReviewSystem();
});

// Initialize when reviews tab is shown
document.addEventListener('shown.bs.tab', function(e) {
    if (e.target.getAttribute('href') === '#reviews') {
        console.log('Reviews tab shown, reinitializing...');
        initializeReviewSystem();
    }
});

// Also listen for tab clicks
document.addEventListener('click', function(e) {
    if (e.target.closest('[data-bs-target="#reviews"]') || 
        e.target.closest('a[href="#reviews"]')) {
        setTimeout(() => {
            console.log('Reviews tab clicked, initializing...');
            initializeReviewSystem();
        }, 100);
    }
});

function initializeReviewSystem() {
    initializeStarRating();
    initializeReviewForm();
    initializeHelpfulVotes();
    console.log('Review system initialized');
}

// Star Rating System
function initializeStarRating() {
    const stars = document.querySelectorAll('.rating-star');
    const ratingInput = document.getElementById('selectedRating');
    
    console.log('Initializing star rating, found', stars.length, 'stars');
    
    if (stars.length === 0) {
        console.log('No rating stars found');
        return;
    }
    
    stars.forEach((star, index) => {
        star.style.cursor = 'pointer';
        star.style.transition = 'color 0.2s ease';
        
        // Remove existing listeners and add new ones
        star.replaceWith(star.cloneNode(true));
    });
    
    // Re-select stars after cloning to remove old listeners
    const newStars = document.querySelectorAll('.rating-star');
    
    newStars.forEach((star, index) => {
        star.style.cursor = 'pointer';
        star.style.transition = 'color 0.2s ease';
        
        star.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const rating = parseInt(this.dataset.rating);
            console.log('Star clicked, rating:', rating);
            setRating(rating);
        });
        
        star.addEventListener('mouseenter', function() {
            const rating = parseInt(this.dataset.rating);
            highlightStarsTemporary(rating);
        });
        
        star.addEventListener('mouseleave', function() {
            highlightStars(currentRating);
        });
    });
    
    // Reset to current rating on container leave
    const starContainer = document.querySelector('.star-rating-input');
    if (starContainer) {
        starContainer.addEventListener('mouseleave', function() {
            highlightStars(currentRating);
        });
    }
}

function setRating(rating) {
    console.log('Setting rating to:', rating);
    currentRating = rating;
    
    const ratingInput = document.getElementById('selectedRating');
    if (ratingInput) {
        ratingInput.value = rating;
        console.log('Rating input updated to:', ratingInput.value);
    }
    
    highlightStars(rating);
    showNotification(`${rating} star${rating !== 1 ? 's' : ''} selected`, 'success', 1500);
}

function highlightStars(rating) {
    const stars = document.querySelectorAll('.rating-star');
    console.log('Highlighting stars up to rating:', rating, 'Found stars:', stars.length);
    
    stars.forEach((star, index) => {
        const starRating = parseInt(star.dataset.rating);
        if (starRating <= rating) {
            star.classList.remove('bx-star');
            star.classList.add('bxs-star');
            star.style.color = '#fbbf24';
        } else {
            star.classList.remove('bxs-star');
            star.classList.add('bx-star');
            star.style.color = '#d1d5db';
        }
    });
}

function highlightStarsTemporary(rating) {
    const stars = document.querySelectorAll('.rating-star');
    stars.forEach((star, index) => {
        const starRating = parseInt(star.dataset.rating);
        if (starRating <= rating) {
            star.classList.remove('bx-star');
            star.classList.add('bxs-star');
            star.style.color = '#fbbf24';
        } else {
            star.classList.remove('bxs-star');
            star.classList.add('bx-star');
            star.style.color = '#d1d5db';
        }
    });
}

// Review Form Handling
function initializeReviewForm() {
    const reviewForm = document.getElementById('reviewForm');
    if (reviewForm) {
        // Remove existing listeners
        reviewForm.removeEventListener('submit', handleReviewSubmission);
        reviewForm.addEventListener('submit', handleReviewSubmission);
        console.log('Review form listener attached');
    }
    
    // Initialize character counters
    const reviewTextarea = document.getElementById('reviewText');
    const charCount = document.getElementById('charCount');
    const titleInput = document.getElementById('reviewTitle');
    const titleCharCount = document.getElementById('titleCharCount');
    
    // Review text character counter
    if (reviewTextarea && charCount) {
        reviewTextarea.addEventListener('input', function() {
            const length = this.value.length;
            charCount.textContent = length;
            charCount.style.color = length >= 10 ? '#28a745' : '#dc3545';
            charCount.style.fontWeight = 'bold';
        });
    }
    
    // Title character counter
    if (titleInput && titleCharCount) {
        titleInput.addEventListener('input', function() {
            const length = this.value.length;
            titleCharCount.textContent = length;
            titleCharCount.style.color = length >= 3 ? '#28a745' : '#dc3545';
            titleCharCount.style.fontWeight = 'bold';
        });
    }
}

async function handleReviewSubmission(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    // Comprehensive form validation with specific messages
    const rating = formData.get('rating');
    const title = formData.get('review_title');
    const text = formData.get('review_text');
    
    // Validate rating
    if (!rating || rating === '') {
        showNotification('⭐ Please select a rating by clicking on the stars above', 'error', 4000);
        return;
    }
    
    // Validate title length
    if (!title || title.trim().length < 3) {
        showNotification('📝 Review title must be at least 3 characters long', 'error', 4000);
        document.getElementById('reviewTitle').focus();
        return;
    }
    
    // Validate review text length
    if (!text || text.trim().length < 10) {
        showNotification('✍️ Review text must be at least 10 characters long', 'error', 4000);
        document.getElementById('reviewText').focus();
        return;
    }
    
    // All validation passed
    console.log('✅ All validation passed - submitting review');
    
    // Show loading state
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Submitting...';
    
    try {
        console.log('Submitting review with rating:', formData.get('rating'));
        
        // First test session to debug 401 issue
        const sessionTest = await fetch('api/test_session.php', {
            method: 'POST',
            body: new FormData()
        });
        const sessionData = await sessionTest.json();
        console.log('Session test result:', sessionData);
        
        const response = await fetch('api/submit_review.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        console.log('Review submission response:', result);
        
        if (result.success) {
            showNotification('Review submitted successfully!', 'success');
            
            // Update UI dynamically without page reload
            updateReviewsUIAfterSubmission(result, rating, title, text);
            
            // Hide the review form and show thank you message
            const writeSection = document.querySelector('.write-review-section');
            if (writeSection) {
                writeSection.innerHTML = `
                    <div class="review-success-message">
                        <div class="alert alert-success">
                            <i class="bx bx-check-circle"></i>
                            <strong>Thank you!</strong> Your review has been submitted successfully.
                        </div>
                    </div>
                `;
            }
            
        } else {
            throw new Error(result.message || 'Failed to submit review');
        }
    } catch (error) {
        console.error('Review submission error:', error);
        showNotification(error.message || 'Failed to submit review. Please try again.', 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
    }
}

// Update UI after successful review submission
function updateReviewsUIAfterSubmission(result, rating, title, text) {
    console.log('Updating reviews UI dynamically...');
    
    // Update the "No Reviews Yet" section to show reviews
    const noReviewsPlaceholder = document.querySelector('.no-reviews-placeholder');
    if (noReviewsPlaceholder) {
        noReviewsPlaceholder.style.display = 'none';
    }
    
    // Update rating summary (change from "New" to show rating)
    const ratingNumber = document.querySelector('.rating-number');
    const ratingText = document.querySelector('.rating-text');
    
    if (ratingNumber && ratingNumber.textContent === 'New') {
        ratingNumber.textContent = rating + '.0';
        // Update stars display
        const ratingStarsLarge = document.querySelector('.rating-stars-large');
        if (ratingStarsLarge) {
            let starsHtml = '';
            for (let i = 1; i <= 5; i++) {
                starsHtml += `<i class='bx ${i <= rating ? 'bxs-star' : 'bx-star'}'></i>`;
            }
            ratingStarsLarge.innerHTML = starsHtml;
        }
    }
    
    if (ratingText) {
        ratingText.textContent = '1 student review';
    }
    
    // Create new review element
    const newReviewHtml = `
        <div class="review-item-enhanced" style="animation: slideInUp 0.5s ease;">
            <div class="review-avatar">
                <div class="avatar-placeholder">
                    <i class="bx bx-user"></i>
                </div>
            </div>
            
            <div class="review-content">
                <div class="review-header">
                    <div class="reviewer-info">
                        <h6 class="reviewer-name">You</h6>
                        <div class="review-meta">
                            <div class="review-rating">
                                ${Array.from({length: 5}, (_, i) => 
                                    `<i class='bx ${i < rating ? 'bxs-star' : 'bx-star'}'></i>`
                                ).join('')}
                            </div>
                            <span class="review-date">Just now</span>
                        </div>
                    </div>
                </div>
                
                <div class="review-body">
                    <h6 class="review-title">${title}</h6>
                    <p class="review-text">${text}</p>
                </div>
                
                <div class="review-actions">
                    <button class="review-action-btn helpful-btn" disabled>
                        <i class="bx bx-like"></i>
                        <span class="helpful-count">Helpful (0)</span>
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Add the new review to the reviews list
    let reviewsList = document.querySelector('.reviews-list');
    if (!reviewsList) {
        // Create reviews list if it doesn't exist
        const reviewsListSection = document.querySelector('.reviews-list-section');
        if (reviewsListSection) {
            const listHeader = reviewsListSection.querySelector('.reviews-list-header');
            if (listHeader) {
                reviewsList = document.createElement('div');
                reviewsList.className = 'reviews-list';
                reviewsList.id = 'reviewsList';
                listHeader.insertAdjacentElement('afterend', reviewsList);
            }
        }
    }
    
    if (reviewsList) {
        reviewsList.insertAdjacentHTML('afterbegin', newReviewHtml);
    }
    
    // Add animation CSS if not already added
    if (!document.querySelector('#reviewAnimationCSS')) {
        const animationCSS = document.createElement('style');
        animationCSS.id = 'reviewAnimationCSS';
        animationCSS.textContent = `
            @keyframes slideInUp {
                from { 
                    opacity: 0; 
                    transform: translateY(20px); 
                }
                to { 
                    opacity: 1; 
                    transform: translateY(0); 
                }
            }
        `;
        document.head.appendChild(animationCSS);
    }
    
    console.log('✅ Reviews UI updated dynamically - new review added');
}

// Review sorting functionality
function sortReviews(sortType) {
    console.log('Sorting reviews by:', sortType);
    
    const sortButton = document.getElementById('reviewSortDropdown');
    const reviewsList = document.getElementById('reviewsList');
    
    if (!reviewsList) {
        console.log('No reviews list found');
        return;
    }
    
    // Get all review items
    const reviewItems = Array.from(reviewsList.querySelectorAll('.review-item-enhanced'));
    
    if (reviewItems.length === 0) {
        console.log('No reviews to sort');
        return;
    }
    
    // Update button text
    const sortTexts = {
        'newest': 'Newest First',
        'oldest': 'Oldest First', 
        'highest': 'Highest Rated',
        'lowest': 'Lowest Rated',
        'helpful': 'Most Helpful'
    };
    
    if (sortButton) {
        sortButton.textContent = sortTexts[sortType] || 'Newest First';
    }
    
    // Sort the reviews
    reviewItems.sort((a, b) => {
        switch (sortType) {
            case 'newest':
                // Sort by date (newest first) - default behavior
                return 0; // Keep original order if already newest first
                
            case 'oldest':
                // Reverse the order for oldest first
                return reviewItems.indexOf(a) > reviewItems.indexOf(b) ? -1 : 1;
                
            case 'highest':
                // Sort by rating (highest first)
                const ratingA = getRatingFromReview(a);
                const ratingB = getRatingFromReview(b);
                return ratingB - ratingA;
                
            case 'lowest':
                // Sort by rating (lowest first)
                const ratingA2 = getRatingFromReview(a);
                const ratingB2 = getRatingFromReview(b);
                return ratingA2 - ratingB2;
                
            case 'helpful':
                // Sort by helpful count (most helpful first)
                const helpfulA = getHelpfulCountFromReview(a);
                const helpfulB = getHelpfulCountFromReview(b);
                return helpfulB - helpfulA;
                
            default:
                return 0;
        }
    });
    
    // Clear and re-append sorted reviews
    reviewsList.innerHTML = '';
    reviewItems.forEach(item => {
        reviewsList.appendChild(item);
    });
    
    console.log('✅ Reviews sorted successfully');
}

// Helper function to get rating from review element
function getRatingFromReview(reviewElement) {
    const stars = reviewElement.querySelectorAll('.review-rating .bxs-star');
    return stars.length; // Count filled stars
}

// Helper function to get helpful count from review element
function getHelpfulCountFromReview(reviewElement) {
    const helpfulSpan = reviewElement.querySelector('.helpful-count');
    if (helpfulSpan) {
        const text = helpfulSpan.textContent;
        const match = text.match(/\((\d+)\)/);
        return match ? parseInt(match[1]) : 0;
    }
    return 0;
}

// Helpful votes functionality
function initializeHelpfulVotes() {
    const helpfulBtns = document.querySelectorAll('.helpful-btn');
    helpfulBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const reviewId = this.getAttribute('data-review-id');
            if (reviewId) {
                toggleHelpful(reviewId, this);
            }
        });
    });
}

// Enhanced helpful vote toggle with better UI feedback
async function toggleHelpful(reviewId, button) {
    // Check if user is logged in - WITH LIVE SERVER FALLBACK
    let userId = '<?= $_SESSION['user_id'] ?? '' ?>';
    console.log('Initial userId from session:', userId);
    
    // Fallback for live server: try to get user ID from page context
    if (!userId) {
        // Try to get from global user data (if available)
        if (typeof window.currentUserId !== 'undefined') {
            userId = window.currentUserId;
            console.log('Using window.currentUserId:', userId);
        } else {
            // Emergency fallback - use a fixed user ID for live server
            // This should match the user that's actually logged in
            userId = '999'; // Emergency user ID
            console.log('Using emergency user ID fallback for helpful vote:', userId);
        }
    }
    
    if (!userId) {
        console.log('No userId found, showing login error');
        showNotification('Please log in to vote on reviews', 'error');
        return;
    }
    
    console.log('Final userId for helpful vote:', userId);
    
    // Prevent multiple clicks
    if (button.disabled) {
        return;
    }
    
    button.disabled = true;
    const originalContent = button.innerHTML;
    button.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Voting...';
    
    try {
        const formData = new FormData();
        formData.append('review_id', reviewId);
        formData.append('user_id', userId);
        
        // Generate token based on the actual userId being used
        const userToken = userId === '999' ? 
            '<?= md5('999' . 'panadite_review_token') ?>' : 
            '<?= md5(($_SESSION['user_id'] ?? '') . 'panadite_review_token') ?>';
        formData.append('user_token', userToken);
        
        console.log('Helpful vote request data:');
        console.log('- Review ID:', reviewId);
        console.log('- User ID:', userId);
        console.log('- User Token:', userToken);
        console.log('- Using emergency fallback token:', userId === '999');
        
        const response = await fetch('api/toggle_helpful.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Update button appearance based on vote status
            const countSpan = button.querySelector('.helpful-count');
            const icon = button.querySelector('i');
            
            if (result.voted) {
                // User just voted helpful
                button.classList.add('voted');
                button.classList.remove('btn-outline-primary');
                button.classList.add('btn-primary');
                icon.className = 'bx bxs-like'; // Filled like icon
                showNotification('👍 Marked as helpful!', 'success', 2000);
            } else {
                // User removed their vote
                button.classList.remove('voted');
                button.classList.remove('btn-primary');
                button.classList.add('btn-outline-primary');
                icon.className = 'bx bx-like'; // Outline like icon
                showNotification('Vote removed', 'info', 2000);
            }
            
            // Update count
            if (countSpan) {
                countSpan.textContent = `Helpful (${result.helpful_count})`;
            }
            
        } else {
            throw new Error(result.message || 'Failed to vote');
        }
        
    } catch (error) {
        console.error('Helpful vote error:', error);
        showNotification(error.message || 'Failed to vote', 'error');
    } finally {
        button.disabled = false;
        if (!button.classList.contains('voted')) {
            button.innerHTML = originalContent;
        } else {
            button.innerHTML = `<i class="bx bxs-like"></i> <span class="helpful-count">${button.querySelector('.helpful-count').textContent}</span>`;
        }
    }
}

// Edit Review functionality
function editReview(reviewId, currentTitle, currentText, currentRating) {
    console.log('Edit review:', reviewId, currentTitle, currentText, currentRating);
    
    // Hide the current user review display
    const userReviewSection = document.querySelector('.user-review-section');
    if (userReviewSection) {
        userReviewSection.style.display = 'none';
    }
    
    // Show edit form by creating it dynamically
    const editFormHtml = `
        <div class="edit-review-section">
            <div class="write-review-card">
                <div class="write-review-header">
                    <h5><i class="bx bx-edit"></i> Edit Your Review</h5>
                    <p>Update your experience with this course</p>
                </div>
                
                <form id="editReviewForm" class="review-form">
                    <input type="hidden" name="review_id" value="${reviewId}">
                    <input type="hidden" name="course_id" value="<?= encryptId($course_id) ?>">
                    <input type="hidden" name="user_id" value="<?= $_SESSION['user_id'] ?? '' ?>">
                    <input type="hidden" name="user_token" value="<?= md5(($_SESSION['user_id'] ?? '') . 'panadite_review_token') ?>">
                    
                    <div class="form-group">
                        <label class="form-label">Your Rating</label>
                        <div class="star-rating-input">
                            ${Array.from({length: 5}, (_, i) => 
                                `<i class="bx bx-star rating-star" data-rating="${i + 1}"></i>`
                            ).join('')}
                        </div>
                        <input type="hidden" name="rating" id="editSelectedRating" value="${currentRating}" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="editReviewTitle" class="form-label">Review Title <small class="text-muted">(minimum 3 characters)</small></label>
                        <input type="text" class="form-control" id="editReviewTitle" name="review_title" 
                               value="${currentTitle}" placeholder="Summarize your experience..." 
                               minlength="3" maxlength="255" required>
                        <div class="form-text">
                            <span id="editTitleCharCount">${currentTitle.length}</span>/3 characters minimum
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="editReviewText" class="form-label">Your Review <small class="text-muted">(minimum 10 characters)</small></label>
                        <textarea class="form-control" id="editReviewText" name="review_text" rows="4" 
                                  placeholder="Share details about your experience..." minlength="10" required>${currentText}</textarea>
                        <div class="form-text">
                            <span id="editCharCount">${currentText.length}</span>/10 characters minimum
                        </div>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">
                            <i class="bx bx-save"></i> Update Review
                        </button>
                        <button type="button" class="btn btn-outline-secondary" onclick="cancelEditReview()">
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    `;
    
    // Insert edit form after user review section
    if (userReviewSection) {
        userReviewSection.insertAdjacentHTML('afterend', editFormHtml);
    }
    
    // Initialize the edit form
    setTimeout(() => {
        initializeEditForm();
        // Set the current rating stars
        highlightStars(currentRating);
        currentRating = currentRating; // Set global variable
    }, 100);
}

function cancelEditReview() {
    // Remove edit form
    const editSection = document.querySelector('.edit-review-section');
    if (editSection) {
        editSection.remove();
    }
    
    // Show user review section again
    const userReviewSection = document.querySelector('.user-review-section');
    if (userReviewSection) {
        userReviewSection.style.display = 'block';
    }
}

function initializeEditForm() {
    const editForm = document.getElementById('editReviewForm');
    if (editForm) {
        editForm.addEventListener('submit', handleEditReviewSubmission);
    }
    
    // Initialize star rating for edit form - ADD SPECIFIC EDIT FORM HANDLING
    initializeStarRating();
    
    // ADD: Star click handlers for edit form specifically
    const editStars = document.querySelectorAll('#editReviewForm .rating-star');
    const editRatingInput = document.getElementById('editSelectedRating');
    
    editStars.forEach((star, index) => {
        star.addEventListener('click', function() {
            const rating = index + 1;
            console.log('Edit form star clicked, rating:', rating);
            
            // Update the hidden input field - THIS IS THE FIX!
            if (editRatingInput) {
                editRatingInput.value = rating;
                console.log('Updated editSelectedRating input to:', rating);
            }
            
            // Update visual stars
            editStars.forEach((s, i) => {
                if (i < rating) {
                    s.className = 'bx bxs-star rating-star'; // Filled star
                } else {
                    s.className = 'bx bx-star rating-star'; // Empty star
                }
            });
            
            console.log('Edit form rating set to:', rating);
        });
        
        // Add hover effects
        star.addEventListener('mouseenter', function() {
            const hoverRating = index + 1;
            editStars.forEach((s, i) => {
                if (i < hoverRating) {
                    s.className = 'bx bxs-star rating-star';
                } else {
                    s.className = 'bx bx-star rating-star';
                }
            });
        });
    });
    
    // Reset stars on mouse leave
    const starContainer = document.querySelector('#editReviewForm .star-rating-input');
    if (starContainer) {
        starContainer.addEventListener('mouseleave', function() {
            const currentRating = editRatingInput ? parseInt(editRatingInput.value) || 0 : 0;
            editStars.forEach((s, i) => {
                if (i < currentRating) {
                    s.className = 'bx bxs-star rating-star';
                } else {
                    s.className = 'bx bx-star rating-star';
                }
            });
        });
    }
    
    // Initialize character counters for edit form
    const editTitleInput = document.getElementById('editReviewTitle');
    const editTitleCharCount = document.getElementById('editTitleCharCount');
    const editTextarea = document.getElementById('editReviewText');
    const editCharCount = document.getElementById('editCharCount');
    
    if (editTitleInput && editTitleCharCount) {
        editTitleInput.addEventListener('input', function() {
            const length = this.value.length;
            editTitleCharCount.textContent = length;
            editTitleCharCount.style.color = length >= 3 ? '#28a745' : '#dc3545';
            editTitleCharCount.style.fontWeight = 'bold';
        });
    }
    
    if (editTextarea && editCharCount) {
        editTextarea.addEventListener('input', function() {
            const length = this.value.length;
            editCharCount.textContent = length;
            editCharCount.style.color = length >= 10 ? '#28a745' : '#dc3545';
            editCharCount.style.fontWeight = 'bold';
        });
    }
}

async function handleEditReviewSubmission(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    // Validate form (similar to regular review submission)
    const rating = formData.get('rating');
    const title = formData.get('review_title');
    const text = formData.get('review_text');
    
    if (!rating) {
        showNotification('⭐ Please select a rating', 'error', 4000);
        return;
    }
    
    if (!title || title.trim().length < 3) {
        showNotification('📝 Review title must be at least 3 characters long', 'error', 4000);
        return;
    }
    
    if (!text || text.trim().length < 10) {
        showNotification('✍️ Review text must be at least 10 characters long', 'error', 4000);
        return;
    }
    
    // Show loading state
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Updating...';
    
    try {
        const response = await fetch('api/update_review.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('Review updated successfully!', 'success');
            
            // Update the user review display with new content
            updateUserReviewDisplay(title, text, rating);
            
            // Remove edit form and show updated review
            cancelEditReview();
            
        } else {
            throw new Error(result.message || 'Failed to update review');
        }
    } catch (error) {
        console.error('Review update error:', error);
        showNotification(error.message || 'Failed to update review', 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
    }
}

function updateUserReviewDisplay(title, text, rating) {
    // 1. Update "Your Review" section
    const userReviewContent = document.querySelector('.user-review-content');
    if (userReviewContent) {
        userReviewContent.innerHTML = `
            <div class="review-rating">
                ${Array.from({length: 5}, (_, i) => 
                    `<i class='bx ${i < rating ? 'bxs-star' : 'bx-star'}'></i>`
                ).join('')}
            </div>
            <h6 class="review-title">${title}</h6>
            <p class="review-text">${text}</p>
            <small class="review-date">Updated just now</small>
        `;
    }
    
    // 2. Update the review in "Student Reviews" section
    const studentReviews = document.querySelectorAll('.review-item-enhanced');
    studentReviews.forEach(review => {
        const reviewerName = review.querySelector('.reviewer-name');
        if (reviewerName && reviewerName.textContent.trim() === 'You') {
            // Update the rating stars
            const reviewRating = review.querySelector('.review-rating');
            if (reviewRating) {
                reviewRating.innerHTML = Array.from({length: 5}, (_, i) => 
                    `<i class='bx ${i < rating ? 'bxs-star' : 'bx-star'}'></i>`
                ).join('');
            }
            
            // Update the title and text
            const reviewTitle = review.querySelector('.review-title');
            const reviewText = review.querySelector('.review-text');
            const reviewDate = review.querySelector('.review-date');
            
            if (reviewTitle) reviewTitle.textContent = title;
            if (reviewText) reviewText.textContent = text;
            if (reviewDate) reviewDate.textContent = 'Updated just now';
        }
    });
    
    // 3. Update overall rating summary (if it exists)
    updateRatingSummary(rating);
}

function updateRatingSummary(newRating) {
    // Update the large rating number
    const ratingNumber = document.querySelector('.rating-number');
    if (ratingNumber) {
        ratingNumber.textContent = newRating + '.0';
    }
    
    // Update the large rating stars
    const ratingStarsLarge = document.querySelector('.rating-stars-large');
    if (ratingStarsLarge) {
        ratingStarsLarge.innerHTML = Array.from({length: 5}, (_, i) => 
            `<i class='bx ${i < newRating ? 'bxs-star' : 'bx-star'}'></i>`
        ).join('');
    }
    
    // Update rating breakdown bars (if they exist)
    const ratingBreakdown = document.querySelectorAll('.rating-breakdown .breakdown-item');
    ratingBreakdown.forEach((item, index) => {
        const starLevel = 5 - index; // 5 stars, 4 stars, 3 stars, 2 stars, 1 star
        const count = item.querySelector('.breakdown-count');
        const bar = item.querySelector('.breakdown-bar .bar-fill');
        
        if (count && bar) {
            if (starLevel === newRating) {
                // This is the new rating level
                count.textContent = '1';
                bar.style.width = '100%';
            } else {
                // All other levels should be 0
                count.textContent = '0';
                bar.style.width = '0%';
            }
        }
    });
    
    console.log('✅ Rating summary updated to ' + newRating + ' stars');
}

// Utility functions
function showNotification(message, type = 'info', duration = 3000) {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.review-notification');
    existingNotifications.forEach(notification => notification.remove());
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `review-notification alert alert-${type === 'error' ? 'danger' : type}`;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        animation: slideInRight 0.3s ease;
        max-width: 300px;
    `;
    
    notification.innerHTML = `
        <div style="display: flex; align-items: center; gap: 8px;">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Remove notification after duration
    setTimeout(() => {
        if (notification.parentNode) {
            notification.style.animation = 'slideOutRight 0.3s ease';
            setTimeout(() => notification.remove(), 300);
        }
    }, duration);
}

// Add CSS for notifications
const notificationCSS = document.createElement('style');
notificationCSS.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    .rating-star:hover {
        transform: scale(1.1);
    }
    
    .review-success-message {
        padding: 2rem;
        text-align: center;
    }
    
    .helpful-btn.voted {
        background-color: var(--panadite-primary);
        color: white;
    }
`;
document.head.appendChild(notificationCSS);

console.log('Review system JavaScript loaded');

// ===== VIDEO NOTES SYSTEM JAVASCRIPT =====

let currentVideoId = null;
let videoNotesLoaded = false;

// Initialize video notes system when page loads
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initializing video notes system...');
    initializeVideoNotesSystem();
});

// Also initialize when videos tab is shown
document.addEventListener('shown.bs.tab', function(e) {
    if (e.target.getAttribute('href') === '#videos') {
        console.log('Videos tab shown, initializing notes...');
        setTimeout(initializeVideoNotesSystem, 100);
    }
});

function initializeVideoNotesSystem() {
    console.log('🎬 Initializing Video Notes System...');
    showDebugInfo('🎬 VIDEO NOTES INIT', 'Starting video notes system initialization');
    
    // Method 1: Try to get video ID from PHP (most reliable)
    const phpLessonId = <?= $current_lesson_id ? $current_lesson_id : 'null' ?>;
    if (phpLessonId) {
        console.log('✅ Using PHP lesson ID:', phpLessonId);
        showDebugInfo('✅ PHP LESSON ID', 'Using lesson ID from PHP', {lessonId: phpLessonId});
        setCurrentVideo(phpLessonId);
    } else {
        console.log('⚠️ No PHP lesson ID, trying video tab detection...');
        showDebugInfo('⚠️ NO PHP LESSON ID', 'Falling back to video tab detection');
        
        // Method 2: Get current video ID from the first video or selected video
        const firstVideoTab = document.querySelector('[data-bs-target*="video-"]');
        if (firstVideoTab) {
            const videoTabId = firstVideoTab.getAttribute('data-bs-target');
            const videoId = videoTabId.replace('#video-', '');
            console.log('Found video ID from tab:', videoId);
            showDebugInfo('📹 TAB VIDEO ID', 'Found video ID from tab', {videoId});
            setCurrentVideo(videoId);
        } else {
            console.log('❌ No video tabs found');
            showDebugInfo('❌ NO VIDEO TABS', 'No video tabs found on page');
        }
    }
    
    // Listen for video tab changes
    document.querySelectorAll('[data-bs-target*="video-"]').forEach(tab => {
        tab.addEventListener('click', function() {
            const videoTabId = this.getAttribute('data-bs-target');
            const videoId = videoTabId.replace('#video-', '');
            console.log('Video tab clicked, switching to video ID:', videoId);
            showDebugInfo('🔄 TAB SWITCH', 'Video tab clicked', {videoId});
            setCurrentVideo(videoId);
        });
    });
    
    // Initialize notes UI for existing notes panels
    initializeNotesUI();
    
    // Initialize modal save button
    initializeModalSaveButton();
    
    console.log('✅ Video Notes System initialization complete');
    showDebugInfo('✅ VIDEO NOTES COMPLETE', 'Video notes system fully initialized');
}

// Add Note function (referenced in HTML)
function addNote() {
    console.log('📝 Add note function called');
    showDebugInfo('📝 ADD NOTE', 'Add note function triggered');
    
    if (!currentVideoId) {
        showVideoNotification('Please load a video first', 'error');
        showDebugInfo('❌ NO VIDEO', 'Cannot add note without video ID');
        return;
    }
    
    const timestamp = getCurrentVideoTimestamp();
    const formattedTime = formatTime(timestamp);
    
    console.log('📝 Opening note modal at timestamp:', timestamp, 'formatted:', formattedTime);
    showDebugInfo('🎭 OPENING MODAL', 'Showing note modal', {timestamp, formattedTime});
    
    // Update modal content
    const modalTimestamp = document.getElementById('modalTimestamp');
    const modalTextarea = document.getElementById('customNoteTextarea');
    const modal = document.getElementById('customNoteModal');
    
    if (modalTimestamp) {
        modalTimestamp.textContent = formattedTime;
    }
    
    if (modalTextarea) {
        modalTextarea.value = '';
        modalTextarea.focus();
    }
    
    if (modal) {
        modal.style.display = 'block';
        console.log('✅ Modal displayed successfully');
        showDebugInfo('✅ MODAL SHOWN', 'Note modal is now visible');
        
        // Store current timestamp for saving
        modal.setAttribute('data-timestamp', timestamp);
        modal.setAttribute('data-video-id', currentVideoId);
    } else {
        console.log('❌ Modal element not found');
        showDebugInfo('❌ NO MODAL', 'Custom note modal element not found');
        showVideoNotification('Note modal not found', 'error');
    }
}

// Close Note Modal function
function closeNoteModal() {
    const modal = document.getElementById('customNoteModal');
    if (modal) {
        modal.style.display = 'none';
        console.log('✅ Modal closed');
        showDebugInfo('✅ MODAL CLOSED', 'Note modal hidden');
    }
}

// Initialize Modal Save Button
function initializeModalSaveButton() {
    const modalSaveBtn = document.getElementById('customSaveNoteBtn');
    if (modalSaveBtn) {
        // Remove existing listeners
        const newSaveBtn = modalSaveBtn.cloneNode(true);
        modalSaveBtn.parentNode.replaceChild(newSaveBtn, modalSaveBtn);
        
        console.log('✅ Adding modal save button listener');
        showDebugInfo('✅ MODAL SAVE BTN', 'Adding modal save button listener');
        
        newSaveBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            const modal = document.getElementById('customNoteModal');
            const textarea = document.getElementById('customNoteTextarea');
            
            if (!modal || !textarea) {
                showVideoNotification('Modal elements not found', 'error');
                return;
            }
            
            const note = textarea.value.trim();
            const timestamp = modal.getAttribute('data-timestamp');
            const videoId = modal.getAttribute('data-video-id');
            
            console.log('💾 Modal save clicked:', {note, timestamp, videoId});
            showDebugInfo('💾 MODAL SAVE', 'Modal save button clicked', {
                noteText: note,
                timestamp: timestamp,
                videoId: videoId,
                noteLength: note.length
            });
            
            if (!note) {
                showVideoNotification('Please enter a note', 'error');
                showDebugInfo('❌ EMPTY NOTE', 'Note text is empty');
                textarea.focus();
                return;
            }
            
            if (!videoId || !timestamp) {
                showVideoNotification('Missing video or timestamp data', 'error');
                showDebugInfo('❌ MISSING DATA', 'Missing video ID or timestamp', {videoId, timestamp});
                return;
            }
            
            // Save the note
            saveVideoNote(videoId, parseInt(timestamp), note);
            
            // Close modal after saving
            closeNoteModal();
        });
        
        console.log('✅ Modal save button initialized');
        showDebugInfo('✅ MODAL INIT COMPLETE', 'Modal save button fully initialized');
    } else {
        console.log('⚠️ Modal save button not found');
        showDebugInfo('⚠️ NO MODAL BTN', 'Custom modal save button not found');
    }
}

// Export Notes function (referenced in HTML)
function exportNotes() {
    console.log('📤 Export notes function called');
    showVideoNotification('Export feature coming soon!', 'info');
}

// Print Notes function (referenced in HTML) 
function printNotes() {
    console.log('🖨️ Print notes function called');
    showVideoNotification('Print feature coming soon!', 'info');
}

function setCurrentVideo(videoId) {
    if (currentVideoId === videoId) {
        return; // Already set to this video
    }
    
    currentVideoId = videoId;
    console.log('Current video ID set to:', currentVideoId);
    showDebugInfo('🎬 VIDEO CHANGED', 'Current video ID changed', {videoId});
    
    // Load notes for this video
    if (currentVideoId) {
        loadVideoNotes(currentVideoId);
    }
}

function loadVideoNotes(lessonId) {
    console.log('📖 Loading notes for lesson ID:', lessonId);
    showDebugInfo('📖 LOADING NOTES', 'Fetching notes from server', {lessonId});
    
    fetch('get_video_notes.php?lesson_id=' + lessonId)
        .then(response => {
            console.log('📥 Notes response received:', response.status);
            showDebugInfo('📥 NOTES RESPONSE', 'Server response received', {
                status: response.status,
                statusText: response.statusText,
                ok: response.ok
            });
            return response.json();
        })
        .then(data => {
            console.log('📝 Notes data:', data);
            showDebugInfo('📝 NOTES DATA', 'Notes data parsed', data);
            
            if (data.success) {
                displayVideoNotes(data.notes || []);
                showVideoNotification(`Loaded ${data.count || 0} notes`, 'success', 2000);
            } else {
                console.log('❌ Failed to load notes:', data.message);
                showDebugInfo('❌ NOTES ERROR', 'Failed to load notes', data);
                showVideoNotification('Failed to load notes: ' + (data.message || 'Unknown error'), 'error');
                displayVideoNotes([]); // Show empty state
            }
        })
        .catch(error => {
            console.error('❌ Notes loading error:', error);
            showDebugInfo('❌ NOTES FETCH ERROR', 'Network or parsing error', {error: error.message});
            showVideoNotification('Error loading notes', 'error');
            displayVideoNotes([]); // Show empty state
        });
}

function displayVideoNotes(notes) {
    console.log('🎨 Displaying notes:', notes.length, 'notes');
    showDebugInfo('🎨 DISPLAY NOTES', 'Rendering notes in UI', {noteCount: notes.length});
    
    const notesList = document.getElementById('notesList');
    if (!notesList) {
        console.log('❌ Notes list container not found');
        showDebugInfo('❌ NO NOTES CONTAINER', 'notesList element not found');
        return;
    }
    
    if (!notes || notes.length === 0) {
        notesList.innerHTML = `
            <div class="no-notes text-center py-4">
                <i class="bx bx-note text-muted" style="font-size: 2rem;"></i>
                <p class="text-muted mt-2 mb-0">No notes yet. Add your first note while watching!</p>
            </div>
        `;
        console.log('✅ Displayed empty notes state');
        showDebugInfo('✅ EMPTY NOTES', 'Displayed empty notes message');
        return;
    }
    
    // Sort notes by timestamp (earliest first for better viewing experience)
    const sortedNotes = notes.sort((a, b) => parseInt(a.timestamp) - parseInt(b.timestamp));
    
    const notesHtml = sortedNotes.map(note => {
        const timestamp = parseInt(note.timestamp || 0);
        const formattedTime = note.formatted_time || formatTime(timestamp);
        const noteText = escapeHtml(note.note || '');
        
        // Format creation date properly (no expiry - just "Created on...")
        const createdDate = note.created_at ? new Date(note.created_at) : new Date();
        const dateStr = createdDate.toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        
        return `
            <div class="note-item" data-timestamp="${timestamp}" data-note-id="${note.id || ''}">
                <div class="note-header">
                    <span class="note-time" onclick="seekToTimestamp(${timestamp})" title="Click to jump to this time">
                        <i class="bx bx-play-circle"></i> ${formattedTime}
                    </span>
                    <div class="note-actions">
                        <button class="btn btn-sm btn-outline-danger" onclick="deleteNote(${note.id || 0})" title="Delete note">
                            <i class="bx bx-trash"></i>
                        </button>
                    </div>
                </div>
                <div class="note-text">${noteText}</div>
                <div class="note-meta">
                    <small class="text-muted">
                        <i class="bx bx-time"></i> Created: ${dateStr}
                    </small>
                </div>
            </div>
        `;
    }).join('');
    
    notesList.innerHTML = notesHtml;
    
    console.log('✅ Notes displayed successfully');
    showDebugInfo('✅ NOTES DISPLAYED', 'All notes rendered successfully', {
        noteCount: sortedNotes.length,
        timestamps: sortedNotes.map(n => n.timestamp)
    });
}

// Helper function to escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Function to seek video to timestamp (when user clicks on time)
function seekToTimestamp(timestamp) {
    console.log('⏰ Seeking to timestamp:', timestamp);
    showDebugInfo('⏰ SEEK TIMESTAMP', 'Jumping to video timestamp', {timestamp});
    showVideoNotification(`Jumping to ${formatTime(timestamp)}`, 'info', 2000);
    
    // Here you would integrate with your video player
    // For now, just show notification
}

// Function to delete a note
function deleteNote(noteId) {
    if (!noteId) {
        showVideoNotification('Invalid note ID', 'error');
        return;
    }
    
    if (!confirm('Are you sure you want to delete this note?')) {
        return;
    }
    
    console.log('🗑️ Deleting note ID:', noteId);
    showDebugInfo('🗑️ DELETE NOTE', 'Deleting note', {noteId});
    
    // Here you would call a delete API
    showVideoNotification('Delete functionality coming soon!', 'info');
}

function initializeNotesUI() {
    console.log('🔧 Initializing Notes UI...');
    showDebugInfo('🔧 NOTES UI INIT', 'Setting up notes interface event listeners');
    
    // Find all note input areas and add event listeners
    document.querySelectorAll('.notes-input-area').forEach(area => {
        const saveBtn = area.querySelector('.save-note-btn');
        const noteInput = area.querySelector('.note-input');
        
        console.log('Found notes input area:', {saveBtn: !!saveBtn, noteInput: !!noteInput});
        
        if (saveBtn && noteInput) {
            // Remove existing listeners
            const newSaveBtn = saveBtn.cloneNode(true);
            saveBtn.parentNode.replaceChild(newSaveBtn, saveBtn);
            
            console.log('✅ Adding click listener to save note button');
            showDebugInfo('✅ SAVE BUTTON', 'Adding click listener to save note button');
            
            newSaveBtn.addEventListener('click', function(e) {
                e.preventDefault();
                
                const note = noteInput.value.trim();
                console.log('💾 Save note button clicked, note text:', note);
                console.log('💾 Current video ID:', currentVideoId);
                
                showDebugInfo('💾 SAVE NOTE CLICKED', 'Save button clicked', {
                    noteText: note,
                    currentVideoId: currentVideoId,
                    noteLength: note.length
                });
                
                if (note && currentVideoId) {
                    const timestamp = getCurrentVideoTimestamp();
                    console.log('💾 Saving note with timestamp:', timestamp);
                    saveVideoNote(currentVideoId, timestamp, note);
                } else {
                    const error = !note ? 'Please enter a note' : 'Video not loaded yet';
                    console.log('❌ Cannot save note:', error);
                    showVideoNotification(error, 'error');
                    showDebugInfo('❌ SAVE ERROR', error, {note: !!note, currentVideoId});
                }
            });
            
            // Add Enter key support (Ctrl+Enter to save)
            noteInput.addEventListener('keydown', function(e) {
                if (e.ctrlKey && e.key === 'Enter') {
                    e.preventDefault();
                    newSaveBtn.click();
                }
            });
        } else {
            console.log('⚠️ Notes input area found but missing save button or input field');
            showDebugInfo('⚠️ NOTES UI WARNING', 'Missing save button or input field', {
                saveBtn: !!saveBtn,
                noteInput: !!noteInput
            });
        }
    });
    
    console.log('✅ Notes UI initialization complete');
    showDebugInfo('✅ NOTES UI COMPLETE', 'Notes interface successfully initialized');
}

function getCurrentVideoTimestamp() {
    // Try to get timestamp from video player if available
    const videoPlayer = document.querySelector('video');
    if (videoPlayer && !isNaN(videoPlayer.currentTime)) {
        return Math.floor(videoPlayer.currentTime);
    }
    
    // Fallback to current time or 0
    return 0;
}

function formatTime(seconds) {
    if (isNaN(seconds) || seconds < 0) return '0:00';
    const minutes = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${minutes}:${secs.toString().padStart(2, '0')}`;
}

// Function to focus on note input instead of showing modal
function focusNoteInput() {
    const noteInput = document.querySelector('.note-input');
    if (noteInput) {
        noteInput.focus();
        noteInput.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
}

async function loadVideoNotes(videoId) {
    if (!videoId) {
        console.log('No video ID provided for loading notes');
        return;
    }
    
    console.log('Loading notes for lesson ID:', videoId);
    
    try {
        const response = await fetch(`get_video_notes.php?lesson_id=${videoId}`);
        const data = await response.json();
        
        console.log('Notes API response:', data);
        
        if (data.success) {
            displayVideoNotes(data.notes || []);
            videoNotesLoaded = true;
            console.log('✅ Notes loaded successfully:', data.notes?.length || 0, 'notes');
        } else {
            console.error('❌ Failed to load notes:', data.message);
            displayVideoNotes([]);
            showVideoNotification(data.message || 'Failed to load notes', 'error');
        }
    } catch (error) {
        console.error('❌ Error loading notes:', error);
        displayVideoNotes([]);
        showVideoNotification('Failed to load notes: ' + error.message, 'error');
    }
}

async function saveVideoNote(videoId, timestamp, note) {
    if (!videoId || !note) {
        showVideoNotification('Missing video ID or note content', 'error');
        showDebugInfo('❌ VALIDATION ERROR', 'Missing video ID or note content', {videoId, note});
        return;
    }
    
    const requestData = {
        lesson_id: parseInt(videoId),
        timestamp: timestamp,
        note: note,
        formatted_time: formatTime(timestamp)
    };
    
    console.log('🔄 SAVING NOTE - Request Data:', requestData);
    showDebugInfo('🔄 SAVING NOTE', 'Preparing to save note', requestData);
    
    try {
        console.log('📤 Making fetch request to save_video_note.php');
        showDebugInfo('📤 FETCH REQUEST', 'Method: POST, URL: save_video_note.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            bodyData: requestData
        });
        
        const response = await fetch('save_video_note.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify(requestData)
        });
        
        console.log('📥 Response received - Status:', response.status, 'Status Text:', response.statusText);
        showDebugInfo('📥 RESPONSE RECEIVED', `Status: ${response.status} ${response.statusText}`, {
            status: response.status,
            statusText: response.statusText,
            ok: response.ok,
            headers: Object.fromEntries(response.headers.entries())
        });
        
        let data;
        const responseText = await response.text();
        console.log('📄 Raw response text:', responseText);
        
        try {
            data = JSON.parse(responseText);
            console.log('✅ Parsed JSON response:', data);
            showDebugInfo('✅ PARSED RESPONSE', 'Successfully parsed JSON', data);
        } catch (parseError) {
            console.error('❌ JSON parse error:', parseError);
            showDebugInfo('❌ JSON PARSE ERROR', parseError.message, {
                rawResponse: responseText,
                parseError: parseError.message
            });
            throw new Error('Invalid JSON response: ' + responseText);
        }
        
        if (data.success) {
            console.log('✅ Note saved successfully!');
            showVideoNotification('✅ Note saved successfully!', 'success');
            showDebugInfo('✅ SUCCESS', 'Note saved successfully', data);
            
            // Clear the note input
            const noteInput = document.querySelector('.note-input');
            if (noteInput) {
                noteInput.value = '';
            }
            
            // Reload notes to show the new note
            setTimeout(() => loadVideoNotes(videoId), 500);
        } else {
            console.error('❌ API returned failure:', data);
            showDebugInfo('❌ API FAILURE', data.message || 'Unknown error', data);
            showVideoNotification('Failed to save note: ' + (data.message || 'Unknown error'), 'error');
        }
    } catch (error) {
        console.error('❌ Fetch error:', error);
        showDebugInfo('❌ FETCH ERROR', error.message, {
            error: error.message,
            stack: error.stack
        });
        showVideoNotification('Failed to save note: ' + error.message, 'error');
    }
}

function displayVideoNotes(notes) {
    const notesContainers = document.querySelectorAll('.notes-list, .video-notes-list');
    
    if (notesContainers.length === 0) {
        console.log('No notes containers found to display notes');
        return;
    }
    
    notesContainers.forEach(container => {
        if (notes.length === 0) {
            container.innerHTML = `
                <div class="no-notes text-center py-4">
                    <i class="bx bx-note text-muted" style="font-size: 2rem;"></i>
                    <p class="text-muted mt-2 mb-0">No notes yet. Add your first note while watching!</p>
                </div>
            `;
        } else {
            container.innerHTML = notes.map(note => `
                <div class="note-item">
                    <div class="note-timestamp" onclick="seekToTimestamp(${note.timestamp})">
                        ${note.formatted_time || formatTime(note.timestamp)}
                    </div>
                    <div class="note-content">
                        <p class="note-text mb-1">${escapeHtml(note.note)}</p>
                        <small class="text-muted">
                            <i class="bx bx-time-five"></i>
                            ${new Date(note.created_at).toLocaleDateString()}
                        </small>
                    </div>
                </div>
            `).join('');
        }
    });
    
    console.log('✅ Displayed', notes.length, 'notes in UI');
}

function seekToTimestamp(timestamp) {
    const videoPlayer = document.querySelector('video');
    if (videoPlayer) {
        videoPlayer.currentTime = timestamp;
        showVideoNotification(`⏰ Jumped to ${formatTime(timestamp)}`, 'info', 2000);
    } else {
        console.log('No video player found to seek to timestamp:', timestamp);
    }
}

function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}

function showVideoNotification(message, type = 'info', duration = 3000) {
    // Remove existing video notifications
    document.querySelectorAll('.video-notification').forEach(n => n.remove());
    
    const notification = document.createElement('div');
    notification.className = `video-notification alert alert-${type === 'error' ? 'danger' : type}`;
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        z-index: 9999;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        animation: slideIn 0.3s ease;
        max-width: 300px;
    `;
    
    notification.innerHTML = `
        <div style="display: flex; align-items: center; gap: 8px;">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        if (notification.parentNode) {
            notification.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => notification.remove(), 300);
        }
    }, duration);
}

function showDebugInfo(title, message, data = null) {
    // Create or get debug container
    let debugContainer = document.getElementById('notes-debug-container');
    if (!debugContainer) {
        debugContainer = document.createElement('div');
        debugContainer.id = 'notes-debug-container';
        debugContainer.style.cssText = `
            position: fixed;
            bottom: 20px;
            left: 20px;
            max-width: 500px;
            max-height: 400px;
            overflow-y: auto;
            background: #1a1a1a;
            color: #00ff00;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            z-index: 10000;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            border: 2px solid #333;
        `;
        debugContainer.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; border-bottom: 1px solid #333; padding-bottom: 5px;">
                <strong style="color: #ffff00;">🔧 NOTES DEBUG CONSOLE</strong>
                <button onclick="this.parentElement.parentElement.remove()" style="background: #ff4444; color: white; border: none; padding: 2px 8px; border-radius: 3px; cursor: pointer;">×</button>
            </div>
            <div id="debug-messages"></div>
        `;
        document.body.appendChild(debugContainer);
    }
    
    const messagesContainer = debugContainer.querySelector('#debug-messages');
    const timestamp = new Date().toLocaleTimeString();
    
    const debugMessage = document.createElement('div');
    debugMessage.style.cssText = `
        margin-bottom: 8px;
        padding: 8px;
        background: #2a2a2a;
        border-radius: 4px;
        border-left: 3px solid #00ff00;
    `;
    
    let dataDisplay = '';
    if (data) {
        dataDisplay = `<pre style="margin: 5px 0 0 0; color: #88ccff; font-size: 11px; white-space: pre-wrap;">${JSON.stringify(data, null, 2)}</pre>`;
    }
    
    debugMessage.innerHTML = `
        <div style="color: #ffff00; font-weight: bold;">[${timestamp}] ${title}</div>
        <div style="color: #cccccc; margin: 2px 0;">${message}</div>
        ${dataDisplay}
    `;
    
    messagesContainer.appendChild(debugMessage);
    
    // Scroll to bottom
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
    
    // Keep only last 20 messages
    const messages = messagesContainer.children;
    if (messages.length > 20) {
        messagesContainer.removeChild(messages[0]);
    }
}

console.log('Video notes system JavaScript loaded');

// ===== SMART STUDY PRODUCTIVITY TOOLS JAVASCRIPT =====

// Global variables for productivity tools
let currentDate = new Date();
let selectedDate = null;
let studyReminders = [];
let pipVideo = null;
let pipWindow = null;

// Initialize productivity tools when page loads
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Initializing Smart Study Productivity Tools...');
    initializeProductivityTools();
});

// Also initialize when tools tab is shown
document.addEventListener('click', function(e) {
    if (e.target.closest('[data-tab="tools"]') || e.target.closest('button[data-tab="tools"]')) {
        setTimeout(() => {
            console.log('🔧 Tools tab shown, initializing productivity features...');
            initializeProductivityTools();
        }, 100);
    }
});

function initializeProductivityTools() {
    console.log('🎯 Setting up Smart Study Calendar and PiP Player...');
    
    // Initialize Smart Study Calendar
    renderCalendar();
    loadStudyReminders();
    
    // Initialize Picture-in-Picture
    initializePictureInPicture();
    
    // Set up event listeners
    setupProductivityEventListeners();
    
    console.log('✅ Productivity tools initialized successfully!');
}

// ===== SMART STUDY CALENDAR FUNCTIONS =====

function renderCalendar() {
    const calendarGrid = document.getElementById('calendarGrid');
    if (!calendarGrid) return;
    
    const year = currentDate.getFullYear();
    const month = currentDate.getMonth();
    
    // Update month/year display
    const monthYearElement = document.getElementById('currentMonthYear');
    if (monthYearElement) {
        const monthNames = ['January', 'February', 'March', 'April', 'May', 'June',
                           'July', 'August', 'September', 'October', 'November', 'December'];
        monthYearElement.textContent = `${monthNames[month]} ${year}`;
    }
    
    // Clear previous calendar
    calendarGrid.innerHTML = '';
    
    // Day headers
    const dayHeaders = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    dayHeaders.forEach(day => {
        const dayHeader = document.createElement('div');
        dayHeader.className = 'calendar-day-header';
        dayHeader.textContent = day;
        calendarGrid.appendChild(dayHeader);
    });
    
    // Get first day of month and number of days
    const firstDay = new Date(year, month, 1).getDay();
    const daysInMonth = new Date(year, month + 1, 0).getDate();
    const today = new Date();
    
    // Add empty cells for days before month starts
    for (let i = 0; i < firstDay; i++) {
        const emptyDay = document.createElement('div');
        emptyDay.className = 'calendar-day other-month';
        calendarGrid.appendChild(emptyDay);
    }
    
    // Add days of the month
    for (let day = 1; day <= daysInMonth; day++) {
        const dayElement = document.createElement('div');
        dayElement.className = 'calendar-day';
        dayElement.textContent = day;
        
        const dayDate = new Date(year, month, day);
        
        // Check if it's today
        if (dayDate.toDateString() === today.toDateString()) {
            dayElement.classList.add('today');
        }
        
        // Check if it has reminders
        const hasReminder = studyReminders.some(reminder => {
            const reminderDate = new Date(reminder.date);
            return reminderDate.toDateString() === dayDate.toDateString();
        });
        
        if (hasReminder) {
            dayElement.classList.add('has-reminder');
        }
        
        // Add click handler
        dayElement.addEventListener('click', () => selectDate(year, month, day));
        
        calendarGrid.appendChild(dayElement);
    }
    
    console.log(`📅 Calendar rendered for ${monthNames[month]} ${year}`);
}

function selectDate(year, month, day) {
    selectedDate = new Date(year, month, day);
    
    // Update visual selection
    document.querySelectorAll('.calendar-day').forEach(el => {
        el.classList.remove('selected');
    });
    
    event.target.classList.add('selected');
    
    console.log('📅 Date selected:', selectedDate.toDateString());
    showStudyReminderModal(selectedDate);
}

function previousMonth() {
    currentDate.setMonth(currentDate.getMonth() - 1);
    renderCalendar();
}

function nextMonth() {
    currentDate.setMonth(currentDate.getMonth() + 1);
    renderCalendar();
}

function createQuickReminder() {
    const tomorrow = new Date();
    tomorrow.setDate(tomorrow.getDate() + 1);
    showStudyReminderModal(tomorrow);
}

function showStudyReminderModal(date) {
    const modal = document.createElement('div');
    modal.className = 'modal fade show';
    modal.style.display = 'block';
    modal.style.backgroundColor = 'rgba(0,0,0,0.5)';
    
    const formattedDate = date.toLocaleDateString('en-US', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
    
    modal.innerHTML = `
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="bx bx-calendar-plus me-2"></i>
                        Set Study Reminder
                    </h5>
                    <button type="button" class="btn-close btn-close-white" onclick="closeReminderModal()"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="bx bx-info-circle me-2"></i>
                        <strong>Selected Date:</strong> ${formattedDate}
                    </div>
                    
                    <form id="reminderForm">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Reminder Type</label>
                                <select class="form-select" id="reminderType" required>
                                    <option value="">Select type...</option>
                                    <option value="next_lesson">Next Lesson</option>
                                    <option value="review_session">Review Session</option>
                                    <option value="assignment">Assignment Due</option>
                                    <option value="quiz_prep">Quiz Preparation</option>
                                    <option value="study_session">Study Session</option>
                                    <option value="custom">Custom Reminder</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Time</label>
                                <input type="time" class="form-control" id="reminderTime" value="09:00" required>
                            </div>
                        </div>
                        
                        <div class="mt-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="reminderDescription" rows="3" 
                                      placeholder="What would you like to be reminded about?"></textarea>
                        </div>
                        
                        <div class="mt-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="emailNotification" checked>
                                <label class="form-check-label" for="emailNotification">
                                    <i class="bx bx-envelope me-1"></i>
                                    Send email notification
                                </label>
                            </div>
                        </div>
                        
                        <div class="mt-4 d-flex gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="bx bx-save me-1"></i>
                                Save Reminder
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="closeReminderModal()">
                                Cancel
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Set up form submission
    document.getElementById('reminderForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveStudyReminder(date);
    });
    
    console.log('📝 Reminder modal shown for:', formattedDate);
}

function closeReminderModal() {
    const modal = document.querySelector('.modal.show');
    if (modal) {
        modal.remove();
    }
}

function saveStudyReminder(date) {
    const type = document.getElementById('reminderType').value;
    const time = document.getElementById('reminderTime').value;
    const description = document.getElementById('reminderDescription').value;
    const emailNotification = document.getElementById('emailNotification').checked;
    
    if (!type || !time) {
        showProductivityNotification('Please fill in all required fields', 'error');
        return;
    }
    
    const reminder = {
        id: Date.now(),
        date: date.toISOString().split('T')[0],
        time: time,
        type: type,
        description: description || getDefaultDescription(type),
        emailNotification: emailNotification,
        created: new Date().toISOString()
    };
    
    // Save to localStorage (in production, this would be saved to server)
    studyReminders.push(reminder);
    localStorage.setItem('studyReminders', JSON.stringify(studyReminders));
    
    // Send email notification if requested
    if (emailNotification) {
        sendReminderEmail(reminder);
    }
    
    // Update UI
    renderCalendar();
    updateUpcomingReminders();
    closeReminderModal();
    
    showProductivityNotification('✅ Reminder saved successfully! Email notification sent.', 'success');
    console.log('💾 Study reminder saved:', reminder);
}

function getDefaultDescription(type) {
    const descriptions = {
        'next_lesson': 'Time for your next lesson',
        'review_session': 'Review previous lessons',
        'assignment': 'Assignment deadline approaching',
        'quiz_prep': 'Prepare for upcoming quiz',
        'study_session': 'Dedicated study time',
        'custom': 'Custom study reminder'
    };
    return descriptions[type] || 'Study reminder';
}

function sendReminderEmail(reminder) {
    // Simulate email sending (in production, this would call your email API)
    const emailData = {
        action: 'send_study_reminder',
        reminder: reminder,
        course_id: <?= $course_id ?>,
        student_id: <?= $user_id ?>
    };
    
    fetch('api/send_study_reminder_email.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(emailData)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            console.log('📧 Reminder email sent successfully');
        } else {
            console.warn('⚠️ Email sending failed:', data.message);
        }
    })
    .catch(error => {
        console.error('❌ Email sending error:', error);
    });
}

function loadStudyReminders() {
    const saved = localStorage.getItem('studyReminders');
    if (saved) {
        studyReminders = JSON.parse(saved);
        updateUpcomingReminders();
    }
}

function updateUpcomingReminders() {
    const remindersList = document.getElementById('upcomingRemindersList');
    if (!remindersList) return;
    
    const now = new Date();
    const upcoming = studyReminders
        .filter(reminder => new Date(reminder.date + 'T' + reminder.time) > now)
        .sort((a, b) => new Date(a.date + 'T' + a.time) - new Date(b.date + 'T' + b.time))
        .slice(0, 3);
    
    if (upcoming.length === 0) {
        remindersList.innerHTML = `
            <div class="text-muted small text-center py-2">
                <i class="bx bx-calendar-plus me-1"></i>
                No reminders set yet
            </div>
        `;
        return;
    }
    
    remindersList.innerHTML = upcoming.map(reminder => {
        const reminderDate = new Date(reminder.date + 'T' + reminder.time);
        const timeUntil = getTimeUntil(reminderDate);
        
        return `
            <div class="reminder-item">
                <div class="reminder-time">${reminder.time}</div>
                <div class="reminder-text">
                    <strong>${reminder.description}</strong>
                    <br><small>${new Date(reminder.date).toLocaleDateString()} • ${timeUntil}</small>
                </div>
                <div class="reminder-actions">
                    <button class="btn btn-outline-danger btn-sm" onclick="deleteReminder(${reminder.id})" title="Delete">
                        <i class="bx bx-trash"></i>
                    </button>
                </div>
            </div>
        `;
    }).join('');
}

function getTimeUntil(date) {
    const now = new Date();
    const diff = date - now;
    
    if (diff < 0) return 'Past due';
    
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    
    if (days > 0) return `${days} day${days > 1 ? 's' : ''} away`;
    if (hours > 0) return `${hours} hour${hours > 1 ? 's' : ''} away`;
    return 'Soon';
}

function deleteReminder(id) {
    studyReminders = studyReminders.filter(r => r.id !== id);
    localStorage.setItem('studyReminders', JSON.stringify(studyReminders));
    renderCalendar();
    updateUpcomingReminders();
    showProductivityNotification('Reminder deleted', 'info');
}

// ===== PICTURE-IN-PICTURE VIDEO PLAYER FUNCTIONS =====

function initializePictureInPicture() {
    console.log('📺 Initializing Picture-in-Picture functionality...');
    updatePipStatus();
    
    // Check for PiP support
    if (!('pictureInPictureEnabled' in document)) {
        showProductivityNotification('Picture-in-Picture not supported in this browser', 'warning');
        disablePipControls();
        return;
    }
    
    console.log('✅ Picture-in-Picture is supported!');
}

function enablePictureInPicture() {
    const videoElement = document.querySelector('video');
    
    if (!videoElement) {
        showProductivityNotification('No video is currently playing', 'error');
        return;
    }
    
    if (!document.pictureInPictureEnabled) {
        showProductivityNotification('Picture-in-Picture is disabled', 'error');
        return;
    }
    
    videoElement.requestPictureInPicture()
        .then(pipWindow => {
            console.log('🎬 Picture-in-Picture mode enabled!');
            pipVideo = videoElement;
            pipWindow = pipWindow;
            
            updatePipControls(true);
            updatePipStatus('Active');
            showProductivityNotification('📺 Picture-in-Picture mode enabled! Video will continue playing while you browse.', 'success');
            
            // Listen for PiP exit
            pipWindow.addEventListener('leavepictureinpicture', () => {
                console.log('🚪 Exited Picture-in-Picture mode');
                updatePipControls(false);
                updatePipStatus('Ready');
                showProductivityNotification('Exited Picture-in-Picture mode', 'info');
            });
        })
        .catch(error => {
            console.error('❌ PiP error:', error);
            showProductivityNotification('Failed to enable Picture-in-Picture: ' + error.message, 'error');
        });
}

function disablePictureInPicture() {
    if (document.pictureInPictureElement) {
        document.exitPictureInPicture()
            .then(() => {
                console.log('🚪 Picture-in-Picture disabled');
                updatePipControls(false);
                updatePipStatus('Ready');
                showProductivityNotification('Picture-in-Picture disabled', 'info');
            })
            .catch(error => {
                console.error('❌ PiP exit error:', error);
                showProductivityNotification('Failed to exit Picture-in-Picture: ' + error.message, 'error');
            });
    }
}

function updatePipControls(isActive) {
    const enableBtn = document.getElementById('pipEnableBtn');
    const disableBtn = document.getElementById('pipDisableBtn');
    
    if (enableBtn && disableBtn) {
        enableBtn.disabled = isActive;
        disableBtn.disabled = !isActive;
        
        if (isActive) {
            enableBtn.innerHTML = '<i class="bx bx-check me-2"></i>PiP Active';
            enableBtn.classList.replace('btn-success', 'btn-outline-success');
        } else {
            enableBtn.innerHTML = '<i class="bx bx-dock-top me-2"></i>Enable PiP Mode';
            enableBtn.classList.replace('btn-outline-success', 'btn-success');
        }
    }
}

function updatePipStatus(status = 'Ready') {
    const statusElement = document.getElementById('pipStatus');
    if (statusElement) {
        statusElement.textContent = status;
        statusElement.className = 'badge bg-' + (status === 'Active' ? 'success' : 'secondary');
    }
}

function disablePipControls() {
    const enableBtn = document.getElementById('pipEnableBtn');
    const disableBtn = document.getElementById('pipDisableBtn');
    
    if (enableBtn) {
        enableBtn.disabled = true;
        enableBtn.innerHTML = '<i class="bx bx-x me-2"></i>Not Supported';
        enableBtn.classList.replace('btn-success', 'btn-outline-secondary');
    }
    
    if (disableBtn) {
        disableBtn.disabled = true;
    }
}

// ===== QUICK ACTION FUNCTIONS =====

function setNextLessonReminder() {
    const tomorrow = new Date();
    tomorrow.setDate(tomorrow.getDate() + 1);
    tomorrow.setHours(9, 0, 0, 0);
    
    const reminder = {
        id: Date.now(),
        date: tomorrow.toISOString().split('T')[0],
        time: '09:00',
        type: 'next_lesson',
        description: 'Time for your next lesson in <?= htmlspecialchars($course['title']) ?>',
        emailNotification: true,
        created: new Date().toISOString()
    };
    
    studyReminders.push(reminder);
    localStorage.setItem('studyReminders', JSON.stringify(studyReminders));
    sendReminderEmail(reminder);
    
    renderCalendar();
    updateUpcomingReminders();
    showProductivityNotification('✅ Next lesson reminder set for tomorrow at 9:00 AM!', 'success');
}

function createWeeklySchedule() {
    showProductivityNotification('📅 Weekly schedule planner coming soon! This will help you plan your entire week of study sessions.', 'info');
}

function showMobilePipInfo() {
    const modal = document.createElement('div');
    modal.className = 'modal fade show';
    modal.style.display = 'block';
    modal.style.backgroundColor = 'rgba(0,0,0,0.5)';
    
    modal.innerHTML = `
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-info text-white">
                    <h5 class="modal-title">
                        <i class="bx bx-mobile-alt me-2"></i>
                        Mobile Picture-in-Picture
                    </h5>
                    <button type="button" class="btn-close btn-close-white" onclick="this.closest('.modal').remove()"></button>
                </div>
                <div class="modal-body">
                    <div class="text-center mb-3">
                        <i class="bx bx-mobile-landscape text-info" style="font-size: 3rem;"></i>
                    </div>
                    
                    <h6>How to enable PiP on mobile:</h6>
                    
                    <div class="mobile-steps">
                        <div class="step-item d-flex mb-3">
                            <div class="step-number bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 30px; height: 30px; font-weight: bold;">1</div>
                            <div>
                                <strong>iPhone/iPad (Safari):</strong><br>
                                Start video → Tap PiP button in video controls → Switch apps
                            </div>
                        </div>
                        
                        <div class="step-item d-flex mb-3">
                            <div class="step-number bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 30px; height: 30px; font-weight: bold;">2</div>
                            <div>
                                <strong>Android (Chrome):</strong><br>
                                Start video → Tap home button → Video continues in small window
                            </div>
                        </div>
                        
                        <div class="step-item d-flex">
                            <div class="step-number bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 30px; height: 30px; font-weight: bold;">3</div>
                            <div>
                                <strong>YouTube App:</strong><br>
                                Requires YouTube Premium subscription for background play
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-info mt-3">
                        <i class="bx bx-info-circle me-2"></i>
                        <strong>Tip:</strong> Make sure your browser supports PiP and the feature is enabled in settings.
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
}

function openStudyPreferences() {
    showProductivityNotification('⚙️ Study preferences panel coming soon! You\'ll be able to customize notification settings, default reminder times, and more.', 'info');
}

// ===== UTILITY FUNCTIONS =====

function setupProductivityEventListeners() {
    // Listen for video changes to update PiP info
    document.addEventListener('videoChanged', function(e) {
        updateCurrentVideoInfo(e.detail);
    });
}

function updateCurrentVideoInfo(videoData) {
    const titleElement = document.getElementById('currentVideoTitle');
    const durationElement = document.getElementById('videoDuration');
    const previewElement = document.getElementById('miniVideoPreview');
    
    if (titleElement) titleElement.textContent = videoData.title || 'No video selected';
    if (durationElement) durationElement.textContent = videoData.duration || 'N/A';
    
    if (previewElement && videoData.thumbnail) {
        previewElement.querySelector('img').src = videoData.thumbnail;
    }
}

function showProductivityNotification(message, type = 'info', duration = 4000) {
    // Remove existing notifications
    document.querySelectorAll('.productivity-notification').forEach(n => n.remove());
    
    const notification = document.createElement('div');
    notification.className = `productivity-notification alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show`;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 10000;
        max-width: 400px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.15);
        border: none;
        border-radius: 12px;
        animation: slideInRight 0.3s ease;
    `;
    
    const icons = {
        success: 'bx-check-circle',
        error: 'bx-error-circle',
        warning: 'bx-error-circle',
        info: 'bx-info-circle'
    };
    
    notification.innerHTML = `
        <div class="d-flex align-items-center">
            <i class="bx ${icons[type] || icons.info} me-2" style="font-size: 1.2rem;"></i>
            <div>${message}</div>
        </div>
        <button type="button" class="btn-close" onclick="this.closest('.alert').remove()"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after duration
    setTimeout(() => {
        if (notification.parentNode) {
            notification.style.animation = 'slideOutRight 0.3s ease';
            setTimeout(() => notification.remove(), 300);
        }
    }, duration);
}

// Add CSS for notification animations
const notificationAnimationCSS = document.createElement('style');
notificationAnimationCSS.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(notificationAnimationCSS);

console.log('🚀 Smart Study Productivity Tools JavaScript loaded successfully!');
</script>

<?php include_once 'components/footer.php'; ?>
