<?php
/**
 * Student Dashboard - Clean Modern Design
 * Professional admin-style interface with clean functionality
 */

// Set current page for navigation
$currentPage = 'dashboard';

// Authentication - Require student role
require_once '../auth/functions.php';
requireRole('student');

// Initialize default values
$enrolledCourses = 0;
$completedCourses = 0;
$assignmentsDue = 0;
$certificatesEarned = 0;
$totalLearningHours = 0;
$upcomingClasses = 0;
$overallProgress = 0;

$userId = $_SESSION['user_id'] ?? 0;
$userName = $_SESSION['first_name'] ?? 'Student';
$userLastName = $_SESSION['last_name'] ?? '';
$fullName = trim($userName . ' ' . $userLastName);

// Database queries for dashboard statistics
try {
    require_once '../config/db_connect.php';
    
    if (isset($conn) && $userId) {
        // Get enrolled courses count (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE student_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $enrolledCourses = $row['count'];
            }
            $stmt->close();
        }
        
        // Get completed courses count (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE student_id = ? AND status = 'completed'");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $completedCourses = $row['count'];
            }
            $stmt->close();
        }
        
        // Get assignments due count (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM assignments a 
                              JOIN enrollments e ON a.course_id = e.course_id 
                              WHERE e.student_id = ? AND a.due_date >= CURDATE() AND a.status != 'submitted'");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $assignmentsDue = $row['count'];
            }
            $stmt->close();
        }
        
        // Get certificates count (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM certificates WHERE student_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $certificatesEarned = $row['count'];
            }
            $stmt->close();
        }
        
        // Calculate learning hours (with error checking)
        $stmt = $conn->prepare("SELECT SUM(c.duration_hours) as hours FROM courses c 
                              JOIN enrollments e ON c.course_id = e.course_id 
                              WHERE e.student_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $totalLearningHours = $row['hours'] ?? 0;
            }
            $stmt->close();
        }
        
        // Get upcoming classes (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM schedules s 
                              JOIN enrollments e ON s.course_id = e.course_id 
                              WHERE e.student_id = ? AND s.schedule_date >= CURDATE()");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $upcomingClasses = $row['count'];
            }
            $stmt->close();
        }
        
        // Calculate real overall progress percentage by averaging all course progress
        $overallProgress = 0;
        if ($enrolledCourses > 0) {
            $progressQuery = $conn->prepare("
                SELECT 
                    AVG(COALESCE(
                        (SELECT 
                            CASE 
                                WHEN COUNT(v.video_id) = 0 THEN 0
                                ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                            END
                        FROM videos v
                        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                        WHERE v.course_id = e.course_id AND v.is_published = 1), 0
                    )) as avg_progress
                FROM enrollments e
                WHERE e.student_id = ?
            ");
            
            if ($progressQuery) {
                $progressQuery->bind_param("i", $userId);
                if ($progressQuery->execute()) {
                    $result = $progressQuery->get_result();
                    if ($row = $result->fetch_assoc()) {
                        $overallProgress = round($row['avg_progress'] ?? 0);
                    }
                }
                $progressQuery->close();
            }
        }
    }
} catch (Exception $e) {
    // Log error but don't display to user
    error_log("Dashboard data error: " . $e->getMessage());
}

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- Clean Modern Dashboard -->
<div class="dashboard-content">


    <!-- Overall Progress Circle -->
    <div class="progress-section">
        <div class="progress-card">
            <div class="progress-circle-container">
                <div class="circular-progress" data-percentage="<?= $overallProgress ?>">
                    <div class="progress-ring">
                        <svg class="progress-svg" viewBox="0 0 120 120">
                            <circle class="progress-bg" cx="60" cy="60" r="54"></circle>
                            <circle class="progress-bar" cx="60" cy="60" r="54" data-progress="<?= $overallProgress ?>"></circle>
                        </svg>
                        <div class="progress-text">
                            <span class="progress-percentage"><?= $overallProgress ?>%</span>
                            <span class="progress-label">Overall Progress</span>
                        </div>
                    </div>
                </div>
            </div>
            <div class="progress-details">
                <h3>Learning Journey</h3>
                <p>Keep up the great work! You're making excellent progress on your learning path.</p>

                
                <!-- Enrolled Courses with Grades -->
                <div class="enrolled-courses-section">
                    <div class="enrolled-courses-card">
                        <div class="cycling-header">
                            <h3><i class='bx bx-book-open'></i> My Enrolled Courses</h3>
                            <div class="courses-badge">
                                <span class="badge-text">Track your progress</span>
                            </div>
                        </div>
                        
                        <!-- Learning Stats Mini Cards -->
                        <div class="learning-stats-mini">
                            <div class="mini-stat-card completed">
                                <div class="mini-stat-icon">
                                    <i class='bx bx-check-circle'></i>
                                </div>
                                <div class="mini-stat-content">
                                    <span class="mini-stat-number"><?= $completedCourses ?></span>
                                    <span class="mini-stat-label">Completed</span>
                                </div>
                            </div>
                            <div class="mini-stat-card progress">
                                <div class="mini-stat-icon">
                                    <i class='bx bx-time-five'></i>
                                </div>
                                <div class="mini-stat-content">
                                    <span class="mini-stat-number"><?= $enrolledCourses - $completedCourses ?></span>
                                    <span class="mini-stat-label">In Progress</span>
                                </div>
                            </div>
                            <div class="mini-stat-card hours">
                                <div class="mini-stat-icon">
                                    <i class='bx bx-time'></i>
                                </div>
                                <div class="mini-stat-content">
                                    <span class="mini-stat-number"><?= $totalLearningHours ?></span>
                                    <span class="mini-stat-label">Hours</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="enrolled-courses-grid" id="enrolledCoursesGrid">
                            <div class="course-loading">
                                <div class="loading-spinner"></div>
                                <p>Loading your courses...</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Featured Courses Section -->
    <div class="featured-courses-section">
        <div class="featured-courses-card">
            <div class="cycling-header">
                <h3><i class='bx bx-sparkles'></i> Featured Courses</h3>
                <div class="cycling-indicator">
                    <span class="indicator-dot active"></span>
                    <span class="indicator-dot"></span>
                    <span class="indicator-dot"></span>
                </div>
            </div>
            <div class="cycling-courses" id="cyclingCourses">
                <!-- Loading placeholder -->
                <div class="course-loading">
                    <div class="loading-spinner"></div>
                    <p>Loading courses...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Statistics Grid -->
    <div class="modern-stats-grid">
        <!-- Enrolled Courses -->
        <div class="modern-stat-card enrolled">
            <div class="stat-header">
                <div class="stat-icon bg-primary">
                    <i class='bx bx-book-open'></i>
                </div>
                <div class="stat-trend">
                    <i class='bx bx-trending-up'></i>
                </div>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $enrolledCourses ?></div>
                <div class="stat-label">Enrolled Courses</div>
                <div class="stat-description">Your active learning journey</div>
            </div>
            <div class="stat-footer">
                <a href="courses.php" class="stat-link">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
        </div>

        <!-- Completed Courses -->
        <div class="modern-stat-card completed">
            <div class="stat-header">
                <div class="stat-icon bg-success">
                    <i class='bx bx-check-circle'></i>
                </div>
                <div class="stat-badge">
                    <?php 
                    $completionRate = $enrolledCourses > 0 ? round(($completedCourses / $enrolledCourses) * 100) : 0;
                    echo $completionRate . '%';
                    ?>
                </div>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $completedCourses ?></div>
                <div class="stat-label">Completed Courses</div>
                <div class="stat-description">Achievement unlocked!</div>
            </div>
            <div class="stat-progress">
                <div class="progress-container">
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?= $completionRate ?>%"></div>
                    </div>
                    <span class="progress-text"><?= $completionRate ?>% Complete</span>
                </div>
            </div>
        </div>

        <!-- Assignments Due -->
        <div class="modern-stat-card assignments">
            <div class="stat-header">
                <div class="stat-icon bg-warning">
                    <i class='bx bx-task'></i>
                </div>
                <?php if ($assignmentsDue > 0): ?>
                <div class="stat-alert">
                    <i class='bx bx-time'></i>
                </div>
                <?php endif; ?>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $assignmentsDue ?></div>
                <div class="stat-label">Assignments Due</div>
                <div class="stat-description"><?= $assignmentsDue > 0 ? 'Pending submissions' : 'All caught up!' ?></div>
            </div>
            <div class="stat-footer">
                <a href="assignments.php" class="stat-link">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
        </div>

        <!-- Certificates Earned -->
        <div class="modern-stat-card certificates">
            <div class="stat-header">
                <div class="stat-icon bg-info">
                    <i class='bx bx-award'></i>
                </div>
                <?php if ($certificatesEarned > 0): ?>
                <div class="stat-celebration">
                    <i class='bx bx-trophy'></i>
                </div>
                <?php endif; ?>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $certificatesEarned ?></div>
                <div class="stat-label">Certificates Earned</div>
                <div class="stat-description"><?= $certificatesEarned > 0 ? 'Well deserved!' : 'Coming soon...' ?></div>
            </div>
            <div class="stat-footer">
                <a href="certificates.php" class="stat-link">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content Grid -->
    <div class="dashboard-grid">
        <!-- Continue Learning Card -->
        <div class="modern-card">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-play-circle'></i> Continue Learning</h3>
                    <span class="course-count"><?= $enrolledCourses ?> courses</span>
                </div>
                <a href="courses.php" class="view-all-btn">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
            <div class="card-content">
                <?php if ($enrolledCourses > 0): ?>
                <div class="course-list">
                    <!-- Placeholder for recent courses -->
                    <div class="modern-empty-state">
                        <div class="empty-illustration">
                            <i class='bx bx-book-reader'></i>
                        </div>
                        <div class="empty-content">
                            <h4>Ready to Continue!</h4>
                            <p>Your enrolled courses are being loaded. Start where you left off.</p>
                            <a href="courses.php" class="modern-btn">
                                <i class='bx bx-book-open'></i>
                                <span>View My Courses</span>
                            </a>
                        </div>
                    </div>
                </div>
                <?php else: ?>
                <div class="modern-empty-state">
                    <div class="empty-illustration">
                        <i class='bx bx-book-add'></i>
                    </div>
                    <div class="empty-content">
                        <h4>Start Your Learning Journey</h4>
                        <p>Discover amazing courses and begin building new skills today!</p>
                        <a href="courses.php" class="modern-btn">
                            <i class='bx bx-search'></i>
                            <span>Browse Courses</span>
                        </a>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Recent Activity Card -->
        <div class="modern-card">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-pulse'></i> Recent Activity</h3>
                    <span class="activity-indicator">
                        <span class="indicator-dot"></span>
                        <span>Live</span>
                    </span>
                </div>
            </div>
            <div class="card-content">
                <div class="modern-empty-state">
                    <div class="empty-illustration">
                        <i class='bx bx-trending-up'></i>
                    </div>
                    <div class="empty-content">
                        <h4>Your Activity Dashboard</h4>
                        <p>Track your learning progress, completions, and achievements here.</p>
                        <div class="activity-preview">
                            <div class="preview-item">
                                <i class='bx bx-check-circle'></i>
                                <span>Course completions</span>
                            </div>
                            <div class="preview-item">
                                <i class='bx bx-task'></i>
                                <span>Assignment submissions</span>
                            </div>
                            <div class="preview-item">
                                <i class='bx bx-trophy'></i>
                                <span>Achievements earned</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Quick Actions -->
    <div class="modern-card full-width">
        <div class="card-header">
            <div class="header-content">
                <h3><i class='bx bx-zap'></i> Quick Actions</h3>
                <span class="subtitle">Everything you need at your fingertips</span>
            </div>
        </div>
        <div class="card-content">
            <div class="modern-actions-grid">
                <a href="courses.php" class="modern-action-card courses">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-book-open'></i>
                    </div>
                    <div class="action-content">
                        <h4>Browse Courses</h4>
                        <p>Discover new learning opportunities</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
                
                <a href="assignments.php" class="modern-action-card assignments">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-task'></i>
                        <?php if ($assignmentsDue > 0): ?>
                        <span class="action-badge"><?= $assignmentsDue ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="action-content">
                        <h4>View Assignments</h4>
                        <p>Check your pending assignments</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
                
                <a href="calendar.php" class="modern-action-card calendar">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-calendar'></i>
                    </div>
                    <div class="action-content">
                        <h4>My Schedule</h4>
                        <p>View your class schedule</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
                
                <a href="certificates.php" class="modern-action-card certificates">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-award'></i>
                        <?php if ($certificatesEarned > 0): ?>
                        <span class="action-badge"><?= $certificatesEarned ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="action-content">
                        <h4>Certificates</h4>
                        <p>View your earned certificates</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
            </div>
        </div>
    </div>


</div>

<!-- Clean Modern CSS Styling -->
<style>
/* Dashboard Content */
.dashboard-content {
    background: whitesmoke;
    min-height: 100vh;
    padding: 1.5rem;
}

/* Welcome Banner */
.welcome-banner {
    background: whitesmoke;
    border-radius: 1rem;
    padding: 2rem;
    margin-bottom: 2rem;
    color: #2c3e50;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    color: #4e73df;
    animation: float 8s ease-in-out infinite;
    transition: opacity 0.3s ease;
}

.floating-icon:nth-child(1) {
    top: 20%;
    left: 85%;
}

.floating-icon:nth-child(2) {
    top: 60%;
    left: 90%;
}

.floating-icon:nth-child(3) {
    top: 15%;
    left: 10%;
}

.floating-icon:nth-child(4) {
    top: 70%;
    left: 15%;
}

.floating-icon:nth-child(5) {
    top: 40%;
    left: 5%;
}

.floating-icon:nth-child(6) {
    top: 30%;
    left: 95%;
}

@keyframes float {
    0%, 100% {
        transform: translateY(0px) translateX(0px);
        opacity: 0.1;
    }
    25% {
        transform: translateY(-10px) translateX(5px);
        opacity: 0.15;
    }
    50% {
        transform: translateY(-5px) translateX(-3px);
        opacity: 0.12;
    }
    75% {
        transform: translateY(-15px) translateX(7px);
        opacity: 0.18;
    }
}

.welcome-banner:hover .floating-icon {
    opacity: 0.2;
}

/* Ensure banner content is above floating icons */
.banner-content {
    position: relative;
    z-index: 2;
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
    gap: 1.5rem;
}

.welcome-text h1 {
    font-size: 2.5rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
}

.welcome-text p {
    font-size: 1.1rem;
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
}

.welcome-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
    justify-content: center;
}

.welcome-actions .btn {
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    border-radius: 0.5rem;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: 2px solid transparent;
}

.welcome-actions .btn-primary {
    background: #4e73df;
    color: white;
    border-color: #4e73df;
}

.welcome-actions .btn-primary:hover {
    background: #2e59d9;
    border-color: #2e59d9;
    transform: translateY(-2px);
}

.welcome-actions .btn-outline-primary {
    background: transparent;
    color: #4e73df;
    border-color: #4e73df;
}

.welcome-actions .btn-outline-primary:hover {
    background: #4e73df;
    color: white;
}

/* Featured Courses Section */
.featured-courses-section {
    margin-bottom: 2rem;
}

.featured-courses-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    border: 1px solid #e3e6f0;
}

.cycling-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.cycling-header h3 {
    margin: 0;
    font-size: 1.2rem;
    font-weight: 700;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #1a2332;
}

.cycling-header h3 i {
    color: #3366ff;
    font-size: 1.3rem;
}

.cycling-indicator {
    display: flex;
    gap: 0.5rem;
}

.indicator-dot {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: rgba(44, 62, 80, 0.3);
    transition: all 0.3s ease;
}

.indicator-dot.active {
    background: #4e73df;
    transform: scale(1.2);
}

/* Cycling Courses Grid */
.cycling-courses {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 0.5rem;
    transition: all 0.5s ease;
    max-width: 800px;
    margin: 0 auto;
}

/* Small Horizontal Course Cards */
.mini-course-card {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 0.5rem;
    padding: 0.5rem;
    color: #333;
    transition: all 0.3s ease;
    cursor: pointer;
    transform: translateY(20px);
    opacity: 0;
    animation: slideInUp 0.5s ease forwards;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    height: 70px;
    min-width: 180px;
}

.mini-course-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    background: white;
}

.course-thumbnail {
    width: 50px;
    height: 50px;
    border-radius: 0.375rem;
    object-fit: cover;
    background: #f8f9fc;
    flex-shrink: 0;
}

.course-info {
    flex: 1;
    display: flex;
    flex-direction: column;
    justify-content: center;
    min-width: 0;
}

.course-info h4 {
    font-size: 0.7rem;
    font-weight: 600;
    margin: 0 0 0.2rem 0;
    color: #2c3e50;
    line-height: 1.1;
    display: -webkit-box;
    -webkit-line-clamp: 1;
    -webkit-box-orient: vertical;
    overflow: hidden;
    white-space: nowrap;
    text-overflow: ellipsis;
}

.course-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 0.6rem;
    margin-bottom: 0.1rem;
    gap: 0.25rem;
}

.course-level {
    background: #4e73df;
    color: white;
    padding: 0.05rem 0.3rem;
    border-radius: 0.2rem;
    font-weight: 500;
    font-size: 0.55rem;
    white-space: nowrap;
}

.course-rating {
    color: #f6c23e;
    font-weight: 600;
    font-size: 0.6rem;
}

.course-price {
    font-size: 0.65rem;
    font-weight: 700;
    color: #1cc88a;
    white-space: nowrap;
}

/* Loading Animation */
.course-loading {
    grid-column: 1 / -1;
    text-align: center;
    padding: 2rem 1rem;
    color: #5a5c69;
}

.loading-spinner {
    width: 30px;
    height: 30px;
    border: 3px solid rgba(78, 115, 223, 0.2);
    border-top: 3px solid #4e73df;
    border-radius: 50%;
    margin: 0 auto 0.5rem;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

@keyframes slideInUp {
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

/* Card Animation Delays */
.mini-course-card:nth-child(1) { animation-delay: 0.1s; }
.mini-course-card:nth-child(2) { animation-delay: 0.2s; }
.mini-course-card:nth-child(3) { animation-delay: 0.3s; }
.mini-course-card:nth-child(4) { animation-delay: 0.4s; }
.mini-course-card:nth-child(5) { animation-delay: 0.5s; }
.mini-course-card:nth-child(6) { animation-delay: 0.6s; }

/* Enrolled Courses Section - Clean Panadite Branding */
.enrolled-courses-section {
    margin-bottom: 2rem;
}

.enrolled-courses-card {
    background: white;
    border-radius: 0.75rem;
    padding: 1.5rem 1rem;
    border: 1px solid #e8ecf4;
    transition: all 0.3s ease;
}

.enrolled-courses-card:hover {
    border-color: #3366ff;
    transform: translateY(-1px);
}

.courses-badge {
    background: #f0f4ff;
    color: #3366ff;
    padding: 0.4rem 1rem;
    border-radius: 2rem;
    font-size: 0.75rem;
    font-weight: 600;
    border: 1px solid #e0ebff;
}

.enrolled-courses-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 0.5rem;
    transition: all 0.5s ease;
    max-width: 100%;
    margin: 0;
    padding: 0;
}

/* Small Enrolled Course Cards - Clean Panadite Style */
.enrolled-mini-card {
    background: #fafbff;
    border-radius: 0.6rem;
    padding: 0.75rem;
    transition: all 0.3s ease;
    opacity: 1;
    transform: translateY(20px);
    opacity: 0;
    animation: slideInUp 0.5s ease forwards;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    min-height: 85px;
    height: auto;
    width: 100%;
    max-width: 280px;
    border: 1px solid #e8ecf4;
}

.enrolled-mini-card:hover {
    transform: translateY(-2px);
    background: #f0f4ff;
    border-color: #3366ff;
    box-shadow: 0 4px 12px rgba(51, 102, 255, 0.08);
}

.enrolled-course-thumbnail {
    width: 56px;
    height: 56px;
    border-radius: 0.5rem;
    object-fit: cover;
    background: #f0f4ff;
    flex-shrink: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 1px solid #e0ebff;
}

.enrolled-course-thumbnail img {
    width: 100%;
    height: 100%;
    border-radius: 0.5rem;
    object-fit: cover;
}

.enrolled-course-thumbnail i {
    font-size: 1.8rem;
    color: #3366ff;
}

.enrolled-course-info {
    flex: 1;
    display: flex;
    flex-direction: column;
    justify-content: center;
    min-width: 0;
    padding: 0.1rem 0;
}

.enrolled-course-info h4 {
    font-size: 0.8rem;
    font-weight: 700;
    margin: 0 0 0.4rem 0;
    color: #1a2332;
    line-height: 1.3;
    overflow: hidden;
    white-space: nowrap;
    text-overflow: ellipsis;
    max-width: 100%;
}

.enrolled-course-meta {
    display: flex;
    justify-content: flex-start;
    align-items: center;
    font-size: 0.65rem;
    margin-bottom: 0.15rem;
    gap: 0.4rem;
    flex-wrap: wrap;
}

.enrolled-course-progress {
    background: #e8f2ff;
    color: #3366ff;
    padding: 0.15rem 0.5rem;
    border-radius: 1rem;
    font-weight: 600;
    font-size: 0.6rem;
    white-space: nowrap;
    border: 1px solid #d4e6ff;
}

.enrolled-course-grade {
    background: #e8f5e8;
    color: #22c55e;
    padding: 0.15rem 0.5rem;
    border-radius: 1rem;
    font-weight: 600;
    font-size: 0.6rem;
    white-space: nowrap;
    border: 1px solid #bbf7d0;
}

.enrolled-course-instructor {
    font-size: 0.7rem;
    color: #6b7280;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.enrolled-course-instructor::before {
    content: "👨‍🏫";
    font-size: 0.7rem;
}

/* Learning Stats Mini Cards */
.learning-stats-mini {
    display: flex;
    gap: 0.5rem;
    margin: 1rem 0 1.5rem 0;
    justify-content: flex-start;
    flex-wrap: wrap;
    width: 100%;
    clear: both;
}

.mini-stat-card {
    background: #fafbff;
    border: 1px solid #e8ecf4;
    border-radius: 0.4rem;
    padding: 0.5rem 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.4rem;
    flex: 1;
    max-width: 140px;
    min-width: 100px;
    height: 56px;
    transition: all 0.3s ease;
    cursor: pointer;
    position: relative;
}

.mini-stat-card:hover {
    transform: translateY(-1px);
    box-shadow: 0 2px 8px rgba(51, 102, 255, 0.06);
}

.mini-stat-card.completed {
    border-color: #bbf7d0;
    background: #f0fdf4;
}

.mini-stat-card.completed:hover {
    border-color: #22c55e;
    background: #ecfdf5;
}

.mini-stat-card.progress {
    border-color: #dbeafe;
    background: #eff6ff;
}

.mini-stat-card.progress:hover {
    border-color: #3366ff;
    background: #dbeafe;
}

.mini-stat-card.hours {
    border-color: #fef3c7;
    background: #fffbeb;
}

.mini-stat-card.hours:hover {
    border-color: #f59e0b;
    background: #fef3c7;
}

.mini-stat-icon {
    width: 28px;
    height: 28px;
    border-radius: 0.3rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.9rem;
    flex-shrink: 0;
}

.mini-stat-card.completed .mini-stat-icon {
    background: #dcfce7;
    color: #22c55e;
}

.mini-stat-card.progress .mini-stat-icon {
    background: #dbeafe;
    color: #3366ff;
}

.mini-stat-card.hours .mini-stat-icon {
    background: #fef3c7;
    color: #f59e0b;
}

.mini-stat-content {
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: flex-start;
    min-width: 0;
    flex: 1;
    height: 100%;
}

.mini-stat-number {
    font-size: 1.1rem;
    font-weight: 700;
    color: #1a2332;
    line-height: 1;
    margin-bottom: 0.1rem;
}

.mini-stat-label {
    font-size: 0.65rem;
    font-weight: 600;
    color: #6b7280;
    text-transform: uppercase;
    letter-spacing: 0.02em;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    max-width: 100%;
}

/* Responsive Mini Stats */
@media (max-width: 768px) {
    .learning-stats-mini {
        justify-content: space-between;
        gap: 0.4rem;
    }
    
    .mini-stat-card {
        min-width: 90px;
        max-width: 120px;
        padding: 0.4rem 0.6rem;
    }
    
    .mini-stat-icon {
        width: 24px;
        height: 24px;
        font-size: 0.8rem;
    }
    
    .mini-stat-number {
        font-size: 1rem;
    }
    
    .mini-stat-label {
        font-size: 0.6rem;
    }
}

@media (max-width: 480px) {
    .learning-stats-mini {
        flex-direction: column;
        gap: 0.5rem;
        align-items: stretch;
    }
    
    .mini-stat-card {
        max-width: none;
        min-width: auto;
        flex: none;
    }
}

/* Responsive Featured Courses */
@media (max-width: 1200px) {
    .cycling-courses {
        grid-template-columns: repeat(3, 1fr);
        max-width: 650px;
    }
}

@media (max-width: 992px) {
    .featured-courses-card {
        padding: 1.5rem;
    }
    
    .cycling-courses {
        grid-template-columns: repeat(2, 1fr);
        max-width: 450px;
        margin: 0 auto;
    }
}

@media (max-width: 768px) {
    .welcome-banner {
        padding: 1.5rem;
    }
    
    .cycling-courses {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.4rem;
        max-width: 400px;
    }
    
    .mini-course-card {
        padding: 0.4rem;
        height: 60px;
        min-width: 160px;
    }
    
    .course-thumbnail {
        width: 40px;
        height: 40px;
    }
    
    .course-info h4 {
        font-size: 0.65rem;
    }
    
    .course-meta {
        font-size: 0.55rem;
    }
    
    .course-price {
        font-size: 0.6rem;
    }
    
    .course-level {
        font-size: 0.5rem;
        padding: 0.03rem 0.25rem;
    }
}

@media (max-width: 576px) {
    .cycling-courses {
        grid-template-columns: 1fr;
        max-width: 220px;
    }
    
    .mini-course-card {
        min-width: 200px;
        height: 55px;
    }
    
    .course-thumbnail {
        width: 35px;
        height: 35px;
    }
    
    .cycling-header h3 {
        font-size: 1rem;
    }
}

/* Overall Progress Section */
.progress-section {
    margin-bottom: 2rem;
}

.progress-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    border: 1px solid #e3e6f0;
    display: grid;
    grid-template-columns: auto 1fr;
    gap: 2rem;
    align-items: center;
}

.progress-circle-container {
    display: flex;
    justify-content: center;
    align-items: center;
}

.circular-progress {
    position: relative;
    width: 140px;
    height: 140px;
}

.progress-ring {
    position: relative;
    width: 100%;
    height: 100%;
}

.progress-svg {
    width: 100%;
    height: 100%;
    transform: rotate(-90deg);
}

.progress-bg {
    fill: none;
    stroke: #f1f3f4;
    stroke-width: 8;
}

.progress-bar {
    fill: none;
    stroke: #4e73df;
    stroke-width: 8;
    stroke-linecap: round;
    stroke-dasharray: 339.29; /* 2 * π * 54 */
    stroke-dashoffset: 339.29;
    transition: stroke-dashoffset 2s ease-in-out;
}

.progress-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    text-align: center;
}

.progress-percentage {
    display: block;
    font-size: 1.8rem;
    font-weight: 700;
    color: #2c3e50;
    line-height: 1;
}

.progress-label {
    display: block;
    font-size: 0.8rem;
    color: #5a5c69;
    margin-top: 0.25rem;
    font-weight: 500;
}

.progress-details h3 {
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
    font-size: 1.5rem;
    font-weight: 600;
}

.progress-details p {
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
    line-height: 1.5;
}

.progress-stats {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 1rem;
}

.progress-stat {
    text-align: center;
    padding: 1rem;
    background: #f8f9fc;
    border-radius: 0.5rem;
    border: 1px solid #e3e6f0;
}

.progress-stat .stat-number {
    display: block;
    font-size: 1.5rem;
    font-weight: 700;
    color: #4e73df;
    line-height: 1;
}

.progress-stat .stat-label {
    display: block;
    font-size: 0.8rem;
    color: #5a5c69;
    margin-top: 0.25rem;
    font-weight: 500;
}

/* Responsive Progress Card */
@media (max-width: 992px) {
    .progress-card {
        grid-template-columns: 1fr;
        text-align: center;
        gap: 1.5rem;
    }
    
    .progress-stats {
        grid-template-columns: repeat(3, 1fr);
        gap: 0.75rem;
    }
}

@media (max-width: 768px) {
    .progress-card {
        padding: 1.5rem;
    }
    
    .circular-progress {
        width: 120px;
        height: 120px;
    }
    
    .progress-percentage {
        font-size: 1.5rem;
    }
    
    .progress-stats {
        grid-template-columns: 1fr;
        gap: 0.5rem;
    }
    
    .progress-stat {
        padding: 0.75rem;
    }
}

/* Modern Card System */
.modern-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.08);
    border: 1px solid #e3e6f0;
    overflow: hidden;
    transition: all 0.3s ease;
    margin-bottom: 2rem;
}

.modern-card:hover {
    box-shadow: 0 0.5rem 2rem rgba(0, 0, 0, 0.12);
    transform: translateY(-2px);
}

.modern-card.full-width {
    grid-column: 1 / -1;
}

/* Card Headers */
.card-header {
    padding: 1.5rem 2rem 1rem 2rem;
    border-bottom: 1px solid #f1f3f4;
    background: linear-gradient(135deg, #f8f9fc 0%, #ffffff 100%);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.header-content h3 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.2rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.header-content h3 i {
    color: #4e73df;
    font-size: 1.3rem;
}

.course-count {
    background: #e3f2fd;
    color: #1976d2;
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 600;
    margin-left: 1rem;
}

.subtitle {
    color: #5a5c69;
    font-size: 0.9rem;
    margin-top: 0.25rem;
    display: block;
}

.view-all-btn {
    color: #4e73df;
    text-decoration: none;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
    font-size: 0.9rem;
}

.view-all-btn:hover {
    background: #4e73df;
    color: white;
    transform: translateX(3px);
}

.activity-indicator {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #1cc88a;
    font-size: 0.8rem;
    font-weight: 600;
}

.indicator-dot {
    width: 8px;
    height: 8px;
    background: #1cc88a;
    border-radius: 50%;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.4; }
}

/* Card Content */
.card-content {
    padding: 2rem;
}

/* Modern Empty States */
.modern-empty-state {
    text-align: center;
    padding: 3rem 2rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1.5rem;
}

.empty-illustration {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
    position: relative;
    overflow: hidden;
}

.empty-illustration::before {
    content: '';
    position: absolute;
    top: -50%;
    left: -50%;
    width: 200%;
    height: 200%;
    background: linear-gradient(45deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shine 3s infinite;
}

@keyframes shine {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.empty-illustration i {
    font-size: 2.5rem;
    color: white;
    z-index: 1;
    position: relative;
}

.empty-content h4 {
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
    font-size: 1.3rem;
    font-weight: 600;
}

.empty-content p {
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
    line-height: 1.6;
    max-width: 300px;
}

.modern-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    background: #4e73df;
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 0.5rem;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(78, 115, 223, 0.3);
}

.modern-btn:hover {
    background: #2e59d9;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(78, 115, 223, 0.4);
    color: white;
}

/* Activity Preview */
.activity-preview {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-top: 1.5rem;
}

.preview-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #f8f9fc;
    border-radius: 0.5rem;
    border-left: 3px solid #4e73df;
}

.preview-item i {
    color: #4e73df;
    font-size: 1.1rem;
}

.preview-item span {
    color: #5a5c69;
    font-size: 0.9rem;
}

/* Modern Actions Grid */
.modern-actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
}

.modern-action-card {
    background: linear-gradient(135deg, #f8f9fc 0%, #ffffff 100%);
    border: 2px solid #e3e6f0;
    border-radius: 1rem;
    padding: 1.5rem;
    text-decoration: none;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 1rem;
    position: relative;
    overflow: hidden;
}

.modern-action-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: #4e73df;
    transform: scaleY(0);
    transition: transform 0.3s ease;
}

.modern-action-card:hover::before {
    transform: scaleY(1);
}

.modern-action-card:hover {
    border-color: #4e73df;
    box-shadow: 0 8px 25px rgba(78, 115, 223, 0.15);
    transform: translateY(-3px);
}

.action-icon-wrapper {
    position: relative;
    width: 60px;
    height: 60px;
    border-radius: 1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

.modern-action-card.courses .action-icon-wrapper {
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
}

.modern-action-card.assignments .action-icon-wrapper {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
}

.modern-action-card.calendar .action-icon-wrapper {
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
}

.modern-action-card.certificates .action-icon-wrapper {
    background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
}

.action-icon-wrapper i {
    font-size: 1.8rem;
    color: white;
}

.action-badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #dc3545;
    color: white;
    font-size: 0.7rem;
    font-weight: 700;
    padding: 0.2rem 0.5rem;
    border-radius: 1rem;
    min-width: 20px;
    text-align: center;
}

.action-content {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.action-content h4 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.1rem;
    font-weight: 600;
}

.action-content p {
    margin: 0;
    color: #5a5c69;
    font-size: 0.9rem;
    line-height: 1.4;
}

.action-arrow {
    color: #4e73df;
    font-size: 1.2rem;
    transition: transform 0.3s ease;
}

.modern-action-card:hover .action-arrow {
    transform: translateX(5px);
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 2rem;
    margin-bottom: 2rem;
}

/* Responsive Design */
@media (max-width: 992px) {
    .dashboard-grid {
        grid-template-columns: 1fr;
    }
    
    .modern-actions-grid {
        grid-template-columns: 1fr;
    }
    
    .card-header {
        padding: 1rem 1.5rem 0.75rem 1.5rem;
    }
    
    .card-content {
        padding: 1.5rem;
    }
}

@media (max-width: 768px) {
    .modern-empty-state {
        padding: 2rem 1rem;
    }
    
    .empty-illustration {
        width: 60px;
        height: 60px;
    }
    
    .empty-illustration i {
        font-size: 2rem;
    }
    
    .modern-action-card {
        padding: 1rem;
    }
    
    .action-icon-wrapper {
        width: 50px;
        height: 50px;
    }
    
    .action-icon-wrapper i {
        font-size: 1.5rem;
    }
}

/* Statistics Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: 0.75rem;
    padding: 1.5rem;
    box-shadow: 0 0.125rem 0.75rem rgba(0, 0, 0, 0.1);
    border: 1px solid #e3e6f0;
    transition: all 0.2s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 0.25rem 1.25rem rgba(0, 0, 0, 0.15);
}

.stat-icon {
    width: 3rem;
    height: 3rem;
    border-radius: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
    font-size: 1.25rem;
    color: white;
}

.stat-icon.bg-primary { background: #4e73df; }
.stat-icon.bg-success { background: #1cc88a; }
.stat-icon.bg-warning { background: #f6c23e; color: #333; }
.stat-icon.bg-info { background: #36b9cc; }

.stat-content {
    margin-bottom: 1rem;
}

.stat-number {
    font-size: 2rem;
    font-weight: 700;
    color: #5a5c69;
    line-height: 1;
}

.stat-label {
    font-size: 0.875rem;
    color: #858796;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-top: 0.25rem;
}

.stat-action a {
    color: #4e73df;
    text-decoration: none;
    font-weight: 500;
    font-size: 0.875rem;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.stat-action a:hover {
    color: #2e59d9;
}

.stat-progress {
    margin-top: 1rem;
}

.progress-bar {
    width: 100%;
    height: 6px;
    background: #eaecf4;
    border-radius: 3px;
    overflow: hidden;
    margin-bottom: 0.5rem;
}

.progress-fill {
    height: 100%;
    background: #1cc88a;
    transition: width 0.3s ease;
}

.progress-text {
    font-size: 0.75rem;
    color: #858796;
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    margin-bottom: 2rem;
}

.dashboard-section {
    background: white;
    border-radius: 0.75rem;
    border: 1px solid #e3e6f0;
    overflow: hidden;
}

.section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid #e3e6f0;
    background: #f8f9fc;
}

.section-header h3 {
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: #5a5c69;
}

.view-all-link {
    color: #4e73df;
    text-decoration: none;
    font-size: 0.875rem;
    font-weight: 500;
}

.view-all-link:hover {
    color: #2e59d9;
}

.section-content {
    padding: 1.5rem;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 2rem 1rem;
}

.empty-icon {
    width: 4rem;
    height: 4rem;
    margin: 0 auto 1rem;
    background: #f8f9fc;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #858796;
    font-size: 1.5rem;
}

.empty-state h4 {
    margin: 0 0 0.5rem 0;
    color: #5a5c69;
    font-weight: 600;
}

.empty-state p {
    margin: 0 0 1rem 0;
    color: #858796;
    font-size: 0.875rem;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
}

/* Quick Actions */
.quick-actions {
    background: white;
    border-radius: 0.75rem;
    padding: 1.5rem;
    border: 1px solid #e3e6f0;
}

.quick-actions h3 {
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
    font-weight: 600;
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.action-card {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border: 1px solid #e3e6f0;
    border-radius: 0.5rem;
    text-decoration: none;
    color: inherit;
    transition: all 0.2s ease;
}

.action-card:hover {
    background: #f8f9fc;
    border-color: #4e73df;
    color: inherit;
    transform: translateY(-1px);
}

.action-icon {
    width: 2.5rem;
    height: 2.5rem;
    background: #4e73df;
    color: white;
    border-radius: 0.375rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    flex-shrink: 0;
}

.action-text h4 {
    margin: 0 0 0.25rem 0;
    font-size: 0.95rem;
    font-weight: 600;
    color: #5a5c69;
}

.action-text p {
    margin: 0;
    font-size: 0.8rem;
    color: #858796;
}

/* Responsive Design */
@media (max-width: 768px) {
    .dashboard-content {
        padding: 1rem;
    }
    
    .banner-content {
        flex-direction: column;
        text-align: center;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .dashboard-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .actions-grid {
        grid-template-columns: 1fr;
    }
    
    .welcome-text h1 {
        font-size: 1.5rem;
    }
}

@media (max-width: 576px) {
    .welcome-actions {
        width: 100%;
        justify-content: center;
    }
    
    .welcome-actions .btn {
        flex: 1;
        justify-content: center;
    }
}

/* Modern Statistics Grid */
.modern-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 2rem;
    margin-bottom: 2.5rem;
}

.modern-stat-card {
    background: white;
    border-radius: 1rem;
    padding: 0;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.08);
    border: 1px solid #e3e6f0;
    transition: all 0.4s ease;
    position: relative;
    overflow: hidden;
    cursor: pointer;
}

.modern-stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #4e73df, #36b9cc);
    transform: scaleX(0);
    transition: transform 0.4s ease;
}

.modern-stat-card:hover::before {
    transform: scaleX(1);
}

.modern-stat-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 1rem 3rem rgba(0, 0, 0, 0.15);
}

/* Stat Card Header */
.stat-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem 1.5rem 0 1.5rem;
    position: relative;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: 1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.stat-icon i {
    font-size: 1.8rem;
    color: white;
}

/* Keep Original Icon Colors */
.bg-primary {
    background: linear-gradient(135deg, #4e73df 0%, #3c65d4 100%);
}

.bg-success {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
}

.bg-warning {
    background: linear-gradient(135deg, #f6c23e 0%, #dda20a 100%);
}

.bg-info {
    background: linear-gradient(135deg, #36b9cc 0%, #2c9faf 100%);
}

/* Stat Header Elements */
.stat-trend {
    color: #1cc88a;
    font-size: 1.2rem;
    opacity: 0.7;
    transition: all 0.3s ease;
}

.modern-stat-card:hover .stat-trend {
    opacity: 1;
    transform: scale(1.1);
}

.stat-badge {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
    color: white;
    padding: 0.4rem 0.8rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 700;
    box-shadow: 0 2px 8px rgba(28, 200, 138, 0.3);
}

.stat-alert {
    color: #f6c23e;
    font-size: 1.2rem;
    animation: pulse-alert 2s infinite;
}

@keyframes pulse-alert {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.6; transform: scale(1.1); }
}

.stat-celebration {
    color: #f1c40f;
    font-size: 1.2rem;
    animation: celebrate 3s infinite;
}

@keyframes celebrate {
    0%, 90%, 100% { transform: scale(1) rotate(0deg); }
    5% { transform: scale(1.2) rotate(10deg); }
    10% { transform: scale(1.1) rotate(-10deg); }
    15% { transform: scale(1.2) rotate(10deg); }
}

/* Stat Card Body */
.stat-body {
    padding: 1rem 1.5rem;
    text-align: left;
}

.stat-number {
    font-size: 3rem;
    font-weight: 700;
    color: #2c3e50;
    line-height: 1;
    margin-bottom: 0.5rem;
    background: linear-gradient(135deg, #2c3e50 0%, #4e73df 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-label {
    font-size: 1rem;
    font-weight: 600;
    color: #5a5c69;
    margin-bottom: 0.25rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.stat-description {
    font-size: 0.85rem;
    color: #858796;
    font-style: italic;
    opacity: 0.8;
}

/* Stat Progress */
.stat-progress {
    padding: 0 1.5rem 1rem 1.5rem;
}

.progress-container {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: #e3e6f0;
    border-radius: 4px;
    overflow: hidden;
    position: relative;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #1cc88a 0%, #17a673 100%);
    border-radius: 4px;
    transition: width 1s ease;
    position: relative;
}

.progress-fill::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

.progress-text {
    font-size: 0.8rem;
    color: #5a5c69;
    font-weight: 600;
    text-align: right;
}

/* Stat Footer */
.stat-footer {
    padding: 0 1.5rem 1.5rem 1.5rem;
    border-top: 1px solid #f1f3f4;
    margin-top: 0.5rem;
    padding-top: 1rem;
}

.stat-link {
    display: flex;
    align-items: center;
    justify-content: space-between;
    color: #4e73df;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.9rem;
    transition: all 0.3s ease;
    padding: 0.5rem 0;
}

.stat-link:hover {
    color: #2e59d9;
    transform: translateX(5px);
}

.stat-link i {
    font-size: 1.1rem;
    transition: transform 0.3s ease;
}

.stat-link:hover i {
    transform: translateX(3px);
}

/* Specific Card Styling */
.modern-stat-card.enrolled:hover {
    border-color: #4e73df;
}

.modern-stat-card.completed:hover {
    border-color: #1cc88a;
}

.modern-stat-card.assignments:hover {
    border-color: #f6c23e;
}

.modern-stat-card.certificates:hover {
    border-color: #36b9cc;
}

/* Responsive Design for Stats */
@media (max-width: 1200px) {
    .modern-stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
    }
}

@media (max-width: 768px) {
    .modern-stats-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .stat-header {
        padding: 1rem 1rem 0 1rem;
    }
    
    .stat-body {
        padding: 0.75rem 1rem;
    }
    
    .stat-footer {
        padding: 0 1rem 1rem 1rem;
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
    }
    
    .stat-icon i {
        font-size: 1.5rem;
    }
    
    .stat-number {
        font-size: 2.5rem;
    }
}

@media (max-width: 480px) {
    .modern-stats-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.75rem;
    }
    
    .modern-stat-card {
        min-height: auto;
    }
    
    .stat-number {
        font-size: 2rem;
    }
    
    .stat-label {
        font-size: 0.85rem;
    }
    
    .stat-description {
        font-size: 0.75rem;
    }
}
</style>

        </div> <!-- Close home-content -->
    </section> <!-- Close home-section -->

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Dashboard JavaScript -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        // Course Cycling Management
        class CourseCycling {
            constructor() {
                this.currentSet = 0;
                this.allCourses = [];
                this.isLoading = false;
                this.cyclingInterval = null;
                this.init();
            }
            
            init() {
                this.loadCourses();
                this.startCycling();
                this.setupResponsive();
            }
            
            // Determine how many courses to show based on screen size
            getCoursesPerSet() {
                const width = $(window).width();
                if (width <= 576) return 2; // Mobile small
                if (width <= 768) return 3; // Mobile
                if (width <= 1200) return 4; // Tablet
                return 6; // Desktop
            }
            
            // Load courses from API
            async loadCourses() {
                if (this.isLoading) return;
                this.isLoading = true;
                
                try {
                    const response = await $.ajax({
                        url: 'api/get_random_courses.php',
                        method: 'GET',
                        data: { limit: 18 }, // Load more courses for cycling
                        dataType: 'json',
                        timeout: 10000
                    });
                    
                    if (response.success && response.courses.length > 0) {
                        this.allCourses = response.courses;
                        this.displayCourses();
                    } else {
                        this.showError('No courses available');
                    }
                } catch (error) {
                    console.error('Course loading error:', error);
                    this.showError('Failed to load courses');
                } finally {
                    this.isLoading = false;
                }
            }
            
            // Display current set of courses
            displayCourses() {
                if (this.allCourses.length === 0) return;
                
                const coursesPerSet = this.getCoursesPerSet();
                const startIndex = (this.currentSet * coursesPerSet) % this.allCourses.length;
                const coursesToShow = [];
                
                // Get courses for current set (with wrapping)
                for (let i = 0; i < coursesPerSet; i++) {
                    const courseIndex = (startIndex + i) % this.allCourses.length;
                    coursesToShow.push(this.allCourses[courseIndex]);
                }
                
                // Create HTML for courses
                const coursesHtml = coursesToShow.map(course => this.createCourseCard(course)).join('');
                
                // Fade out, update content, fade in
                $('#cyclingCourses').fadeOut(300, () => {
                    $('#cyclingCourses').html(coursesHtml).fadeIn(500);
                });
                
                // Update indicators
                this.updateIndicators();
            }
            
            // Create individual course card HTML
            createCourseCard(course) {
                const priceDisplay = course.price > 0 ? `R${course.price}` : 'Free';
                
                return `
                    <div class="mini-course-card" data-course-id="${course.id}">
                        <img src="${course.thumbnail}" alt="${course.title}" class="course-thumbnail" 
                             onerror="this.src='/assets/img/courses/default-course.jpg'">
                        <div class="course-info">
                            <h4 title="${course.title}">${course.title}</h4>
                            <div class="course-meta">
                                <span class="course-level">${course.level}</span>
                                <span class="course-rating">★ ${course.rating}</span>
                            </div>
                            <div class="course-price">${priceDisplay}</div>
                        </div>
                    </div>
                `;
            }
            
            // Update cycling indicators
            updateIndicators() {
                const maxSets = Math.ceil(this.allCourses.length / this.getCoursesPerSet());
                const activeIndex = this.currentSet % 3; // Always show 3 dots
                
                $('.indicator-dot').removeClass('active').eq(activeIndex).addClass('active');
            }
            
            // Start automatic cycling
            startCycling() {
                this.stopCycling(); // Clear any existing interval
                
                this.cyclingInterval = setInterval(() => {
                    if (this.allCourses.length > 0) {
                        this.nextSet();
                    }
                }, 4000); // Change every 4 seconds
            }
            
            // Stop cycling
            stopCycling() {
                if (this.cyclingInterval) {
                    clearInterval(this.cyclingInterval);
                    this.cyclingInterval = null;
                }
            }
            
            // Move to next set of courses
            nextSet() {
                const maxSets = Math.ceil(this.allCourses.length / this.getCoursesPerSet());
                this.currentSet = (this.currentSet + 1) % Math.max(maxSets, 1);
                this.displayCourses();
            }
            
            // Handle responsive changes
            setupResponsive() {
                let resizeTimer;
                $(window).on('resize', () => {
                    clearTimeout(resizeTimer);
                    resizeTimer = setTimeout(() => {
                        this.displayCourses(); // Refresh layout on resize
                    }, 300);
                });
            }
            
            // Show error message
            showError(message) {
                $('#cyclingCourses').html(`
                    <div class="course-loading">
                        <i class='bx bx-error-circle' style="font-size: 2rem; margin-bottom: 0.5rem;"></i>
                        <p>${message}</p>
                    </div>
                `);
            }
        }

        // Circular Progress Animation
        function animateProgress() {
            const progressBars = document.querySelectorAll('.progress-bar');
            
            progressBars.forEach(bar => {
                const progress = parseInt(bar.getAttribute('data-progress'));
                const circumference = 2 * Math.PI * 54; // radius = 54
                const progressOffset = circumference - (progress / 100) * circumference;
                
                // Start with full offset (0% progress)
                bar.style.strokeDashoffset = circumference;
                
                // Animate to actual progress after a short delay
                setTimeout(() => {
                    bar.style.strokeDashoffset = progressOffset;
                }, 500);
            });
        }

        // Initialize floating icon animations
        function initFloatingIcons() {
            const floatingIcons = document.querySelectorAll('.floating-icon');
            
            floatingIcons.forEach(icon => {
                const delay = icon.getAttribute('data-delay') || '0s';
                icon.style.animationDelay = delay;
            });
        }

        // Load enrolled courses with grades
        function loadEnrolledCourses() {
            $.ajax({
                url: 'api/get_enrolled_courses.php',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.success && response.courses) {
                        displayEnrolledCourses(response.courses);
                    } else {
                        showCoursesError('Failed to load courses');
                    }
                },
                error: function() {
                    showCoursesError('Error connecting to server');
                }
            });
        }

        // Helper function to get course icon based on title
        function getCourseIcon(title) {
            const titleLower = title.toLowerCase();
            if (titleLower.includes('programming') || titleLower.includes('code')) return 'bx-code-alt';
            if (titleLower.includes('design') || titleLower.includes('ui')) return 'bx-palette';
            if (titleLower.includes('data') || titleLower.includes('analytics')) return 'bx-bar-chart-alt-2';
            if (titleLower.includes('web') || titleLower.includes('html')) return 'bx-world';
            if (titleLower.includes('mobile') || titleLower.includes('app')) return 'bx-mobile-alt';
            if (titleLower.includes('security') || titleLower.includes('cyber')) return 'bx-shield-quarter';
            if (titleLower.includes('ai') || titleLower.includes('machine learning')) return 'bx-brain';
            if (titleLower.includes('network') || titleLower.includes('cloud')) return 'bx-cloud';
            if (titleLower.includes('database') || titleLower.includes('sql')) return 'bx-data';
            if (titleLower.includes('javascript') || titleLower.includes('js')) return 'bx-code-block';
            if (titleLower.includes('python')) return 'bx-code-curly';
            if (titleLower.includes('java')) return 'bx-coffee';
            if (titleLower.includes('business') || titleLower.includes('management')) return 'bx-briefcase';
            if (titleLower.includes('marketing') || titleLower.includes('social')) return 'bx-trending-up';
            if (titleLower.includes('powerbi') || titleLower.includes('power bi')) return 'bx-bar-chart-alt-2';
            if (titleLower.includes('cyber') || titleLower.includes('security')) return 'bx-shield-quarter';
            if (titleLower.includes('entrepreneur') || titleLower.includes('venture')) return 'bx-briefcase';
            if (titleLower.includes('assessment') || titleLower.includes('facilitator')) return 'bx-check-square';
            if (titleLower.includes('computer') || titleLower.includes('science')) return 'bx-desktop';
            return 'bx-book-open'; // Default icon
        }

        // Display enrolled courses in compact cards with clean Panadite styling
        function displayEnrolledCourses(courses) {
            const container = $('#enrolledCoursesGrid');
            
            if (courses.length === 0) {
                container.html(`
                    <div class="course-loading">
                        <div class="loading-spinner"></div>
                        <p>No enrolled courses yet. <a href="courses.php" style="color: #3366ff; text-decoration: none; font-weight: 600;">Browse courses</a> to get started!</p>
                    </div>
                `);
                return;
            }

            let coursesHtml = '';
            courses.forEach((course, index) => {
                // Handle thumbnail path - clean up any relative paths and duplicates
                let thumbnailUrl = null;
                if (course.thumbnail) {
                    let cleanPath = course.thumbnail;
                    
                    // Remove any ../ prefix
                    if (cleanPath.startsWith('../')) {
                        cleanPath = cleanPath.substring(3);
                    }
                    
                    // If path already starts with uploads/thumbnails/, just prepend /
                    if (cleanPath.startsWith('uploads/thumbnails/')) {
                        thumbnailUrl = '/' + cleanPath;
                    }
                    // If it starts with /, use as-is
                    else if (cleanPath.startsWith('/')) {
                        thumbnailUrl = cleanPath;
                    }
                    // Otherwise, assume it's just the filename and prepend full path
                    else {
                        thumbnailUrl = `/uploads/thumbnails/${cleanPath}`;
                    }
                }
                
                // Determine course icon based on title
                const courseIcon = getCourseIcon(course.title);
                
                const thumbnailContent = thumbnailUrl ? 
                    `<img src="${thumbnailUrl}" alt="${course.title}" onerror="this.parentElement.innerHTML='<i class=\\'bx ${courseIcon}\\'></i>'">` :
                    `<i class='bx ${courseIcon}'></i>`;

                coursesHtml += `
                    <div class="enrolled-mini-card" onclick="window.location.href='course-grade-details.php?course_id=${course.course_id}'" style="animation-delay: ${index * 0.1}s;">
                        <div class="enrolled-course-thumbnail">
                            ${thumbnailContent}
                        </div>
                        <div class="enrolled-course-info">
                            <h4>${course.title}</h4>
                            <div class="enrolled-course-meta">
                                <span class="enrolled-course-progress">${course.progress_percentage}% Progress</span>
                                <span class="enrolled-course-grade">${course.final_grade}% ${course.letter_grade}</span>
                            </div>
                            <div class="enrolled-course-instructor">${course.instructor}</div>
                        </div>
                    </div>
                `;
            });

            container.html(coursesHtml);
        }

        $(document).ready(function() {
            const courseCycling = new CourseCycling();
            
            // Load enrolled courses
            loadEnrolledCourses();
            
            // Animate progress circles
            animateProgress();
            
            // Initialize floating icon animations
            initFloatingIcons();
            
            // Handle course card clicks
            $(document).on('click', '.mini-course-card', function() {
                const courseId = $(this).data('course-id');
                if (courseId) {
                    window.location.href = `course-details.php?id=${courseId}`;
                }
            });
            
            // Pause cycling on hover
            $(document).on('mouseenter', '.course-cycling-container', function() {
                courseCycling.stopCycling();
            });
            
            // Resume cycling when not hovering
            $(document).on('mouseleave', '.course-cycling-container', function() {
                courseCycling.startCycling();
            });
            
            // Manual indicator clicks
            $(document).on('click', '.indicator-dot', function() {
                const index = $(this).index();
                courseCycling.currentSet = index;
                courseCycling.displayCourses();
            });
        });

        // Smooth scrolling for internal links
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('a[href^="#"]').forEach(anchor => {
                anchor.addEventListener('click', function (e) {
                    e.preventDefault();
                    const target = document.querySelector(this.getAttribute('href'));
                    if (target) {
                        target.scrollIntoView({
                            behavior: 'smooth',
                            block: 'start'
                        });
                    }
                });
            });
            
            console.log('Clean dashboard loaded successfully!');
        });
    </script>
</body>
</html>
