<?php
session_start();
require_once '../../includes/config.php';
require_once '../../includes/functions.php';

// Ensure user is logged in as student
requireRole('student');

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get POST data
$video_id = intval($_POST['video_id'] ?? 0);
$lesson_id = intval($_POST['lesson_id'] ?? 0);
$course_id = intval($_POST['course_id'] ?? 0);
$student_id = $_SESSION['user_id'];

try {
    // Check video progress in database
    $progress_query = $conn->prepare("
        SELECT 
            vp.*,
            v.title as video_title,
            v.duration as video_duration
        FROM video_progress vp
        LEFT JOIN videos v ON vp.video_id = v.video_id
        WHERE vp.student_id = ? AND vp.video_id = ?
    ");
    $progress_query->bind_param("ii", $student_id, $video_id);
    $progress_query->execute();
    $progress_result = $progress_query->get_result()->fetch_assoc();
    
    // Check video details
    $video_query = $conn->prepare("
        SELECT v.*, 
               COALESCE(vp.progress_percentage, 0) as watch_progress,
               CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END as completed,
               CASE WHEN vp.last_position > 0 THEN vp.last_position ELSE 0 END as last_position
        FROM videos v
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
        WHERE v.video_id = ?
    ");
    $video_query->bind_param("ii", $student_id, $video_id);
    $video_query->execute();
    $video_result = $video_query->get_result()->fetch_assoc();
    
    // Check all videos in course for comparison
    $all_videos_query = $conn->prepare("
        SELECT 
            v.video_id,
            v.title,
            COALESCE(vp.progress_percentage, 0) as progress,
            CASE WHEN vp.completed = 1 THEN 'YES' ELSE 'NO' END as completed,
            vp.last_watched
        FROM videos v
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
        WHERE v.course_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC
    ");
    $all_videos_query->bind_param("ii", $student_id, $course_id);
    $all_videos_query->execute();
    $all_videos_result = $all_videos_query->get_result();
    
    $all_videos = [];
    while ($row = $all_videos_result->fetch_assoc()) {
        $all_videos[] = $row;
    }
    
    // Format badge status from database
    $is_completed = ($progress_result['completed'] ?? 0) == 1;
    $current_progress = $progress_result['progress_percentage'] ?? 0;
    
    $badge_status = 'NONE';
    if ($is_completed) {
        $badge_status = 'COMPLETED ✅';
    } elseif ($current_progress > 0) {
        $badge_status = 'WATCHING (' . $current_progress . '%)';
    }
    
    echo json_encode([
        'success' => true,
        'badge_status' => $badge_status,
        'completion_status' => $is_completed ? 'COMPLETED' : 'NOT COMPLETED',
        'debug_info' => [
            'student_id' => $student_id,
            'video_id' => $video_id,
            'lesson_id' => $lesson_id,
            'course_id' => $course_id,
            'badge_info' => [
                'status' => $badge_status,
                'is_completed' => $is_completed,
                'progress_percentage' => $current_progress,
                'completion_threshold' => 90
            ],
            'current_video_progress' => $progress_result,
            'current_video_details' => $video_result,
            'all_course_videos' => $all_videos,
            'completion_threshold' => '90%',
            'analysis' => [
                'has_progress_record' => !empty($progress_result),
                'current_progress' => $current_progress,
                'is_completed_in_db' => $is_completed,
                'should_be_completed' => $current_progress >= 90
            ]
        ]
    ]);

} catch (Exception $e) {
    error_log("Debug video completion error: " . $e->getMessage());
    echo json_encode([
        'error' => 'Debug failed',
        'details' => $e->getMessage()
    ]);
}
?>
