<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session to get student ID
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Not authenticated']);
    exit;
}

$student_id = $_SESSION['user_id'];
$assignment_id = isset($_GET['assignment_id']) ? intval($_GET['assignment_id']) : 0;

if (!$assignment_id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid assignment ID']);
    exit;
}

require_once '../../config/db_connect.php';

try {
    // Get the latest submission and its grading results
    $stmt = $conn->prepare("
        SELECT 
            submission_id,
            ai_score,
            ai_feedback,
            similarity_score,
            keyword_match_score,
            structure_score,
            quality_score,
            ai_confidence,
            processing_status,
            review_needed
        FROM assignment_submissions 
        WHERE assignment_id = ? AND student_id = ? 
        ORDER BY submission_id DESC 
        LIMIT 1
    ");
    
    $stmt->bind_param('ii', $assignment_id, $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $submission = $result->fetch_assoc();
    
    if (!$submission) {
        echo json_encode(['success' => false, 'error' => 'No submission found']);
        exit;
    }
    
    if ($submission['processing_status'] !== 'completed') {
        echo json_encode(['success' => false, 'error' => 'Grading not completed yet']);
        exit;
    }
    
    // Format the grade data
    $gradeData = [
        'ai_score' => round(floatval($submission['ai_score'] ?? 0), 1),
        'similarity_score' => round(floatval($submission['similarity_score'] ?? 0), 1),
        'keyword_match_score' => round(floatval($submission['keyword_match_score'] ?? 0), 1),
        'structure_score' => round(floatval($submission['structure_score'] ?? 0), 1),
        'quality_score' => round(floatval($submission['quality_score'] ?? 0), 1),
        'ai_confidence' => round(floatval($submission['ai_confidence'] ?? 0), 1),
        'ai_feedback' => $submission['ai_feedback'] ?? 'No feedback available.',
        'review_needed' => $submission['review_needed'] ? true : false
    ];
    
    echo json_encode([
        'success' => true,
        'grade_data' => $gradeData,
        'processing_status' => $submission['processing_status']
    ]);
    
} catch (Exception $e) {
    error_log("Error fetching latest grade: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Internal server error']);
}
?>
