<?php
/**
 * AJAX Endpoint: Get Lesson Data
 * Simplified version for testing AJAX video navigation
 */

// Suppress all PHP errors from displaying 
error_reporting(0);
ini_set('display_errors', 0);

// Set JSON response header immediately
header('Content-Type: application/json');

// Start output buffering to catch any unwanted output
ob_start();

// Simple session start without complex error handling
@session_start();

// Add debug logging function
function debug_log($message) {
    error_log("[AJAX DEBUG] " . $message);
}

debug_log("AJAX endpoint called");

try {
    // Include required files for proper functionality (correct paths from student/ajax/)
    require_once '../../includes/config.php';
    require_once '../../includes/id_encryption.php';
    debug_log("Included config and encryption system");

    // Get JSON input
    $raw_input = file_get_contents('php://input');
    debug_log("Raw input: " . $raw_input);
    
    $input = json_decode($raw_input, true);
    debug_log("Parsed input: " . json_encode($input));
    
    if (!$input || !isset($input['lesson_id'])) {
        throw new Exception('Invalid input data - missing lesson_id');
    }
    
    $lesson_id = (int)$input['lesson_id'];
    $encrypted_course_id = $input['course_id'] ?? '';
    $user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 1; // Default user ID
    
    debug_log("Received: Lesson ID: $lesson_id, Encrypted Course ID: $encrypted_course_id, User ID: $user_id");
    
    // Decrypt the course_id using the proper encryption system
    if (!empty($encrypted_course_id)) {
        debug_log("Attempting to decrypt: $encrypted_course_id");
        debug_log("Function exists? " . (function_exists('decryptId') ? 'YES' : 'NO'));
        
        if (function_exists('decryptId')) {
            $course_id = decryptId($encrypted_course_id);
            debug_log("Decryption result: " . var_export($course_id, true));
        } else {
            debug_log("decryptId function not available, encryption includes failed");
            $course_id = false;
        }
        
        if (!$course_id || !is_numeric($course_id)) {
            debug_log("Failed to decrypt course_id (result: $course_id), using fallback");
            $course_id = 1; // Fallback
        }
    } else {
        debug_log("No encrypted course_id provided, using fallback");
        $course_id = 1; // Fallback
    }
    
    debug_log("Using course_id: $course_id for lesson: $lesson_id");
    
    // Get actual lesson data from database or file structure
    debug_log("Looking for lesson: $lesson_id in course: $course_id");
    
    // QUERY REAL DATABASE: Get actual lesson data from videos table
    debug_log("Querying database for real lesson data: lesson_id = $lesson_id");
    
    $lesson_data = null;
    
    try {
        // Connect to database with correct credentials
        $conn = new mysqli('localhost', 'twpurnyc_149586Alain', '149586Alain#', 'twpurnyc_panadite_DB');
        if ($conn->connect_error) {
            debug_log("Database connection failed: " . $conn->connect_error);
            throw new Exception("Database connection failed");
        }
        
        debug_log("Database connected successfully");
        
        // Query the videos table for real lesson data - use video_id column only
        $query = "SELECT * FROM videos WHERE video_id = ? LIMIT 1";
        if ($stmt = $conn->prepare($query)) {
            $stmt->bind_param("i", $lesson_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($video = $result->fetch_assoc()) {
                debug_log("FOUND real video data: " . json_encode($video));
                
                // Handle both YouTube URLs and file uploads
                $video_url = '';
                $video_type = $video['video_type'] ?? 'url';
                
                if ($video_type === 'url' && !empty($video['external_url'])) {
                    // YouTube/External URL videos
                    $video_url = $video['external_url'];
                    if (strpos($video_url, 'youtu.be/') !== false) {
                        // Convert youtu.be to embed URL
                        preg_match('/youtu\.be\/([^?&]+)/', $video_url, $matches);
                        if (isset($matches[1])) {
                            $video_url = 'https://www.youtube.com/embed/' . $matches[1];
                        }
                    } elseif (strpos($video_url, 'youtube.com/watch') !== false) {
                        // Convert youtube.com/watch to embed URL
                        preg_match('/v=([^&]+)/', $video_url, $matches);
                        if (isset($matches[1])) {
                            $video_url = 'https://www.youtube.com/embed/' . $matches[1];
                        }
                    }
                } elseif ($video_type === 'file' && !empty($video['video_url'])) {
                    // File upload videos - convert relative path to absolute URL
                    $video_url = $video['video_url'];
                    
                    // Remove any leading '../' or './' 
                    $video_url = str_replace('../', '', $video_url);
                    $video_url = str_replace('./', '', $video_url);
                    
                    // Ensure it starts with uploads/ (PHP 7.x compatible)
                    if (strpos($video_url, 'uploads/') !== 0) {
                        $video_url = 'uploads/' . ltrim($video_url, '/');
                    }
                    
                    // Convert to absolute URL (assuming the site base URL)
                    $video_url = 'https://panaditeacademy.com/' . $video_url;
                    
                    debug_log("File upload video converted: " . $video['video_url'] . " -> " . $video_url);
                }
                
                debug_log("Final video URL: " . $video_url . " (type: " . $video_type . ")");
                
                $lesson_data = [
                    'lesson_id' => $lesson_id,
                    'video_id' => $video['video_id'],
                    'title' => $video['title'],
                    'duration' => $video['duration'] ?? "10:30",
                    'video_url' => $video_url,
                    'video_path' => $video['video_path'] ?? '',
                    'thumbnail' => $video['thumbnail'] ?? '',
                    'description' => $video['description'] ?? '',
                    'completed' => false,
                    'has_assignment' => (bool)$video['has_assignment'],
                    'has_quiz' => (bool)$video['has_quiz'],
                    'has_materials' => false,
                    'watch_progress' => 0,
                    'created_at' => $video['created_at'] ?? date('Y-m-d H:i:s'),
                    'updated_at' => $video['updated_at'] ?? date('Y-m-d H:i:s'),
                    'video_order' => $video['video_order'] ?? $lesson_id,
                    'is_published' => (bool)$video['is_published'],
                    'last_position' => 0,
                    'video_type' => $video['video_type'] ?? 'youtube'
                ];
                debug_log("Created lesson data from real video: " . json_encode($lesson_data));
            } else {
                debug_log("No video found in database for video_id: $lesson_id");
            }
            $stmt->close();
        }
        $conn->close();
        
    } catch (Exception $e) {
        debug_log("Database error: " . $e->getMessage());
    }
    
    // FALLBACK: If no real data found, create basic lesson data to maintain AJAX functionality
    if (!$lesson_data) {
        debug_log("Using fallback lesson data for lesson_id: $lesson_id");
        $lesson_data = [
            'lesson_id' => $lesson_id,
            'video_id' => $lesson_id,
            'title' => "Lesson $lesson_id",
            'duration' => "10:30",
            'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
            'video_path' => '',
            'thumbnail' => '',
            'description' => "Lesson content for lesson $lesson_id",
            'completed' => false,
            'has_assignment' => true,
            'has_quiz' => false,
            'has_materials' => false,
            'watch_progress' => 0,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'video_order' => $lesson_id,
            'is_published' => 1,
            'last_position' => 0,
            'video_type' => 'youtube'
        ];
    }
    
    debug_log("Final lesson data: " . json_encode($lesson_data));
    
    debug_log("Final lesson data: " . json_encode($lesson_data));
    
    // Process video URL for external videos
    if ($lesson_data['video_type'] === 'youtube' && $lesson_data['video_url']) {
        // Convert YouTube watch URLs to embed URLs
        if (strpos($lesson_data['video_url'], 'youtube.com/watch') !== false) {
            parse_str(parse_url($lesson_data['video_url'], PHP_URL_QUERY), $params);
            if (isset($params['v'])) {
                $lesson_data['video_url'] = 'https://www.youtube.com/embed/' . $params['v'];
            }
        } elseif (strpos($lesson_data['video_url'], 'youtu.be/') !== false) {
            $video_id = substr(parse_url($lesson_data['video_url'], PHP_URL_PATH), 1);
            $lesson_data['video_url'] = 'https://www.youtube.com/embed/' . $video_id;
        }
    }
    
    // Mock course progress data for testing
    $courseProgress = [
        'total_lessons' => 7,
        'completed_lessons' => 3,
        'progress_percentage' => 43
    ];
    
    // Mock next lesson data
    $nextLesson = [
        'lesson_id' => $lesson_id + 1,
        'title' => 'Next Lesson ' . ($lesson_id + 1)
    ];
    
    debug_log("Preparing JSON response");
    
    // Clean any output buffer before sending JSON
    ob_clean();
    
    // Success response with lesson and progress data
    $response = [
        'success' => true,
        'message' => 'Lesson loaded successfully via AJAX',
        'lesson' => $lesson_data,
        'course_progress' => $courseProgress,
        'next_lesson' => $nextLesson,
        'debug_info' => [
            'lesson_id' => $lesson_id,
            'user_id' => $user_id,
            'course_id' => $course_id,
            'video_type' => $lesson_data['video_type'],
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($response);
    debug_log("JSON response sent: " . json_encode($response));
    
} catch (Exception $e) {
    debug_log("Error occurred: " . $e->getMessage());
    
    // Clean output buffer
    ob_clean();
    
    // Return error response
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'debug_info' => [
            'lesson_id' => isset($lesson_id) ? $lesson_id : 'unknown',
            'course_id' => isset($encrypted_course_id) ? $encrypted_course_id : 'unknown',
            'user_id' => isset($user_id) ? $user_id : 'unknown'
        ]
    ]);
}
?>
