<?php
/**
 * Live AI Grading Endpoint
 * Provides real-time feedback during assignment grading
 */

// Start output buffering to prevent any HTML output
ob_start();

require_once '../../includes/session_start.php';
require_once '../../auth/functions.php';
require_once '../../config/db_connect.php';
require_once '../../includes/ai_grading_service.php';

header('Content-Type: application/json');
header('Cache-Control: no-cache');

if (!isLoggedIn()) {
    ob_clean();
    echo json_encode(['error' => 'Not authenticated']);
    exit;
}

$student_id = $_SESSION['user_id'];
$assignment_id = intval($_POST['assignment_id'] ?? 0);
$step = $_POST['step'] ?? 'start';

if (!$assignment_id) {
    ob_clean();
    echo json_encode(['error' => 'Invalid assignment ID']);
    exit;
}

// Get assignment details
$stmt = $conn->prepare("
    SELECT a.*, c.title as course_title 
    FROM assignments a 
    JOIN courses c ON a.course_id = c.course_id 
    WHERE a.assignment_id = ?
");

if (!$stmt) {
    ob_clean();
    echo json_encode(['error' => 'Database error: Failed to prepare assignment query']);
    exit;
}

$stmt->bind_param('i', $assignment_id);
$stmt->execute();
$assignment = $stmt->get_result()->fetch_assoc();

if (!$assignment) {
    ob_clean();
    echo json_encode(['error' => 'Assignment not found']);
    exit;
}

// Get the latest submission for this student
$stmt = $conn->prepare("
    SELECT * FROM assignment_submissions 
    WHERE assignment_id = ? AND student_id = ? 
    ORDER BY submitted_at DESC 
    LIMIT 1
");

if (!$stmt) {
    ob_clean();
    echo json_encode(['error' => 'Database error: Failed to prepare submission query']);
    exit;
}

$stmt->bind_param('ii', $assignment_id, $student_id);
$stmt->execute();
$submission = $stmt->get_result()->fetch_assoc();

if (!$submission) {
    ob_clean();
    echo json_encode(['error' => 'No submission found']);
    exit;
}

$response = ['step' => $step, 'progress' => 0, 'message' => '', 'details' => []];

try {
    switch ($step) {
        case 'start':
            $response['progress'] = 10;
            $response['message'] = 'Starting AI grading process...';
            $response['details'] = [
                'assignment_title' => $assignment['title'],
                'course' => $assignment['course_title'],
                'submission_id' => $submission['submission_id'],
                'has_file' => !empty($submission['file_url']),
                'has_text' => !empty($submission['submission_text'])
            ];
            break;

        case 'extract_text':
            $response['progress'] = 25;
            $response['message'] = 'Extracting text from submission...';
            
            $extractedText = '';
            $details = [];
            
            // Extract from text submission
            if (!empty($submission['submission_text'])) {
                $extractedText .= $submission['submission_text'];
                $details['text_length'] = strlen($submission['submission_text']);
                $details['text_preview'] = substr($submission['submission_text'], 0, 100) . '...';
            }
            
            // Extract from uploaded file
            if (!empty($submission['file_url'])) {
                $filePath = '../../uploads/assignments/' . basename($submission['file_url']);
                
                if (file_exists($filePath)) {
                    $details['file_path'] = $filePath;
                    $details['file_size'] = filesize($filePath);
                    $details['file_type'] = pathinfo($filePath, PATHINFO_EXTENSION);
                    
                    try {
                        require_once '../../includes/robust_document_processor.php';
                        $processor = new RobustDocumentProcessor(true);
                        $fileText = $processor->extractText($filePath);
                        
                        if (!empty($fileText)) {
                            $extractedText .= "\n\n" . $fileText;
                            $details['extracted_length'] = strlen($fileText);
                            $details['extracted_preview'] = substr($fileText, 0, 200) . '...';
                            $details['extraction_success'] = true;
                            $details['extraction_method'] = 'robust_processor';
                        }
                    } catch (Exception $e) {
                        $details['extraction_error'] = $e->getMessage();
                        $details['extraction_success'] = false;
                        error_log('Live grading text extraction failed: ' . $e->getMessage());
                    }
                } else {
                    $details['file_missing'] = true;
                }
            }
            
            $response['details'] = $details;
            $response['extracted_text_length'] = strlen($extractedText);
            
            // Update database with extracted text
            $updateStmt = $conn->prepare("UPDATE assignment_submissions SET extracted_text = ? WHERE submission_id = ?");
            $updateStmt->bind_param('si', $extractedText, $submission['submission_id']);
            $updateStmt->execute();
            
            break;

        case 'check_memorandum':
            $response['progress'] = 40;
            $response['message'] = 'Checking memorandum and grading criteria...';
            
            $details = [
                'has_memorandum' => !empty($assignment['memorandum_content']),
                'memorandum_length' => strlen($assignment['memorandum_content'] ?? ''),
                'ai_enabled' => (bool)$assignment['ai_grading_enabled'],
                'pass_mark' => $assignment['pass_mark'] ?? 50
            ];
            
            if (!empty($assignment['memorandum_content'])) {
                $details['memorandum_preview'] = substr($assignment['memorandum_content'], 0, 200) . '...';
            } else {
                $details['error'] = 'No memorandum content available for grading';
            }
            
            $response['details'] = $details;
            break;

        case 'ai_analysis':
            $response['progress'] = 60;
            $response['message'] = 'Performing AI analysis and scoring...';
            
            if (empty($assignment['memorandum_content'])) {
                throw new Exception('No memorandum available for comparison');
            }
            
            $fullText = $submission['extracted_text'] ?: $submission['submission_text'];
            if (empty($fullText)) {
                throw new Exception('No text content to analyze');
            }
            
            // Suppress any PHP warnings/errors that could output HTML
            $oldErrorReporting = error_reporting(0);
            
            try {
                // Check if PHPAIGradingService class exists
                if (!class_exists('PHPAIGradingService')) {
                    throw new Exception('AI Grading Service not available');
                }
                
                // Try to instantiate with maximum error protection
                try {
                    $gradingService = new PHPAIGradingService($conn, true);
                    if (!$gradingService) {
                        throw new Exception('AI service instantiation returned null');
                    }
                } catch (Throwable $serviceError) {
                    // Catch both Exception and Error (PHP 7+)
                    throw new Exception('Failed to initialize AI grading service: ' . $serviceError->getMessage());
                }
            } catch (Exception $initError) {
                error_reporting($oldErrorReporting);
                // Return a response with error details instead of crashing
                $response['details'] = [
                    'error' => 'AI service initialization failed',
                    'message' => $initError->getMessage(),
                    'fallback_analysis' => true
                ];
                error_reporting($oldErrorReporting);
                break; // Exit this case gracefully
            }
            
            // Get detailed analysis
            $analysis = [
                'text_length' => strlen($fullText),
                'word_count' => str_word_count($fullText),
                'memorandum_length' => strlen($assignment['memorandum_content']),
                'memorandum_words' => str_word_count($assignment['memorandum_content'])
            ];
            
            // Perform basic similarity check with maximum protection
            try {
                if (method_exists($gradingService, 'calculateSimilarity')) {
                    $similarity = @$gradingService->calculateSimilarity($fullText, $assignment['memorandum_content']);
                    $analysis['similarity_score'] = is_numeric($similarity) ? round($similarity, 2) : 0;
                } else {
                    $analysis['similarity_score'] = 0;
                    $analysis['similarity_error'] = 'Method not available';
                }
            } catch (Throwable $e) {
                $analysis['similarity_score'] = 0;
                $analysis['similarity_error'] = 'Calculation failed: ' . $e->getMessage();
            }
            
            // Check for key terms with maximum protection
            try {
                if (method_exists($gradingService, 'extractKeywords') && method_exists($gradingService, 'checkKeywordMatches')) {
                    $keywords = @$gradingService->extractKeywords($assignment['memorandum_content']);
                    if (is_array($keywords)) {
                        $keywordMatches = @$gradingService->checkKeywordMatches($fullText, $keywords);
                        $analysis['keyword_matches'] = is_array($keywordMatches) ? count($keywordMatches) : 0;
                        $analysis['total_keywords'] = count($keywords);
                        $analysis['keyword_score'] = count($keywords) > 0 ? round(($analysis['keyword_matches'] / count($keywords)) * 100, 2) : 0;
                    } else {
                        $analysis['keyword_matches'] = 0;
                        $analysis['total_keywords'] = 0;
                        $analysis['keyword_score'] = 0;
                        $analysis['keyword_error'] = 'Keyword extraction returned invalid data';
                    }
                } else {
                    $analysis['keyword_matches'] = 0;
                    $analysis['total_keywords'] = 0;
                    $analysis['keyword_score'] = 0;
                    $analysis['keyword_error'] = 'Methods not available';
                }
            } catch (Throwable $e) {
                $analysis['keyword_matches'] = 0;
                $analysis['total_keywords'] = 0;
                $analysis['keyword_score'] = 0;
                $analysis['keyword_error'] = 'Analysis failed: ' . $e->getMessage();
            }
            
            // Restore error reporting
            error_reporting($oldErrorReporting);
            
            $response['details'] = $analysis;
            break;

        case 'generate_grade':
            $response['progress'] = 85;
            $response['message'] = 'Generating final grade and feedback...';
            
            $fullText = $submission['extracted_text'] ?: $submission['submission_text'];
            
            if (empty($assignment['memorandum_content']) || empty($fullText)) {
                throw new Exception('Missing content for grading');
            }
            
            // Suppress any PHP warnings/errors that could output HTML
            $oldErrorReporting = error_reporting(0);
            
            try {
                // Check if class and method exist
                if (!class_exists('PHPAIGradingService')) {
                    throw new Exception('AI Grading Service not available');
                }
                
                $gradingService = new PHPAIGradingService($conn, true);
                
                if (!method_exists($gradingService, 'gradeAssignment')) {
                    throw new Exception('gradeAssignment method not available');
                }
                
                $gradingResult = @$gradingService->gradeAssignment($fullText, $assignment['memorandum_content'], $assignment_id);
                
                // Validate grading result
                if (!is_array($gradingResult)) {
                    throw new Exception('Invalid grading result format');
                }
                
                // Ensure all required fields exist with defaults
                $gradingResult = array_merge([
                    'ai_score' => 0,
                    'ai_feedback' => 'Grading completed with basic analysis',
                    'similarity_score' => 0,
                    'plagiarism_score' => 0,
                    'keyword_match_score' => 0,
                    'structure_score' => 0,
                    'quality_score' => 0,
                    'review_needed' => 1,
                    'ai_confidence' => 0.5
                ], $gradingResult);
                
            } catch (Throwable $gradingError) {
                error_reporting($oldErrorReporting);
                // Create fallback grading result instead of crashing
                $gradingResult = [
                    'ai_score' => 0,
                    'ai_feedback' => 'AI grading failed: ' . $gradingError->getMessage() . '. Manual review required.',
                    'similarity_score' => 0,
                    'plagiarism_score' => 0,
                    'keyword_match_score' => 0,
                    'structure_score' => 0,
                    'quality_score' => 0,
                    'review_needed' => 1,
                    'ai_confidence' => 0
                ];
            }
            
            // Restore error reporting
            error_reporting($oldErrorReporting);
            
            // Update submission with grading results
            $updateStmt = $conn->prepare("
                UPDATE assignment_submissions SET 
                    ai_score = ?, ai_feedback = ?, similarity_score = ?, 
                    plagiarism_score = ?, keyword_match_score = ?, 
                    structure_score = ?, quality_score = ?, 
                    processing_status = 'completed', review_needed = ?, ai_confidence = ?
                WHERE submission_id = ?
            ");
            
            $updateStmt->bind_param(
                'dsdddddidd',
                $gradingResult['ai_score'],
                $gradingResult['ai_feedback'],
                $gradingResult['similarity_score'],
                $gradingResult['plagiarism_score'],
                $gradingResult['keyword_match_score'],
                $gradingResult['structure_score'],
                $gradingResult['quality_score'],
                $gradingResult['review_needed'],
                $gradingResult['ai_confidence'],
                $submission['submission_id']
            );
            $updateStmt->execute();
            
            $response['details'] = [
                'ai_score' => round($gradingResult['ai_score'], 1),
                'similarity_score' => round($gradingResult['similarity_score'], 1),
                'keyword_match_score' => round($gradingResult['keyword_match_score'], 1),
                'structure_score' => round($gradingResult['structure_score'], 1),
                'quality_score' => round($gradingResult['quality_score'], 1),
                'ai_confidence' => round($gradingResult['ai_confidence'], 1),
                'ai_feedback' => $gradingResult['ai_feedback'], // Full feedback for modal display
                'review_needed' => $gradingResult['review_needed'],
                'feedback_length' => strlen($gradingResult['ai_feedback']),
                'feedback_preview' => substr($gradingResult['ai_feedback'], 0, 200) . '...'
            ];
            break;

        case 'complete':
            $response['progress'] = 100;
            $response['message'] = 'AI grading completed successfully!';
            
            // Update course progress
            try {
                require_once '../../includes/course_progress_service.php';
                $progressService = new CourseProgressService($conn);
                $updatedProgress = $progressService->calculateCourseProgress($student_id, $assignment['course_id']);
                
                $response['details'] = [
                    'course_progress' => round($updatedProgress['overall_percentage'], 1),
                    'completion_time' => date('Y-m-d H:i:s')
                ];
            } catch (Exception $progressError) {
                // Don't fail the whole grading if progress update fails
                error_log('Course progress update failed: ' . $progressError->getMessage());
                $response['details'] = [
                    'course_progress' => 0,
                    'completion_time' => date('Y-m-d H:i:s'),
                    'progress_warning' => 'Progress tracking unavailable'
                ];
            }
            break;

        default:
            throw new Exception('Invalid processing step');
    }

} catch (Exception $e) {
    $response['error'] = $e->getMessage();
    $response['step'] = 'error';
    
    // Update submission as failed
    if (isset($submission['submission_id'])) {
        $updateStmt = $conn->prepare("
            UPDATE assignment_submissions SET 
                processing_status = 'failed', 
                ai_feedback = 'AI grading failed: " . $conn->real_escape_string($e->getMessage()) . "',
                review_needed = 1
            WHERE submission_id = ?
        ");
        $updateStmt->bind_param('i', $submission['submission_id']);
        $updateStmt->execute();
    }
}

// Clean any output buffer content and send only JSON
ob_clean();
echo json_encode($response);
?>
