/**
 * PANADITE ACADEMY - VIDEO PROGRESS TRACKER
 * Isolated, Dynamic Progress Tracking System
 * Handles both file uploads and URL videos (YouTube, Vimeo, etc.)
 * 
 * Author: Cascade AI
 * Version: 2.0
 * Created: 2025-11-29
 */

class VideoProgressTracker {
    constructor(options = {}) {
        this.videoId = options.videoId;
        this.lessonId = options.lessonId;
        this.courseId = options.courseId;
        this.studentId = options.studentId;
        this.videoElement = null;
        this.progressInterval = null;
        this.lastSavedProgress = 0;
        this.isCompleted = false;
        this.updateFrequency = options.updateFrequency || 2000; // 2 seconds
        this.completionThreshold = options.completionThreshold || 90; // 90% = completed
        
        // Callback functions
        this.onProgressUpdate = options.onProgressUpdate || function() {};
        this.onCompletion = options.onCompletion || function() {};
        this.onError = options.onError || function() {};
        
        // Debug mode
        this.debug = options.debug || false;
        
        this.log('📹 VideoProgressTracker initialized', {
            videoId: this.videoId,
            lessonId: this.lessonId,
            courseId: this.courseId,
            studentId: this.studentId
        });
        
        this.initializeTracking();
    }
    
    log(message, data = null) {
        if (this.config.debug) {
            console.log(`[VideoProgressTracker] ${message}`, data || '');
        }
        
        // 🔍 CRITICAL: Also show important messages in the visible debug panel
        if (message.includes('❌') || message.includes('Error') || message.includes('Failed') || message.includes('YouTube Progress')) {
            this.updateVisibleDebug(message, data);
        }
    }
    
    /**
     * Update visible debug panel on page
     */
    updateVisibleDebug(message, data = null) {
        // Try multiple debug panel selectors to find the visible console
        const debugSelectors = [
            '.youtube-debug-log',
            '.debug-console', 
            '[class*="debug-log"]',
            '.console'
        ];
        
        let debugPanel = null;
        for (const selector of debugSelectors) {
            debugPanel = document.querySelector(selector);
            if (debugPanel) break;
        }
        
        if (debugPanel) {
            const timestamp = new Date().toLocaleTimeString();
            const debugEntry = `[${timestamp}] ${message}` + (data ? ` ${JSON.stringify(data)}` : '');
            
            // Add to debug panel
            if (debugPanel.tagName === 'TEXTAREA' || debugPanel.tagName === 'PRE') {
                debugPanel.textContent = debugEntry + '\n' + debugPanel.textContent;
            } else {
                const newEntry = document.createElement('div');
                newEntry.textContent = debugEntry;
                newEntry.style.fontSize = '11px';
                newEntry.style.marginBottom = '2px';
                if (message.includes('❌')) newEntry.style.color = 'red';
                if (message.includes('✅')) newEntry.style.color = 'green';
                debugPanel.insertBefore(newEntry, debugPanel.firstChild);
            }
            
            // Keep only last 30 entries
            const entries = debugPanel.children;
            if (entries.length > 30) {
                for (let i = entries.length - 1; i >= 30; i--) {
                    entries[i].remove();
                }
            }
        }
        
        // Also console.error for critical issues
        if (message.includes('❌')) {
            console.error('[VideoProgressTracker]', message, data);
        }
    }
    
    /**
     * Initialize progress tracking based on video type
     */
    initializeTracking() {
        // Try to find video element (HTML5 video)
        this.videoElement = document.getElementById('lessonVideo');
        
        if (this.videoElement) {
            if (this.videoElement.tagName.toLowerCase() === 'video') {
                this.log('🎬 HTML5 Video detected - initializing file video tracking');
                this.initializeFileVideoTracking();
            } else if (this.videoElement.tagName.toLowerCase() === 'iframe') {
                this.log('🌐 External Video detected - initializing URL video tracking');
                this.initializeURLVideoTracking();
            }
        } else {
            this.log('❌ No video element found', 'error');
            this.onError('No video element found');
        }
    }
    
    /**
     * Initialize tracking for uploaded video files (MP4, etc.)
     */
    initializeFileVideoTracking() {
        const video = this.videoElement;
        
        // Wait for video metadata to load
        video.addEventListener('loadedmetadata', () => {
            this.log('📊 Video metadata loaded', {
                duration: video.duration,
                videoWidth: video.videoWidth,
                videoHeight: video.videoHeight
            });
        });
        
        // Track when video starts playing
        video.addEventListener('play', () => {
            this.log('▶️ Video started playing');
            this.startProgressTracking();
        });
        
        // Track when video is paused
        video.addEventListener('pause', () => {
            this.log('⏸️ Video paused');
            this.saveCurrentProgress();
        });
        
        // Track when video ends
        video.addEventListener('ended', () => {
            this.log('🏁 Video ended');
            this.handleVideoCompletion();
        });
        
        // Track time updates
        video.addEventListener('timeupdate', () => {
            this.updateProgress();
        });
        
        // Handle video errors
        video.addEventListener('error', (e) => {
            this.log('❌ Video error occurred', e);
            this.onError('Video playback error');
        });
    }
    
    /**
     * Initialize tracking for external URLs (YouTube, Vimeo, etc.)
     */
    initializeURLVideoTracking() {
        const iframe = this.videoElement;
        const src = iframe.src;
        
        if (src.includes('youtube.com/embed')) {
            this.log('📺 YouTube video detected');
            this.initializeYouTubeTracking(iframe);
        } else if (src.includes('vimeo.com')) {
            this.log('🎭 Vimeo video detected');
            this.initializeVimeoTracking(iframe);
        } else {
            this.log('🌐 Generic external video - using fallback tracking');
            this.initializeFallbackTracking(iframe);
        }
    }
    
    /**
     * YouTube-specific tracking using PostMessage API
     */
    initializeYouTubeTracking(iframe) {
        this.youtubeDuration = 0;
        this.youtubeCurrentTime = 0;
        
        // Listen for YouTube PostMessage events
        window.addEventListener('message', (event) => {
            if (event.origin !== 'https://www.youtube.com') return;
            
            try {
                const data = typeof event.data === 'string' ? JSON.parse(event.data) : event.data;
                this.log('📡 YouTube Message', data);
                
                // Handle state changes
                if (data.event === 'onStateChange') {
                    this.handleYouTubeStateChange(data.info);
                }
                
                // Handle time/duration updates
                if (data.event === 'infoDelivery' && data.info) {
                    if (data.info.duration) {
                        this.youtubeDuration = data.info.duration;
                        this.log('⏱️ YouTube duration received', this.youtubeDuration);
                    }
                    if (data.info.currentTime !== undefined && this.youtubeDuration > 0) {
                        this.youtubeCurrentTime = data.info.currentTime;
                        this.updateYouTubeProgress();
                    }
                }
            } catch (error) {
                // Ignore invalid JSON messages
            }
        });
        
        // Start requesting YouTube data
        this.requestYouTubeData(iframe);
    }
    
    /**
     * Update YouTube progress based on current time and duration
     */
    updateYouTubeProgress() {
        if (this.youtubeDuration > 0 && this.youtubeCurrentTime !== null) {
            const progressPercent = Math.round((this.youtubeCurrentTime / this.youtubeDuration) * 100);
            this.log(`📊 YouTube Progress: ${progressPercent}% (${this.youtubeCurrentTime}/${this.youtubeDuration}s)`);
            
            // Update progress UI callback
            if (this.config.onProgressUpdate) {
                this.config.onProgressUpdate({
                    progress: progressPercent,
                    currentTime: this.youtubeCurrentTime,
                    duration: this.youtubeDuration,
                    completed: progressPercent >= 90
                });
            }
            
            // Save progress every 5% or on significant milestones
            if (this.shouldSaveProgress(progressPercent)) {
                this.saveCurrentProgress(progressPercent);
            }
        } else {
            // 🔍 DEBUG: Show why progress isn't updating in visible console
            const debugInfo = {
                duration: this.youtubeDuration,
                currentTime: this.youtubeCurrentTime,
                durationValid: this.youtubeDuration > 0,
                timeValid: this.youtubeCurrentTime !== null
            };
            this.log(`❌ YouTube Progress Stuck - Duration: ${this.youtubeDuration}s, Time: ${this.youtubeCurrentTime}s`, debugInfo);
        }
    }
    
    handleYouTubeStateChange(state) {
        const states = {
            '-1': 'Unstarted',
            '0': 'Ended',
            '1': 'Playing', 
            '2': 'Paused',
            '3': 'Buffering',
            '5': 'Cued'
        };
        
        this.log(`🎮 YouTube State: ${states[state] || 'Unknown'} (${state})`);
        
        if (state === 1) { // Playing
            this.startProgressTracking();
        } else if (state === 0) { // Ended
            this.handleVideoCompletion();
        } else if (state === 2) { // Paused
            this.saveCurrentProgress();
        }
    }
    
    requestYouTubeData(iframe) {
        // Request current time and duration every 2 seconds
        const requestData = () => {
            try {
                iframe.contentWindow.postMessage('{"event":"listening"}', 'https://www.youtube.com');
                iframe.contentWindow.postMessage('{"event":"command","func":"getCurrentTime","args":""}', 'https://www.youtube.com');
                iframe.contentWindow.postMessage('{"event":"command","func":"getDuration","args":""}', 'https://www.youtube.com');
            } catch (error) {
                this.log('❌ YouTube PostMessage error', error.message);
            }
        };
        
        // Start requesting data
        setInterval(requestData, this.updateFrequency);
        
        // Initial request after iframe loads
        setTimeout(requestData, 3000);
    }
    
    /**
     * Vimeo-specific tracking
     */
    initializeVimeoTracking(iframe) {
        // Vimeo Player API implementation would go here
        this.log('⚠️ Vimeo tracking not yet implemented - using fallback');
        this.initializeFallbackTracking(iframe);
    }
    
    /**
     * Fallback tracking for unknown external videos
     */
    initializeFallbackTracking(iframe) {
        this.log('🔄 Using fallback progress tracking');
        
        // For external videos without API, we can only track engagement time
        let startTime = Date.now();
        let isVisible = true;
        
        // Track if iframe is in viewport
        const observer = new IntersectionObserver((entries) => {
            isVisible = entries[0].isIntersecting;
            if (isVisible) {
                startTime = Date.now();
            }
        });
        
        observer.observe(iframe);
        
        // Simulate progress based on engagement time
        setInterval(() => {
            if (isVisible) {
                const engagementTime = (Date.now() - startTime) / 1000; // seconds
                const estimatedProgress = Math.min(Math.round((engagementTime / 300) * 100), 95); // Assume 5min video
                
                if (estimatedProgress > this.lastSavedProgress) {
                    this.onProgressUpdate({
                        progress: estimatedProgress,
                        currentTime: engagementTime,
                        duration: 300, // estimated
                        completed: false // Can't determine completion for fallback
                    });
                    
                    this.saveProgress(estimatedProgress, engagementTime);
                }
            }
        }, this.updateFrequency);
    }
    
    /**
     * Start progress tracking interval
     */
    startProgressTracking() {
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
        }
        
        this.log('🚀 Starting progress tracking interval');
        this.lastSaveTime = Date.now();
        
        this.progressInterval = setInterval(() => {
            this.updateProgress();
        }, this.updateFrequency);
    }
    
    /**
     * Update progress for HTML5 videos
     */
    updateProgress() {
        if (!this.videoElement || this.videoElement.tagName.toLowerCase() !== 'video') return;
        
        const video = this.videoElement;
        if (!video.duration || isNaN(video.duration)) return;
        
        const currentTime = video.currentTime;
        const duration = video.duration;
        const progressPercentage = Math.round((currentTime / duration) * 100);
        
        this.log(`📊 Video Progress: ${progressPercentage}% (${currentTime}/${duration}s)`);
        
        // Update UI
        this.onProgressUpdate({
            progress: progressPercentage,
            currentTime: currentTime,
            duration: duration,
            completed: progressPercentage >= this.completionThreshold
        });
        
        // Save progress periodically (every 5% change or every 10 seconds)
        if (Math.abs(progressPercentage - this.lastSavedProgress) >= 5 || 
            Date.now() - this.lastSaveTime > 10000) {
            this.saveProgress(progressPercentage, currentTime);
        }
        
        // Check for completion
        if (progressPercentage >= this.completionThreshold && !this.isCompleted) {
            this.handleVideoCompletion();
        }
    }
    
    /**
     * Save current progress without checking intervals
     */
    saveCurrentProgress() {
        if (this.videoElement && this.videoElement.tagName.toLowerCase() === 'video') {
            const video = this.videoElement;
            if (video.duration && !isNaN(video.duration)) {
                const progressPercentage = Math.round((video.currentTime / video.duration) * 100);
                this.saveProgress(progressPercentage, video.currentTime);
            }
        } else {
            // For YouTube videos, use current tracked time
            if (this.youtubeDuration > 0) {
                const progressPercentage = Math.round((this.youtubeCurrentTime / this.youtubeDuration) * 100);
                this.saveProgress(progressPercentage, this.youtubeCurrentTime);
            }
        }
    }
    
    /**
     * Save progress to database - FIXED FOR YOUTUBE VIDEOS
     */
    saveProgress(progressPercentage, currentTime) {
        // Validate required data
        if (!this.courseId || !this.studentId) {
            this.log('❌ Cannot save progress: Missing courseId or studentId');
            return;
        }
        
        // ⭐ CRITICAL FIX: For URL videos without video_id, use lesson_id as video_id
        const videoIdToUse = this.videoId || this.lessonId;
        if (!videoIdToUse) {
            this.log('❌ Cannot save progress: No video_id or lesson_id available');
            return;
        }

        this.log('💾 Saving progress...', {
            videoId: videoIdToUse,
            lessonId: this.lessonId,
            courseId: this.courseId,
            progress: progressPercentage,
            currentTime: currentTime
        });
        
        // Make AJAX request to save progress
        fetch('ajax/video-progress-handler.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'save_progress',
                video_id: videoIdToUse,
                lesson_id: this.lessonId,
                course_id: this.courseId,
                student_id: this.studentId,
                progress_percentage: progressPercentage,
                current_time: currentTime,
                completed: progressPercentage >= this.completionThreshold ? 1 : 0
            })
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                this.log('✅ Progress saved successfully', data);
                this.lastSavedProgress = progressPercentage;
                this.lastSaveTime = Date.now();
                
                // 🎯 CRITICAL FIX: Update main course progress directly from save response  
                if (data.course_progress !== undefined) {
                    const courseProgressPercentage = Math.round(data.course_progress);
                    
                    console.log(`🎯 ATTEMPTING COURSE PROGRESS UPDATE: ${courseProgressPercentage}%`);
                    
                    // 🎯 Target the EXACT elements found in semantic search
                    const mainProgressText = document.getElementById('main-course-progress-percentage');
                    const mainProgressBar = document.getElementById('main-course-progress-bar');
                    
                    console.log('🔍 DOM ELEMENTS CHECK:', {
                        progressText: !!mainProgressText,
                        progressBar: !!mainProgressBar,
                        textContent: mainProgressText ? mainProgressText.textContent : 'not found',
                        barWidth: mainProgressBar ? mainProgressBar.style.width : 'not found'
                    });
                    
                    // Update the percentage text
                    if (mainProgressText) {
                        const oldText = mainProgressText.textContent;
                        mainProgressText.textContent = `${courseProgressPercentage}%`;
                        console.log(`✅ UPDATED PROGRESS TEXT: "${oldText}" → "${courseProgressPercentage}%"`);
                        this.log(`🎯 Updated percentage text: ${courseProgressPercentage}%`);
                    } else {
                        console.error(`❌ Could not find #main-course-progress-percentage element`);
                        this.log(`❌ Could not find main-course-progress-percentage element`);
                    }
                    
                    // Update the progress bar  
                    if (mainProgressBar) {
                        const oldWidth = mainProgressBar.style.width;
                        mainProgressBar.style.width = `${courseProgressPercentage}%`;
                        console.log(`✅ UPDATED PROGRESS BAR: "${oldWidth}" → "${courseProgressPercentage}%"`);
                        this.log(`🎯 Updated progress bar width: ${courseProgressPercentage}%`);
                    } else {
                        console.error(`❌ Could not find #main-course-progress-bar element`);
                        this.log(`❌ Could not find main-course-progress-bar element`);
                    }
                    
                    this.log(`🎯 Course progress update completed: ${courseProgressPercentage}%`);
                }
                
                // Update UI callback
                this.onProgressUpdate({
                    progress: progressPercentage,
                    currentTime: currentTime,
                    completed: data.completed,
                    first_time_completion: data.first_time_completion
                });
            } else {
                throw new Error(data.error || 'Failed to save progress');
            }
        })
        .catch(error => {
            this.log('❌ Error saving progress', error.message);
            this.onError(`Network error while saving progress: ${error.message}`);
        });
    }
    
    /**
     * Handle video completion
     */
    handleVideoCompletion() {
        this.log('🎉 Video completion detected');
        
        // Save final progress as 100% if not already completed
        if (!this.isCompleted) {
            const finalTime = this.videoElement && this.videoElement.tagName.toLowerCase() === 'video' 
                ? this.videoElement.duration || this.youtubeCurrentTime || 0
                : this.youtubeCurrentTime || 0;
                
            this.saveProgress(100, finalTime);
            this.isCompleted = true;
            
            // Trigger completion callback
            this.onCompletion({
                videoId: this.videoId,
                lessonId: this.lessonId,
                courseId: this.courseId,
                finalProgress: 100
            });
        }
        
        // Stop tracking
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
            this.progressInterval = null;
        }
    }
    
    /**
     * Get current progress from database
     */
    async getCurrentProgress() {
        try {
            const response = await fetch('ajax/video-progress-handler.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    action: 'get_progress',
                    video_id: this.videoId,
                    student_id: this.studentId
                })
            });
            
            const data = await response.json();
            if (data.success) {
                return data.progress;
            } else {
                this.log('❌ Failed to get current progress', data.message);
                return null;
            }
        } catch (error) {
            this.log('❌ Network error getting progress', error.message);
            return null;
        }
    }
    
    /**
     * Stop all tracking
     */
    destroy() {
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
            this.progressInterval = null;
        }
        
        this.log('🛑 VideoProgressTracker destroyed');
    }
}

// Export for use in other files
window.VideoProgressTracker = VideoProgressTracker;
