<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Show only quizzes from courses the student is enrolled in
$quizzes_query = "
    SELECT 
        q.quiz_id,
        q.title,
        q.description,
        q.time_limit,
        q.passing_score,
        q.is_published,
        q.created_at,
        q.contributes_to_final_grade,
        c.course_id,
        c.title as course_title,
        u.first_name,
        u.last_name,
        'available' as status
    FROM quizzes q
    JOIN courses c ON q.course_id = c.course_id
    JOIN users u ON c.teacher_id = u.user_id
    JOIN enrollments e ON c.course_id = e.course_id AND e.student_id = ?
    WHERE q.is_published = 1
    ORDER BY q.created_at DESC
";

// Use MySQLi connection to execute the query with parameters
$stmt = $conn->prepare($quizzes_query);

// Add error handling for query preparation
if (!$stmt) {
    die("Query preparation failed: " . $conn->error);
}

$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$quizzes = [];

if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        // Add question count separately
        $quiz_id = $row['quiz_id'];
        $question_count_query = "SELECT COUNT(*) as question_count FROM quiz_questions WHERE quiz_id = $quiz_id";
        $question_result = $conn->query($question_count_query);
        $question_count = $question_result ? $question_result->fetch_assoc()['question_count'] : 0;
        $row['question_count'] = $question_count;
        
        $quizzes[] = $row;
    }
}

$page_title = "All Quizzes";

// Set current page for navigation
$currentPage = 'assignments';

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- All Quizzes Content -->
<div class="all-quizzes-content">
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1e40af;
            --panadite-accent: #f59e0b;
            --bg-light: #f8fafc;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        /* Header */
        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .page-subtitle {
            font-size: 1.125rem;
            opacity: 0.9;
            margin: 0;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        .back-btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--panadite-primary);
            text-decoration: none;
            font-weight: 500;
            margin-bottom: 2rem;
            transition: color 0.3s ease;
        }

        .back-btn:hover {
            color: var(--panadite-secondary);
        }

        .quizzes-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 2rem;
        }

        .quiz-card {
            background: white;
            border-radius: 15px;
            padding: 1.5rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            border-left: 4px solid var(--panadite-accent);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }

        .quiz-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }

        .quiz-header h3 {
            color: var(--panadite-primary);
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }

        .grade-badge {
            display: flex;
            align-items: center;
            gap: 0.3rem;
            padding: 0.4rem 0.8rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            white-space: nowrap;
            cursor: help;
            transition: all 0.3s ease;
        }

        .grade-counts {
            background: linear-gradient(135deg, #fef3e2 0%, #fde68a 100%);
            color: #d97706;
            border: 2px solid #f59e0b;
        }

        .grade-counts:hover {
            background: linear-gradient(135deg, #fde68a 0%, #f59e0b 100%);
            color: white;
            transform: scale(1.05);
        }

        .grade-practice {
            background: linear-gradient(135deg, #e0f2fe 0%, #b3e5fc 100%);
            color: #0277bd;
            border: 2px solid #03a9f4;
        }

        .grade-practice:hover {
            background: linear-gradient(135deg, #b3e5fc 0%, #03a9f4 100%);
            color: white;
            transform: scale(1.05);
        }

        .quiz-course {
            color: var(--text-secondary);
            font-size: 0.875rem;
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .course-badge {
            background: var(--panadite-primary);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .quiz-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            margin: 1rem 0;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
            color: var(--text-secondary);
        }

        .meta-icon {
            color: var(--panadite-primary);
        }

        .quiz-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 1.5rem;
        }

        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 25px;
            font-size: 0.875rem;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .status-available {
            background: linear-gradient(135deg, var(--panadite-primary), var(--panadite-secondary));
            color: white;
        }

        .status-completed {
            background: linear-gradient(135deg, var(--success-color), #059669);
            color: white;
        }

        .status-in_progress {
            background: linear-gradient(135deg, var(--warning-color), #d97706);
            color: white;
        }

        .quiz-btn {
            padding: 0.75rem 1.5rem;
            border-radius: 10px;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
        }

        .btn-primary {
            background: var(--panadite-primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--panadite-secondary);
            color: white;
            transform: translateY(-2px);
        }

        .btn-outline {
            border: 2px solid var(--panadite-primary);
            color: var(--panadite-primary);
        }

        .btn-outline:hover {
            background: var(--panadite-primary);
            color: white;
        }

        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            background: white;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }

        .empty-icon {
            font-size: 4rem;
            color: var(--text-secondary);
            margin-bottom: 1rem;
        }

        .empty-title {
            color: var(--text-primary);
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }

        .empty-description {
            color: var(--text-secondary);
        }
        @media (max-width: 768px) {
            body {
                margin-left: 0;
            }
            
            .page-title {
                font-size: 2rem;
            }
            
            .quizzes-grid {
                grid-template-columns: 1fr;
            }
        }
        @media (max-width: 768px) {
            body {
                margin-left: 0;
            }
            
            .page-title {
                font-size: 2rem;
            }
            
            .quizzes-grid {
                grid-template-columns: 1fr;
            }
            
            .container {
                padding: 0 0.5rem 2rem;
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 0 0.5rem 2rem;
            }
            
            .page-header {
                padding: 1.5rem 0;
            }
            
            .quiz-card {
                padding: 1rem;
            }
        }
    </style>

    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">
                <i class="bx bx-help-circle"></i>
                All Quizzes
            </h1>
            <p class="page-subtitle">Take quizzes from all your enrolled courses</p>
        </div>
    </div>

    <div class="container">
        <a href="assignments.php" class="back-btn">
            <i class="bx bx-arrow-back"></i>
            Back to Assignments
        </a>

        <?php if (!empty($quizzes)): ?>
            <div class="quizzes-grid">
                <?php foreach($quizzes as $quiz): ?>
                    <div class="quiz-card">
                        <div class="quiz-header">
                            <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 0.5rem;">
                                <h3 style="margin: 0; flex: 1;"><?= htmlspecialchars($quiz['title']) ?></h3>
                                <?php if($quiz['contributes_to_final_grade']): ?>
                                    <div class="grade-badge grade-counts" title="This quiz contributes to your final grade">
                                        <i class="bx bx-medal"></i>
                                        <span>Counts to Final</span>
                                    </div>
                                <?php else: ?>
                                    <div class="grade-badge grade-practice" title="This is a practice quiz - does not affect your grade">
                                        <i class="bx bx-dumbbell"></i>
                                        <span>Practice Only</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="quiz-course">
                                <i class="bx bx-book-alt"></i>
                                <span><?= htmlspecialchars($quiz['course_title']) ?></span>
                                <span>by <?= htmlspecialchars($quiz['first_name'] . ' ' . $quiz['last_name']) ?></span>
                            </div>
                        </div>
                        
                        <div class="quiz-content">
                            <p><?= htmlspecialchars($quiz['description']) ?></p>
                            
                            <div class="quiz-meta">
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-help-circle"></i>
                                    <span><?= $quiz['question_count'] ?> Questions</span>
                                </div>
                                <?php if($quiz['time_limit'] > 0): ?>
                                    <div class="meta-item">
                                        <i class="meta-icon bx bx-time"></i>
                                        <span><?= $quiz['time_limit'] ?> minutes</span>
                                    </div>
                                <?php endif; ?>
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-target-lock"></i>
                                    <span>Pass: <?= $quiz['passing_score'] ?>%</span>
                                </div>
                                <?php if($quiz['status'] === 'completed'): ?>
                                    <div class="meta-item">
                                        <i class="meta-icon bx bx-check-circle"></i>
                                        <span>Score: <?= round($quiz['score'], 1) ?>%</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="quiz-footer">
                                <div class="status-badge status-<?= $quiz['status'] ?>">
                                    <?php 
                                    switch($quiz['status']) {
                                        case 'completed':
                                            echo '<i class="bx bx-check-circle"></i> Completed';
                                            break;
                                        case 'available':
                                            echo '<i class="bx bx-play-circle"></i> Available';
                                            break;
                                        case 'in_progress':
                                            echo '<i class="bx bx-time"></i> In Progress';
                                            break;
                                        default:
                                            echo '<i class="bx bx-lock"></i> Not Available';
                                    }
                                    ?>
                                </div>
                                
                                <div class="action-buttons">
                                    <?php if($quiz['status'] === 'completed'): ?>
                                        <a href="quiz_results.php?attempt_id=<?= $quiz['attempt_id'] ?>" class="quiz-btn btn-outline">
                                            <i class="bx bx-bar-chart"></i>
                                            View Results
                                        </a>
                                    <?php elseif($quiz['status'] === 'available'): ?>
                                        <a href="take_quiz.php?id=<?= $quiz['quiz_id'] ?>" class="quiz-btn btn-primary">
                                            <i class="bx bx-play"></i>
                                            Take Quiz
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="bx bx-help-circle"></i>
                </div>
                <h3 class="empty-title">No Quizzes Available</h3>
                <p class="empty-description">
                    You don't have any quizzes at the moment. Check back later for new quizzes from your instructors.
                </p>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Animate cards on load
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.quiz-card');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                
                setTimeout(() => {
                    card.style.transition = 'all 0.6s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
    </script>
</body>
</html>
