<?php
/**
 * Check Real Student Data with Valid User Matches
 * Find students that actually have matching user records
 */

error_reporting(-1);
ini_set('display_errors', 1);

echo "<h1>🔍 Real Student-User Data Check</h1>";

try {
    require_once 'config/db_connect.php';
    echo "✅ Database connected<br>";
} catch (Exception $e) {
    echo "❌ Database error: " . $e->getMessage() . "<br>";
    exit;
}

echo "<h2>📋 Students with Valid User Matches</h2>";

try {
    // Find students that have matching user records
    $valid_students = $pdo->query("
        SELECT 
            s.student_id, 
            s.first_name, 
            s.last_name, 
            s.user_id,
            u.email,
            u.username,
            u.user_role
        FROM students s 
        JOIN users u ON u.user_id = s.user_id
        ORDER BY s.student_id
    ");
    
    if ($valid_students && $valid_students->rowCount() > 0) {
        echo "✅ Found students with valid user matches:<br>";
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>";
        echo "<tr style='background: #d4edda;'>";
        echo "<th>Student ID</th><th>Name</th><th>User ID</th><th>Email</th><th>Username</th><th>Role</th>";
        echo "</tr>";
        
        $real_students = [];
        while ($row = $valid_students->fetch(PDO::FETCH_ASSOC)) {
            $real_students[] = $row;
            echo "<tr>";
            echo "<td><strong>{$row['student_id']}</strong></td>";
            echo "<td>{$row['first_name']} {$row['last_name']}</td>";
            echo "<td>{$row['user_id']}</td>";
            echo "<td><strong>{$row['email']}</strong></td>";
            echo "<td>{$row['username']}</td>";
            echo "<td>{$row['user_role']}</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Test API with real data
        if (!empty($real_students)) {
            $test_student = $real_students[0]; // Use first valid student
            
            echo "<h2>🧪 Test API with Real Student Data</h2>";
            echo "<div style='background: #e7f3ff; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
            echo "<strong>Using Real Student:</strong><br>";
            echo "Student ID: {$test_student['student_id']}<br>";
            echo "Name: {$test_student['first_name']} {$test_student['last_name']}<br>";
            echo "Email: {$test_student['email']}<br>";
            echo "</div>";
            
            // Get a valid course
            $course_query = $pdo->query("SELECT course_id, title FROM courses WHERE is_published = 1 LIMIT 1");
            $course = $course_query ? $course_query->fetch(PDO::FETCH_ASSOC) : null;
            
            if ($course) {
                echo "<div style='background: #e7f3ff; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
                echo "<strong>Using Course:</strong><br>";
                echo "Course ID: {$course['course_id']}<br>";
                echo "Title: {$course['title']}<br>";
                echo "</div>";
                
                // Test the API with real data
                $real_test_data = [
                    'action' => 'send_study_reminder',
                    'reminder' => [
                        'id' => time(),
                        'date' => '2025-11-23',
                        'time' => '10:00',
                        'type' => 'study_session',
                        'description' => 'Real student test - should work now!',
                        'emailNotification' => true
                    ],
                    'course_id' => (int)$course['course_id'],
                    'student_id' => (int)$test_student['student_id']
                ];
                
                echo "<h3>📤 Testing API Call</h3>";
                echo "<div style='background: #f8f9fa; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
                echo "<strong>API Data:</strong><br>";
                echo "<pre>" . json_encode($real_test_data, JSON_PRETTY_PRINT) . "</pre>";
                echo "</div>";
                
                $api_url = 'http://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']) . '/student/api/send_study_reminder_email.php';
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $api_url);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($real_test_data));
                curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                echo "<h3>📥 API Response</h3>";
                
                if ($http_code == 200) {
                    if (!empty($response)) {
                        $json_data = json_decode($response, true);
                        
                        if ($json_data !== null) {
                            echo "<div style='background: #d4edda; color: #155724; padding: 15px; border-radius: 5px;'>";
                            echo "🎉 <strong>SUCCESS! API is working with real data!</strong><br>";
                            echo "<pre>" . json_encode($json_data, JSON_PRETTY_PRINT) . "</pre>";
                            echo "</div>";
                            
                            echo "<div style='background: #d1ecf1; color: #0c5460; padding: 15px; border-radius: 5px; margin-top: 10px;'>";
                            echo "✅ <strong>Email should be sent to: {$test_student['email']}</strong><br>";
                            echo "📧 Check your email inbox and spam folder<br>";
                            echo "⏰ Automatic reminder scheduled for 2025-11-23 10:00<br>";
                            echo "</div>";
                            
                        } else {
                            echo "<div style='background: #fff3cd; color: #856404; padding: 15px; border-radius: 5px;'>";
                            echo "⚠️ <strong>Invalid JSON response</strong><br>";
                            echo "Raw response: <pre>" . htmlspecialchars($response) . "</pre>";
                            echo "</div>";
                        }
                    } else {
                        echo "<div style='background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px;'>";
                        echo "❌ <strong>Still empty response - need to fix API JOIN query</strong><br>";
                        echo "The API probably still has the wrong JOIN syntax.";
                        echo "</div>";
                    }
                } else {
                    echo "<div style='background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px;'>";
                    echo "❌ <strong>HTTP Error: $http_code</strong><br>";
                    if (!empty($response)) {
                        echo "Response: <pre>" . htmlspecialchars($response) . "</pre>";
                    }
                    echo "</div>";
                }
                
            } else {
                echo "❌ No published courses found<br>";
            }
        }
        
    } else {
        echo "❌ No students found with matching user records<br>";
        
        // Show orphaned students
        $orphaned = $pdo->query("
            SELECT s.student_id, s.first_name, s.last_name, s.user_id
            FROM students s 
            LEFT JOIN users u ON u.user_id = s.user_id
            WHERE u.user_id IS NULL
        ");
        
        if ($orphaned && $orphaned->rowCount() > 0) {
            echo "<h3>⚠️ Orphaned Students (no matching user):</h3>";
            echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>";
            echo "<tr style='background: #fff3cd;'>";
            echo "<th>Student ID</th><th>Name</th><th>User ID</th><th>Status</th>";
            echo "</tr>";
            
            while ($row = $orphaned->fetch(PDO::FETCH_ASSOC)) {
                echo "<tr>";
                echo "<td>{$row['student_id']}</td>";
                echo "<td>{$row['first_name']} {$row['last_name']}</td>";
                echo "<td>{$row['user_id']}</td>";
                echo "<td>❌ No matching user</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

?>

<style>
body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 1000px; margin: 20px auto; padding: 20px; }
h1 { color: #2c3e50; }
h2, h3 { color: #34495e; }
table { margin: 10px 0; }
th, td { padding: 6px 10px; text-align: left; border: 1px solid #ddd; }
th { background-color: #f8f9fa; }
pre { font-size: 12px; }
</style>
