<?php
/**
 * Student Courses Page - Clean Modern Implementation
 * Panadite Academy
 * 
 * Features:
 * - Enrolled courses with progress tracking
 * - Available courses with payment integration
 * - Modern responsive design
 * - Course filtering and statistics
 * - Clean error handling
 */

// Authentication and session
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');

// Initialize variables
$enrolledCourses = [];
$availableCourses = [];
$coursesInProgress = 0;
$completedCourses = 0;
$totalCourseHours = 0;
$userId = $_SESSION['user_id'] ?? 0;

// Database operations with comprehensive error handling
try {
    require_once '../config/db_connect.php';
    
    if (isset($conn) && $userId) {
        // Fetch enrolled courses with comprehensive data
        $enrolledQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, 
                   u.first_name, u.last_name, e.enrolled_date, 
                   COALESCE(e.progress, 0) as progress, 
                   'General' as category,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students
            FROM enrollments e
            JOIN courses c ON e.course_id = c.course_id
            JOIN users u ON c.teacher_id = u.user_id
            WHERE e.student_id = ?
            ORDER BY e.enrolled_date DESC
        ";
        
        $stmt = $conn->prepare($enrolledQuery);
        if (!$stmt) {
            error_log("Enrolled courses SQL prepare failed: " . $conn->error);
            error_log("Query was: " . $enrolledQuery);
            throw new Exception("Enrolled courses database query preparation failed");
        }
        
        $stmt->bind_param("i", $userId);
        if (!$stmt->execute()) {
            error_log("Enrolled courses SQL execute failed: " . $stmt->error);
            throw new Exception("Enrolled courses database query execution failed");
        }
        
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $status = $row['progress'] >= 100 ? 'completed' : 'in_progress';
            
            $enrolledCourses[] = [
                'id' => $row['id'],
                'title' => $row['title'],
                'description' => $row['description'],
                'instructor' => $row['first_name'] . ' ' . $row['last_name'],
                'thumbnail' => !empty($row['thumbnail']) ? $row['thumbnail'] : './assets/img/courses/default.jpg',
                'progress' => $row['progress'],
                'total_modules' => 10,
                'completed_modules' => ceil($row['progress'] / 10),
                'start_date' => $row['enrolled_date'],
                'category' => $row['category'],
                'status' => $status,
                'students' => $row['total_students']
            ];
            
            if ($status === 'in_progress') {
                $coursesInProgress++;
            } else {
                $completedCourses++;
            }
        }
        
        $totalCourseHours = count($enrolledCourses) * 10;
        
        // Fetch available courses (not enrolled)
        $availableQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, c.price,
                   u.first_name, u.last_name, 'General' as category,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students,
                   COALESCE(4.5, 4.5) as rating,
                   '10 hours' as duration, 
                   'Beginner' as level
            FROM courses c
            JOIN users u ON c.teacher_id = u.user_id
            WHERE c.is_published = 1
            AND c.course_id NOT IN (
                SELECT course_id FROM enrollments WHERE student_id = ? AND course_id IS NOT NULL
            )
            ORDER BY c.created_at DESC
        ";
        
        $stmt = $conn->prepare($availableQuery);
        if (!$stmt) {
            error_log("SQL prepare failed: " . $conn->error);
            error_log("Query was: " . $availableQuery);
            throw new Exception("Database query preparation failed");
        }
        
        $stmt->bind_param("i", $userId);
        if (!$stmt->execute()) {
            error_log("SQL execute failed: " . $stmt->error);
            throw new Exception("Database query execution failed");
        }
        
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $availableCourses[] = [
                'id' => $row['id'],
                'title' => $row['title'],
                'description' => $row['description'],
                'instructor' => $row['first_name'] . ' ' . $row['last_name'],
                'thumbnail' => !empty($row['thumbnail']) ? $row['thumbnail'] : './assets/img/courses/default.jpg',
                'category' => $row['category'],
                'students' => $row['total_students'],
                'price' => $row['price'] ?? 0,
                'rating' => round($row['rating'], 1),
                'duration' => $row['duration'] ?? '10 hours',
                'level' => $row['level'] ?? 'Beginner'
            ];
        }
    }
} catch (Exception $e) {
    error_log("Courses page error: " . $e->getMessage());
}

// Calculate statistics
$totalEnrolled = count($enrolledCourses);
$completionRate = $totalEnrolled > 0 ? round(($completedCourses / $totalEnrolled) * 100) : 0;

// Handle session notifications
$enrollmentSuccess = $_SESSION['enrollment_success'] ?? null;
$enrollmentError = $_SESSION['enrollment_error'] ?? null;
unset($_SESSION['enrollment_success'], $_SESSION['enrollment_error']);

include_once 'components/header.php';
?>

<!-- Enrollment Notifications -->
<?php if ($enrollmentSuccess): ?>
<div class="container-fluid pt-3">
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class='bx bx-check-circle me-2'></i>
        <?= htmlspecialchars($enrollmentSuccess) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
</div>
<?php endif; ?>

<?php if ($enrollmentError): ?>
<div class="container-fluid pt-3">
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class='bx bx-error-circle me-2'></i>
        <?= htmlspecialchars($enrollmentError) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
</div>
<?php endif; ?>

<!-- Main Content -->
<div class="dashboard-content">
    <!-- Learning Dashboard Header -->
    <div class="dashboard-header">
        <h1 class="dashboard-title">
            <i class='bx bx-graduation'></i>
            My Learning Dashboard
        </h1>
        <p class="dashboard-subtitle">
            Track your progress and continue your learning journey with our comprehensive course catalog
        </p>
    </div>

    <!-- Compact Learning Stats Banner -->
    <div class="compact-stats-banner">
        <div class="stats-container">
            <!-- Active Learning Stat -->
            <div class="stat-item">
                <div class="stat-icon-wrapper primary">
                    <i class='bx bx-book-bookmark'></i>
                </div>
                <div class="stat-details">
                    <div class="stat-badge">
                        <?php if ($coursesInProgress > 0): ?>
                            <span class="pulse-indicator"></span>
                            Active Learning
                        <?php else: ?>
                            Ready to Start
                        <?php endif; ?>
                    </div>
                    <div class="stat-number"><?= count($enrolledCourses) ?></div>
                    <div class="stat-label">Enrolled Courses</div>
                    <div class="stat-subtitle">Your active learning journey</div>
                    <a href="#enrolled-courses" class="stat-action">
                        View Courses <i class='bx bx-right-arrow-alt'></i>
                    </a>
                </div>
            </div>

            <!-- Completed Courses Stat -->
            <div class="stat-item">
                <div class="stat-icon-wrapper success">
                    <i class='bx bx-check-circle'></i>
                </div>
                <div class="stat-details">
                    <div class="stat-badge">
                        <?php if ($completedCourses > 0): ?>
                            <span class="celebration-indicator"></span>
                            Achieved
                        <?php else: ?>
                            Getting Started
                        <?php endif; ?>
                    </div>
                    <div class="stat-number"><?= $completedCourses ?></div>
                    <div class="stat-label">Completed Courses</div>
                    <div class="stat-subtitle">
                        <?php 
                        $completionRate = count($enrolledCourses) > 0 ? round(($completedCourses / count($enrolledCourses)) * 100) : 0;
                        echo $completionRate; ?>% Completion Rate
                    </div>
                </div>
            </div>

            <!-- Learning Hours Stat -->
            <div class="stat-item">
                <div class="stat-icon-wrapper warning">
                    <i class='bx bx-time'></i>
                </div>
                <div class="stat-details">
                    <div class="stat-badge">Time Invested</div>
                    <div class="stat-number"><?= $totalCourseHours ?></div>
                    <div class="stat-label">Learning Hours</div>
                    <div class="stat-subtitle">Keep growing your expertise</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Enrolled Courses Section -->
    <section class="modern-courses-section" id="enrolled-courses">
        <div class="modern-card full-width">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-book-open'></i> My Enrolled Courses</h3>
                    <span class="subtitle">Continue your learning journey</span>
                </div>
                <div class="course-filters">
                    <button class="filter-btn active" data-filter="all">
                        <i class='bx bx-list-ul'></i>
                        All Courses
                    </button>
                    <button class="filter-btn" data-filter="in_progress">
                        <i class='bx bx-play-circle'></i>
                        In Progress
                    </button>
                    <button class="filter-btn" data-filter="completed">
                        <i class='bx bx-check-circle'></i>
                        Completed
                    </button>
                </div>
            </div>
            <div class="card-content">
                <?php if (empty($enrolledCourses)): ?>
                    <div class="modern-empty-state">
                        <div class="empty-illustration">
                            <i class='bx bx-book-add'></i>
                        </div>
                        <div class="empty-content">
                            <h4>No Courses Enrolled Yet</h4>
                            <p>Start your learning journey by enrolling in courses that match your interests and career goals.</p>
                            <a href="#available-courses" class="modern-btn primary">
                                <i class='bx bx-search'></i>
                                Browse Available Courses
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="enrolled-courses-grid">
                        <?php foreach ($enrolledCourses as $index => $course): ?>
                            <div class="enrolled-course-card" 
                                 data-course-id="<?= $course['id'] ?>"
                                 data-status="<?= $course['status'] ?>" 
                                 style="animation-delay: <?= $index * 0.1 ?>s">
                                
                                <!-- Course Image & Status -->
                                <div class="enrolled-course-header">
                                    <div class="course-image-section">
                                        <img src="<?= $course['thumbnail'] ?>" alt="<?= htmlspecialchars($course['title']) ?>" class="enrolled-course-image">
                                        <div class="course-status-badge <?= $course['status'] ?>">
                                            <i class='bx <?= $course['status'] === 'completed' ? 'bx-check-circle' : 'bx-play-circle' ?>'></i>
                                            <?= ucfirst(str_replace('_', ' ', $course['status'])) ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Course Info -->
                                <div class="enrolled-course-content">
                                    <h4 class="enrolled-course-title" title="<?= htmlspecialchars($course['title']) ?>">
                                        <?= htmlspecialchars($course['title']) ?>
                                    </h4>
                                    
                                    <div class="course-instructor-info">
                                        <i class='bx bx-user-circle'></i>
                                        <span><?= htmlspecialchars($course['instructor']) ?></span>
                                    </div>
                                    
                                    <!-- Progress Range Detector -->
                                    <div class="progress-range-section">
                                        <div class="progress-header">
                                            <span class="progress-label">Progress</span>
                                            <span class="progress-value"><?= $course['progress'] ?>%</span>
                                        </div>
                                        <div class="progress-range-container">
                                            <input type="range" 
                                                   class="progress-range" 
                                                   value="<?= $course['progress'] ?>" 
                                                   min="0" 
                                                   max="100" 
                                                   disabled>
                                            <div class="progress-range-bg">
                                                <div class="progress-range-fill" style="width: <?= $course['progress'] ?>%"></div>
                                            </div>
                                        </div>
                                        <div class="modules-info">
                                            <i class='bx bx-list-check'></i>
                                            <span><?= $course['completed_modules'] ?>/<?= $course['total_modules'] ?> modules completed</span>
                                        </div>
                                    </div>
                                    
                                    <!-- Enrollment Date -->
                                    <div class="enrollment-info">
                                        <i class='bx bx-calendar'></i>
                                        <span>Enrolled: <?= date('M j, Y', strtotime($course['start_date'])) ?></span>
                                    </div>
                                </div>
                                
                                <!-- Action Buttons -->
                                <div class="enrolled-course-actions">
                                    <div class="primary-actions">
                                        <a href="<?= generateSecureUrl('course-details.php', $course['id']) ?>" class="action-btn details-btn" title="View Course Details">
                                            <i class='bx bx-show'></i>
                                        </a>
                                        <button class="action-btn continue-btn" onclick="continueCourse(<?= $course['id'] ?>)" title="Continue Learning">
                                            <i class='bx <?= $course['status'] === 'completed' ? 'bx-book' : 'bx-play-circle' ?>'></i>
                                            <?= $course['status'] === 'completed' ? 'Review' : 'Continue' ?>
                                        </button>
                                    </div>
                                    <div class="secondary-actions">
                                        <button class="action-btn unenroll-btn" onclick="unenrollCourse(<?= $course['id'] ?>)" title="Unenroll from Course">
                                            <i class='bx bx-user-minus'></i>
                                        </button>
                                        <button class="action-btn delete-btn" onclick="deleteCourse(<?= $course['id'] ?>)" title="Delete Course">
                                            <i class='bx bx-trash'></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </section>

    <!-- Available Courses Section -->
    <section class="modern-courses-section" id="available-courses">
        <div class="modern-card full-width">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-compass'></i> Discover New Courses</h3>
                    <span class="subtitle">Expand your skills with our curated course collection</span>
                </div>
                <a href="course-catalog.php" class="view-all-btn">
                    <span>View All Courses</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
            <div class="card-content">
                <?php if (empty($availableCourses)): ?>
                    <div class="modern-empty-state">
                        <div class="empty-illustration">
                            <i class='bx bx-search-alt'></i>
                        </div>
                        <div class="empty-content">
                            <h4>More Courses Coming Soon</h4>
                            <p>We're constantly adding new courses to help you grow. Check back soon for exciting new learning opportunities!</p>
                            <a href="course-catalog.php" class="modern-btn">
                                <i class='bx bx-bell'></i>
                                Get Notified
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="udemy-style-courses-grid">
                        <?php foreach ($availableCourses as $index => $course): ?>
                            <div class="udemy-course-card" data-course-id="<?= IdEncryption::encrypt($course['id']) ?>" style="animation-delay: <?= $index * 0.1 ?>s">
                                <!-- Course Thumbnail -->
                                <div class="course-image-container">
                                    <img src="<?= $course['thumbnail'] ?>" alt="<?= htmlspecialchars($course['title']) ?>" class="course-image">
                                    <?php if ($course['price'] <= 0): ?>
                                        <div class="course-badge free-badge">FREE</div>
                                    <?php else: ?>
                                        <div class="course-badge price-badge">R<?= number_format($course['price'], 0) ?></div>
                                    <?php endif; ?>
                                </div>
                                
                                <!-- Course Content -->
                                <div class="course-card-content">
                                    <!-- Course Title (single line with ellipsis) -->
                                    <h3 class="course-card-title" title="<?= htmlspecialchars($course['title']) ?>">
                                        <?= htmlspecialchars($course['title']) ?>
                                    </h3>
                                    
                                    <!-- Course Description (2 lines max with ellipsis) -->
                                    <p class="course-card-description" title="<?= htmlspecialchars($course['description']) ?>">
                                        <?= htmlspecialchars($course['description']) ?>
                                    </p>
                                    
                                    <!-- Instructor -->
                                    <div class="course-instructor-name">
                                        <?= htmlspecialchars($course['instructor']) ?>
                                    </div>
                                    
                                    <!-- Rating and Students -->
                                    <div class="course-rating-info">
                                        <div class="rating-stars">
                                            <span class="rating-value"><?= $course['rating'] ?></span>
                                            <div class="stars">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i class='bx <?= $i <= $course['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                <?php endfor; ?>
                                            </div>
                                            <span class="rating-count">(<?= $course['students'] ?>)</span>
                                        </div>
                                    </div>
                                    
                                    <!-- Course Meta -->
                                    <div class="course-meta-info">
                                        <span class="course-duration"><?= htmlspecialchars($course['duration']) ?></span>
                                        <span class="course-level"><?= htmlspecialchars($course['level']) ?></span>
                                    </div>
                                    
                                    <!-- Price and Action Buttons -->
                                    <div class="course-card-footer">
                                        <div class="course-price-section">
                                            <?php if ($course['price'] > 0): ?>
                                                <span class="course-price">R<?= number_format($course['price'], 2) ?></span>
                                            <?php else: ?>
                                                <span class="course-price free">Free</span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="course-action-buttons">
                                            <a href="<?= generateSecureUrl('course-details.php', $course['id']) ?>" class="action-btn view-btn" title="View Course Details">
                                                <i class='bx bx-show'></i>
                                            </a>
                                            <button class="action-btn cart-btn" onclick="addToCart('<?= IdEncryption::encrypt($course['id']) ?>', this)" title="Add to Cart">
                                                <i class='bx bx-cart-add'></i>
                                            </button>
                                            <button class="action-btn enroll-btn" onclick="enrollCourse(<?= $course['id'] ?>)" title="Enroll Now">
                                                <i class='bx bx-user-plus'></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </section>
</div>

<!-- Modern Courses Page CSS -->
<style>
/* Modern Courses Page Styling */
.dashboard-content {
    margin: 2rem auto;
    max-width: 1400px;
    padding: 0 1.5rem;
}

/* Dashboard Header */
.dashboard-header {
    text-align: center;
    margin-bottom: 2rem;
    padding: 1rem 0;
}

.dashboard-title {
    font-size: 2.5rem;
    font-weight: 700;
    color: #4A90E2;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
    line-height: 1.2;
}

.dashboard-title i {
    font-size: 2.25rem;
    color: #FF9500;
    background: linear-gradient(135deg, #FF9500, #E8860C);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    filter: drop-shadow(0 2px 4px rgba(255, 149, 0, 0.2));
}

.dashboard-subtitle {
    font-size: 1.1rem;
    color: #7f8c8d;
    margin: 0;
    font-weight: 400;
    line-height: 1.5;
    max-width: 600px;
    margin: 0 auto;
}

/* Responsive Dashboard Header */
@media (max-width: 768px) {
    .dashboard-title {
        font-size: 2rem;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .dashboard-title i {
        font-size: 2rem;
    }
    
    .dashboard-subtitle {
        font-size: 1rem;
        padding: 0 1rem;
    }
}

/* Compact Learning Stats Banner */
.compact-stats-banner {
    background: linear-gradient(135deg, #fff5e6 0%, #ffecd1 100%);
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 10px 30px rgba(255, 149, 0, 0.15);
    position: relative;
    overflow: hidden;
    border: 1px solid rgba(255, 149, 0, 0.1);
}

.compact-stats-banner::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url("data:image/svg+xml,%3Csvg width='100' height='100' viewBox='0 0 100 100' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M11 18c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm48 25c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm-43-7c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm63 31c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zM34 90c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm56-76c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3z' fill='%23ff9500' fill-opacity='0.05' fill-rule='evenodd'/%3E%3C/svg%3E");
    opacity: 0.5;
    z-index: 1;
}

.stats-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 2rem;
    position: relative;
    z-index: 2;
}

.stat-item {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    background: rgba(255, 255, 255, 0.8);
    backdrop-filter: blur(10px);
    border-radius: 16px;
    padding: 1.5rem;
    border: 1px solid rgba(74, 144, 226, 0.2);
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(74, 144, 226, 0.15);
    background: rgba(255, 255, 255, 0.9);
}

.stat-icon-wrapper {
    width: 50px;
    height: 50px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
    position: relative;
}

.stat-icon-wrapper.primary {
    background: linear-gradient(135deg, #4A90E2, #357ABD);
    box-shadow: 0 4px 15px rgba(74, 144, 226, 0.3);
}

.stat-icon-wrapper.success {
    background: linear-gradient(135deg, #4A90E2, #357ABD);
    box-shadow: 0 4px 15px rgba(74, 144, 226, 0.3);
}

.stat-icon-wrapper.warning {
    background: linear-gradient(135deg, #FF9500, #E8860C);
    box-shadow: 0 4px 15px rgba(255, 149, 0, 0.3);
}

.stat-icon-wrapper i {
    font-size: 1.5rem;
    color: white;
}

.stat-details {
    flex: 1;
    color: #2c3e50;
}

.stat-badge {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: #4A90E2;
    margin-bottom: 0.5rem;
}

.pulse-indicator, .celebration-indicator {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: #4A90E2;
    animation: pulse 2s ease-in-out infinite;
}

.celebration-indicator {
    background: #FF9500;
    animation: celebrate 2s ease-in-out infinite;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 1; }
    50% { transform: scale(1.2); opacity: 0.7; }
}

@keyframes celebrate {
    0%, 100% { transform: scale(1) rotate(0deg); opacity: 1; }
    25% { transform: scale(1.1) rotate(90deg); opacity: 0.8; }
    50% { transform: scale(1.2) rotate(180deg); opacity: 0.6; }
    75% { transform: scale(1.1) rotate(270deg); opacity: 0.8; }
}

.stat-number {
    font-size: 2.5rem;
    font-weight: 700;
    color: #4A90E2;
    line-height: 1;
    margin-bottom: 0.25rem;
    text-shadow: 0 2px 4px rgba(74, 144, 226, 0.1);
}

.stat-label {
    font-size: 1.1rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.25rem;
}

.stat-subtitle {
    font-size: 0.9rem;
    color: #7f8c8d;
    margin-bottom: 0.75rem;
    line-height: 1.4;
}

.stat-action {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    color: white;
    text-decoration: none;
    font-size: 0.9rem;
    font-weight: 600;
    padding: 0.5rem 1rem;
    background: linear-gradient(135deg, #4A90E2, #357ABD);
    border-radius: 8px;
    border: 1px solid rgba(74, 144, 226, 0.3);
    transition: all 0.3s ease;
    box-shadow: 0 2px 8px rgba(74, 144, 226, 0.2);
}

.stat-action:hover {
    background: linear-gradient(135deg, #357ABD, #2E6DA4);
    transform: translateX(4px);
    color: white;
    text-decoration: none;
    box-shadow: 0 4px 12px rgba(74, 144, 226, 0.3);
}

.stat-action i {
    transition: transform 0.3s ease;
}

.stat-action:hover i {
    transform: translateX(2px);
}

/* Responsive Design for Compact Banner */
@media (max-width: 768px) {
    .compact-stats-banner {
        padding: 1.5rem;
        margin-bottom: 1.5rem;
    }
    
    .stats-container {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .stat-item {
        padding: 1rem;
    }
    
    .stat-number {
        font-size: 2rem;
    }
    
    .stat-icon-wrapper {
        width: 40px;
        height: 40px;
    }
    
    .stat-icon-wrapper i {
        font-size: 1.25rem;
    }
}

/* Modern Welcome Banner */
.modern-welcome-banner {
    background: whitesmoke;
    border-radius: 1rem;
    padding: 2.5rem;
    margin-bottom: 2rem;
    color: #2c3e50;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    position: relative;
    overflow: hidden;
    border: 1px solid #e3e6f0;
}

.banner-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    z-index: 2;
}

.welcome-text h1 {
    font-size: 2.5rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    color: #2c3e50;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.welcome-text h1 i {
    color: #4e73df;
    font-size: 2.8rem;
}

.welcome-text p {
    font-size: 1.1rem;
    color: #5a5c69;
    margin: 0;
}

.banner-actions {
    display: flex;
    gap: 1rem;
    flex-shrink: 0;
}

.modern-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 0.5rem;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    font-size: 0.9rem;
}

.modern-btn.primary {
    background: #4e73df;
    color: white;
    box-shadow: 0 4px 15px rgba(78, 115, 223, 0.3);
}

.modern-btn.primary:hover {
    background: #2e59d9;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(78, 115, 223, 0.4);
    color: white;
}

.modern-btn.secondary {
    background: white;
    color: #4e73df;
    border: 2px solid #4e73df;
}

.modern-btn.secondary:hover {
    background: #4e73df;
    color: white;
    transform: translateY(-2px);
}

.modern-btn.success {
    background: #1cc88a;
    color: white;
    box-shadow: 0 4px 15px rgba(28, 200, 138, 0.3);
}

.modern-btn.success:hover {
    background: #17a673;
    transform: translateY(-2px);
    color: white;
}

.modern-btn.full-width {
    width: 100%;
    justify-content: center;
}

/* Floating Learning Icons */
.floating-icons {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 1;
    pointer-events: none;
}

.floating-icon {
    position: absolute;
    font-size: 2rem;
    color: #4e73df;
    opacity: 0.1;
    animation: float 6s ease-in-out infinite;
}

.floating-icon.icon-1 { top: 20%; left: 15%; animation-delay: 0s; }
.floating-icon.icon-2 { top: 30%; right: 20%; animation-delay: 1s; }
.floating-icon.icon-3 { bottom: 40%; left: 25%; animation-delay: 2s; }
.floating-icon.icon-4 { bottom: 20%; right: 15%; animation-delay: 3s; }
.floating-icon.icon-5 { top: 60%; left: 40%; animation-delay: 4s; }
.floating-icon.icon-6 { top: 40%; right: 40%; animation-delay: 5s; }

@keyframes float {
    0%, 100% { transform: translateY(0px) rotate(0deg); }
    33% { transform: translateY(-10px) rotate(5deg); }
    66% { transform: translateY(5px) rotate(-3deg); }
}

/* Modern Statistics Grid */
.modern-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 2rem;
    margin-bottom: 2rem;
}

.modern-stat-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.08);
    border: 1px solid #e3e6f0;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.modern-stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 1rem 3rem rgba(0, 0, 0, 0.15);
}

.stat-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.stat-icon.bg-primary { background: linear-gradient(135deg, #4e73df, #224abe); }
.stat-icon.bg-success { background: linear-gradient(135deg, #1cc88a, #17a673); }
.stat-icon.bg-warning { background: linear-gradient(135deg, #f6c23e, #dda20a); }

.stat-badge {
    background: #f8f9fc;
    color: #5a5c69;
    padding: 0.3rem 0.8rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.pulse-dot {
    width: 8px;
    height: 8px;
    background: #1cc88a;
    border-radius: 50%;
    animation: pulse 2s infinite;
}

.celebration-dot {
    width: 8px;
    height: 8px;
    background: #f6c23e;
    border-radius: 50%;
    animation: celebrate 3s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.5; transform: scale(1.2); }
}

@keyframes celebrate {
    0%, 100% { opacity: 1; transform: scale(1) rotate(0deg); }
    25% { opacity: 0.8; transform: scale(1.1) rotate(90deg); }
    50% { opacity: 0.6; transform: scale(1.2) rotate(180deg); }
    75% { opacity: 0.8; transform: scale(1.1) rotate(270deg); }
}

.stat-content {
    text-align: center;
}

.stat-number {
    font-size: 2.5rem;
    font-weight: 700;
    background: linear-gradient(135deg, #4e73df, #224abe);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 1.1rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.3rem;
}

.stat-sublabel {
    color: #5a5c69;
    font-size: 0.9rem;
    margin-bottom: 1rem;
}

.stat-link {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    color: #4e73df;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.9rem;
    transition: all 0.3s ease;
}

.stat-link:hover {
    color: #2e59d9;
    transform: translateX(5px);
}

/* Progress Elements */
.completion-progress, .hours-breakdown {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.progress-ring {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: conic-gradient(#1cc88a calc(var(--progress) * 1%), #e3e6f0 0);
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
}

.progress-ring::before {
    content: '';
    width: 30px;
    height: 30px;
    border-radius: 50%;
    background: white;
}

.shimmer-progress {
    flex: 1;
    height: 6px;
    background: #e3e6f0;
    border-radius: 3px;
    overflow: hidden;
    position: relative;
}

.shimmer-fill {
    height: 100%;
    background: linear-gradient(90deg, #f6c23e, #dda20a);
    border-radius: 3px;
    position: relative;
    overflow: hidden;
}

.shimmer-fill::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.6), transparent);
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { left: -100%; }
    100% { left: 100%; }
}

/* Modern Course Cards */
.modern-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.08);
    border: 1px solid #e3e6f0;
    margin-bottom: 2rem;
}

.modern-card.full-width {
    width: 100%;
}

.card-header {
    padding: 2rem 2rem 1rem 2rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid #e3e6f0;
}

.header-content h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #2c3e50;
    margin: 0 0 0.5rem 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.header-content h3 i {
    color: #4e73df;
    font-size: 1.7rem;
}

.subtitle {
    color: #5a5c69;
    font-size: 0.9rem;
}

.view-all-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #4e73df;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.9rem;
    transition: all 0.3s ease;
}

.view-all-btn:hover {
    color: #2e59d9;
    transform: translateX(5px);
}

.card-content {
    padding: 2rem;
}

/* Course Filters */
.course-filters {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.filter-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: white;
    color: #5a5c69;
    border: 1px solid #e3e6f0;
    border-radius: 0.5rem;
    font-size: 0.85rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-btn:hover, .filter-btn.active {
    background: #4e73df;
    color: white;
    border-color: #4e73df;
    transform: translateY(-2px);
}

/* Modern Courses Grid - 5 Cards Per Row */
.modern-courses-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 1.5rem;
    margin-top: 1.5rem;
}

.modern-course-card {
    background: linear-gradient(145deg, #ffffff, #f8f9fa);
    border-radius: 16px;
    overflow: hidden;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    transition: all 0.3s ease;
    border: 1px solid rgba(233, 236, 239, 0.8);
    position: relative;
    display: flex;
    flex-direction: column;
    height: 100%;
    opacity: 0;
    transform: translateY(20px);
    animation: slideUp 0.6s ease forwards;
}

@keyframes slideUp {
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.modern-course-card:hover {
    box-shadow: 0 1rem 3rem rgba(0, 0, 0, 0.15);
    transform: translateY(-8px);
}

/* Course Thumbnail */
.course-thumbnail-wrapper {
    position: relative;
    height: 200px;
    overflow: hidden;
}

.course-thumbnail {
    width: 100%;
    height: 100%;
    background-size: cover;
    background-position: center;
    background-repeat: no-repeat;
    position: relative;
}

.course-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(0,0,0,0.3), rgba(0,0,0,0.1));
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 1rem;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.modern-course-card:hover .course-overlay {
    opacity: 1;
}

.status-badge {
    background: rgba(255, 255, 255, 0.95);
    color: #2c3e50;
    padding: 0.4rem 0.8rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.status-badge.completed {
    background: rgba(28, 200, 138, 0.95);
    color: white;
}

.status-badge.in_progress {
    background: rgba(78, 115, 223, 0.95);
    color: white;
}

.category-badge {
    background: rgba(255, 255, 255, 0.95);
    color: #2c3e50;
    padding: 0.3rem 0.6rem;
    border-radius: 0.5rem;
    font-size: 0.8rem;
    font-weight: 600;
}

.course-action-btn {
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.95);
    color: #4e73df;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    text-decoration: none;
    transition: all 0.3s ease;
}

.course-action-btn:hover {
    background: #4e73df;
    color: white;
    transform: scale(1.1);
}

/* Course Progress Ring */
.course-progress-ring {
    position: absolute;
    bottom: -30px;
    right: 20px;
    width: 60px;
    height: 60px;
    background: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 3px solid white;
}

.progress-ring {
    width: 60px;
    height: 60px;
    transform: rotate(-90deg);
}

.progress-ring-circle {
    fill: transparent;
    stroke: #1cc88a;
    stroke-width: 4;
    stroke-dasharray: 163.36;
    stroke-dashoffset: calc(163.36 - (163.36 * var(--progress)) / 100);
    transition: stroke-dashoffset 1s ease;
}

.progress-text {
    position: absolute;
    font-size: 0.8rem;
    font-weight: 700;
    color: #2c3e50;
}

/* Course Price Badge */
.course-price-badge {
    position: absolute;
    top: 15px;
    right: 15px;
    padding: 0.4rem 0.8rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 700;
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.course-price-badge.free {
    background: rgba(28, 200, 138, 0.95);
    color: white;
}

.course-price-badge.paid {
    background: rgba(78, 115, 223, 0.95);
    color: white;
}

/* Course Content */
.course-content {
    padding: 2rem 1.5rem 1.5rem 1.5rem;
    display: flex;
    flex-direction: column;
    flex-grow: 1;
}

.course-meta, .course-rating-level {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.course-category {
    background: #f8f9fc;
    color: #5a5c69;
    padding: 0.3rem 0.6rem;
    border-radius: 0.5rem;
    font-size: 0.8rem;
    font-weight: 600;
}

.enrollment-date {
    color: #858796;
    font-size: 0.8rem;
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.course-rating {
    display: flex;
    align-items: center;
    gap: 0.3rem;
    color: #f6c23e;
    font-weight: 600;
}

.students-count {
    color: #858796;
    font-size: 0.8rem;
}

.course-level {
    background: #e3f2fd;
    color: #1976d2;
    padding: 0.2rem 0.5rem;
    border-radius: 0.3rem;
    font-size: 0.8rem;
    font-weight: 600;
}

.course-title {
    margin: 0 0 1rem 0;
    font-size: 1.2rem;
    font-weight: 600;
    line-height: 1.4;
}

.course-title a {
    color: #2c3e50;
    text-decoration: none;
    transition: color 0.3s ease;
}

.course-title a:hover {
    color: #4e73df;
}

.course-instructor {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #5a5c69;
    margin-bottom: 1rem;
    font-size: 0.9rem;
}

.course-instructor i {
    color: #4e73df;
    font-size: 1.1rem;
}

.course-meta {
    display: flex;
    gap: 1rem;
    margin-bottom: 1rem;
    color: #858796;
    font-size: 0.85rem;
}

.course-meta span {
    display: flex;
    align-items: center;
    gap: 0.3rem;
}

.course-description {
    color: #5a5c69;
    line-height: 1.6;
    margin-bottom: 1.5rem;
    font-size: 0.9rem;
    flex-grow: 1;
}

/* Course Progress Info */
.course-progress-info {
    margin-bottom: 1.5rem;
}

.progress-details {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.5rem;
    font-size: 0.85rem;
}

.modules-completed {
    display: flex;
    align-items: center;
    gap: 0.3rem;
    color: #5a5c69;
}

.progress-percentage {
    color: #1cc88a;
    font-weight: 600;
}

.modern-progress-bar {
    width: 100%;
    height: 6px;
    background: #e3e6f0;
    border-radius: 3px;
    overflow: hidden;
}

.modern-progress-bar .progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #1cc88a 0%, #17a673 100%);
    border-radius: 3px;
    transition: width 1s ease;
    position: relative;
}

/* Course Actions Footer */
.course-actions-footer {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-top: auto;
    padding-top: 1rem;
}

.price-display {
    flex: 1;
}

.price-display .price {
    font-size: 1.3rem;
    font-weight: 700;
    color: #4e73df;
}

/* Empty State */
.modern-empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-illustration i {
    font-size: 5rem;
    color: #e3e6f0;
    margin-bottom: 2rem;
}

.empty-content h4 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 1rem;
}

.empty-content p {
    color: #5a5c69;
    font-size: 1.1rem;
    margin-bottom: 2rem;
    max-width: 500px;
    margin-left: auto;
    margin-right: auto;
}

/* Responsive Design for 5-Card Layout */
@media (max-width: 1600px) {
    .modern-courses-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 1.25rem;
    }
}

@media (max-width: 1200px) {
    .banner-content {
        flex-direction: column;
        gap: 2rem;
        text-align: center;
    }
    
    .banner-actions {
        width: 100%;
        justify-content: center;
    }
    
    .modern-courses-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 1.25rem;
    }
    
    .card-header {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
}

@media (max-width: 992px) {
    .modern-courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
    }
    
    .course-thumbnail-wrapper {
        height: 180px;
    }
}

@media (max-width: 768px) {
    .welcome-text h1 {
        font-size: 2rem;
    }
    
    .modern-courses-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .banner-actions {
        flex-direction: column;
        width: 100%;
    }
    
    .course-filters {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .modern-stats-grid {
        grid-template-columns: 1fr;
    }
    
    .course-thumbnail-wrapper {
        height: 200px;
    }
    

}

@media (max-width: 480px) {
    .dashboard-content {
        padding: 1rem;
    }
    
    .modern-welcome-banner {
        padding: 1.5rem;
    }
    
    .course-content {
        padding: 1.5rem 1rem 1rem 1rem;
    }
    
    .course-actions-footer {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .modern-btn {
        width: 100%;
        justify-content: center;
    }
    
    .card-header, .card-content {
        padding: 1.5rem;
    }
}

/* Udemy-Style Course Cards */
.udemy-style-courses-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 1rem;
    padding: 0;
}

.udemy-course-card {
    background: #ffffff;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    overflow: hidden;
    transition: all 0.3s ease;
    animation: slideUp 0.6s ease forwards;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
}

.udemy-course-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
    border-color: #cbd5e1;
}

/* Course Image */
.course-image-container {
    position: relative;
    width: 100%;
    height: 160px;
    overflow: hidden;
}

.course-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.udemy-course-card:hover .course-image {
    transform: scale(1.05);
}

.course-badge {
    position: absolute;
    top: 12px;
    right: 12px;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.free-badge {
    background: #10b981;
    color: white;
}

.price-badge {
    background: #3b82f6;
    color: white;
}

/* Course Content */
.course-card-content {
    padding: 16px;
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.course-card-title {
    font-size: 16px;
    font-weight: 700;
    color: #1f2937;
    line-height: 1.4;
    margin: 0;
    display: -webkit-box;
    -webkit-line-clamp: 1;
    -webkit-box-orient: vertical;
    overflow: hidden;
    text-overflow: ellipsis;
    cursor: pointer;
}

.course-card-title:hover {
    color: #3b82f6;
}

.course-card-description {
    font-size: 14px;
    color: #6b7280;
    line-height: 1.4;
    margin: 0;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
    text-overflow: ellipsis;
    height: 2.8em; /* Fixed height for 2 lines */
}

.course-instructor-name {
    font-size: 12px;
    color: #9ca3af;
    font-weight: 500;
    margin: 4px 0;
}

/* Rating Section */
.course-rating-info {
    margin: 4px 0;
}

.rating-stars {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 14px;
}

.rating-value {
    color: #f59e0b;
    font-weight: 700;
    font-size: 14px;
}

.stars {
    display: flex;
    gap: 1px;
}

.stars i {
    color: #f59e0b;
    font-size: 12px;
}

.rating-count {
    color: #9ca3af;
    font-size: 12px;
    margin-left: 4px;
}

/* Course Meta Info */
.course-meta-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 12px;
    color: #6b7280;
    margin: 4px 0;
}

.course-duration, .course-level {
    font-weight: 500;
}

/* Footer with Price and Actions */
.course-card-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 12px;
    padding-top: 12px;
    border-top: 1px solid #f1f5f9;
}

.course-price-section {
    flex: 1;
}

.course-price {
    font-size: 18px;
    font-weight: 800;
    color: #1f2937;
}

.course-price.free {
    color: #10b981;
}

/* Action Buttons */
.course-action-buttons {
    display: flex;
    gap: 6px;
    align-items: center;
}

.action-btn {
    width: 36px;
    height: 36px;
    border: none;
    border-radius: 6px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
    font-size: 16px;
}

.view-btn {
    background: #f1f5f9;
    color: #64748b;
}

.view-btn:hover {
    background: #e2e8f0;
    color: #475569;
    transform: translateY(-1px);
}

.cart-btn {
    background: #fef3c7;
    color: #d97706;
}

.cart-btn:hover {
    background: #fde68a;
    color: #92400e;
    transform: translateY(-1px);
}

.enroll-btn {
    background: #3b82f6;
    color: white;
}

.enroll-btn:hover {
    background: #2563eb;
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.4);
}

/* Responsive Design for Udemy Cards */
@media (max-width: 1400px) {
    .udemy-style-courses-grid {
        grid-template-columns: repeat(4, 1fr);
    }
}

@media (max-width: 1100px) {
    .udemy-style-courses-grid {
        grid-template-columns: repeat(3, 1fr);
    }
}

@media (max-width: 768px) {
    .udemy-style-courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.8rem;
    }
    
    .course-card-content {
        padding: 12px;
    }
    
    .course-card-title {
        font-size: 15px;
    }
    
    .course-card-description {
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .udemy-style-courses-grid {
        grid-template-columns: 1fr;
    }
    
    .course-action-buttons {
        gap: 4px;
    }
    
    .action-btn {
        width: 32px;
        height: 32px;
        font-size: 14px;
    }
}

/* Responsive Design for Enrolled Courses */
@media (max-width: 1400px) {
    .enrolled-courses-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 0.9rem;
    }
}

@media (max-width: 1100px) {
    .enrolled-courses-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 0.8rem;
    }
}

@media (max-width: 768px) {
    .enrolled-courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.7rem;
    }
    
    .enrolled-course-header {
        height: 120px; /* Slightly taller for better mobile display */
    }
    
    .enrolled-course-image {
        width: 100%;
        height: 100%;
        object-fit: contain;
        object-position: center;
        display: block;
        background: #f8fafc;
    }
    
    .enrolled-course-content {
        padding: 10px;
    }
    
    .enrolled-course-title {
        font-size: 13px;
        line-height: 1.3;
    }
}

@media (max-width: 480px) {
    .enrolled-courses-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .enrolled-course-card {
        max-height: none; /* Remove height restriction on mobile */
        aspect-ratio: auto; /* Allow natural sizing */
        min-height: 200px; /* Ensure minimum height */
    }
    
    .enrolled-course-header {
        height: 160px; /* Larger image area on mobile */
    }
    
    .enrolled-course-image {
        width: 100%;
        height: 100%;
        object-fit: contain;
        object-position: center;
        display: block;
        background: #f8fafc;
        border-radius: 0;
    }
    
    .course-image-section {
        width: 100%;
        height: 100%;
        position: relative;
        overflow: hidden;
    }
    
    .enrolled-course-content {
        padding: 15px;
        flex: none; /* Don't flex on mobile */
    }
    
    .enrolled-course-title {
        font-size: 16px;
        font-weight: 700;
        margin-bottom: 8px;
    }
    
    .course-instructor-info {
        font-size: 12px;
        margin-bottom: 10px;
    }
    
    .progress-range-section {
        padding: 10px;
        margin-bottom: 10px;
    }
    
    .enrolled-course-actions {
        padding: 0 15px 15px 15px;
    }
    
    .primary-actions {
        justify-content: center;
    }
    
    .action-btn {
        padding: 8px 16px;
        font-size: 14px;
        min-width: 44px;
        min-height: 44px;
        display: flex !important;
        align-items: center;
        justify-content: center;
    }
    
    .secondary-actions {
        display: flex !important;
        gap: 8px !important;
        justify-content: center;
    }
    
    .details-btn {
        display: flex !important;
        align-items: center;
        justify-content: center;
        min-width: 44px;
        min-height: 44px;
    }
}

/* Enrolled Courses - Square Card Design */
.enrolled-courses-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 1rem;
    padding: 0;
}

.enrolled-course-card {
    background: #ffffff;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    overflow: hidden;
    transition: all 0.3s ease;
    animation: slideUp 0.6s ease forwards;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    aspect-ratio: 1 / 1.1; /* Slightly taller than square for better content fit */
    display: flex;
    flex-direction: column;
    max-height: 280px; /* Limit maximum height */
}

.enrolled-course-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
    border-color: #cbd5e1;
}

/* Course Header with Image */
.enrolled-course-header {
    position: relative;
    height: 100px;
    overflow: hidden;
    background: #f1f5f9;
}

.course-image-section {
    position: relative;
    width: 100%;
    height: 100%;
}

.enrolled-course-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
    display: block;
    background: #e2e8f0;
}

.enrolled-course-card:hover .enrolled-course-image {
    transform: scale(1.05);
}

.course-status-badge {
    position: absolute;
    top: 12px;
    right: 12px;
    padding: 6px 10px;
    border-radius: 20px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    display: flex;
    align-items: center;
    gap: 4px;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.course-status-badge.completed {
    background: rgba(16, 185, 129, 0.9);
    color: white;
}

.course-status-badge.in_progress {
    background: rgba(59, 130, 246, 0.9);
    color: white;
}

/* Course Content */
.enrolled-course-content {
    padding: 8px;
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.enrolled-course-title {
    font-size: 12px;
    font-weight: 700;
    color: #1f2937;
    line-height: 1.2;
    margin: 0;
    display: -webkit-box;
    -webkit-line-clamp: 1;
    -webkit-box-orient: vertical;
    overflow: hidden;
    text-overflow: ellipsis;
    cursor: pointer;
}

.enrolled-course-title:hover {
    color: #3b82f6;
}

.course-instructor-info {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 10px;
    color: #6b7280;
}

.course-instructor-info i {
    font-size: 12px;
}

/* Progress Range Section */
.progress-range-section {
    background: #f8fafc;
    padding: 6px;
    border-radius: 4px;
    border: 1px solid #f1f5f9;
}

.progress-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2px;
}

.progress-label {
    font-size: 9px;
    font-weight: 600;
    color: #374151;
}

.progress-value {
    font-size: 10px;
    font-weight: 700;
    color: #3b82f6;
}

.progress-range-container {
    position: relative;
    margin-bottom: 4px;
}

.progress-range {
    width: 100%;
    height: 6px;
    background: #e2e8f0;
    border-radius: 3px;
    outline: none;
    opacity: 0;
    cursor: pointer;
    position: relative;
    z-index: 2;
}

.progress-range-bg {
    position: absolute;
    top: 50%;
    left: 0;
    right: 0;
    height: 6px;
    background: #e2e8f0;
    border-radius: 3px;
    transform: translateY(-50%);
    z-index: 1;
}

.progress-range-fill {
    height: 100%;
    background: linear-gradient(90deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: 3px;
    transition: width 0.3s ease;
    position: relative;
}

.progress-range-fill::after {
    content: '';
    position: absolute;
    right: -6px;
    top: 50%;
    width: 12px;
    height: 12px;
    background: #3b82f6;
    border-radius: 50%;
    transform: translateY(-50%);
    box-shadow: 0 2px 4px rgba(59, 130, 246, 0.4);
}



/* Loading spinner animation */
.bx-spin {
    animation: cartSpin 1s linear infinite;
}

@keyframes cartSpin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Cart notification styles */
.cart-notification {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 9999;
    min-width: 300px;
    max-width: 400px;
}

.cart-notification .alert {
    margin-bottom: 10px;
    border-radius: 10px;
    padding: 12px 16px;
    font-weight: 500;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    animation: slideInRight 0.3s ease-out;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes slideOutRight {
    from {
        transform: translateX(0);
        opacity: 1;
    }
    to {
        transform: translateX(100%);
        opacity: 0;
    }
}

.modules-info {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 11px;
    color: #6b7280;
}

.modules-info i {
    font-size: 12px;
}

/* Enrollment Info */
.enrollment-info {
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 11px;
    color: #9ca3af;
}

.enrollment-info i {
    font-size: 12px;
}

/* Action Buttons */
.enrolled-course-actions {
    padding: 12px 16px;
    border-top: 1px solid #f1f5f9;
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 8px;
}

.primary-actions {
    flex: 1;
}

.secondary-actions {
    display: flex;
    gap: 4px;
}

.enrolled-course-actions .action-btn {
    border: none;
    border-radius: 6px;
    font-size: 12px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 4px;
}

.continue-btn {
    background: #3b82f6;
    color: white;
    padding: 8px 16px;
    width: 100%;
}

.continue-btn:hover {
    background: #2563eb;
    transform: translateY(-1px);
}

.unenroll-btn {
    background: #fef3c7;
    color: #d97706;
    padding: 8px;
    width: 32px;
    height: 32px;
}

.unenroll-btn:hover {
    background: #fde68a;
    color: #92400e;
    transform: translateY(-1px);
}

.delete-btn {
    background: #fee2e2;
    color: #dc2626;
    padding: 8px;
    width: 32px;
    height: 32px;
}

.delete-btn:hover {
    background: #fecaca;
    color: #b91c1c;
    transform: translateY(-1px);
}

.details-btn {
    background: #f1f5f9;
    color: #64748b;
    padding: 8px;
    width: 32px;
    height: 32px;
    text-decoration: none;
}

.details-btn:hover {
    background: #e2e8f0;
    color: #475569;
    transform: translateY(-1px);
    text-decoration: none;
}

/* Responsive Design for Enrolled Cards - Enhanced */
@media (max-width: 1400px) {
    .enrolled-courses-grid {
        grid-template-columns: repeat(4, 1fr);
    }
}

@media (max-width: 1100px) {
    .enrolled-courses-grid {
        grid-template-columns: repeat(3, 1fr);
    }
}
</style>

<!-- Course Filtering and Interaction JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize cart system
    initializeCart();
    
    // Course filtering functionality
    const filterButtons = document.querySelectorAll('.filter-btn');
    const courseCards = document.querySelectorAll('.enrolled-course');
    
    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            filterButtons.forEach(btn => btn.classList.remove('active'));
            // Add active class to clicked button
            this.classList.add('active');
            
            const filter = this.dataset.filter;
            
            courseCards.forEach(card => {
                if (filter === 'all' || card.dataset.status === filter) {
                    card.style.display = 'block';
                    card.style.animation = 'slideUp 0.6s ease forwards';
                } else {
                    card.style.display = 'none';
                }
            });
        });
    });
    
    // Smooth scrolling for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
    
    // Enhanced button hover effects
    document.querySelectorAll('.modern-btn').forEach(btn => {
        btn.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px) scale(1.02)';
        });
        
        btn.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
    
    // Progress ring animation
    document.querySelectorAll('.progress-ring-circle').forEach(circle => {
        const progressRing = circle.closest('.course-progress-ring');
        if (progressRing) {
            const progress = progressRing.style.getPropertyValue('--progress');
            if (progress) {
                setTimeout(() => {
                    circle.style.strokeDashoffset = `calc(163.36 - (163.36 * ${progress}) / 100)`;
                }, 500);
            }
        }
    });
});

// Course Action Functions
function viewCourse(courseId) {
    // Navigate to course detail page
    window.location.href = `course-detail.php?id=${courseId}`;
}



// Update navbar cart count
function updateNavbarCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cart.length;
    
    // Update cart badge in navbar
    const cartBadge = document.querySelector('.cart-count');
    if (cartBadge) {
        cartBadge.textContent = cartCount;
        cartBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    // Update cart icon badge (alternative selector)
    const cartIconBadge = document.querySelector('#cart-count');
    if (cartIconBadge) {
        cartIconBadge.textContent = cartCount;
        cartIconBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    console.log('Cart count updated:', cartCount);
}

// Initialize cart on page load
function initializeCart() {
    // Create notification container if it doesn't exist
    if (!document.querySelector('.cart-notification')) {
        const notificationContainer = document.createElement('div');
        notificationContainer.className = 'cart-notification';
        document.body.appendChild(notificationContainer);
    }
    
    // Update cart count on page load
    updateNavbarCartCount();
    
    // Mark courses already in cart
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    cart.forEach(cartItem => {
        const cartButton = document.querySelector(`[data-course-id="${cartItem.id}"] .cart-btn`);
        if (cartButton) {
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            cartButton.disabled = true;
            cartButton.classList.add('in-cart');
            cartButton.title = 'Added to Cart';
        }
    });
}

// Enhanced notification system
function showNotification(message, type = 'info', duration = 4000) {
    const container = document.querySelector('.cart-notification') || createNotificationContainer();
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type} alert-dismissible`;
    alert.style.opacity = '0';
    
    const alertTypes = {
        'success': 'alert-success',
        'error': 'alert-danger', 
        'warning': 'alert-warning',
        'info': 'alert-info'
    };
    
    alert.className = `alert ${alertTypes[type] || 'alert-info'} alert-dismissible`;
    
    alert.innerHTML = `
        <div class="d-flex align-items-center">
            <div class="flex-grow-1">${message}</div>
            <button type="button" class="btn-close btn-close-white" aria-label="Close"></button>
        </div>
    `;
    
    container.appendChild(alert);
    
    // Animate in
    setTimeout(() => {
        alert.style.opacity = '1';
    }, 10);
    
    // Auto dismiss
    const timeoutId = setTimeout(() => {
        dismissNotification(alert);
    }, duration);
    
    // Manual dismiss
    const closeBtn = alert.querySelector('.btn-close');
    closeBtn.addEventListener('click', () => {
        clearTimeout(timeoutId);
        dismissNotification(alert);
    });
}

function createNotificationContainer() {
    const container = document.createElement('div');
    container.className = 'cart-notification';
    document.body.appendChild(container);
    return container;
}

function dismissNotification(alert) {
    alert.style.animation = 'slideOutRight 0.3s ease-in';
    setTimeout(() => {
        if (alert.parentNode) {
            alert.parentNode.removeChild(alert);
        }
    }, 300);
}

function addToCart(courseId, buttonElement) {
    console.log('addToCart called with courseId:', courseId, 'buttonElement:', buttonElement);
    
    // Get the cart button - either passed as parameter or find by course ID
    let cartButton = buttonElement;
    
    if (!cartButton) {
        // Try to find the button by course ID in the card
        cartButton = document.querySelector(`[data-course-id="${courseId}"] .cart-btn`);
        console.log('Searched for cart button with selector:', `[data-course-id="${courseId}"] .cart-btn`);
    }
    
    if (!cartButton) {
        console.error('Cart button not found for course:', courseId);
        console.log('Available course cards with data-course-id:');
        document.querySelectorAll('[data-course-id]').forEach(card => {
            console.log('- Card ID:', card.getAttribute('data-course-id'));
        });
        showNotification('❌ Unable to add course to cart - button not found', 'error');
        return;
    }
    
    console.log('Cart button found:', cartButton);
    
    // Check if course is already in cart
    const existingCart = JSON.parse(localStorage.getItem('cart') || '[]');
    if (existingCart.find(item => item.id == courseId)) {
        showNotification('📚 Course is already in your cart!', 'warning');
        return;
    }
    
    // Visual feedback - show loading state
    const originalHTML = cartButton.innerHTML;
    const originalBg = cartButton.style.background;
    const originalColor = cartButton.style.color;
    
    cartButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    cartButton.disabled = true;
    cartButton.style.background = '#6c757d';
    
    // Make API call to validate and get course data
    fetch('api/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            course_id: courseId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.success && data.course_data) {
            // Add course to localStorage cart
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            cart.push(data.course_data);
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Success visual feedback
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            
            // Show success notification
            showNotification(`🎉 "${data.course_data.title}" added to cart successfully!`, 'success');
            
            // Update cart count in navbar
            updateNavbarCartCount();
            
            // Change button to "In Cart" state after delay
            setTimeout(() => {
                cartButton.innerHTML = '<i class="bx bx-check"></i>';
                cartButton.style.background = '#10b981';
                cartButton.style.color = 'white';
                cartButton.disabled = true;
                cartButton.classList.add('in-cart');
                cartButton.title = 'Added to Cart';
            }, 1500);
            
        } else {
            throw new Error(data.message || 'Failed to add course to cart');
        }
    })
    .catch(error => {
        console.error('Add to cart error:', error);
        
        // Error visual feedback
        cartButton.innerHTML = '<i class="bx bx-error-circle"></i>';
        cartButton.style.background = '#ef4444';
        cartButton.style.color = 'white';
        
        // Show error notification
        const errorMessage = error.message.includes('Authentication') 
            ? 'Please log in to add courses to cart' 
            : error.message || 'Failed to add course to cart';
        showNotification(`❌ ${errorMessage}`, 'error');
        
        // Reset button after delay
        setTimeout(() => {
            cartButton.innerHTML = originalHTML;
            cartButton.style.background = originalBg;
            cartButton.style.color = originalColor;
            cartButton.disabled = false;
        }, 3000);
    });
}

function enrollCourse(courseId) {
    // Direct enrollment functionality
    const enrollButton = document.querySelector(`[data-course-id="${courseId}"] .enroll-btn`);
    
    // Visual feedback
    const originalHTML = enrollButton.innerHTML;
    enrollButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    enrollButton.disabled = true;
    
    // Check if course is free or paid
    const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
    const priceElement = courseCard.querySelector('.course-price');
    const isFree = priceElement.textContent.toLowerCase().includes('free');
    
    if (isFree) {
        // Direct enrollment for free courses
        fetch('api/enroll-course.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                course_id: courseId,
                action: 'enroll_free'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Success feedback
                enrollButton.innerHTML = '<i class="bx bx-check"></i>';
                enrollButton.style.background = '#10b981';
                
                showNotification('Successfully enrolled in course!', 'success');
                
                // Redirect to course or refresh page after delay
                setTimeout(() => {
                    window.location.href = `course-detail.php?id=${courseId}`;
                }, 1500);
            } else {
                handleEnrollmentError(enrollButton, originalHTML, data.message);
            }
        })
        .catch(error => {
            console.error('Enrollment error:', error);
            handleEnrollmentError(enrollButton, originalHTML, 'An error occurred. Please try again.');
        });
    } else {
        // Redirect to payment page for paid courses
        enrollButton.innerHTML = originalHTML;
        enrollButton.disabled = false;
        window.location.href = `course-payment.php?id=${courseId}`;
    }
}

function handleEnrollmentError(button, originalHTML, message) {
    button.innerHTML = '<i class="bx bx-error-circle"></i>';
    button.style.background = '#ef4444';
    
    showNotification(message || 'Failed to enroll in course', 'error');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.disabled = false;
        button.style.background = '';
    }, 2000);
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `course-notification ${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add notification styles if not already present
    if (!document.querySelector('#notification-styles')) {
        const style = document.createElement('style');
        style.id = 'notification-styles';
        style.textContent = `
            .course-notification {
                position: fixed;
                top: 100px;
                right: 20px;
                background: white;
                padding: 16px 20px;
                border-radius: 8px;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
                border-left: 4px solid;
                z-index: 10000;
                animation: slideInRight 0.3s ease;
                max-width: 400px;
            }
            .course-notification.success { border-left-color: #10b981; }
            .course-notification.error { border-left-color: #ef4444; }
            .course-notification.info { border-left-color: #3b82f6; }
            .notification-content {
                display: flex;
                align-items: center;
                gap: 12px;
                font-size: 14px;
                font-weight: 500;
            }
            .notification-content i {
                font-size: 18px;
                color: inherit;
            }
            .course-notification.success .notification-content i { color: #10b981; }
            .course-notification.error .notification-content i { color: #ef4444; }
            .course-notification.info .notification-content i { color: #3b82f6; }
            @keyframes slideInRight {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOutRight {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto remove after 4 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 4000);
}

// Encrypted URL mapping - populated server-side
const encryptedUrls = {};
<?php foreach ($enrolledCourses as $course): ?>
encryptedUrls[<?= $course['id'] ?>] = '<?= generateSecureUrl('course-detail.php', $course['id']) ?>';
<?php endforeach; ?>

// Enrolled Course Action Functions
function continueCourse(courseId) {
    console.log('Continue course clicked for ID:', courseId);
    console.log('Available encrypted URLs:', encryptedUrls);
    
    // Navigate to course detail page with encrypted URL
    const encryptedUrl = encryptedUrls[courseId];
    console.log('Found encrypted URL:', encryptedUrl);
    
    if (encryptedUrl) {
        console.log('Navigating to:', encryptedUrl);
        window.location.href = encryptedUrl;
    } else {
        console.log('No encrypted URL found, using fallback');
        // Fallback for safety - try the correct page name
        window.location.href = 'courses.php';
    }
}

function unenrollCourse(courseId) {
    // Confirm unenrollment
    if (!confirm('Are you sure you want to unenroll from this course? You will lose access to all course materials and your progress will be saved but not accessible until you re-enroll.')) {
        return;
    }
    
    const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
    if (!courseCard) {
        console.error('Course card not found for ID:', courseId);
        showNotification('Error: Course not found', 'error');
        return;
    }
    
    const unenrollButton = courseCard.querySelector('.unenroll-btn');
    if (!unenrollButton) {
        console.error('Unenroll button not found for course:', courseId);
        showNotification('Error: Button not found', 'error');
        return;
    }
    
    // Visual feedback
    const originalHTML = unenrollButton.innerHTML;
    unenrollButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    unenrollButton.disabled = true;
    
    // Simulate API call
    fetch('api/unenroll-course.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            course_id: courseId,
            action: 'unenroll'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Success feedback
            unenrollButton.innerHTML = '<i class="bx bx-check"></i>';
            unenrollButton.style.background = '#10b981';
            unenrollButton.style.color = 'white';
            
            showNotification('Successfully unenrolled from course!', 'success');
            
            // Remove course card with animation
            setTimeout(() => {
                courseCard.style.animation = 'slideOut 0.5s ease forwards';
                setTimeout(() => {
                    courseCard.remove();
                    
                    // Check if no enrolled courses left
                    const remainingCourses = document.querySelectorAll('.enrolled-course-card');
                    if (remainingCourses.length === 0) {
                        location.reload(); // Reload to show empty state
                    }
                }, 500);
            }, 1000);
        } else {
            handleUnenrollmentError(unenrollButton, originalHTML, data.message);
        }
    })
    .catch(error => {
        console.error('Unenrollment error:', error);
        handleUnenrollmentError(unenrollButton, originalHTML, 'An error occurred. Please try again.');
    });
}

function deleteCourse(courseId) {
    // Confirm deletion
    if (!confirm('Are you sure you want to delete this course from your account? This will permanently remove all your progress and course data. This action cannot be undone.')) {
        return;
    }
    
    const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
    if (!courseCard) {
        console.error('Course card not found for ID:', courseId);
        showNotification('Error: Course not found', 'error');
        return;
    }
    
    const deleteButton = courseCard.querySelector('.delete-btn');
    if (!deleteButton) {
        console.error('Delete button not found for course:', courseId);
        showNotification('Error: Button not found', 'error');
        return;
    }
    
    // Visual feedback
    const originalHTML = deleteButton.innerHTML;
    deleteButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    deleteButton.disabled = true;
    
    // Simulate API call
    fetch('api/delete-course-enrollment.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            course_id: courseId,
            action: 'delete'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Success feedback
            deleteButton.innerHTML = '<i class="bx bx-check"></i>';
            deleteButton.style.background = '#10b981';
            deleteButton.style.color = 'white';
            
            showNotification('Course deleted successfully!', 'success');
            
            // Remove course card with animation
            setTimeout(() => {
                courseCard.style.animation = 'slideOut 0.5s ease forwards';
                setTimeout(() => {
                    courseCard.remove();
                    
                    // Check if no enrolled courses left
                    const remainingCourses = document.querySelectorAll('.enrolled-course-card');
                    if (remainingCourses.length === 0) {
                        location.reload(); // Reload to show empty state
                    }
                }, 500);
            }, 1000);
        } else {
            handleDeletionError(deleteButton, originalHTML, data.message);
        }
    })
    .catch(error => {
        console.error('Deletion error:', error);
        handleDeletionError(deleteButton, originalHTML, 'An error occurred. Please try again.');
    });
}

function handleUnenrollmentError(button, originalHTML, message) {
    button.innerHTML = '<i class="bx bx-error-circle"></i>';
    button.style.background = '#ef4444';
    button.style.color = 'white';
    
    showNotification(message || 'Failed to unenroll from course', 'error');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.disabled = false;
        button.style.background = '';
        button.style.color = '';
    }, 2000);
}

function handleDeletionError(button, originalHTML, message) {
    button.innerHTML = '<i class="bx bx-error-circle"></i>';
    button.style.background = '#ef4444';
    button.style.color = 'white';
    
    showNotification(message || 'Failed to delete course', 'error');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.disabled = false;
        button.style.background = '';
        button.style.color = '';
    }, 2000);
}

// Add slide out animation for card removal
document.addEventListener('DOMContentLoaded', function() {
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideOut {
            0% {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
            100% {
                opacity: 0;
                transform: translateY(-20px) scale(0.9);
            }
        }
    `;
    document.head.appendChild(style);
});
</script>

<?php include_once 'components/footer.php'; ?>
