<?php
/**
 * API Endpoint: Get Random Courses for Dashboard Cycling
 * Returns randomly selected courses for the banner cycling feature
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Authentication check
session_start();
require_once '../../auth/functions.php';

try {
    // Require student authentication
    requireRole('student');
    
    // Get number of courses to fetch (default 6)
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 6;
    $limit = min($limit, 12); // Max 12 courses
    
    require_once '../../config/db_connect.php';
    
    if (!isset($conn)) {
        throw new Exception('Database connection not available');
    }
    
    // Fetch random published courses using working query structure
    $query = "
        SELECT 
            c.course_id as id,
            c.title,
            c.description,
            c.thumbnail,
            c.price,
            u.first_name,
            u.last_name,
            cat.category_name as category,
            (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students
        FROM courses c
        JOIN users u ON c.teacher_id = u.user_id
        JOIN categories cat ON c.category_id = cat.category_id
        WHERE c.is_published = 1
        ORDER BY RAND()
        LIMIT ?
    ";
    
    $stmt = $conn->prepare($query);
    if (!$stmt) {
        throw new Exception('Query preparation failed');
    }
    
    $stmt->bind_param("i", $limit);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $courses = [];
    while ($row = $result->fetch_assoc()) {
        // Format course data using available columns
        $instructor = trim($row['first_name'] . ' ' . $row['last_name']);
        $description = !empty($row['description']) ? substr($row['description'], 0, 100) . '...' : 'Course description';
        
        $courses[] = [
            'id' => (int)$row['id'],
            'title' => htmlspecialchars($row['title']),
            'description' => htmlspecialchars($description),
            'instructor' => htmlspecialchars($instructor ?: 'Instructor'),
            'thumbnail' => !empty($row['thumbnail']) 
                ? htmlspecialchars($row['thumbnail']) 
                : './assets/img/courses/default.jpg',
            'price' => number_format((float)($row['price'] ?? 0), 2),
            'level' => 'Intermediate', // Default level since column doesn't exist
            'category' => htmlspecialchars($row['category'] ?? 'General'),
            'rating' => 4.5, // Default rating since ratings table doesn't exist yet
            'students' => (int)$row['total_students'],
            'duration' => '8 hours' // Default duration since column doesn't exist
        ];
    }
    
    // Success response
    echo json_encode([
        'success' => true,
        'courses' => $courses,
        'total' => count($courses)
    ]);
    
} catch (Exception $e) {
    // Error response
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'courses' => []
    ]);
}
?>
