<?php
/**
 * Student Navbar Component - Airbnb Style Design
 * Clean, modern navbar matching Airbnb's design exactly
 */

// Get user information
$userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
$userInitials = 'ST';
$currentUserName = 'Student';
$currentUserRole = 'Student';

if ($userId && isset($conn) && $conn) {
    try {
        $stmt = $conn->prepare("SELECT first_name, last_name, email, profile_picture FROM users WHERE user_id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($user = $result->fetch_assoc()) {
            $currentUserName = $user['first_name'] . ' ' . $user['last_name'];
            $userInitials = strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1));
        }
    } catch (Exception $e) {
        // Use defaults
    }
}
?>

<!-- Airbnb-Style Navbar -->
<nav class="airbnb-navbar">
    <div class="navbar-container">
        <!-- Left: Menu Toggle -->
        <div class="navbar-left">
            <button class="menu-toggle-btn" onclick="toggleSidebar()">
                <i class="bx bx-menu"></i>
            </button>
        </div>

        <!-- Center: Search Bar -->
        <div class="navbar-center">
            <!-- Boxicons -->
            <link href='https://unpkg.com/boxicons@2.0.9/css/boxicons.min.css' rel='stylesheet'>
            
            <!-- Google Fonts -->
            <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
            
            <!-- Dedicated Search Dropdown CSS -->
            <link rel="stylesheet" href="/assets/css/search-dropdown.css">
            <div class="search-container" id="searchContainer">
                <div class="search-segment">
                    <label>What</label>
                    <input type="text" placeholder="Search courses, topics" id="search-what">
                </div>
                <div class="search-divider"></div>
                <div class="search-segment">
                    <label>Level</label>
                    <select id="search-level">
                        <option value="">All levels</option>
                        <option value="beginner">Beginner</option>
                        <option value="intermediate">Intermediate</option>
                        <option value="advanced">Advanced</option>
                    </select>
                </div>
                <div class="search-button-container">
                    <button class="search-button" type="button" onclick="performSearch()">
                        <i class="bx bx-search"></i>
                    </button>
                </div>
                
                <!-- Dynamic Search Dropdown -->
                <div class="navbar-search-dropdown" id="searchDropdown">
                    <div class="navbar-search-results" id="searchResults">
                        <!-- Dynamic results will be inserted here -->
                    </div>
                </div>
            </div>
        </div>

        <!-- Right: User Menu -->
        <div class="navbar-right">
            <a href="#" class="become-instructor">Become an Instructor</a>
            
            <div class="cart-container">
                <button class="cart-btn" onclick="toggleCart()">
                    <i class="bx bx-shopping-bag"></i>
                    <span class="cart-count" id="cartCount" style="display: none;"></span>
                </button>
            </div>
            
            <div class="user-menu-container">
                <button class="user-menu-toggle" onclick="toggleUserMenu()">
                    <i class="bx bx-menu"></i>
                    <div class="user-avatar">
                        <span><?php echo $userInitials; ?></span>
                    </div>
                </button>
                
                <div class="user-dropdown" id="userDropdown">
                    <div class="dropdown-header">
                        <div class="user-info">
                            <div class="user-avatar-large">
                                <span><?php echo $userInitials; ?></span>
                            </div>
                            <div class="user-details">
                                <div class="user-name"><?php echo htmlspecialchars($currentUserName); ?></div>
                                <div class="user-role"><?php echo $currentUserRole; ?></div>
                            </div>
                        </div>
                    </div>
                    <div class="dropdown-divider"></div>
                    <div class="dropdown-menu">
                        <a href="dashboard.php" class="dropdown-item">
                            <i class="bx bx-home"></i>
                            <span>Dashboard</span>
                        </a>
                        <a href="courses.php" class="dropdown-item">
                            <i class="bx bx-book"></i>
                            <span>My Courses</span>
                        </a>
                        <a href="assignments.php" class="dropdown-item">
                            <i class="bx bx-task"></i>
                            <span>Assignments</span>
                        </a>
                        <a href="#" class="dropdown-item">
                            <i class="bx bx-user"></i>
                            <span>Profile</span>
                        </a>
                        <a href="#" class="dropdown-item">
                            <i class="bx bx-cog"></i>
                            <span>Settings</span>
                        </a>
                        <div class="dropdown-divider"></div>
                        <a href="../logout.php" class="dropdown-item logout">
                            <i class="bx bx-log-out"></i>
                            <span>Sign out</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</nav>

<style>
/* Airbnb-Style Navbar CSS */
.airbnb-navbar {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    height: 80px;
    background: #fff;
    border-bottom: 1px solid #e0e0e0;
    z-index: 1000;
    box-shadow: 0 1px 8px rgba(0, 0, 0, 0.08);
}

.navbar-container {
    max-width: 1200px;
    margin: 0 auto;
    height: 100%;
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 0 24px;
}

/* Menu Toggle Section */
.navbar-left {
    flex: 0 0 auto;
}

.menu-toggle-btn {
    background: none;
    border: none;
    font-size: 24px;
    color: #222;
    cursor: pointer;
    padding: 8px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: background-color 0.2s ease;
}

.menu-toggle-btn:hover {
    background-color: #f7f7f7;
}

.menu-toggle-btn i {
    font-size: 24px;
}

/* Search Section */
.navbar-center {
    flex: 1;
    display: flex;
    justify-content: center;
    margin: 0 40px;
}

.search-container {
    display: flex;
    align-items: center;
    background: #f7f7f7;
    border: 1px solid #ddd;
    border-radius: 40px;
    height: 48px;
    max-width: 400px;
    width: 100%;
    overflow: hidden;
    transition: box-shadow 0.2s ease;
}

.search-container:hover {
    box-shadow: 0 2px 16px rgba(0, 0, 0, 0.12);
}

.search-segment {
    flex: 1;
    padding: 0 16px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    cursor: pointer;
    transition: background-color 0.2s ease;
}

.search-segment:hover {
    background-color: #ebebeb;
}

.search-segment label {
    font-size: 12px;
    font-weight: 600;
    color: #222;
    margin-bottom: 2px;
    text-transform: uppercase;
}

.search-segment input,
.search-segment select {
    border: none;
    background: transparent;
    font-size: 14px;
    color: #717171;
    outline: none;
    width: 100%;
}

.search-segment input::placeholder {
    color: #717171;
}

.search-divider {
    width: 1px;
    height: 24px;
    background: #ddd;
}

.search-button-container {
    padding: 0 8px;
}

.search-button {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: #f59e0b;
    border: none;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: background-color 0.2s ease;
}

.search-button:hover {
    background: #d97706;
}

.search-button i {
    color: white;
    font-size: 16px;
}

/* Right Section */
.navbar-right {
    flex: 0 0 auto;
    display: flex;
    align-items: center;
    gap: 16px;
}

.become-instructor {
    text-decoration: none;
    color: #222;
    font-weight: 600;
    font-size: 14px;
    padding: 12px 16px;
    border-radius: 22px;
    transition: background-color 0.2s ease;
}

.become-instructor:hover {
    background-color: #f7f7f7;
    color: #222;
}

/* Cart Button */
.cart-container {
    position: relative;
}

.cart-btn {
    background: none;
    border: none;
    padding: 8px;
    border-radius: 8px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    transition: background-color 0.2s ease;
}

.cart-btn:hover {
    background-color: #f7f7f7;
}

.cart-btn i {
    font-size: 20px;
    color: #222;
}

.cart-count {
    position: absolute;
    top: -2px;
    right: -2px;
    background: #f59e0b;
    color: white;
    font-size: 11px;
    font-weight: 600;
    padding: 2px 6px;
    border-radius: 10px;
    min-width: 16px;
    height: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    line-height: 1;
}

.cart-count:empty,
.cart-count[data-count="0"] {
    display: none;
}

/* User Menu */
.user-menu-container {
    position: relative;
}

.user-menu-toggle {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 6px 6px 6px 12px;
    border: 1px solid #ddd;
    border-radius: 21px;
    background: white;
    cursor: pointer;
    transition: box-shadow 0.2s ease;
}

.user-menu-toggle:hover {
    box-shadow: 0 2px 16px rgba(0, 0, 0, 0.12);
}

.user-menu-toggle i {
    font-size: 16px;
    color: #717171;
}

.user-avatar {
    width: 30px;
    height: 30px;
    border-radius: 50%;
    background: #3b82f6;
    display: flex;
    align-items: center;
    justify-content: center;
}

.user-avatar span {
    color: white;
    font-size: 12px;
    font-weight: 600;
}

/* Dropdown Menu */
.user-dropdown {
    position: absolute;
    top: 100%;
    right: 0;
    margin-top: 8px;
    width: 240px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 8px 28px rgba(0, 0, 0, 0.28);
    display: none;
    z-index: 1001;
}

.user-dropdown.show {
    display: block;
}

.dropdown-header {
    padding: 16px;
}

.user-info {
    display: flex;
    align-items: center;
    gap: 12px;
}

.user-avatar-large {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #3b82f6;
    display: flex;
    align-items: center;
    justify-content: center;
}

.user-avatar-large span {
    color: white;
    font-size: 16px;
    font-weight: 600;
}

.user-name {
    font-weight: 600;
    color: #222;
    font-size: 14px;
}

.user-role {
    color: #717171;
    font-size: 12px;
}

.dropdown-divider {
    height: 1px;
    background: #ebebeb;
    margin: 0 16px;
}

.dropdown-menu {
    padding: 8px 0;
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 16px;
    text-decoration: none;
    color: #222;
    font-size: 14px;
    transition: background-color 0.2s ease;
}

.dropdown-item:hover {
    background-color: #f7f7f7;
    color: #222;
}

.dropdown-item i {
    font-size: 16px;
    color: #717171;
    width: 16px;
}

.dropdown-item.logout {
    color: #ff385c;
}

.dropdown-item.logout i {
    color: #ff385c;
}

/* Responsive Design */
@media (max-width: 768px) {
    .navbar-container {
        padding: 0 16px;
    }
    
    .navbar-center {
        margin: 0 20px;
    }
    
    .search-container {
        max-width: 280px;
    }
    
    .search-segment {
        padding: 0 12px;
    }
    
    .become-instructor {
        display: none;
    }
}

@media (max-width: 576px) {
    .search-container {
        max-width: 200px;
    }
    
    .search-segment label {
        font-size: 10px;
    }
    
    .search-segment input,
    .search-segment select {
        font-size: 12px;
    }
}
</style>

<script>
// Sidebar toggle functionality
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        if (window.innerWidth <= 768) {
            // Mobile: toggle between hidden and fully open
            sidebar.classList.toggle('open');
        } else {
            // Desktop: toggle between full and collapsed (icons only)
            sidebar.classList.toggle('close');
        }
    }
}

// Cart toggle functionality
function toggleCart() {
    // Redirect to cart page or show cart dropdown
    window.location.href = 'cart.php';
}

// Update cart count
function updateCartCount() {
    // Get cart count from localStorage or make AJAX call
    const cartItems = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cartItems.length;
    const cartCountElement = document.getElementById('cartCount');
    
    if (cartCountElement) {
        cartCountElement.textContent = cartCount;
        cartCountElement.setAttribute('data-count', cartCount);
        
        // Show/hide count badge
        if (cartCount > 0) {
            cartCountElement.style.display = 'flex';
        } else {
            cartCountElement.style.display = 'none';
        }
    }
}

// User menu toggle functionality
function toggleUserMenu() {
    const dropdown = document.getElementById('userDropdown');
    dropdown.classList.toggle('show');
}

// Close dropdown when clicking outside
document.addEventListener('click', function(event) {
    const userMenu = document.querySelector('.user-menu-container');
    const dropdown = document.getElementById('userDropdown');
    
    if (!userMenu.contains(event.target)) {
        dropdown.classList.remove('show');
    }
});

// Dynamic Search functionality
let searchTimeout;
let currentSearchRequest;

function performDynamicSearch() {
    const what = document.getElementById('search-what').value.trim();
    const level = document.getElementById('search-level').value;
    const dropdown = document.getElementById('searchDropdown');
    const results = document.getElementById('searchResults');
    
    // Clear previous timeout
    clearTimeout(searchTimeout);
    
    // Cancel previous request
    if (currentSearchRequest) {
        currentSearchRequest.abort();
    }
    
    // If no search query, hide dropdown
    if (!what && !level) {
        dropdown.classList.remove('show');
        return;
    }
    
    // Show loading state
    dropdown.classList.add('show');
    results.innerHTML = `
        <div class="search-loading">
            <i class="bx bx-loader-alt"></i>
            <div>Searching courses...</div>
        </div>
    `;
    
    // Debounce search
    searchTimeout = setTimeout(() => {
        const params = new URLSearchParams();
        if (what) params.append('q', what);
        if (level) params.append('level', level);
        params.append('limit', '6');
        
        currentSearchRequest = new AbortController();
        
        fetch(`api/navbar_search.php?${params.toString()}`, {
            signal: currentSearchRequest.signal
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data.courses.length > 0) {
                displaySearchResults(data.data.courses, what, level);
            } else {
                showNoResults();
            }
        })
        .catch(error => {
            if (error.name !== 'AbortError') {
                console.error('Search error:', error);
                showSearchError();
            }
        });
    }, 300);
}

function displaySearchResults(courses, query, level) {
    const results = document.getElementById('searchResults');
    
    let html = '';
    
    courses.forEach(course => {
        const courseInitials = course.title ? course.title.substring(0, 2).toUpperCase() : 'CO';
        
        html += `
            <a href="${course.url}" class="navbar-search-item">
                <div class="navbar-course-thumb">${courseInitials}</div>
                <div class="navbar-course-info">
                    <div class="navbar-course-title">${course.title}</div>
                    <div class="navbar-course-meta">
                        <span class="navbar-course-price">${course.price}</span>
                        <span class="navbar-meta-separator">•</span>
                        <span class="navbar-course-category">${course.category}</span>
                        <span class="navbar-meta-separator">•</span>
                        <span class="navbar-course-teacher">${course.teacher_name}</span>
                    </div>
                </div>
            </a>
        `;
    });
    
    // Add "View All Results" button if there are results
    if (courses.length > 0) {
        html += `
            <div class="navbar-search-footer">
                <button class="navbar-view-all-btn" onclick="viewAllResults('${query}', '')">
                    View All Results
                </button>
            </div>
        `;
    }
    
    results.innerHTML = html;
}

function showSearchLoading() {
    const results = document.getElementById('searchResults');
    results.innerHTML = `
        <div class="navbar-search-loading">
            <i class="bx bx-loader-alt"></i>
            <div>Searching courses...</div>
        </div>
    `;
}

function showNoResults() {
    const results = document.getElementById('searchResults');
    results.innerHTML = `
        <div class="navbar-search-empty">
            <i class="bx bx-search-alt"></i>
            <div>No courses found</div>
            <div>Try adjusting your search terms</div>
        </div>
    `;
}

function showSearchError() {
    const results = document.getElementById('searchResults');
    results.innerHTML = `
        <div class="search-no-results">
            <i class="bx bx-error"></i>
            <div>Search temporarily unavailable</div>
            <div style="font-size: 12px; margin-top: 4px;">Please try again later</div>
        </div>
    `;
}

function viewAllResults(query, level) {
    let url = 'courses.php?';
    if (query) url += 'search=' + encodeURIComponent(query) + '&';
    if (level) url += 'level=' + encodeURIComponent(level) + '&';
    window.location.href = url.slice(0, -1);
}

function performSearch() {
    // Keep the old function for the search button - redirect to full results
    const what = document.getElementById('search-what').value.trim();
    const level = document.getElementById('search-level').value;
    viewAllResults(what, level);
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('search-what');
    const searchLevel = document.getElementById('search-level');
    const searchContainer = document.getElementById('searchContainer');
    const dropdown = document.getElementById('searchDropdown');
    
    if (searchInput) {
        // Real-time search as user types
        searchInput.addEventListener('input', performDynamicSearch);
        
        // Show dropdown when input is focused
        searchInput.addEventListener('focus', function() {
            if (this.value.trim()) {
                performDynamicSearch();
            }
        });
        
        // Enter key search
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch();
            }
        });
    }
    
    if (searchLevel) {
        searchLevel.addEventListener('change', performDynamicSearch);
    }
    
    // Hide dropdown when clicking outside
    document.addEventListener('click', function(event) {
        if (!searchContainer.contains(event.target)) {
            dropdown.classList.remove('show');
        }
    });
    
    // Initialize cart count on page load
    updateCartCount();
});
</script>
