<?php
/**
 * Study Reminder Email API - Proven Working Version
 * Handles creating study reminders and sending email notifications
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Include database connection
    require_once '../../config/db_connect.php';
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['action']) || $input['action'] !== 'send_study_reminder') {
        throw new Exception('Invalid request');
    }
    
    $reminder = $input['reminder'] ?? [];
    $course_id = (int)($input['course_id'] ?? 0);
    $student_id = (int)($input['student_id'] ?? 0);
    
    if (!$reminder || !$course_id || !$student_id) {
        throw new Exception('Missing required parameters');
    }
    
    // Get student and course information using correct database structure
    // Note: In this system, student_id in enrollments = user_id, no separate students table needed
    $stmt = $pdo->prepare("
        SELECT 
            u.user_id,
            u.first_name, 
            u.last_name, 
            u.email, 
            c.title as course_title,
            e.student_id,
            e.enrollment_id
        FROM users u 
        JOIN enrollments e ON e.user_id = u.user_id
        JOIN courses c ON c.course_id = e.course_id 
        WHERE u.user_id = ? AND c.course_id = ? AND u.user_role = 'student'
        LIMIT 1
    ");
    $stmt->execute([$student_id, $course_id]);
    $student = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$student) {
        throw new Exception("Student not found or not enrolled in this course. User ID: $student_id, Course ID: $course_id");
    }
    
    // Save reminder to database using the student_id from enrollments (which equals user_id)
    $insert_stmt = $pdo->prepare("
        INSERT INTO study_reminders 
        (student_id, course_id, reminder_date, reminder_time, reminder_type, description, email_sent, reminder_sent, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, 0, 0, NOW())
    ");
    
    $insert_result = $insert_stmt->execute([
        $student['student_id'], // Use the student_id from the enrollment record (= user_id)
        $course_id, 
        $reminder['date'], 
        $reminder['time'], 
        $reminder['type'], 
        $reminder['description']
    ]);
    
    if (!$insert_result) {
        throw new Exception('Failed to save reminder to database');
    }
    
    $reminder_id = $pdo->lastInsertId();
    
    // Use centralized email service function - CLEAN & WORKING
    $email_sent = false;
    try {
        // Include email service
        require_once '../../config/fast_email_service.php';
        
        // Prepare student name
        $student_name = $student['first_name'] . ' ' . $student['last_name'];
        
        // Send email using the dedicated study reminder function from fast_email_service.php
        $email_result = sendStudyReminderEmail(
            $student['email'],
            $student_name,
            $student['course_title'],
            $reminder['date'],
            $reminder['time'],
            $reminder['type'],
            $reminder['description']
        );
        
        // Handle array return from email service properly
        if (is_array($email_result)) {
            $email_sent = $email_result['success'] ?? false;
        } else {
            $email_sent = (bool)$email_result;
        }
        
        // Update email_sent flag if successful
        if ($email_sent) {
            $update_stmt = $pdo->prepare("UPDATE study_reminders SET email_sent = 1 WHERE reminder_id = ?");
            $update_stmt->execute([$reminder_id]);
        }
        
    } catch (Exception $email_error) {
        // Email failed but reminder is still saved - this is okay
        error_log("Email sending failed: " . $email_error->getMessage());
    }
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'Study reminder created successfully' . ($email_sent ? ' and email sent' : ''),
        'reminder_id' => $reminder_id,
        'scheduled_datetime' => $reminder['date'] . ' ' . $reminder['time'],
        'email_sent' => $email_sent
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'error_details' => $e->getFile() . ':' . $e->getLine()
    ]);
}
?>

