<?php
/**
 * Simple Study Reminder Email API - Minimal Working Version
 * This version focuses on core functionality without complex templates
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Include database connection
    require_once '../../config/db_connect.php';
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['action']) || $input['action'] !== 'send_study_reminder') {
        throw new Exception('Invalid request');
    }
    
    $reminder = $input['reminder'] ?? [];
    $course_id = (int)($input['course_id'] ?? 0);
    $student_id = (int)($input['student_id'] ?? 0);
    
    if (!$reminder || !$course_id || !$student_id) {
        throw new Exception('Missing required parameters');
    }
    
    // Get student and course information
    $stmt = $pdo->prepare("
        SELECT 
            s.student_id, 
            s.first_name, 
            s.last_name, 
            u.email, 
            c.title as course_title
        FROM students s 
        JOIN users u ON u.user_id = s.user_id
        JOIN enrollments e ON e.student_id = s.student_id
        JOIN courses c ON c.course_id = e.course_id 
        WHERE s.student_id = ? AND c.course_id = ?
        LIMIT 1
    ");
    $stmt->execute([$student_id, $course_id]);
    $student = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$student) {
        throw new Exception('Student not found or not enrolled in this course');
    }
    
    // Save reminder to database
    $insert_stmt = $pdo->prepare("
        INSERT INTO study_reminders 
        (student_id, course_id, reminder_date, reminder_time, reminder_type, description, email_sent, reminder_sent, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, 0, 0, NOW())
    ");
    
    $insert_result = $insert_stmt->execute([
        $student_id, 
        $course_id, 
        $reminder['date'], 
        $reminder['time'], 
        $reminder['type'], 
        $reminder['description']
    ]);
    
    if (!$insert_result) {
        throw new Exception('Failed to save reminder to database');
    }
    
    $reminder_id = $pdo->lastInsertId();
    
    // Simple email sending (without complex templates)
    $email_sent = false;
    try {
        // Include email service
        require_once '../../config/fast_email_service.php';
        
        $subject = "Study Reminder Set - " . $student['course_title'];
        $message = "Hi " . $student['first_name'] . ",\n\n";
        $message .= "Your study reminder has been set:\n";
        $message .= "Course: " . $student['course_title'] . "\n";
        $message .= "Date: " . $reminder['date'] . "\n";
        $message .= "Time: " . $reminder['time'] . "\n";
        $message .= "Description: " . $reminder['description'] . "\n\n";
        $message .= "We'll send you another reminder at the scheduled time.\n\n";
        $message .= "Best regards,\nPanadite Academy";
        
        // Try to send email using the email service
        if (function_exists('sendEmailNow')) {
            $email_sent = sendEmailNow($student['email'], $subject, $message);
        } else if (function_exists('queueEmail')) {
            $email_sent = queueEmail($student['email'], $subject, $message, 'study_reminder', 'high');
        }
        
        // Update email_sent flag if successful
        if ($email_sent) {
            $update_stmt = $pdo->prepare("UPDATE study_reminders SET email_sent = 1 WHERE reminder_id = ?");
            $update_stmt->execute([$reminder_id]);
        }
        
    } catch (Exception $email_error) {
        // Email failed but reminder is still saved - this is okay
        error_log("Email sending failed: " . $email_error->getMessage());
    }
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'Study reminder created successfully' . ($email_sent ? ' and email sent' : ''),
        'reminder_id' => $reminder_id,
        'scheduled_datetime' => $reminder['date'] . ' ' . $reminder['time'],
        'email_sent' => $email_sent
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'error_details' => $e->getFile() . ':' . $e->getLine()
    ]);
}
?>
