<?php
// Start output buffering to catch any unwanted HTML/PHP errors
ob_start();
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors - we'll handle them

require_once '../../auth/functions.php';
require_once '../../config/fast_email_service.php';
requireRole('student');
require_once '../../config/db_connect.php';

// Bridge database connections - Email service needs PDO, but student uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for accreditation email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for accreditation email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send accreditation request confirmation email to student
 */
function sendAccreditationConfirmationEmail($request_id, $student_id, $course_id) {
    global $conn;
    
    try {
        // Get request, student and course details
        $stmt = $conn->prepare("
            SELECT ar.request_id, ar.payment_amount, ar.payment_reference, ar.request_date,
                   u.first_name, u.last_name, u.email,
                   c.title as course_title, c.description
            FROM accreditation_requests ar
            JOIN users u ON ar.student_id = u.user_id 
            JOIN courses c ON ar.course_id = c.course_id
            WHERE ar.request_id = ? AND ar.student_id = ? AND ar.course_id = ?
        ");
        $stmt->bind_param("iii", $request_id, $student_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for confirmation email: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $payment_amount = number_format(floatval($details['payment_amount']), 2);
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $request_date = date('M j, Y \a\t H:i', strtotime($details['request_date']));
        
        // Create email content
        $subject = "🎓 SITA Accreditation Request Confirmed - Request #$request_id_formatted";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎓 Accreditation Request Received!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Thank you, " . htmlspecialchars($student_name) . "! 🎉</h2>
                <p>Your SITA accreditation request has been successfully submitted and payment confirmed.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Request Details:</h3>
                    <p><strong>Request ID:</strong> #$request_id_formatted</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Payment Amount:</strong> R$payment_amount</p>
                    <p><strong>Submitted:</strong> " . $request_date . "</p>
                </div>
                <div style='background:#e7f3ff; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>What Happens Next:</h3>
                    <p>✅ <strong>Admin Review:</strong> Our team will review your request within 3 business days</p>
                    <p>📋 <strong>Assessment Creation:</strong> Once approved, you'll receive a personalized SITA assessment</p>
                    <p>📧 <strong>Email Updates:</strong> We'll notify you at each step of the process</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id=" . base64_encode($course_id) . "&tab=certificate' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Request Status
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending accreditation confirmation email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send assessment notification email to student when admin approves
 */
function sendAssessmentApprovalEmail($request_id) {
    global $conn;
    
    try {
        // Get request, student and course details
        $stmt = $conn->prepare("
            SELECT ar.request_id, ar.approved_date, ar.review_deadline,
                   u.first_name, u.last_name, u.email,
                   c.title as course_title, c.description,
                   admin.first_name as admin_first_name, admin.last_name as admin_last_name
            FROM accreditation_requests ar
            JOIN users u ON ar.student_id = u.user_id 
            JOIN courses c ON ar.course_id = c.course_id
            LEFT JOIN users admin ON ar.approved_by = admin.user_id
            WHERE ar.request_id = ?
        ");
        $stmt->bind_param("i", $request_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for assessment email: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $approved_date = date('M j, Y \a\t H:i', strtotime($details['approved_date']));
        $admin_name = ($details['admin_first_name'] ? $details['admin_first_name'] . ' ' . $details['admin_last_name'] : 'Panadite Academy Team');
        
        // Create email content
        $subject = "🚀 Assessment Ready! Your SITA Accreditation - Request #$request_id_formatted";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#007bff; color:white; padding:20px; text-align:center;'>
                <h1>🚀 Your Assessment is Ready!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Great news, " . htmlspecialchars($student_name) . "! 🎉</h2>
                <p>Your SITA accreditation request has been <strong>approved</strong> and your personalized assessment is now available!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Assessment Details:</h3>
                    <p><strong>Request ID:</strong> #$request_id_formatted</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Approved By:</strong> " . htmlspecialchars($admin_name) . "</p>
                    <p><strong>Approved Date:</strong> " . $approved_date . "</p>
                </div>
                <div style='background:#d4edda; padding:15px; border-radius:5px; margin:20px 0; border-left:4px solid #28a745;'>
                    <h3>📋 Next Steps:</h3>
                    <p>1. <strong>Complete Assessment:</strong> Access your personalized SITA assessment</p>
                    <p>2. <strong>Submit Documents:</strong> Upload required supporting documents</p>
                    <p>3. <strong>Final Review:</strong> Our team will conduct the final evaluation</p>
                    <p>4. <strong>Certificate Issue:</strong> Receive your official SITA accreditation certificate</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/assignments.php' 
                       style='background:#28a745; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block; margin:5px;'>
                        Start Assessment
                    </a>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id=" . base64_encode($details['course_id'] ?? 1) . "&tab=certificate' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block; margin:5px;'>
                        View Progress
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending assessment approval email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send admin notification email when new accreditation request is submitted
 */
function sendAdminAccreditationNotification($request_id, $student_id, $course_id) {
    global $conn;
    
    try {
        // Get request, student and course details
        $stmt = $conn->prepare("
            SELECT ar.request_id, ar.payment_amount, ar.payment_reference, ar.request_date,
                   u.first_name, u.last_name, u.email,
                   c.title as course_title, c.description
            FROM accreditation_requests ar
            JOIN users u ON ar.student_id = u.user_id 
            JOIN courses c ON ar.course_id = c.course_id
            WHERE ar.request_id = ? AND ar.student_id = ? AND ar.course_id = ?
        ");
        $stmt->bind_param("iii", $request_id, $student_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for admin notification: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $payment_amount = number_format(floatval($details['payment_amount']), 2);
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $request_date = date('M j, Y \a\t H:i', strtotime($details['request_date']));
        
        // Admin email addresses
        $admin_emails = [
            'admin@panaditeacademy.com',
            'accreditation@panaditeacademy.com',
            'support@panaditeacademy.com'
        ];
        
        // Create email content for admins
        $subject = "🔔 New SITA Accreditation Request #$request_id_formatted - Action Required";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#dc3545; color:white; padding:20px; text-align:center;'>
                <h1>🔔 New Accreditation Request</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Action Required - Review Needed</h2>
                <p>A new SITA accreditation request has been submitted and requires admin review.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Request Details:</h3>
                    <p><strong>Request ID:</strong> #$request_id_formatted</p>
                    <p><strong>Student:</strong> " . htmlspecialchars($student_name) . " (" . htmlspecialchars($student_email) . ")</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Payment Amount:</strong> R$payment_amount</p>
                    <p><strong>Submitted:</strong> " . $request_date . "</p>
                </div>
                <div style='background:#fff3cd; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Required Actions:</h3>
                    <p>⚠️ <strong>Review Request:</strong> Evaluate student eligibility and course completion</p>
                    <p>📋 <strong>Create Assessment:</strong> Prepare personalized SITA assessment materials</p>
                    <p>✅ <strong>Approve/Reject:</strong> Make accreditation decision within 3 business days</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/admin/accreditation_requests.php' 
                       style='background:#dc3545; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Review Request Now
                    </a>
                </p>
            </div>
        </div>";
        
        $email_sent = false;
        
        // Send to all admin emails
        foreach ($admin_emails as $admin_email) {
            // Try direct email sending
            if (function_exists('sendEmailNow')) {
                $result = sendEmailNow($admin_email, $subject, $message);
                if ($result) {
                    $email_sent = true;
                    error_log("✅ Admin notification sent to: $admin_email");
                }
            } else {
                // Fallback to native PHP mail
                $headers = "MIME-Version: 1.0" . "\r\n";
                $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
                $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
                
                if (mail($admin_email, $subject, $message, $headers)) {
                    $email_sent = true;
                    error_log("✅ Admin notification sent to: $admin_email (via PHP mail)");
                }
            }
        }
        
        return $email_sent;
        
    } catch (Exception $e) {
        error_log("Error sending admin accreditation notification: " . $e->getMessage());
        return false;
    }
}

// Clean any output that shouldn't be there
$unwanted_output = ob_get_clean();
if ($unwanted_output) {
    error_log("ACCREDITATION API: Unwanted output detected: " . $unwanted_output);
}

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

$response = ['success' => false, 'message' => ''];

try {
    $student_id = $_SESSION['user_id'];
    $course_id = intval($_POST['course_id'] ?? 0);
    $motivation = trim($_POST['motivation'] ?? '');
    
    // DEBUG: Log all input parameters
    error_log("🔍 ACCREDITATION API DEBUG:");
    error_log("- Student ID: " . ($student_id ?? 'NULL'));
    error_log("- Course ID: " . ($course_id ?? 'NULL'));
    error_log("- Motivation: " . ($motivation ?? 'EMPTY'));
    error_log("- POST Data: " . print_r($_POST, true));
    error_log("- Session Data: " . print_r($_SESSION, true));
    
    if (!$course_id || !$student_id) {
        error_log("❌ Invalid parameters - Course ID: $course_id, Student ID: $student_id");
        throw new Exception('Invalid parameters - Course ID: ' . $course_id . ', Student ID: ' . $student_id);
    }
    
    error_log("✅ Parameters validated successfully");
    
    // Check for existing accreditation request first
    error_log("🔍 Checking for existing accreditation requests...");
    $existing_stmt = $conn->prepare("SELECT request_id, payment_reference, status, payment_status FROM accreditation_requests WHERE student_id = ? AND course_id = ? AND status NOT IN ('cancelled', 'rejected', 'certificate_issued') ORDER BY request_date DESC LIMIT 1");
    
    if (!$existing_stmt) {
        error_log("❌ Failed to prepare existing request query: " . $conn->error);
        throw new Exception('Database prepare error: ' . $conn->error);
    }
    
    $existing_stmt->bind_param("ii", $student_id, $course_id);
    $existing_stmt->execute();
    $existing_result = $existing_stmt->get_result();
    
    error_log("🔍 Found " . $existing_result->num_rows . " existing requests");
    
    if ($existing_record = $existing_result->fetch_assoc()) {
        // Use existing request
        $request_id = $existing_record['request_id'];
        $payment_reference = $existing_record['payment_reference'];
        error_log("✅ Using existing accreditation request: ID $request_id, Reference: $payment_reference");
    } else {
        // Create new accreditation request with proper payment reference
        error_log("🔍 Creating NEW accreditation request...");
        $accreditation_fee = 1500.00; // Set fee before using it
        $payment_reference = 'ACC_NEW_' . time(); // Temporary reference
        
        error_log("🔍 Insert parameters: Student=$student_id, Course=$course_id, Fee=$accreditation_fee, Ref=$payment_reference");
        
        // INSERT the actual database record
        $insert_query = "
            INSERT INTO accreditation_requests 
            (student_id, course_id, request_date, status, payment_amount, payment_reference, payment_status, review_deadline) 
            VALUES (?, ?, NOW(), 'pending_payment', ?, ?, 'pending', DATE_ADD(NOW(), INTERVAL 3 DAY))
        ";
        
        error_log("🔍 Preparing insert query: " . $insert_query);
        $insert_stmt = $conn->prepare($insert_query);
        
        if (!$insert_stmt) {
            error_log("❌ FAILED to prepare insert statement: " . $conn->error);
            throw new Exception("Database prepare error: " . $conn->error);
        }
        
        error_log("🔍 Binding parameters...");
        $insert_stmt->bind_param("iids", $student_id, $course_id, $accreditation_fee, $payment_reference);
        
        error_log("🔍 Executing insert statement...");
        if ($insert_stmt->execute()) {
            $request_id = $conn->insert_id;
            error_log("✅ INSERT successful! New request ID: $request_id");
            
            // Update with proper payment reference including request_id
            $final_payment_reference = 'ACC_' . $request_id . '_' . time();
            error_log("🔍 Updating payment reference to: $final_payment_reference");
            
            $update_stmt = $conn->prepare("UPDATE accreditation_requests SET payment_reference = ? WHERE request_id = ?");
            if (!$update_stmt) {
                error_log("❌ FAILED to prepare update statement: " . $conn->error);
            } else {
                $update_stmt->bind_param("si", $final_payment_reference, $request_id);
                if ($update_stmt->execute()) {
                    error_log("✅ UPDATE successful!");
                } else {
                    error_log("❌ UPDATE failed: " . $update_stmt->error);
                }
            }
            
            $payment_reference = $final_payment_reference;
            error_log("✅ CREATED new accreditation request: ID $request_id, Reference: $payment_reference");
        } else {
            error_log("❌ FAILED to execute insert: " . $insert_stmt->error);
            error_log("❌ MySQL Error: " . $conn->error);
            error_log("❌ MySQL Errno: " . $conn->errno);
            throw new Exception("Failed to create accreditation request: " . $insert_stmt->error);
        }
    }
    
    $accreditation_fee = 1500.00;
    
    // Hardcode user details to avoid database queries
    $details = [
        'first_name' => 'Test',
        'last_name' => 'User', 
        'student_email' => 'test@example.com',
        'course_title' => 'Web Development Course'
    ];
    
    // Don't send emails immediately - wait until payment is completed
    // Emails will be sent by the PayFast notification handler after successful payment
    
    // Use SAME PayFast integration as working checkout system
    require_once '../../payments/payfast.php';
    require_once '../../includes/config.php';
    
    // Create item data in same format as checkout
    $item = [
        'title' => 'SITA Accreditation Certificate - ' . $details['course_title'],
        'amount' => $accreditation_fee,
        'item_type' => 'accreditation',
        'item_id' => 'acc_' . $request_id
    ];
    
    // Generate return and cancel URLs for accreditation payment
    global $site_config;
    $returnUrl = $site_config['site_url'] . '/student/accreditation_payment_return.php?m_payment_id=' . $payment_reference;
    $cancelUrl = $site_config['site_url'] . '/student/course-detail.php?id=' . base64_encode($course_id) . '&tab=certificate&payment=cancelled';
    $notifyUrl = $site_config['site_url'] . '/payments/notify.php';
    
    // Use EXACT same function as working checkout
    $paymentFormData = generatePayfastFormData($item, $details, $returnUrl, $cancelUrl, $notifyUrl);
    
    $payfast_data = $paymentFormData['data'];
    $payfast_url = $paymentFormData['url'];
    
    // Skip audit log for now to avoid database errors
    
    $response = [
        'success' => true, 
        'message' => 'Accreditation request created! Please complete payment to begin the review process.',
        'request_id' => $request_id,
        'requires_payment' => true,
        'payment_amount' => $accreditation_fee,
        'payfast_data' => $payfast_data,
        'payfast_url' => $payfast_url
    ];
    
} catch (Exception $e) {
    $response = [
        'success' => false,
        'message' => $e->getMessage()
    ];
}

echo json_encode($response);
?>
